/*
  seek.c: extract rows at selected offsets from an indexed
  NoSQL table.

  Original code Copyright (c) 1998 Maurizio Sartori

  Changes for NoSQL Copyright (c) 2004 Carlo Strozzi

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  $Id: seek.c,v 1.2 2004/08/08 21:09:06 carlo Exp $

*/

#include <stdio.h>
#include <ctype.h>
#include <unistd.h>

#ifndef HELPDIR
#define HELPDIR "/usr/local/nosql/help"
#endif
#ifndef HELPFILE
#define HELPFILE (HELPDIR "/seek.txt")
#endif

/****************************************************************************/
/*                          Function prototypes                             */
/****************************************************************************/
void LineOutput( FILE * file, const char * pszStr );
void LineSkip( FILE * file, const char * pszStr );
void LineInfo( FILE * file, const int fBlank, const char * pszStr );

/****************************************************************************/
/*                              LineOutput ()                               */
/****************************************************************************/

void LineOutput( FILE * file, const char * pszStr )
{
   int chChar;

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      fputc( chChar, stdout );
      chChar = fgetc( file );
   }
   fputc( chChar, stdout );
}

/****************************************************************************/
/*                              LineInfo ()                                 */
/****************************************************************************/

void LineInfo( FILE * file, const int fBlank, const char * pszStr )
{
   int chChar;
   long iStart = 0;
   long iEnd = 0;
   long iSize = 0;
   char cBlank = 'y';

   iStart = ftell( file );

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      /* Check whether it is a blank row */

      if ( chChar != ' ' && chChar != '\t' ) cBlank = 'n';

      chChar = fgetc( file );
   }
   iEnd = ftell( file );

   /* Exclude the trailing newline/EOF from line size */
   iEnd -=2;
   iSize = iEnd - iStart;

   if ( fBlank )
   {
     if ( cBlank == 'y' )
     {
       printf( "\001%ld\t\001%ld\t\001%ld\t\001%c\n", \
				iStart, iEnd, iSize, cBlank );
     }
   }
   else printf( "\001%ld\t\001%ld\t\001%ld\t\001%c\n", \
				iStart, iEnd, iSize, cBlank );
}

/****************************************************************************/
/*                               LineSkip ()                                */
/****************************************************************************/
void LineSkip( FILE * file, const char * pszStr )
{
   int chChar;

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      chChar = fgetc( file );
   }
}

/****************************************************************************/
/*                                  main ()                                 */
/****************************************************************************/
int main( int argc, char **argv )
{
   int    fRaw = 0;
   int    fHdr = 1;
   int    fInfo = 0;
   int    fBlank = 0;
   int	  some_input = 0;
   FILE * fileDb = NULL;
   int    iInx;
   int    chChar;
   long   lOffset;
   int    iValue[256];
   int    iBase;

   for ( iInx = 0; iInx < 256; iInx++ )
   {
      iValue[iInx] = 0;
   }

   iValue['0'] = 1;
   iValue['1'] = 2;
   iValue['2'] = 3;
   iValue['3'] = 4;
   iValue['4'] = 5;
   iValue['5'] = 6;
   iValue['6'] = 7;
   iValue['7'] = 8;
   iValue['8'] = 9;
   iValue['9'] = 10;
   iBase = 10;

   for ( iInx = 1; iInx < argc; iInx++ )
   {
      if ( !strcmp( argv[iInx], "-R" )
        || !strcmp( argv[iInx], "--raw-input" ) )
      {
         fRaw = 1;
      }
      else if ( !strcmp ( argv[iInx], "-X" )
            || !strcmp ( argv[iInx], "--hex" ) )
      {
         iValue['a'] = iValue['A'] = 11;
         iValue['b'] = iValue['B'] = 11;
         iValue['c'] = iValue['C'] = 11;
         iValue['d'] = iValue['D'] = 11;
         iValue['e'] = iValue['E'] = 11;
         iValue['f'] = iValue['F'] = 11;
         iBase = 16;
      }
      else if ( !strcmp( argv[iInx], "-N" )
        || !strcmp( argv[iInx], "--no-header" ) )
      {
         fHdr = 0;
      }
      else if ( !strcmp( argv[iInx], "-r" )
        || !strcmp( argv[iInx], "--info" ) )
      {
         fInfo = 1;
      }
      else if ( !strcmp( argv[iInx], "-b" )
        || !strcmp( argv[iInx], "--blank" ) )
      {
         fBlank = 1;
         fInfo = 1;
      }
      else if (!strcmp(argv[iInx], "-h") ||
	       !strcmp(argv[iInx], "--help")) {
	      execlp("grep","grep","-v","^#",HELPFILE,(char *) 0);
	      perror("grep");
	      exit(0);
      }
      else if ( fileDb )
      {
         fprintf( stderr, "seek: invalid number of parameters\n" );
         return 1;
      }
      else
      {
         fileDb = fopen( argv[iInx], "rt" );
         if ( !fileDb )
         {
            fprintf( stderr, "seek: cannot open \"%s\"\n", argv[iInx] );
            return 1;
         }
      }
   }

   if ( !fileDb )
   {
      fprintf( stderr, "seek: no database specified\n" );
      return 1;
   }

   /*************************************************************************/
   /* Print table header                                                    */
   /*************************************************************************/
   if ( fHdr && !fInfo )
   {
     LineOutput( fileDb, "Invalid Database Header\n" );
   }

   /*************************************************************************/
   /* Skip header if not Raw Mode                                          */
   /*************************************************************************/
   if ( !fRaw )
   {
      LineSkip( stdin, "seek: invalid input table header\n" );
   }

   /* Print the output table header if '--info' was requested. */

   if ( fHdr && fInfo )
     printf("Start\tEnd\tSize\tBlank\n-----\t---\t----\t-----\n");

   /*************************************************************************/
   /* Process offsets                                                       */
   /*************************************************************************/
   lOffset = 0;
   chChar = fgetc( stdin );
   while ( chChar != EOF )
   {
      if (iValue [chChar])
      {
         lOffset *= iBase;
         lOffset += iValue[chChar] - 1;
      }
      else if ( isspace( chChar ) )
      {
         if ( lOffset )
         {
            if ( fseek( fileDb, lOffset, SEEK_SET ) )
            {
               fprintf( stderr, "seek: cannot seek\n" );
               exit(1);
            }

	    some_input = 1;

            if ( !fInfo )
              LineOutput( fileDb, "seek: invalid file line\n" );
            else
              LineInfo( fileDb, fBlank, "seek: invalid file line\n" );
         }

         lOffset = 0;
      }
      else
      {
         fprintf( stderr, "seek: invalid character \"%c\"\n", chChar );
         exit(1);
      }

      chChar = fgetc( stdin );
   }

   if (!some_input) exit(1);
   else exit(0);
}

/* End of program */
