/*
 *   NVRAM WakeUp
 *   Copyright (C) 2001-2004, Sergei Haller.
 *   Copyright (C) 2002 Bernhard "Bero" Rosenkraenzer <bero@arklinux.org>
 *
 *   $Id: nvram-wakeup-mb.c,v 1.218 2004/07/21 12:01:02 bistr-o-math Exp $
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#define CVSREV_nvram_wakeup_mb_c \
     "$Id: nvram-wakeup-mb.c,v 1.218 2004/07/21 12:01:02 bistr-o-math Exp $"

#include <stdio.h>
#include <string.h>

#include "nvram-wakeup.h"

/*
 *  If you can't find your motherboard here, try at the MB tracker
 *    http://sourceforge.net/tracker/index.php?group_id=35022&atid=412959
 *  or the latest CVS revision of this file
 *    http://cvs.sourceforge.net/cgi-bin/viewcvs.cgi/nvram-wakeup/nvram-wakeup/nvram-wakeup-mb.c?rev=HEAD
 *  or
 *    read README.mb.
 */

typedef void (*info_writer)(struct biosinfo *b);
     
struct mainboard {
     info_writer infowriter;

     char *iw_name;     /* the function name of the infowriter */

     char *vendor;
     char *type;
     char *version;

     char *biosvendor;
     char *biosversion;
     char *biosrelease;
};


/*
 *  look into nvram-wakeup.conf(5) for the documentation of the options
 */
 
/* 
 *  By far the most common way of storing the timing information is having
 *  it in subsequent bytes in the day-hour-min-sec order. #define this
 *  here to save code later on
 */
 
#define TIMEREGS(b, base) b->addr_day  = base;  \
                          b->addr_hour = base+1;\
                          b->addr_min  = base+2;\
                          b->addr_sec  = base+3

/* 
 *  Virtually every board keeps the low and high bytes of the checksum in
 *  subsequent bytes - save some more code
 */
 
#define CHECKSUM(b, base) b->addr_chk_h = base;\
                          b->addr_chk_l = base+1

#define   BCD(b) b->nr_mon      = 5;\
                 b->nr_day      = 6;\
                 b->nr_hour     = 6;\
                 b->nr_min      = 7;\
                 b->nr_sec      = 7;\
                 b->nr_rtc_mon  = 5;\
                 b->nr_rtc_day  = 6;\
                 b->nr_wdays    = 7;\
                 b->bcd         = ON

#define NOBCD(b) b->nr_mon      = 4;\
                 b->nr_day      = 5;\
                 b->nr_hour     = 5;\
                 b->nr_min      = 6;\
                 b->nr_sec      = 6;\
                 b->nr_rtc_day  = 6;\
                 b->nr_rtc_mon  = 5;\
                 b->nr_wdays    = 7

void msi_ms_6351(struct biosinfo *b) 
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_stat  = 3;
}

void m6vcg_p6bat_8kta3_common(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     b->need_reboot = ON_STAT;
     b->addr_stat   = 0x47;
     b->shift_stat  = 5;
     b->addr_hour   = 0x41;
     b->addr_min    = 0x40;
     b->addr_sec    = 0x3F;
}

void chaintech_6aja4(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x43;
     b->need_reboot = ON_ANY_CHANGE;
}

void ecs_p6bat(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x4F;
}

void dfi_ak74_ec(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x37;
}

void shuttle_mvp3(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x4F;
     b->need_reboot = ON_ANY_CHANGE;
}

void tekram_p5mvp3(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x4F;
     b->rtc_time    = ON;
}

void bio_m6vcg(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x51;
}

void msi_ms_6323(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x51;
     b->rtc_time    = ON;
}

void ecs_k7vza(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x51;
     b->addr_sec    = 0; /* no seconds */
}

void msi_ms_6368(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x51;
     b->need_reboot = ON_ANY_CHANGE;
}

void epx_8kta3(struct biosinfo *b)
{
     m6vcg_p6bat_8kta3_common(b);
     b->addr_day    = 0x53;
}

void abit_bp6(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     TIMEREGS(b, 0x47);
     b->addr_stat   = 0x46;
}

void abit_bh6(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     TIMEREGS(b, 0x47);
     b->addr_stat   = 0x46;
     b->need_reboot = ON_ANY_CHANGE;
}

void aopen_ax4pe_max(struct biosinfo *b)
{
     TIMEREGS(b, 0x47);
     b->addr_stat   = 0x46;
     b->need_reboot = ON_ANY_CHANGE;
}

void ga_6bxc_qdi_common(struct biosinfo *b)
{
     CHECKSUM(b, 0x6F);
     TIMEREGS(b, 0x47);
     b->addr_stat   = 0x46;
}

void gig_ga_6bxc(struct biosinfo *b)
{
     ga_6bxc_qdi_common(b);
     b->need_reboot = ON_ANY_CHANGE;
}

void qdi_brlx1(struct biosinfo *b)
{
     ga_6bxc_qdi_common(b);
     b->need_reboot = ON_STAT;
}

void msi_ms_6160(struct biosinfo *b)
{
     ga_6bxc_qdi_common(b);
     b->need_reboot = ON_ANY_CHANGE;
     b->rtc_time    = ON;
}

void epox_4g4a(struct biosinfo *b)
{
     ga_6bxc_qdi_common(b);
     b->addr_sec = 0;
}

void asus_tusl2(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     b->need_reboot = OFF;
     b->rtc_time    = ON;
     b->reset_day   = ON;
     b->addr_stat   = 0x56;
     b->rtc_day     = 0x7F;
     b->shift_stat  = 4;
}

void asus_cubx(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     b->need_reboot = ON_ANY_CHANGE;
     b->rtc_time    = ON;
     b->reset_day   = ON;
     b->addr_stat   = 0x56;
     b->rtc_day     = 0x7F;
     b->shift_stat  = 4;
}

void asus_cusi_fx(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     b->addr_stat       = 0x56;
     b->shift_stat      = 4;

     b->addr_day        = 0x57;

     b->rtc_time        = ON;

     b->rtc_day         = 0x70;
     b->reset_day       = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void msi_ms_6120(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x59;
     b->shift_stat  = 5;
}

void msi_ms_6593(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x59;
     b->shift_stat  = 3;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void msi_ms_6380e(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x59;
     b->shift_stat  = 5;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void msi_ms_6590(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x59;
     b->shift_stat  = 6;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void asus_cuv4xe(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     b->need_reboot = OFF; /* ??? */
     b->rtc_time    = ON;
     b->reset_day   = ON;
     b->addr_stat   = 0x56;
     b->rtc_day     = 0x6F;
     b->shift_stat  = 4;
}

void asus_a7s_vm(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF; /* ??? */
     b->rtc_time    = ON;
     b->addr_stat   = 0x4B;
     b->addr_day    = 0x4C;
     b->shift_stat  = 7;
}

void asus_p4s533(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->rtc_time    = ON;
     b->addr_stat   = 0x4A;
     b->addr_day    = 0x4A;
     b->shift_stat  = 1;
     b->shift_day   = 2;
}

void asus_p4s533_x(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->rtc_time    = ON;
     b->addr_stat   = 0x4B;
     b->addr_day    = 0x4B;
     b->shift_stat  = 1;
     b->shift_day   = 2;
}

void asus_p4s8l(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF; /* ??? */
     b->rtc_time    = ON;
     b->addr_stat   = 0x49;
     b->addr_day    = 0x49;
     b->shift_stat  = 1;
     b->shift_day   = 2;
}

void asus_a7v600(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->rtc_time    = ON;
     b->addr_stat   = 0x4F;
     b->addr_day    = 0x4F;
     b->shift_stat  = 1;
     b->shift_day   = 2;
}

void asus_p2l97s(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     b->need_reboot = OFF;
     b->rtc_time    = ON;
     b->addr_stat   = 0x40;
     b->rtc_day     = 0x7F; /* Needs kernel patch */
     b->shift_stat  = 1;
}

void asus_tx97_xe(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     b->need_reboot = OFF;
     b->rtc_time    = ON;
     b->addr_stat   = 0x41;
     b->rtc_day     = 0x7F; /* Needs kernel patch */
     b->shift_stat  = 2;
}

void pb_bora_pro(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = OFF; /* ??? */
     b->addr_stat   = 0x5A;
     b->shift_day   = 1;
}

void fic_va_503p(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     TIMEREGS(b, 0x49);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x48;
     b->addr_mon    = 0x4D;
     b->shift_stat  = 7;
     b->shift_mon   = 4;
}

void gig_ga_7zx_common(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x57);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x57;
     b->rtc_time    = ON;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void gig_ga_7zx(struct biosinfo *b)
{
     gig_ga_7zx_common(b);
     b->shift_stat  = 1;
     b->shift_day   = 2;
}

void gig_ga_7zxe(struct biosinfo *b)
{
     gig_ga_7zx_common(b);
     b->shift_day   = 1;
}

void gig_ga_7vtxh(struct biosinfo *b)
{
     gig_ga_7zx_common(b);
     TIMEREGS(b, 0x59);
     b->shift_stat  = 7;
}


void msi_ms_6570(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x46;
     b->shift_stat  = 3;
     b->addr_day    = 0x48;
     b->addr_hour   = 0x49;
     b->addr_min    = 0x4A;
}

void abit_kg7(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x46;
     b->shift_stat  = 3;
     b->addr_day    = 0x47;
     b->addr_hour   = 0x48;
     b->addr_min    = 0x49;
}

void shttl_fn41_common(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     b->addr_stat   = 0x46;
     b->shift_stat  = 3;
     b->addr_hour   = 0x48;
     b->addr_min    = 0x49;
     b->addr_sec    = 0x4A;
}

void shttl_fn41(struct biosinfo *b)
{
     shttl_fn41_common(b);
     b->need_reboot = ON_ANY_CHANGE;
}

void shttl_fn41sp(struct biosinfo *b)
{
     shttl_fn41_common(b);
     b->need_reboot = ON_STAT;
}

void chain_7nif2(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x46;
     b->shift_stat  = 3;
     TIMEREGS(b, 0x48);
}

void msi_ms_6380(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x59);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x58;
     b->shift_stat  = 3;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void msi_ms_6728(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x59);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x58;
     b->shift_stat  = 3;
     b->rtc_time    = ON;
     b->rtc_day     = 0x7F;
     b->reset_day   = ON;
}

void msi_ms_6712_1(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x58;
     b->shift_stat  = 7;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void msi_ms_6712_2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x5A;
     b->shift_day   = 1;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void msi_ms_6712_3(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5A);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x5A;
     b->shift_stat  = 2;
     b->shift_day   = 3;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void msi_ms_6580(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x59);
     b->need_reboot = OFF;  /* ??? */
     b->addr_stat   = 0x59;
     b->shift_stat  = 1;
     b->shift_day   = 2;
     b->rtc_day     = 0x7F;
     b->reset_day   = ON;
     b->rtc_time    = ON;
     
     b->upper_method = INTEL;
}

void asrock_k7vt2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x59);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x59;
     b->shift_stat  = 1;
     b->shift_day   = 2;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void asrock_k7vt4(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->need_reboot = ON_ANY_CHANGE; /* ??? */
     b->addr_stat   = 0x55;
     b->shift_stat  = 5;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void tosh_eq_3300m(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_stat  = 6;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void asrock_k7vm2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x59);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x59;
     b->shift_day   = 1;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void asrock_k7vm2_new(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x57);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x56;
     b->shift_stat  = 6;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void asrock_k7s8xe(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x57);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_stat  = 6;
}

void asrock_k7s8x(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x54);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x51;
     b->shift_stat  = 6;
}

void asrock_k7s8xe2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x50;
     b->shift_stat  = 7;
}

void asrock_k7s8x2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x53);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x50;
     b->shift_stat  = 6;
     b->shift_day   = 3;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void asrock_k7s8x3(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = OFF;
     b->addr_stat   = 0x50;
     b->shift_stat  = 6;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void gig_7vrx(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x57);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x56;
     b->shift_stat  = 7;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void asrock_k7vm4(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->need_reboot = ON_ANY_CHANGE; /* ??? */
     b->addr_stat   = 0x56;
     b->shift_stat  = 2;
     b->shift_day   = 3;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void asus_sp98_x(struct biosinfo *b)
{
     CHECKSUM(b, 0x6C);
     b->need_reboot = ON_STAT|ON_WDAYS;
     b->rtc_time    = ON;
     b->addr_stat   = 0x3F;
     b->addr_wdays  = 0x41;
     b->addr_hour   = 0x42;
     b->addr_min    = 0x43;
     b->addr_sec    = 0x44;
     b->shift_wdays = 1;
}

void soyo_sy5ssm(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x34);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->addr_mon    = 0x32;
     b->addr_day    = 0x33;
     b->shift_mon   = 4;
}

void msi_ms_6524(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
}

void gig_ga_8simlh(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = OFF; /* ??? */
     b->addr_stat   = 0x32;
}

void sis_730(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
}

void msi_ms_6760(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->shift_stat  = 3;

     b->rtc_time    = ON;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day  = 8;
}

void ecs_k7s5a(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x51);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x50;
     b->shift_stat  = 7;
     b->shift_day   = 3;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;

     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void ecs_p4s5a(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x51);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x50;
     b->shift_stat  = 7;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_time    = ON;

     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void ecs_k7som_p(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x51);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x4F;
     b->shift_stat  = 7;
     b->shift_day   = 3;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;

     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void ecs_k7s5al(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x50;
     b->shift_stat  = 7;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;

     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void ecs_k7s5a_1(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x51;
     b->shift_stat  = 6;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;

     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void ecs_k7s5a_2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x51);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x50;
     b->shift_stat  = 6;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;

     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void ecs_k7s5au(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x51;
     b->shift_stat  = 7;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_time    = ON;

     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void ecs_p4s8ag(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);

     b->need_reboot = OFF;  /* ??? */

     b->addr_stat   = 0x50;
     b->shift_stat  = 7;

     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;

     b->rtc_time    = ON;
}

void gigabyte_5aa(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x51;
     b->shift_stat  = 7;
}

void msi_ms_5169(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x51;
     b->shift_stat  = 5;
}

void gig_ga_7vkml(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x57);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x56;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->shift_stat  = 3;
}

void gig_ga_k7vm4(struct biosinfo *b)
{
     gig_ga_7vkml(b);
     b->rtc_time    = ON;
}

void gig_ga_7vkml_2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->shift_day   = 2;
     b->addr_stat   = 0x54;
     b->shift_stat  = 7;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void asus_k7m(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x57);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x56;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->shift_stat  = 4;
}

void gig_ga_7zxr30(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x57);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x56;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->shift_stat  = 5;
}

void fic_sd_11(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->need_reboot = ON_STAT;
     b->addr_stat   = 0x55;
     b->shift_stat  = 4;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void shuttle_mv43vn(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_stat  = 7;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void gig_ga_5ax(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     BCD(b);
     b->addr_chk_h2 = 0x2B; /* Checksum for the status byte */
     b->addr_chk_l2 = 0x2C; /* Checksum for the status byte */
     b->need_reboot = ON_ANY_CHANGE;
     b->rtc_time    = ON;
     b->addr_stat   = 0x2A;
     b->addr_day    = 0x44;
     b->addr_hour   = 0x45;
     b->addr_min    = 0x46;
     b->shift_stat  = 7;
}

void mp_avt_121(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x4F);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x4E;
     b->shift_stat  = 2;
}

void mp_avt_122(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x53);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x53;
     b->shift_stat  = 2;
     b->shift_day   = 3;
}

void msi_6535_20b(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x53;
     b->shift_stat  = 7;
     b->shift_day   = 3;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
}

void asrock_g_pro(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_day   = 1;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
}

void asrock_g_pro_new(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x59);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x58;
     b->shift_stat  = 7;
     b->shift_day   = 3;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
}

void msi_ms_6398e(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_day   = 1;
}

void asus_p4t533(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF; /* ??? */
     b->addr_stat   = 0x4C;
     b->addr_day    = 0x4C;
     b->shift_stat  = 5;
     b->shift_day   = 2;
     b->rtc_time    = ON;
     b->day_hack    = 2;
}

void asus_p4b533(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF; /* ??? */

     b->addr_stat   = 0x4C;
     b->shift_stat  = 5;

     b->addr_day    = 0x4D;
     b->day_hack    = 2;

     b->addr_hour   = 0x68;
     b->addr_min    = 0x69;
     b->addr_sec    = 0x6A;
     
     BCD(b);
     b->nr_day      = 5;
/*   DAY is not (!) in BCD. This does not matter, since a special 
     (day_hack) handling required anyway. just nr_day  must be correct! */
}

void asus_p4xp_x(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF; /* ??? */

     b->addr_stat   = 0x4D;
     b->shift_stat  = 3;

     b->addr_day    = 0x4E;
     b->day_hack    = 4;

     b->addr_hour   = 0x68;
     b->addr_min    = 0x69;
     b->addr_sec    = 0x6A;
     
     BCD(b);
     b->nr_day      = 5;
/*   DAY is not (!) in BCD. This does not matter, since a special 
     (day_hack) handling required anyway. just nr_day  must be correct! */
}

void asus_a7v8x(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);

     b->need_reboot = OFF;

     b->addr_stat   = 0x4D;
     b->shift_stat  = 5;

     b->addr_day    = 0x4E;
     b->day_hack    = 2;

     b->rtc_time    = ON;
}

void asus_p4t533_c(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->addr_stat   = 0x4C;
     b->shift_stat  = 5;
     b->rtc_time    = ON;
     b->addr_day    = 0x4D;
     b->day_hack    = 2;
}

void asus_p4s8x(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF; /* ??? */
     b->addr_stat   = 0x4C;
     b->shift_stat  = 3;
     b->rtc_time    = ON;
     b->addr_day    = 0x4D;
     b->day_hack    = 4;
}

void asus_p4pe(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     TIMEREGS(b, 0x67);
     b->need_reboot = OFF;
     b->addr_stat   = 0x4E;
     b->shift_stat  = 3;
     b->addr_day    = 0x4F;
     b->day_hack    = 4;
     BCD(b);
     b->nr_day      = 5;
/*   DAY is not (!) in BCD. This does not matter, since a special 
     (day_hack) handling required anyway. just nr_day  must be correct! */
}

void ecs_k7som(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x52);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x52;
     b->shift_day   = 1;
}

void via_p4ma(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     TIMEREGS(b, 0x48);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x47;
     b->shift_stat  = 3;
}

void dfi_ak70(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     TIMEREGS(b, 0x48);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x46;
     b->shift_stat  = 2;
}

void via_epia_m(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     TIMEREGS(b, 0x4C);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x4B;
     b->shift_stat  = 3;
}

void msi_694tpro(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x58);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x57;
     b->shift_stat  = 5;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void asrock_m266a(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x58);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x57;
     b->shift_stat  = 3;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void gig_ga_7vtx(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_stat  = 2;
     b->shift_day   = 3;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void gig_ga_7ixe4(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x54;
     b->shift_stat  = 6;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void acer_s58m(struct biosinfo *b)
{
     /* This board doesn't have a checksum */
     BCD(b);
     b->need_reboot = OFF;
     b->addr_stat   = 0x41;
     b->addr_hour   = 0x54;
     b->addr_min    = 0x55;
     b->addr_sec    = 0x56;
     b->rtc_day     = 0x5C;
     b->shift_stat  = 4;
}

void msi_k7pro_k7pro18_ecs_common(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x58);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x57;
}

void msi_k7_pro(struct biosinfo *b)
{
     msi_k7pro_k7pro18_ecs_common(b);
}

void ecs_k7vmm(struct biosinfo *b)
{
     msi_k7pro_k7pro18_ecs_common(b);
     b->shift_stat  = 3;
}

void msi_ms_6243(struct biosinfo *b)
{
     msi_k7pro_k7pro18_ecs_common(b);
     b->shift_stat  = 5;
}

void msi_k7pro18(struct biosinfo *b)
{
     msi_k7pro_k7pro18_ecs_common(b);
     b->shift_stat  = 7;
}

void ecs_k7vmmp(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x58);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x56;
     b->shift_stat  = 7;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
     b->rtc_time    = ON;
}

void msi_6535_oldbios(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->addr_stat   = 0x54;
     b->shift_day   = 2;
     b->shift_stat  = 7;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void msi_6535(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x55;
     b->shift_stat  = 4;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void asrock_ge_pro(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x55);
     b->addr_stat   = 0x54;
     b->shift_stat  = 6;
     
     b->rtc_time    = ON;
     
     b->rtc_day     = 0x70;
     b->nr_rtc_day  = 8;
     b->reset_day   = ON;
     b->rtc_day_0_is_c0 = ON;
}

void asrock_k7s41gx(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x56);
     b->addr_stat   = 0x54;
     b->shift_stat  = 7;
     b->rtc_time    = ON;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->nr_rtc_day  = 7;
     b->rtc_day_0_is_c0 = ON;
}

void ecs_k7sem(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->addr_mon    = 0x32;
     b->shift_mon   = 4;
}

void gig_sis_645(struct biosinfo *b)
{
     /* ACPI must be disabled in Kernel, otherwise PC would'nt wake up! */
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->addr_mon    = 0x32;
     b->shift_mon   = 4;

     b->rtc_time    = ON;
}

void gig_sis_645_new(struct biosinfo *b)
{
     /* ACPI must be disabled in Kernel, otherwise PC would'nt wake up! */
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->addr_mon    = 0x32;
     b->shift_mon   = 4;
}

void sis_630s(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->addr_mon    = 0x32;
     b->shift_mon   = 4;

     b->rtc_time    = ON;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void shuttle_fs40(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->addr_mon    = 0x32;
     b->shift_mon   = 4;
     b->rtc_time    = ON;
     b->rtc_day     = 0x70;
     b->rtc_mon     = 0x71;
     b->reset_day   = ON;
     b->reset_mon   = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
     b->rtc_mon_0_is_c0 = ON;
     b->nr_rtc_mon      = 8;
}

void ecs_k7s6a(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     TIMEREGS(b, 0x33);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x32;
     b->rtc_time    = ON;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
     b->rtc_day_0_is_c0 = ON;
     b->nr_rtc_day      = 8;
}

void chaintech_8363(struct biosinfo *b)
{
     CHECKSUM(b, 0x6D);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_min    = 0x40;
     b->addr_hour   = 0x41;
     b->addr_day    = 0x51;
     /* addr_stat is missing -- ??? */
}

void asus_a7v333(struct biosinfo *b)
{
     CHECKSUM(b,0x6B);
     b->need_reboot = OFF; /* ??? */
     b->addr_stat   = 0x4B;
     b->shift_stat  = 1;
     b->addr_day    = 0x4B;
     b->shift_day   = 2;
     b->rtc_time    = ON;
}

void asus_a7s333(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF; /* ??? */
     b->addr_stat   = 0x4B;
     b->shift_stat  = 6;
     b->addr_day    = 0x4C;
     b->rtc_time    = ON;
}

void asus_a4s333(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->addr_stat   = 0x4B;
     b->shift_stat  = 6;
     b->addr_day    = 0x4C;
     b->day_hack    = 1;
     b->rtc_time    = ON;
}

void asus_p4b266(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->addr_stat   = 0x4B;
     b->shift_stat  = 5;
     b->addr_day    = 0x4C;
     b->day_hack    = 2;
     b->rtc_time    = ON;
}

void asus_p4b266_e(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->addr_stat   = 0x4B;
     b->shift_stat  = 3;
     b->addr_day    = 0x4C;
     b->day_hack    = 4;
     b->rtc_time    = ON;
}

void asus_p4b266_m(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->need_reboot = OFF;
     b->addr_stat   = 0x4A;
     b->shift_stat  = 4;
     b->addr_day    = 0x4B;
     b->day_hack    = 3;
     b->rtc_time    = ON;
}

void pcchips_810lmr(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5B);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x5A;
     b->shift_stat  = 3;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
}

void msi_ms_7021(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5B);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x5A;
     b->shift_stat  = 3;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void msi_ms_6764(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5B);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x5A;
     b->shift_stat  = 5;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void msi_ms_6764_brkn(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x5B);
     b->need_reboot = ON_ANY_CHANGE;
     b->addr_stat   = 0x59;
     b->shift_stat  = 7;
     b->rtc_day     = 0x6F;
     b->reset_day   = ON;
}

void dell_optiplex_dxa(struct biosinfo *b)
{
     CHECKSUM(b, 0x5F);
     
     b->chk_method = DELL;
     
     b->addr_stat  = 0x4B;
     b->shift_stat = 6;  /* this bit switches between "off" and "everyday"        */

     BCD(b);

     b->addr_hour  = 0x4C;
     b->addr_min   = 0x4D;
}

void asus_p4bgl(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->addr_stat  = 0x4C;
     b->shift_stat = 7;

     b->addr_day   = 0x4D;

     BCD(b);
     b->day_no_bcd = ON;  /* the day is not in BCD. */
     b->nr_day     = 5;

     b->addr_hour  = 0x68;
     b->addr_min   = 0x69;
     b->addr_sec   = 0x6A;
}

void asus_p4b533_e(struct biosinfo *b)
{
     CHECKSUM(b, 0x6B);
     b->addr_stat  = 0x4C;

     b->addr_day   = 0x4C;
     b->shift_day  = 1;

     BCD(b);
     b->day_no_bcd = ON;  /* the day is not in BCD. */
     b->nr_day     = 5;

     b->addr_hour  = 0x68;
     b->addr_min   = 0x69;
     b->addr_sec   = 0x6A;
}

void aopen_ax45_4d(struct biosinfo *b)
{
     CHECKSUM(b, 0x6E);
     b->addr_stat  = 0x32;
     b->need_reboot = ON_ANY_CHANGE;

     b->addr_day   = 0x33;
     b->addr_hour  = 0x40;
     b->addr_min   = 0x41;
     b->addr_sec   = 0x36;

     b->rtc_time    = ON;
     b->rtc_day     = 0x70;
     b->reset_day   = ON;
}

void asus_p4p800_vm(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     b->need_reboot  = OFF;  /* ??? */
     b->upper_method = INTEL;

/* note that below this line are the real addresses, */
/* not the addresses from /dev/nvram                 */

     b->addr_stat    = 0x97;
     b->shift_stat   = 1;

     TIMEREGS(b, 0x97);
     b->shift_day    = 2;
}

void asus_p4p800_vm_2(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     b->need_reboot  = ON_ANY_CHANGE;
     b->upper_method = INTEL;

/* note that below this line are the real addresses, */
/* not the addresses from /dev/nvram                 */

     b->addr_stat    = 0x98;
     b->shift_stat   = 4;

     TIMEREGS(b, 0x99);
}

void asus_p4c800_e(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     b->need_reboot  = ON_ANY_CHANGE;
     b->upper_method = INTEL;

     b->addr_stat    = 0x9B;
     b->shift_stat   = 6;

     TIMEREGS(b, 0x9C);
}

void asus_p4c800(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     b->need_reboot  = ON_ANY_CHANGE;
     b->upper_method = INTEL;

     b->addr_stat    = 0x9B;
     b->shift_stat   = 5;

     TIMEREGS(b, 0x9C);
}

void asus_p4p800(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     b->need_reboot  = ON_ANY_CHANGE;
     b->upper_method = INTEL;

     b->addr_stat    = 0x9B;
     b->shift_stat   = 4;

     TIMEREGS(b, 0x9C);
}

void springdale(struct biosinfo *b)
{
     CHECKSUM(b, 0x30);
     b->need_reboot  = OFF; /* ??? */
     b->upper_method = INTEL;

     TIMEREGS(b, 0x92);
     b->shift_day    = 1;
     b->addr_stat    = 0x92;
}

void aopen_mk73le_n(struct biosinfo *b)
{
     b->upper_method = VT82Cxxx;

     b->addr_stat    = 0xD2;
     b->shift_stat   = 5;

     TIMEREGS(b, 0xD8);
}


void ibm_pc_300pl(struct biosinfo *b)
{
     b->upper_method = VT82Cxxx;

     b->addr_stat    = 0x9B;

     BCD(b);
     b->addr_day     = 0x9A;
     b->addr_hour    = 0x99;
     b->addr_min     = 0x98;
}

void asus_a7n8x_vm400(struct biosinfo *b)
{
     b->upper_method = VT82Cxxx;

     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0x89);
     
     b->addr_stat    = 0x88;
     b->shift_stat   = 5;
     b->need_reboot  = ON_ANY_CHANGE;
}

void gig_ga_7vt600(struct biosinfo *b)
{
     b->upper_method = VT8235_37;

     TIMEREGS(b, 0xD8);

     b->addr_stat    = 0xD2;
     b->shift_stat   = 5;
     b->rtc_time     = ON;
}

void aopen_ax37(struct biosinfo *b)
{
     b->upper_method = VT8235_37;

     TIMEREGS(b, 0xD8);

     b->addr_stat    = 0xD2;
     b->shift_stat   = 5;
}

void shuttle_fx41(struct biosinfo *b)
{
     b->upper_method = VT8235_37;

     TIMEREGS(b, 0xD8);
     b->addr_sec     = 0; /* no seconds */

     b->addr_stat    = 0xD2;
     b->shift_stat   = 5;
}

void shuttle_sk43g(struct biosinfo *b)
{
     b->upper_method = VT8235_37;

     CHECKSUM(b, 0xDE);
     TIMEREGS(b, 0xD8);

     b->addr_stat    = 0xD2;
     b->shift_stat   = 5;
     
     b->need_reboot  = ON_STAT;
}

void asus_k8vsedx(struct biosinfo *b)
{
     b->upper_method = VT8235_37;

     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0xE5);

     b->addr_stat    = 0xE3;
     b->shift_stat   = 3;

     b->rtc_time     = ON;
     b->rtc_day      = 0x6F;
     b->reset_day    = ON;

     b->need_reboot  = ON_ANY_CHANGE;
}

void asus_k8v(struct biosinfo *b)
{
     b->upper_method = VT8235_37;

     CHECKSUM(b, 0x30);
     TIMEREGS(b, 0xE4);
     b->shift_day    = 2;

     b->addr_stat    = 0xE2;
     b->shift_stat   = 7;

     b->rtc_time     = ON;
     b->rtc_day      = 0x6F;
     b->reset_day    = ON;
}

/*
 * Definitions of the mainboards below are in the format
 *
 *             { infowriter,        iwname,             boardvendor, boardtype, boardversion, biosvendor, biosversion, biosrelease }
 *
 * Where NULL means "Don't care" and "" is a real empty field.
 */
static struct mainboard boards[] = {
/* MB =  1 */  { msi_ms_6351,      "msi_ms_6351",      "MSI",             "MS-6351", NULL, NULL, NULL, NULL},
               { msi_ms_6351,      "msi_ms_6351",      "Micro-Star Inc.", "MS-6351", "1.0", "American Megatrends Inc.", "62710", "12/04/01" },
               { msi_ms_6351,      "msi_ms_6351",      "Micro-Star Inc.", "MS-6351", "1.0", "American Megatrends Inc.", "62710", "01/14/02" },
               { msi_ms_6351,      "msi_ms_6351",      "Micro-Star Inc.", "MS-6351", "1.0", "American Megatrends Inc.", "62710", "04/03/02" },

/* MB =  2 */  { bio_m6vcg,        "bio_m6vcg",        "Abit <http://www.abit.com.tw/>", "KT7", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Abit <http://www.abit.com.tw/>", "KA7-100", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Abit <http://www.abit.com.tw/>", "VP6", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "<http://www.abit.com.tw>", "8363-686A(KT7[A][-RAID],KT7E)", "v1.0 ~ v1.2", NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "<http://www.abit.com.tw>", "8363-686A(KT7,KT7A,KT7-RAID,KT7A-RAID)", "", "Award Software International, Inc.", "6.00 PG", "07/11/2002" },
               { bio_m6vcg,        "bio_m6vcg",        "Acer", "M19A", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "AOPEN", "AK33", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        /* Aopen AK73 */ "", "07", "9189310401", NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Biostar", "M6VBE", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Biostar Group", "M6VBE-A", "1.x", "Award Software International, Inc.", "4.51 PG", "03/23/00" },
               /* 
                *  Unfortunately BIOSTAR M6VCG has no valid DMI entries
                *  (at least bios rev. 07/31/2000). Thus,
                *
                *     --iwname bio_m6vcg
                *
                *  option is your friend.
                */
               { bio_m6vcg,        "bio_m6vcg",        "Biostar", "M6VCG", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "",        "M6VCG", "", "Award Software International, Inc.", "6.00 PG", "09/21/2000" },
               { bio_m6vcg,        "bio_m6vcg",        /* Biostar M7VKB */    "", "8363-686A", "", "Award Software International, Inc.", "6.00 PG", "05/24/2001" },
               { bio_m6vcg,        "bio_m6vcg",        /* Chaintech 6AIV2T */ "",  "601-686A", "", "Award Software International, Inc.", "6.00 PG", "09/26/2001" },
               { bio_m6vcg,        "bio_m6vcg",        "Elitegroup", "K7VZA", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Elitegroup", "P6VAP", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Epox", "8KTA", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        /* Freetech P6S215 */ "", "8604-686B", "", "Phoenix Technologies, LTD", "6.00 PG", "06/25/2003" },
               { bio_m6vcg,        "bio_m6vcg",        "GigaByte Technology Co., Ltd.", "601-686B", "", "Award Software International, Inc.", "6.00 PG", "08/16/2002" },
               { bio_m6vcg,        "bio_m6vcg",        "Legend QDI", "Advance-10T", "", "Award Software International, Inc.", "6.00 PG", "08/23/2001" },
               { bio_m6vcg,        "bio_m6vcg",        "Legend.QDI,Inc", "KINETIZ 7T/7B", "V1.X", "Award Software International, Inc.", "6.00 PG", "12/20/2000" },
               { bio_m6vcg,        "bio_m6vcg",        /* MSI */ "", "MS-6318", "", "Award Software International, Inc.", "6.00 PG", "02/23/00" },
               { bio_m6vcg,        "bio_m6vcg",        /* MSI */ "", "MS-6318", "", "Award Software International, Inc.", "6.00 PG", "06/13/01" },
               { bio_m6vcg,        "bio_m6vcg",        /* NMC/ENMIC 7VAX */ "", "8371-686A", "", "Award Software International, Inc.", "6.00 PG", "02/21/2001" },
               { bio_m6vcg,        "bio_m6vcg",                             "", "8371-686A", "", "Award Software International, Inc.", "6.00 PG", "07/25/2000" },
               { bio_m6vcg,        "bio_m6vcg",        "Shuttle", "FV24", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Shuttle", "MV25N", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "Tyan", "S1834", NULL, NULL, NULL, NULL },
               { bio_m6vcg,        "bio_m6vcg",        "", "TwisterT-686B", "", "Award Software International, Inc.", "6.00 PG", "05/09/2002" },

               { msi_ms_6323,      "msi_ms_6323",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6323", "", "Award Software International, Inc.", "6.00 PG", "06/30/00" },

               { ecs_k7vza,        "ecs_k7vza",        /* Elitegroup K7VZA */ "", "8363-686A", "", "Award Software International, Inc.", "6.00 PG", "12/06/2000" },
               { ecs_k7vza,        "ecs_k7vza",        "<http://www.abit.com.tw>", "8363-686A(KT7,KT7A,KT7A-RAID,KT7E)", "", NULL, NULL, NULL },


/* MB =  3 */  { gig_ga_6bxc,      "gig_ga_6bxc",      /* Abit TH7-Raid  */ "ABIT", "i850-W83627HF", "Rev:0.1", "Award Software International, Inc.", "6.00 PG", "12/04/2001" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Acorp 6A815EPD */ "", "i815-W83627", "", "Award Software International, Inc.", "6.00 PG", "08/03/2001" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* AOpen MX3W     */ "", "", "", "Award Software International, Inc.", "6.00 PG", "10/11/99" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* AOpen MX3W Pro */ "", "", "", "Award Software International, Inc.", "6.00 PG", "07/27/2001" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "AOpen", "MX4GVR", "918B410101", "Phoenix/Award Technologies, LTD", "6.00 PG", "02/13/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "AOpen", "1244",   "918B410201", "Phoenix/Award Technologies, LTD", "6.00 PG", "04/03/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "BIOSTAR Group", "M6TWG", "1.X", "Award Software International, Inc.", "6.00 PG", "05/14/01" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Chaintech 6LTM2 */ "", "i440LX-8671", "", "Award Software International, Inc.", "4.51 PG", "07/20/99" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* ECS P6BXT-A+    */ "EliteGroup Computer Systems Co. Ltd.", "i440BX-W977EF", "1.0", "Award Software International, Inc.", "4.51 PG", "06/23/00" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Gigabyte GA-6BA", "i440BX-W977", "1.0", "Award Software International, Inc.", "4.51 PG", "02/05/01" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Gigabyte", "GA-6BXC", NULL, NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Gigabyte", "GA-6BXD", NULL, NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Gigabyte GA-6BXC */ "Gigabyte Technology Co. Ltd.",  "i440BX-8671", "", "Award Software International, Inc.", "4.51 PG", "09/17/98" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Gigabyte GA-6BXC */ "Giga-Byte Technology CO., LTD", "i440BX-8671", "", "Award Software International, Inc.", "4.51 PG", "01/22/02" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Gigabyte GA-6BXE */ "Giga-Byte Technology CO., LTD", "i440BX-8671", "", "Award Software International, Inc.", "4.51 PG", "02/06/01" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Gigabyte Technology Co., Ltd.", "6IEM", "1.x", "Award Software International, Inc.", "6.00 PG", "01/18/2002" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Gigabyte", "GA-8GE667 Pro", NULL, NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Gigabyte Technology Co., Ltd.", "8IK1100",   "1.x", "Award Software International, Inc.", "FA", "08/13/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Gigabyte Technology Co., Ltd.", "8IPE1000P", "1.x", "Award Software International, Inc.", "Fa", "07/29/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Gigabyte GA-686DL2 */ "Gigabyte Technology Co. Ltd.", "i440LX-8661", "", "Award Software International, Inc.", "4.51 PG", "03/02/98" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Gigabyte GA-BX2000 */ "Corporation Name", "i440BX-8671", "1.0", NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Gigabyte GA-6WMMC7 */ "Gigabyte Technology Co., Ltd.", "i810-47B27X", "", "Award Software International, Inc.", "6.00 PG", "10/22/01" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "MediaPortal", "QDI Winnex9", NULL, NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* MediaPortal */ "", "i810-W83627HF", "", "Award Software International, Inc.", "6.00 PG", "09/30/2000" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* MediaPortal */ "", "i810-W83627HF", "", "Award Software International, Inc.", "6.00 PG", "11/13/2000" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* MediaPortal */ "", "i810-W83627HF", "", "Award Software International, Inc.", "6.00 PG", "11/28/2000" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6116 (i440BX)", "1.X", NULL, "", "" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "MSI", "MS-6117", NULL, NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "MSI", "MS-6156", NULL, NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6159 (i440LX)", "1.X", "Award Software International, Inc.", "4.51 PG", "04/20/99" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6337", "5.0", "Award Software International, Inc.", "6.00 PG", "01/11/2002" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "MICRO-STAR INTL, CO.,LTD.", "MS-6743", "Ver1.1A", "Phoenix Technologies, LTD", "6.00 PG", "06/24/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Shuttle AE23 */ "", "i815E-627HF", "", "Award Software International, Inc.", "6.00 PG", "09/11/2001" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Shuttle Inc", "FB50", /* Shuttle SB51G  */ "", "Phoenix Technologies, LTD", "6.00 PG", "11/14/2002" }, 
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Shuttle Inc", "FB50", /* Shuttle SB51G  */ "", "Phoenix Technologies, LTD", "6.00 PG", "11/22/2002" }, 
               { gig_ga_6bxc,      "gig_ga_6bxc",      "SHUTTLE INC", "FB52", /* Shuttle SB52G2 */ "", "Phoenix Technologies, LTD", "6.00 PG", "01/20/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Shuttle Inc", "FB61", "", "Phoenix Technologies, LTD", "6.00 PG", "04/28/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Shuttle Inc", "FB61", "", "Phoenix Technologies, LTD", "6.00 PG", "05/16/2003" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Shuttle Inc.", "(HOT-661)Intel i440BX", "2A69KH2B", NULL, "", NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Tekram", "P6B40-A4X", NULL, NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "Tekram Technology Co., Ltd.", "P6B40-A4X-i440BX", "Rev. 1.0", NULL, NULL, NULL },
               { gig_ga_6bxc,      "gig_ga_6bxc",      "", "i815EP-ITE8712", "", "Award Software International, Inc.", "6.00 PG", "04/27/2001" },
               { gig_ga_6bxc,      "gig_ga_6bxc",      /* Lex */ "", "i815E-ITE", "", "Award Software International, Inc.", "6.00 PG", "06/06/2002" },


/* MB =  4 */  { epx_8kta3,        "epx_8kta3",        "EPox", "8KTA3", NULL, NULL, NULL, NULL },
               { epx_8kta3,        "epx_8kta3",        /* EPox 8KTA2      */ "", "8363-686A",  "", "Award Software International, Inc.", "6.00 PG", "10/23/2001" },
               { epx_8kta3,        "epx_8kta3",        /* EPox 8KTA3      */ "", "8363A-686B", "", "Award Software International, Inc.", "6.00 PG", "03/05/2001" },
               { epx_8kta3,        "epx_8kta3",        /* ENMIC/NMC 8TTX+ */ "", "8363A-686B", "", "Award Software International, Inc.", "6.00 PG", "08/02/2002" },

/* MB =  5 */  { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "TUSL2", NULL, NULL, NULL, NULL },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "TUSL2-C", NULL, NULL, NULL, NULL },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "CUSL2", NULL, NULL, NULL, NULL },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "CUSL2-C", "REV 1.xx", "Award Software, Inc.", "ASUS CUSL2-C ACPI BIOS Revision 1003", "10/04/2000" },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "CUSL2-C", NULL, NULL, NULL, NULL },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "P3B-F", NULL, NULL, NULL, NULL },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "<P3B-F>", NULL, NULL, NULL, NULL },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "P3C-E",    "REV 1.xx", "Award Software, Inc.", "ASUS P3C-E ACPI BIOS Revision 1023",            "10/12/2000" },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "<CUBX-L>", "REV 1.xx", "Award Software, Inc.", "ASUS CUBX-L/CUBX-E ACPI BIOS Revision 1007.A",  "07/26/2000" },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "<CUWE>",   "REV 1.xx", "Award Software, Inc.", "ASUS CUWE ACPI BIOS Revision 1017 Beta 003",    "10/11/2001" },
               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "CUWE-RM",  "REV 1.xx", "Award Software, Inc.", "ASUS CUWE-RM ACPI BIOS Revision 1017 Beta 003", "10/11/2001" },

               { asus_tusl2,       "asus_tusl2",       "ASUSTeK Computer INC.", "<CUBX>", "REV 1.xx", "Award Software, Inc.", "ASUS CUBX ACPI BIOS Revision 1008 Beta 004", "10/15/2001" },
               { asus_cubx,        "asus_cubx",        "ASUSTeK Computer INC.", "<CUBX>", "REV 1.xx", "Award Software, Inc.", "ASUS CUBX ACPI BIOS Revision 1008B4 (Tualatin) CMD 1.9.16", "10/15/2001" },
               { asus_cubx,        "asus_cubx",        "ASUSTeK Computer INC.", "<CUBX>", "REV 1.xx", "Award Software, Inc.", "ASUS CUBX ACPI BIOS Revision 1006", "06/27/2000" },

               { asus_cusi_fx,     "asus_cusi_fx",     "ASUSTeK Computer INC.", "CUSI-FX", "REV 1.xx", "Award Software, Inc.", "ASUS CUSI-FX ACPI BIOS Revision 1005", "12/11/2000" },

/* MB =  6 */  { ecs_p6bat,        "ecs_p6bat",        "AOPEN", "AX59PRO", NULL, NULL, NULL, NULL },
               { ecs_p6bat,        "ecs_p6bat",        "DFI", "K6BV3+", "", "Award Software International, Inc.", "4.51 PG", "01/15/01" },
               { ecs_p6bat,        "ecs_p6bat",        "DFI", "K6BV3+/66", NULL, NULL, NULL, NULL },
               { ecs_p6bat,        "ecs_p6bat",        "Apollo", "MVP3", NULL, NULL, NULL, NULL },
               { ecs_p6bat,        "ecs_p6bat",        /* ECS Elitegroup P6BAT-A+, P6BAP-Me */ "", "693-596-W977EF", "", "Award Software International, Inc.", "4.51 PG", "06/22/00" },
               { ecs_p6bat,        "ecs_p6bat",        "Elitegroup", "P6BAT-A+", NULL, NULL, NULL, NULL },
               /* 
                *  Unfortunately EPOX MVP3C-M has no valid DMI entries
                *  (at least bios rev. 1.01). Thus,
                *
                *     --iwname ecs_p6bat
                *
                *  option is your friend.
                */
               { ecs_p6bat,        "ecs_p6bat",        "EPOX", "MVP3C-M (EP-58)", "Rev 1.0", "Award", NULL, NULL },
               { ecs_p6bat,        "ecs_p6bat",        /* GA-6VXE7+ */ "Gigabyte Technology Co., Ltd.", "693A-596B-8671", "", "Award Software International, Inc.", "4.51 PG", "03/21/01" },
               { ecs_p6bat,        "ecs_p6bat",        "NMC", "5VMXe", NULL, NULL, NULL, NULL },
               { ecs_p6bat,        "ecs_p6bat",        "Shuttle", "HOT-661V", NULL, NULL, NULL, NULL },
               { ecs_p6bat,        "ecs_p6bat",        "Shuttle", "AV61", NULL, NULL, NULL, NULL },
     
               { dfi_ak74_ec,      "dfi_ak74_ec",      "DFI AK74-EC", "8363-686A", "", "Award Software International, Inc.", "6.00 PG", "11/05/2001" },

               { shuttle_mvp3,     "shuttle_mvp3",     "Shuttle Inc.", "VIA APOLLO MVP3 (HOT-591P)", "2A5LEH2A", NULL, NULL, NULL },

               { tekram_p5mvp3,    "tekram_p5mvp3",    "Tekram Technology Co., Ltd.", "P5MVP3", "Rev 1.X", "Award Software International, Inc.", "4.51 PG", "06/16/00" },
               { tekram_p5mvp3,    "tekram_p5mvp3",    "", "MVP3-596-W877", "", "Award Software International, Inc.", "4.51 PG", "12/14/00" },

/* MB =  7 */  { msi_ms_6120,      "msi_ms_6120",      "MSI", "MS-6120", NULL, NULL, NULL, NULL },

/* MB =  8 */  { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "CUV4X-E", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "A7M266", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "A7V-133", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "<A7V133>", "REV 1.xx", "Award Software, Inc.", "ASUS A7V-133 ACPI BIOS Revision 1009", "04/23/2002" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "<A7V133>", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "A7V266", "REV 1.xx", "Award Software, Inc.", "ASUS A7V266 ACPI BIOS Rev 1011", "08/20/2002" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "A7V266-E", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "<K7V>", "REV 1.xx", "Award Software, Inc.", "ASUS K7V 1007", "07/26/2000" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "<K7V-T>", "REV 1.xx", "Award Software, Inc.", "ASUS K7V-T ACPI BIOS Revision 1007", "07/26/2000" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "MED 2001", "REV 1.xx", "Award Software, Inc.", "ASUS MED 2001 ACPI BIOS Revision 1006A", "02/01/2001" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "MED2001", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "P3V4X", "REV 1.xx", "Award Software, Inc.", "ASUS P3V4X ACPI BIOS Revision 1005",            "06/12/2000" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "P3V4X", "REV 1.xx", "Award Software, Inc.", "ASUS P3V4X ACPI BIOS Revision 1006 Beta 002",   "10/06/2000" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "TUV4X", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "Gigabyte", "GA-7DXR", NULL, NULL, NULL, NULL },

               /*
                *  the lower two lines of the next few are for the same board with different BIOS revisions.
                *  If you have the missing information on the BIOS entries, please submit
                */
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "<A7V>", "REV 1.xx", "Award Software, Inc.", "ASUS A7V ACPI BIOS Revision 1008", "07/17/2001" },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "<A7V>", "REV 1.xx", "Award Software, Inc.", "ASUS A7V ACPI BIOS Revision 1011", "04/23/2002" },
               { bio_m6vcg,        "bio_m6vcg",        "ASUSTeK Computer INC.", "A7V", NULL, NULL, NULL, NULL },
               { asus_cuv4xe,      "asus_cuv4xe",      "ASUSTeK Computer INC.", "A7V", NULL, NULL, NULL, NULL },

/* MB =  9 */  { asus_a7s_vm,      "asus_a7s_vm",      "ASUSTeK Computer INC.", "A7S-VM", NULL, NULL, NULL, NULL },
               { asus_a7s_vm,      "asus_a7s_vm",      "ASUSTeK Computer INC.", "A7A266-E", NULL, NULL, NULL, NULL },

/* MB = 10 */  { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P2L97-S", NULL, NULL, NULL, NULL },
               { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P2B", NULL, NULL, NULL, NULL },
               { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P2B-B", NULL, NULL, NULL, NULL },
               { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P2B-F", NULL, NULL, NULL, NULL },
               { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P2B-LS", "REV 1.xx", "Award Software, Inc.", "ASUS P2B-LS ACPI BIOS Revision 1012", "03/03/00" },
               { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P5A", NULL, NULL, NULL, NULL },
               { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P5A-B", "REV 1.XX", "Award Software, Inc.", "ASUS P5A-B ACPI BIOS Revision 1011 Beta 004", "12/14/00" },
               { asus_p2l97s,      "asus_p2l97s",      "ASUSTeK Computer INC.", "P5A-B", "REV 1.XX", NULL, NULL, NULL },
     
               { asus_tx97_xe,     "asus_tx97_xe",     "ASUSTeK Computer INC.", "TX97-XE", NULL, NULL, NULL, NULL },
     
/* MB = 11 */  { pb_bora_pro,      "pb_bora_pro",      "Packard Bell", "Bora Pro", NULL, NULL, NULL, NULL },
     
/* MB = 12 */  { fic_va_503p,      "fic_va_503p",      "FIC", "VA 503+", NULL, NULL, NULL, NULL },

/* MB = 13 */  { msi_k7_pro,       "msi_k7_pro",       "MSI", "K7 Pro", NULL, NULL, NULL, NULL },
     
/*  These boards don't support wake-on-timer at all */
/*
 *             { sci_pro_c5,       "sci_pro_c5",       "Sciencic", "Pro C5", NULL, NULL, NULL, NULL },
 *             { sci_pro_c5,       "sci_pro_c5",       "Sciencic", "Pro C6", NULL, NULL, NULL, NULL }, 
 */
     
/* MB = 15 */  { msi_ms_6368,      "msi_ms_6368",      "MSI", "MS-6368", NULL, NULL, NULL, NULL },
               { msi_ms_6368,      "msi_ms_6368",      /* MSI MS-6368 Ver. 5.0 */ "", "601-686A", "", "Award Software International, Inc.", "6.00 PG", "03/18/2002" },
               { msi_ms_6368,      "msi_ms_6368",      "",                                  "MS-6318", "", "Award Software International, Inc.", "6.00 PG", "09/28/00" },
               { msi_ms_6368,      "msi_ms_6368",      "MSI",                               "MS-6330", "v5.0", NULL, NULL, NULL },
               { msi_ms_6368,      "msi_ms_6368",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6330", "", "Award Software International, Inc.", "6.00 PG", "05/11/2001" },
               { msi_ms_6368,      "msi_ms_6368",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6378(VT8361)", "", NULL, NULL, NULL },
               { msi_ms_6368,      "msi_ms_6368",      "",     "694X-686B", "",         "Award Software International, Inc.", "6.00 PG", "04/26/2001" },
               { msi_ms_6368,      "msi_ms_6368",      "Abit", "694X-686B", "VH6T 1.1", "Award Software International, Inc.", "6.00 PG", "03/15/2002" },
               { msi_ms_6368,      "msi_ms_6368",      "Biostar Group", "M6VBE-A", "1.x", "Award Software International, Inc.", "4.51 PG", "03/23/00" },
               { msi_ms_6368,      "msi_ms_6368",      "Legend.QDI,Inc", "8363-686A", "KINETIZ 7T",  "Award Software International, Inc.", "6.00 PG", "03/07/2003" },
               { msi_ms_6368,      "msi_ms_6368",      "Legend.QDI,Inc", "KINETIZ 7T/7B/7E", "V2.X", "Award Software International, Inc.", "6.00 PG", "03/07/2003" },
               { msi_ms_6368,      "msi_ms_6368",      "Legend.QDI,Inc", "KINETIZ 7T/7B/7E", "V2.X", "Award Software International, Inc.", "6.00 PG", "02/24/2004" },


/* MB = 16 */  { gig_ga_7zx,       "gig_ga_7zx",       "Gigabyte",                  "GA-7ZX", NULL, NULL, NULL, NULL },
               { gig_ga_7zx,       "gig_ga_7zx",       "Gigabyte Technology Co. Ltd.", "7VX", "1.0", "American Megatrends Inc.", "62710", "07/15/97" },

/* MB = 17 */  { asus_sp98_x,      "asus_sp98_x",      "ASUSTeK Computer INC.", "SP98AGP-X", NULL, NULL, NULL, NULL },
     
/* MB = 18 */  { soyo_sy5ssm,      "soyo_sy5ssm",      "Soyo", "SY-5SSM", NULL, NULL, NULL, NULL },
               { soyo_sy5ssm,      "soyo_sy5ssm",      "Gigabyte", "GA-586STX2", NULL, NULL, NULL, NULL },

/* MB = 19 */  { msi_k7pro18,      "msi_k7pro18",      "MSI", "K7 Pro", NULL, NULL, NULL, NULL },
               { msi_k7pro18,      "msi_k7pro18",      /* MSI-Barebone Hermes 845GV   (MS-6557) */ "Micro Star", "MS-6557", "3.00", "American Megatrends Inc.", "V3.3", "07/03/03" },

/* MB = 20 */  { qdi_brlx1,        "qdi_brlx1",        "QDI", "Legend", NULL, NULL, NULL, NULL },
               { qdi_brlx1,        "qdi_brlx1",        "QDI", "BrillianX I", NULL, NULL, NULL, NULL },
               { qdi_brlx1,        "qdi_brlx1",        "QDI", "P6I440BX/B1", NULL, NULL, NULL, NULL },

               { ga_6bxc_qdi_common,"ga_6bxc_qdi_common","MICRO-STAR INTERNATIONAL CO., LTD", "MS-6391", "", "Award Software International, Inc.", "6.00 PG", "12/31/2001" },

               { epox_4g4a,        "epox_4g4a",        /* Epox 4G4A+ */ "", "i845G-W83627", "", "Award Software International, Inc.", "6.00 PG", "11/04/2002" },
               { epox_4g4a,        "epox_4g4a",        "Epox", "Springdale-G", "EX5-300 / EP-4GPF", "Phoenix Technologies, LTD", "6.00 PG", "04/27/2004" },


               /* 
                * Unfortunately ECS changed the BIOS without increasing the BIOS release number
                * or revision date, so the right thing to do can't be detected automatically.
                *
                * if it doesn't work for you out-of-the-box, try one of the command line options: 
                *   --iwname ecs_k7s5a_1
                *   --iwname ecs_k7s5a_2
                *   --iwname ecs_k7s5au
                */
               { ecs_k7s5a,        "ecs_k7s5a",        "ECS", "K7S5A", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },
               { ecs_k7s5a_1,      "ecs_k7s5a_1",      "ECS", "K7S5A", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },
               { ecs_k7s5a_2,      "ecs_k7s5a_2",      "ECS", "K7S5A", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },
               { ecs_k7s5au,       "ecs_k7s5au",       "ECS", "K7S5A", "",    "American Megatrends Inc.", "07.00T", "04/02/01" },
               { ecs_k7s5au,       "ecs_k7s5au",       "ECS", "K7S5A", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },

/* MB = 21 */  { ecs_k7s5a,        "ecs_k7s5a",        "ECS", "K7S5A",  NULL, NULL, NULL, NULL },
/* MB = 24 */  { ecs_k7s5al,       "ecs_k7s5al",       "ECS", "K7S5AL", NULL, NULL, NULL, NULL },
/* MB = 43 */  { ecs_k7s5au,       "ecs_k7s5au",       "ECS", "K7S5AU", NULL, NULL, NULL, NULL },

               { ecs_p4s5a,        "ecs_p4s5a",        "ECS", "P4S5A",  "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },

               { ecs_p4s8ag,       "ecs_p4s8ag",       "ECS", "P4S8AG", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },

/* MB = 22 */  { gig_ga_7vkml,     "gig_ga_7vkml",     "Gigabyte", "GA-7VKML", NULL, NULL, NULL, NULL },
               { gig_ga_7vkml_2,   "gig_ga_7vkml_2",   "Gigabyte Technology Co., Ltd.", "7VKML", "1.0", "American Megatrends Inc.", "07.00T", "02/24/0103" },
               { gig_ga_k7vm4,     "gig_ga_k7vm4",     /* Gigabyte */ "", "K7VM4", "1.00", "American Megatrends Inc.", "P2.10", "11/14/2003" },

/* MB = 23 */  { fic_sd_11,        "fic_sd_11",        "FIC", "SD 11", NULL, NULL, NULL, NULL },
               { fic_sd_11,        "fic_sd_11",        "FIC", "SD11", NULL, NULL, NULL, NULL },

/* MB = 25 */  { gig_ga_7zxe,      "gig_ga_7zxe",      "Gigabyte", "GA-7ZXE", NULL, NULL, NULL, NULL },
               { gig_ga_7zxe,      "gig_ga_7zxe",      "Gigabyte Technology Co., Ltd.", "7IXEH", "1.x", "American Megatrends Inc.", "62710", "10/08/2001" },

/* MB = 26 */  { gig_ga_5ax,       "gig_ga_5ax",       "Gigabyte", "GA-5AX", NULL, NULL, NULL, NULL },

/* MB = 27 */  { mp_avt_121,       "mp_avt_121",       "MediaPortal", "AVT", "i810 bios version 1.21", NULL, NULL, NULL },
               { mp_avt_121,       "mp_avt_121",       "ZIDA", "Instant810C", "1.0", "American Megatrends Inc.", "62710", "05/20/99" },

/* MB = 28 */  { mp_avt_122,       "mp_avt_122",       "MediaPortal", "AVT", "i810 bios version 1.22", NULL, NULL, NULL },
               { mp_avt_122,       "mp_avt_122",       "INTEL", "WHITNEY", "1.0", "American Megatrends Inc.", "62710", "07/15/97" },

/* MB = 30 */  { asus_p4t533_c,    "asus_p4t533_c",    "ASUSTeK Computer INC.", "P4T533-C", NULL, NULL, NULL, NULL },
               { asus_p4t533_c,    "asus_p4t533_c",    "ASUSTeK Computer INC.", "P4S800MX", "REV 1.xx", "Award Software, Inc.", "ASUS P4S800MX ACPI BIOS Revision 1002", "10/06/2003" },
               { asus_p4t533_c,    "asus_p4t533_c",    "ASUSTeK Computer INC.", "<P4B>", "REV 1.xx", "Award Software, Inc.", "ASUS P4B ACPI BIOS Revision 1010", "05/03/2002" },

/* MB = 46 */  { asus_p4pe,        "asus_p4pe",        "ASUSTeK Computer INC.", "P4PE",      NULL, NULL, NULL, NULL },
               { asus_p4pe,        "asus_p4pe",        "ASUSTeK Computer INC.", "P4PE-X",    "REV 1.xx", "Award Software, Inc.", "ASUS P4PE-X ACPI BIOS Revision 1003",    "05/01/2003" },
               { asus_p4pe,        "asus_p4pe",        "ASUSTeK Computer INC.", "P4PE-X",    "REV 1.xx", "Award Software, Inc.", "ASUS P4PE-X ACPI BIOS Revision 1004",    "05/21/2003" },
               { asus_p4pe,        "asus_p4pe",        "ASUSTeK Computer INC.", "P4PE-X/TE", "REV 1.xx", "Award Software, Inc.", "ASUS P4PE-X/TE ACPI BIOS Revision 1006", "01/05/2004" },
               { asus_p4pe,        "asus_p4pe",        "ASUSTeK Computer INC.", "P4GE-V",    "REV 1.xx", "Award Software, Inc.", "ASUS P4GE-V ACPI BIOS Revision 1001",    "09/16/2002" },

/* MB = 31 */  { gig_ga_7vtxh,     "gig_ga_7vtxh",     "Gigabyte", "GA-7VTXH", NULL, NULL, NULL, NULL },
               { gig_ga_7vtxh,     "gig_ga_7vtxh",     "Gigabyte Technology Co., Ltd.", "7VTXH", "1.0", "American Megatrends Inc.", "062710", "12/19/2001" },
               { gig_ga_7vtxh,     "gig_ga_7vtxh",     "Gigabyte", "GA-7VTXE", NULL, NULL, NULL, NULL },

/* MB = 33 */  { msi_ms_6380,      "msi_ms_6380",      "MSI", "MS-6380", NULL, NULL, NULL, NULL },
     
/* MB = 34 */  { ecs_k7som,        "ecs_k7som",        "Elitegroup", "K7SOM", NULL, NULL, NULL, NULL },

/* MB = 35 */  { via_p4ma,         "via_p4ma",         "VIA", "P4MA", NULL, NULL, NULL, NULL },
               { via_p4ma,         "via_p4ma",         "", "P4MA", "", "Award Software International, Inc.", "6.00 PG", "05/23/2002" },

/* MB = 36 */  { msi_694tpro,      "msi_694tpro",      "MSI", "MS-6309", NULL, NULL, NULL, NULL },

/* MB = 37 */  { gig_ga_7vtx,      "gig_ga_7vtx",      "Gigabyte", "GA-7VTXH", NULL, NULL, NULL, NULL },
               { gig_ga_7vtx,      "gig_ga_7vtx",      "Gigabyte Technology Co., Ltd.", "7VTXE", "1.0", "American Megatrends Inc.", "07.00T", "06/28/2002" },

/* MB = 38 */  { acer_s58m,        "acer_s58m",        "Acer", "S58M", NULL, NULL, NULL, NULL },
     
/* MB = 39 */  { ecs_k7vmm,        "ecs_k7vmm",        "ECS", "K7VMM", NULL, NULL, NULL, NULL },
               { ecs_k7vmmp,       "ecs_k7vmmp",       "ECS", "K7VMM+", "3.1", "American Megatrends Inc.", "07.00T", "04/02/01" },


               /* 
                * Unfortunately MSI changed the BIOS without increasing the BIOS release number
                * or revision date, so the right thing to do can't be detected automatically.
                *
                * The first line is for BIOS 1.2, the second one is for 1.0 or 1.1
                *
                * if it doesn't work for you out-of-the-box, either update the BIOS 
                * or use msi_6535_oldbios (--iwname option is your friend). 
                */
               { msi_6535,         "msi_6535",         "MSI", "MS-6535", "1.00", "American Megatrends Inc.", "07.00T", "12/14/01" },
               { msi_6535_oldbios, "msi_6535_oldbios", "MSI", "MS-6535", "1.00", "American Megatrends Inc.", "07.00T", "12/14/01" },

               { msi_6535_20b,     "msi_6535_20b",     "MSI", "MS-6535", "2.0B", "American Megatrends Inc.", "Version 07.00T", "04/06/02" },

               { sis_630s,         "sis_630s",         "", "SiS-630S", "", "Award Software International, Inc.", "6.00 PG", "04/08/2002" },

               { chaintech_8363,   "chaintech_8363",   /* Chaintech CT-7AJA-2 */ "", "8363-686A", "", "Award Software International, Inc.", "6.00 PG", "04/23/2001" },

/* MB = 40 */  { asus_a7v333,      "asus_a7v333",      "ASUSTeK Computer INC.", "A7V333", NULL, NULL, NULL, NULL },
     
               { asus_a7s333,      "asus_a7s333",      "ASUSTeK Computer INC.", "A7S333", "REV 1.xx", "Award Software, Inc.", "ASUS A7S333 ACPI BIOS Revision 1005 Beta 003", "10/30/2002" },

/* MB = 42 */  { asus_p4t533,      "asus_p4t533",      "ASUSTeK Computer INC.", "P4T533/R-UAY", NULL, NULL, NULL, NULL },

/* MB = 44 */  { via_epia_m,       "via_epia_m",       "VIA", "Epia M6000", NULL, NULL, NULL, NULL },
               { via_epia_m,       "via_epia_m",       "VIA", "Epia M9000", NULL, NULL, NULL, NULL },
               { via_epia_m,       "via_epia_m",       /*         M10000             */ "", "EPIA-M", "", "Award Software International, Inc.", "6.00 PG", "11/19/2002" },
               { via_epia_m,       "via_epia_m",       /*         M10000             */ "", "EPIA-M", "", "Award Software International, Inc.", "6.00 PG", "11/26/2002" },
               { via_epia_m,       "via_epia_m",       /* M9000 + M10000             */ "", "EPIA-M", "", "Award Software International, Inc.", "6.00 PG", "03/17/2003" },
               { via_epia_m,       "via_epia_m",       /*                    CL6000E */ "", "EPIA-M", "", "Award Software International, Inc.", "6.00 PG", "08/01/2003" },
               { via_epia_m,       "via_epia_m",       /*                  MII-60000 */ "", "EPIA-M", "", "Award Software International, Inc.", "6.00 PG", "11/25/2003" },
               { via_epia_m,       "via_epia_m",       /* M6000 + M10000 + MII-10000 */ "", "EPIA-M", "", "Award Software International, Inc.", "6.00 PG", "12/11/2003" },
               { via_epia_m,       "via_epia_m",       /* M6000 + M10000 + MII-10000 */ "", "EPIA-M", "", "Award Software International, Inc.", "6.00 PG", NULL },
               { via_epia_m,       "via_epia_m",       "", "C3M266", "", "Award Software International, Inc.", "6.00 PG", "03/11/2003" },

/* MB = 45 */  { gig_ga_7zxr30,    "gig_ga_7zxr30",    "Gigabyte", "GA-7ZXR30", NULL, NULL, NULL, NULL },
               { gig_ga_7zxr30,    "gig_ga_7zxr30",    "Gigabyte Technology Co., Ltd.", "7ZM",  "1.0", "American Megatrends Inc.", "62710", "10/16/2001" },
               { gig_ga_7zxr30,    "gig_ga_7zxr30",    "Gigabyte Technology Co., Ltd.", "7ZXR", "1.0", "American Megatrends Inc.", "62710", "09/10/2002" },

/* MB = 47 */  { gig_ga_7ixe4,     "gig_ga_7ixe4",     "Gigabyte Technology Co., Ltd.", "GA-7IXE4", NULL, NULL, NULL, NULL },
               { gig_ga_7ixe4,     "gig_ga_7ixe4",     "Gigabyte Technology Co., Ltd.", "7IXE",  "1.1", "American Megatrends Inc.", "62710", "06/01/2000" },
               { gig_ga_7ixe4,     "gig_ga_7ixe4",     "Gigabyte Technology Co., Ltd.", "7IXE4", "1.1", "American Megatrends Inc.", "62710", "06/01/2000" },

/* MB = 48 */  { ecs_k7sem,        "ecs_k7sem",        "Elitegroup", "K7SEM", NULL, NULL, NULL, NULL },
               { ecs_k7sem,        "ecs_k7sem",        /* Elitegroup K7SEM */ "", "SiS-730", "", "Award Software International, Inc.", "6.00 PG", "02/28/2002" },

               { shuttle_fs40,     "shuttle_fs40",     /* Shuttle FS40 */ "", "", "", "Phoenix Technologies, LTD", "6.00 PG", "04/30/2002" },
               { shuttle_fs40,     "shuttle_fs40",     /* Shuttle FS40 */ "", "FS40", "", "Phoenix Technologies, LTD", "6.00 PG", "01/21/2003" },

               { ecs_k7s6a,        "ecs_k7s6a",        /* ECS K7S6A */    "", "SiS-745", "", "Award Software International, Inc.", "6.00 PG", "02/19/2002" },
               { ecs_k7s6a,        "ecs_k7s6a",        /* Shuttle FS50 */ "", "FS50",    "", "Award Software International, Inc.", "6.00 PG", "07/05/2002" },

               { gig_sis_645,      "gig_sis_645",      "Gigabyte Technology Co., Ltd.", "SiS-645", "1.x", "Award Software International, Inc.", "6.00 PG", "02/06/2002" },
               { gig_sis_645_new,  "gig_sis_645_new",  "Gigabyte Technology Co., Ltd.", "SiS-645", "1.x", "Award Software International, Inc.", "6.00 PG", "09/16/2002" },

               { msi_ms_6243,      "msi_ms_6243",      /* MSI-Barebone Hermes 845GL   (MS-6243) */ "INTEL", "BROKDALE-G", "1.0", "American Megatrends Inc.", "V1.1", "06/10/02" },
               { msi_ms_6243,      "msi_ms_6243",      /* MSI-Barebone Hermes 845GL-P (MS-6557) */ "INTEL", "BROKDALE-G", NULL, NULL, NULL, NULL },
               { msi_ms_6243,      "msi_ms_6243",      /* MSI-Barebone Hermes 845GL   (MS-6243) */ "Micro Star", "MS-6557", "1.00", "American Megatrends Inc.", "V1.2B6", "12/03/02" },
               { msi_ms_6243,      "msi_ms_6243",      "MICRO-STAR INC.", "MS-6765", "100", "American Megatrends Inc.", "V1.1 on 07.00T", "09/02/03" },


               { asrock_k7vt2,     "asrock_k7vt2",     "ASRock",              "K7VT2", "2.00", "American Megatrends Inc.", "P1.00", "09/24/2002" },
               { asrock_k7vt2,     "asrock_k7vt2",     /* ASRock K7VT2 */ "", "K7VT2", "2.00", "American Megatrends Inc.", "P1.20", "12/10/2002" },

               { asrock_k7vt4,     "asrock_k7vt4",     /* ASRock K7VT4 */ "", "K7VT4-4X.", "3.0X", "American Megatrends Inc.", "P1.70", "10/02/2003" },

               { tosh_eq_3300m,    "tosh_eq_3300m",    /* Toshiba Equium 3300M (Typ: GVC MR824) */ "VIA Technologies Inc.", "VIA Apollo Pro 133", "1.0", "American Megatrends Inc.", "1.08.03", "07/10/01" },

               { asrock_k7vm2,     "asrock_k7vm2",     "ASRock", "K7VM2", "2.00", "American Megatrends Inc.", "P1.30", "12/13/2002" },
               { asrock_k7vm2,     "asrock_k7vm2",           "", "K7VM2", "2.00", "American Megatrends Inc.", "P1.30", "12/13/2002" },
               { asrock_k7vm2_new, "asrock_k7vm2_new", "ASRock", "K7VM2", "2.00", "American Megatrends Inc.", "P1.90", "08/28/2003" },
               { asrock_k7vm2_new, "asrock_k7vm2_new",       "", "K7VM2", "2.00", "American Megatrends Inc.", "P1.90", "08/28/2003" },
               { asrock_k7vm2_new, "asrock_k7vm2_new",       "", "K7VM2.", "3.0X", "American Megatrends Inc.", "P1.80", "07/28/2003" },
               { asrock_k7vm2_new, "asrock_k7vm2_new", "FIRST INTERNALTIONAL COMPUTER", "VB-601-V", "VER:1.3", "American Megatrends, Inc.", "0627", "03/09/99" },
               { asrock_k7vm2_new, "asrock_k7vm2_new", "MicroStar Inc.", "MS-6340", "1.0", "American Megatrends Inc.", "62710", "07/15/97" },

               { asrock_k7s8xe2,   "asrock_k7s8xe2",         "", "K7S8XE+", "1.0", "American Megatrends Inc.", "P1.40", "11/11/2003" },
               { asrock_k7s8xe2,   "asrock_k7s8xe2",         "", "K7S8XE+", "1.0", "American Megatrends Inc.", "P1.60", "01/29/2004" },
               { asrock_k7s8xe,    "asrock_k7s8xe",          "", "K7S8XE+", "1.0", "American Megatrends Inc.", "M1.44", "12/16/2003" },
               { asrock_k7s8x,     "asrock_k7s8x",           "", "K7S8X.",  "3.0", "American Megatrends Inc.", "P1.40", "10/07/2003" },
               { asrock_k7s8x2,    "asrock_k7s8x2",          "", "K7S8X.",  "3.0", "American Megatrends Inc.", "P2.20", "10/07/2003" },
               { asrock_k7s8x3,    "asrock_k7s8x3",          "", "K7S8X",   "1.0", "American Megatrends Inc.", "P1.20", "03/10/2003" },
               { asrock_k7s8x2,    "asrock_k7s8x2",          "", "K7S8X",   "1.0", "American Megatrends Inc.", "P2.20", "10/07/2003" },

               { gig_7vrx,         "gig_7vrx",         "Gigabyte Technology Co., Ltd.", "7VRX", "1.0", "American Megatrends Inc.", "07.00T", "08/23/2002" },
               { gig_7vrx,         "gig_7vrx",         "MSI", /* K7T266 Pro */ "MS-6380", "0.A", "American Megatrends Inc.", "062710", "07/15/97" },

               { asrock_k7vm4,     "asrock_k7vm4",     "ASRock", "K7VM4", "1.00", "American Megatrends Inc.", "P1.40", "07/18/2003" },
               { asrock_k7vm4,     "asrock_k7vm4",           "", "K7VM4", "1.00", "American Megatrends Inc.", "P1.70", "08/09/2003" },
               { asrock_k7vm4,     "asrock_k7vm4",           "", "K7VM4", "1.00", "American Megatrends Inc.", "P1.90", "09/19/2003" },
               { asus_k7m,         "asus_k7m",               "", "K7VM4", "1.00", "American Megatrends Inc.", "P2.30", "12/30/2003" },
               { asus_k7m,         "asus_k7m",         "ASUSteK Computer INC.", "K7M", "1.XX", "American Megatrends Inc.", "627.10", "02/29/2000" },

               { asus_p4s8l,       "asus_p4s8l",       "ASUSTeK Computer INC.", "P4S8L", "REV 1.xx", "Award Software, Inc.", "ASUS P4S8L ACPI BIOS Revision 1004", "03/27/2003" },
               { asus_p4s8l,       "asus_p4s8l",       "ASUSTeK Computer INC.", "P4S8L", "REV 1.xx", "Award Software, Inc.", "ASUS P4S8L ACPI BIOS Revision 1005", "06/11/2003" },

               { asus_a7v600,      "asus_a7v600",      "ASUSTeK Computer INC.", "A7V600", "REV 1.xx", "Award Software, Inc.", "ASUS A7V600 ACPI BIOS Revision 1006", "11/12/2003" },
               
               { asus_p4s8x,       "asus_p4s8x",       "ASUSTeK Computer INC.", "P4S8X", "REV 1.xx", "Award Software, Inc.", "ASUS P4S8X ACPI BIOS Revision 1005 Beta 009", "02/21/2003" },

               { msi_ms_6524,      "msi_ms_6524",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6524", "", "Award Software International, Inc.", "6.00 PG", "04/26/2002" },

               { pcchips_810lmr,   "pcchips_810lmr",   /* pcchips 810LMR version 7.1a */ "", "", "", "American Megatrends Inc.", "062710", "07/15/97" },

               { msi_ms_6764,      "msi_ms_6764",      "MSI", "MS-6764", "1.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },

               /* 
                * Unfortunately MSI has wrong entries in the DMI table of the following board
                * so the following FOUR boards have actually the same DMI entries, but are different boards.
                *
                * The first one has correct entries, 
                * the second and the third ones are differend bios versions with wrong DMI entries.
                * the third one is actually a MS-6764 with completely wrong DMI entries.
                *
                * the automatic selection mechanism will select the first one.
                * in the case you want to use either one of the oter two,
                * the --iwname option is your friend.
                */
               { msi_ms_6712_1,    "msi_ms_6712_1",    "MSI", "MS-6712", "1.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },
               { msi_ms_6712_2,    "msi_ms_6712_2",    "MSI", "MS-6712", "1.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },
               { msi_ms_6712_3,    "msi_ms_6712_3",    "MSI", "MS-6712", "1.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },
               { msi_ms_6764_brkn, "msi_ms_6764_brkn", "MSI", "MS-6712", "1.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },
               { msi_ms_6593,      "msi_ms_6593",      "MSI", "MS-6712", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },

               { msi_ms_6593,      "msi_ms_6593",      "MSI", "MS-6593", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },

               { msi_ms_7021,      "msi_ms_7021",      "MSI", "MS-7021", "1.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },
    
               { msi_ms_6160,      "msi_ms_6160",      /* Fujitsu-Siemens-PC */ "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6160 (i440LX)", "1.X", "Award Software International, Inc.", "4.51 PG", "10/20/98" },

               { asus_a7v8x,       "asus_a7v8x",       "ASUSTeK Computer INC.", "A7V8X",   "REV 1.xx", "Award Software, Inc.", "ASUS A7V8X ACPI BIOS Revision 1007",   "10/04/2002" },
               { asus_a7v8x,       "asus_a7v8x",       "ASUSTeK Computer INC.", "A7V8X",   "REV 1.xx", "Award Software, Inc.", "ASUS A7V8X ACPI BIOS Revision 1010",   "11/15/2002" },
               { asus_a7v8x,       "asus_a7v8x",       "ASUSTeK Computer INC.", "A7V8X",   "REV 1.xx", "Award Software, Inc.", "ASUS A7V8X ACPI BIOS Revision 1012",   "05/08/2003" },
               { asus_a7v8x,       "asus_a7v8x",       "ASUSTeK Computer INC.", "A7V8X-X", "REV 1.xx", "Award Software, Inc.", "ASUS A7V8X-X ACPI BIOS Revision 1005", "05/08/2003" },
               { asus_a7v8x,       "asus_a7v8x",       "ASUSTeK Computer INC.", "A7V8X-X", "REV 1.xx", "Award Software, Inc.", "ASUS A7V8X-X ACPI BIOS Revision 1006", "08/06/2003" },
               { asus_a7v8x,       "asus_a7v8x",       "ASUSTeK Computer INC.", "A7V8X-X", "REV 1.xx", "Award Software, Inc.", "ASUS A7V8X-X ACPI BIOS Revision 1007", "11/13/2003" },

               { abit_bp6,         "abit_bp6",         "<http://www.abit.com.tw>", "i440BX-W83977 (BP6)",   "", "Award Software International, Inc.", "4.51 PG", "04/20/00"   },
               { abit_bp6,         "abit_bp6",         "<http://www.abit.com.tw>", "i440BX-W977 (BX6R2.0)", "", "Award Software International, Inc.", "4.51 PG", "04/26/2000" },

               { abit_bh6,         "abit_bh6",         "<http://www.abit.com.tw>", "i440BX-W83977 (BH6)",   "", "Award Software International, Inc.", "4.51 PG", "06/28/2000" },
               
               { aopen_ax4pe_max,  "aopen_ax4pe_max",  "AOpen", "AX4PE Max", "", "Phoenix/Award Technologies, LTD", "6.00 PG", "12/23/2002" },             
               
               { msi_ms_6570,      "msi_ms_6570",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6570", "", "Phoenix Technologies, LTD", "6.00 PG", "04/18/2003" },
               { msi_ms_6570,      "msi_ms_6570",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6570", "", "Phoenix Technologies, LTD", "6.00 PG", "08/12/2003" },
               { msi_ms_6570,      "msi_ms_6570",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6570", "", "Phoenix Technologies, LTD", "6.00 PG", "11/07/2003" },
               { msi_ms_6570,      "msi_ms_6570",      "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6570", "", "Phoenix Technologies, LTD", "6.00 PG", "03/29/2004" },
               { msi_ms_6570,      "msi_ms_6570",      /* EPoX EP-8RDA3+            */ "", "nVidia-nForce", "", "Phoenix Technologies, LTD", "6.00 PG", "10/17/2003" },
               { msi_ms_6570,      "msi_ms_6570",      /* Biostar M7NCG rev. 1.4    */ "", "nVidia-nForce", "", "Phoenix Technologies, LTD", "6.00 PG", "10/17/2003" },
               { msi_ms_6570,      "msi_ms_6570",      /* Chaintech 7NIF2 (nforce2) */ "", "nVidia-nForce", "", "Phoenix Technologies, LTD", "6.00 PG", "04/18/2003" },
               { chain_7nif2,      "chain_7nif2",      /* Chaintech 7NIF2 (nforce2) */ "", "nVidia-nForce", "", "Phoenix Technologies, LTD", "6.00 PG", "09/05/2003" },
               { chain_7nif2,      "chain_7nif2",      /* ECS Elitegroup N2U400-A   */ "", "nVidia-nForce", "", "Phoenix Technologies, LTD", "6.00 PG", "10/02/2003" },

               { msi_ms_6570,      "msi_ms_6570",      /* Shuttle XPC SN41G2 */ "Shuttle Inc", "FN41",   "", "Phoenix Technologies, LTD", "6.00 PG", "06/17/2003" },
               { shttl_fn41,       "shttl_fn41",       /* Shuttle XPC SN41G2 */ "Shuttle Inc", "FN41",   "", "Phoenix Technologies, LTD", "6.00 PG", "10/03/2003" },
               { shttl_fn41,       "shttl_fn41",                                "Shuttle Inc", "FN41",   "", "Phoenix Technologies, LTD", "6.00 PG", "01/15/2003" },
               { shttl_fn41sp,     "shttl_fn41sp",                              "Shuttle Inc", "FN41SP", "", "Phoenix Technologies, LTD", "6.00 PG", "10/03/2003" },
               { shttl_fn41,       "shttl_fn41",       "ASUSTeK Computer INC.", "A7N8X-E",  "REV 2.xx", "Phoenix Technologies, LTD", "ASUS A7N8X-E Deluxe ACPI BIOS Rev 1008", "12/05/2003" },
               { shttl_fn41,       "shttl_fn41",       "ASUSTeK Computer INC.", "A7N8X-E",  "REV 2.xx", "Phoenix Technologies, LTD", "ASUS A7N8X-E Deluxe ACPI BIOS Rev 1010", "03/16/2004" },
               { shttl_fn41,       "shttl_fn41",       "ASUSTeK Computer INC.", "A7N8X",    "REV 1.xx", "Phoenix Technologies, LTD", "ASUS A7N8X Deluxe ACPI BIOS Rev 1002.A", "02/25/2003" },
               { shttl_fn41,       "shttl_fn41",       "ASUSTeK Computer INC.", "A7N8X2.0", "REV 2.xx", "Phoenix Technologies, LTD", "ASUS A7N8X2.0 Deluxe ACPI BIOS Rev 1007", "10/06/2003" },

               { ecs_k7som_p,      "ecs_k7som_p",      "ECS", "K7SOM+", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },

               { asus_a4s333,      "asus_a4s333",      "ASUSTeK Computer INC.", "P4S333VF", "REV 1.xx", "Award Software, Inc.", "ASUS P4S333-VF ACPI BIOS Revision 1005", "03/05/2002" },

               { dell_optiplex_dxa,"dell_optiplex_dxa",/* Dell Optiplex DXa */ NULL, NULL, NULL, "Dell Computer Corporation", "A10", "06/23/00" },
               { dell_optiplex_dxa,"dell_optiplex_dxa",/* Dell Optiplex Gn  */ NULL, NULL, NULL, "Dell Computer Corporation", "A12", "06/23/00" },
               { dell_optiplex_dxa,"dell_optiplex_dxa",/* Dell Optiplex G1  */ NULL, NULL, NULL, "Dell Computer Corporation", "A10", "04/29/03" },
               { dell_optiplex_dxa,"dell_optiplex_dxa","Dell Computer Corporation", "Dimension 4300", NULL, "Dell Computer Corporation", "A05", "03/28/2002" },
               { dell_optiplex_dxa,"dell_optiplex_dxa","Dell Computer Corporation", "OptiPlex GX1 333L+", "", "Dell Computer Corporation", "A10", "08/01/01" },

               { asus_p4bgl,       "asus_p4bgl",       "ASUSTeK Computer INC.", "P4BGL-MX", "REV 1.xx", "Award Software, Inc.", "ASUS P4BGL-MX ACPI BIOS Revision 1006", "02/27/2003" },
               { asus_p4bgl,       "asus_p4bgl",       "ASUSTeK Computer INC.", "P4GE-FSC", "REV 1.xx", "Award Software, Inc.", "ASUS P4GE-FSC ACPI BIOS Revision 1004", "07/14/2003" },

               { asus_p4b266,      "asus_p4b266",      "ASUSTeK Computer INC.", "P4B266",   "REV 2.xx", "Award Software, Inc.", "ASUS P4B266 ACPI BIOS Revision 1010", "08/06/2002" },
               { asus_p4b266_m,    "asus_p4b266_m",    "ASUSTeK Computer INC.", "P4B266-M", "REV 1.xx", "Award Software, Inc.", "ASUS P4B266-M ACPI BIOS Revision 1008 Beta 001", "04/03/2003" },
               { asus_p4b266_e,    "asus_p4b266_e",    "ASUSTeK Computer INC.", "P4B266-E", "REV 1.xx", "Award Software, Inc.", "ASUS P4B266-E ACPI BIOS Revision 1003", "08/19/2002" },
               
/* MB = 41 */  { msi_ms_6380e,     "msi_ms_6380e",     "MSI", "MS-6380E", NULL, NULL, NULL, NULL },
               { msi_ms_6380e,     "msi_ms_6380e",     "MSI", "MS-6590", "2.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },
               { msi_ms_6590,      "msi_ms_6590",      "MSI", "MS-6590", "1.0", "American Megatrends Inc.", "Version 07.00T", "04/02/01" },

               { chaintech_6aja4,  "chaintech_6aja4",  /* Chaintech CT-6AJA4/G100 */ "", "694X-686A", "", "Award Software International, Inc.", "6.00 PG", "05/17/2001" },

               { dfi_ak70,         "dfi_ak70",         /* DFI AK70 Rev. A+ */ "", "AMD-75X-W977", "", "Award Software International, Inc.", "6.00 PG", "05/25/2000" },
               { dfi_ak70,         "dfi_ak70",         "Gigabyte Technology Co.,LTD.", "GA-7IX", "1.X", "Award Software International, Inc.", "6.00 PG", "04/25/00" },
               { dfi_ak70,         "dfi_ak70",         "", "HOT_AI61 AMD-75X-W977", "6A6S2H29", "Award Software International, Inc.", "6.00 PG", "05/03/01" },
               
               { asrock_g_pro,     "asrock_g_pro",     "ASRock", "G Pro / GV Pro", "1.00", "American Megatrends Inc.", "P1.40", "04/07/2003" },
               { asrock_g_pro_new, "asrock_g_pro_new", "ASRock", "G Pro / GV Pro", "1.00", "American Megatrends Inc.", "P1.00", "10/09/2002" },
               
               { asus_p4s533,      "asus_p4s533",      "ASUSTeK Computer INC.", "P4S533MX", "REV 1.xx", "Award Software, Inc.", "ASUS P4S533-MX ACPI BIOS Revision 1004", "01/24/2003" },
               { asus_p4s533,      "asus_p4s533",      "ASUSTeK Computer INC.", "P4S533MX", "REV 1.xx", "Award Software, Inc.", "ASUS P4S533-MX ACPI BIOS Revision 1005", "04/18/2003" },
               { asus_p4s533,      "asus_p4s533",      "ASUSTeK Computer INC.", "P4S533MX", "REV 1.xx", "Award Software, Inc.", "ASUS P4S533-MX ACPI BIOS Revision 1006", "06/11/2003" },
               { asus_p4s533,      "asus_p4s533",      "ASUSTeK Computer INC.", "P4S533MX", "REV 1.xx", "Award Software, Inc.", NULL, NULL },
               { asus_p4s533,      "asus_p4s533",      "ASUSTeK Computer INC.", "P4SP-MX",  "REV 1.xx", "Award Software, Inc.", "ASUS P4SP-MX ACPI BIOS Revision 1001",   "12/31/2003" },


               { asus_p4s533_x,    "asus_p4s533_x",    "ASUSTeK Computer INC.", "P4S533-X", "REV 1.xx", "Award Software, Inc.", "ASUS P4S533-X ACPI BIOS Revision 1004", "02/24/2003" },

               { asrock_m266a,     "asrock_m266a",     /* ASRock M266a */ "", "M266", "2.00", "American Megatrends Inc.", "P1.20", "02/26/2003" },
               { asrock_m266a,     "asrock_m266a",     /* ASRock M266a */ "", "M266", "2.00", "American Megatrends Inc.", "P1.60", "07/18/2003" },
               { asrock_m266a,     "asrock_m266a",     /* ASRock M266a */ "", "M266", "2.00", "American Megatrends Inc.", "P1.90", "09/15/2003" },
               { asrock_m266a,     "asrock_m266a",     /* MSI 694T Pro */ "MSI", "MS-6309", "5.0", "American Megatrends Inc.", "062710", "07/15/97" },

               { msi_ms_6398e,     "msi_ms_6398e",     "Micro-Star Inc.", "MS-6398E", "1.0", "American Megatrends Inc.", "07.00T", "04/02/01" },

               { shuttle_mv43vn,   "shuttle_mv43vn",   "Shuttle Inc", "MV43V", "", "American Megatrends Inc.", "07.00T", "04/02/01" },

               { gigabyte_5aa,     "gigabyte_5aa",     "GIGABYTE", "5AA", "", "...waiting for strings...", "", "" },

               { asus_p4b533,      "asus_p4b533",      "ASUSTeK Computer INC.", "P4B533",   "REV 1.xx", "Award Software, Inc.", "ASUS P4B533 ACPI BIOS Revision 1011", "09/27/2002" },
               { asus_p4b533_e,    "asus_p4b533_e",    "ASUSTeK Computer INC.", "P4B533-E", "REV 1.xx", "Award Software, Inc.", "ASUS P4B533-E ACPI BIOS Revision 1012 Beta 004", "12/11/2002" },

               { msi_ms_5169,      "msi_ms_5169",      "MSI INC.", "ALADDIN5", "VER:1.0", "American Megatrends Inc. AMI", NULL, NULL },

               { aopen_ax45_4d,    "aopen_ax45_4d",    "AOpen", "AX45-4D Max", "918A410001", NULL, NULL, NULL },

               { asus_p4p800,      "asus_p4p800",      "ASUSTeK Computer Inc.", "P4P800",    "Rev 1.xx", "American Megatrends Inc.", "080009",   "11/06/2003" },
               { asus_p4p800,      "asus_p4p800",      "ASUSTeK Computer Inc.", "P4P800",    "Rev 1.xx", "American Megatrends Inc.", "080009",   "12/12/2003" },
               { asus_p4c800,      "asus_p4c800",      "ASUSTeK Computer Inc.", "P4C800",    "Rev 1.xx", "American Megatrends Inc.", "1016.001", "02/23/2004" },
               { asus_p4c800_e,    "asus_p4c800_e",    "ASUSTeK Computer Inc.", "P4C800-E",  "Rev 1.xx", "American Megatrends Inc.", "080009",   "11/21/2003" },
               { asus_p4p800_vm,   "asus_p4p800_vm",   "ASUSTeK Computer INC.", "P4P800-VM", "Rev 1.xx", "American Megatrends Inc.", "1007.001", "07/01/2003" },
               { asus_p4p800_vm_2, "asus_p4p800_vm_2", "ASUSTeK Computer INC.", "P4P800-VM", "Rev 1.xx", "American Megatrends Inc.", "1008.008", "10/31/2003" },
               { asus_p4p800_vm_2, "asus_p4p800_vm_2", "ASUSTeK Computer INC.", "P4P800-VM", "Rev 1.xx", "American Megatrends Inc.", "1012.002", "03/22/2004" },

               { aopen_mk73le_n,   "aopen_mk73le_n",   /* Aopen MK73LE-N */ "", "6", "", "Phoenix Technologies, LTD", "6.00 PG", "02/19/2002" },

               { ibm_pc_300pl,     "ibm_pc_300pl",     /* IBM PC 300PL */ "IBM", NULL, NULL, "IBM", "PMKT16AUS", "02/24/2000" },

               { springdale,       "springdale",       "INTEL", "SpringDale-G", "A1", "American Megatrends Inc.", "080009", "10/20/2003" },
               { springdale,       "springdale",       /* Soltek SL-B8E-F */ "INTEL", "SpringDale-G", "A1", "American Megatrends Inc.", "080009", "12/18/2003" },

               { gig_ga_8simlh,    "gig_ga_8simlh",    "Gigabyte Technology Co., Ltd.", "GA-8SIMLH", "1.x", "Award Software International, Inc.", "6.00 PG", "11/19/2003" },

               { sis_730,          "sis_730",          "", "SiS-730", "", "Award Software International, Inc.", "6.00 PG", "09/28/2001" },
               { sis_730,          "sis_730",          "Gigabyte Technology Co., Ltd.", "GA-8SQ800", "1.x", "Award Software International, Inc.", "F5", "06/28/2003" },

               { asus_a7n8x_vm400, "asus_a7n8x_vm400", "ASUSTeK Computer INC.", "A7NVM400", "Rev 2.xx", "American Megatrends Inc.", "1.00", "11/25/2003" },
               { asus_a7n8x_vm400, "asus_a7n8x_vm400", "ASUSTeK Computer INC.", "A7NVM400", "Rev 2.xx", "American Megatrends Inc.", "2.00", "03/24/2004" },
               { asus_a7n8x_vm400, "asus_a7n8x_vm400", "ASUSTeK Computer INC.", "P4R8L", "1.xx", "American Megatrends Inc.", "1002.003", "03/10/2004" },

               { gig_ga_7vt600,    "gig_ga_7vt600",    /* Bios F4 */ "Gigabyte Technology Co., Ltd.", "GA-7VT600 1394", "15X", "Award Software International, Inc.", "F3", "09/24/2003" },
               { gig_ga_7vt600,    "gig_ga_7vt600",    /* Elitegroup L7VTA2 */ "", "KT400-8235", "", "Phoenix Technologies, LTD", "6.00 PG", "06/26/2003" },

               { shuttle_fx41,     "shuttle_fx41",     "Shuttle Inc", "Fx41", "", "Phoenix Technologies, LTD", "6.00 PG", "09/08/2003" },

               /* 
                *  Unfortunately Shuttle XPC SK43G has no valid DMI entries
                *  (at least bios rev. FX43S40R, 07/11/2003). Thus, the command line option
                *
                *        --iwname shuttle_sk43g
                *
                *  is your friend.
                */
               { shuttle_sk43g,    "shuttle_sk43g",    "Shuttle Inc", "SK43G", "", "", "", "" },
               { shuttle_sk43g,    "shuttle_sk43g",    "ASUS", "A7V8X-MX SE", "1.03", "Phoenix Technologies, LTD", "ASUS A7V8X-MX SE ACPI BIOS Revision 1003", "03/12/2004" },
               { shuttle_sk43g,    "shuttle_sk43g",    "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6734", "", "Phoenix Technologies, LTD", "6.00 PG", "01/16/2004" },

               { asus_k8vsedx,     "asus_k8vsedx",     "ASUSTeK Computer Inc.", "K8VSEDX", "Rev 1.xx", "American Megatrends Inc.", "1001.005", "02/09/2004" },
               { asus_k8vsedx,     "asus_k8vsedx",     "ASUSTeK Computer Inc.", "K8VSEDX", "Rev 2.00", "American Megatrends Inc.", "1003.002", "05/18/2004" },
               { asus_k8vsedx,     "asus_k8vsedx",     "ASUSTek Computer Inc.", "K8V",     "Rev 1.xx", "American Megatrends Inc.", "1005.011", "01/15/2004" },

               { asus_k8v,         "asus_k8v",         "ASUSTeK Computer Inc.", "K8V",     "Rev 1.xx", "American Megatrends Inc.", "1003.027", "11/27/2003" },

               { asus_p4xp_x,      "asus_p4xp_x",      "ASUSTeK Computer INC.", "P4XP-X", "REV 1.xx", "Award Software, Inc.", "ASUS P4XP-X ACPI BIOS Revision 1003", "01/27/2003" },

               { msi_ms_6580,      "msi_ms_6580",      "MICRO-STAR INC.", /* MSI 845GE Max */ "MS-6580", "20A", "American Megatrends Inc.", "V3.9  0", "09/09/2003" },

               { aopen_ax37,       "aopen_ax37",       /* AOpen AX37 Pro/Plus */ "", "07", "9189910203", "Award Software International, Inc.", "6.00 PG", "01/29/2001" },
               { aopen_ax37,       "aopen_ax37",       "ECS Elitegroup", "L7VMM2", "", "Phoenix Technologies, LTD", "6.00 PG", "09/09/2003" },
               { aopen_ax37,       "aopen_ax37",       /* Elito-Epox 8K5A2+ */ "", "VT8367-8235", "", "Award Software International, Inc.", "6.00 PG", "04/07/2003" },

               { asrock_k7s41gx,   "asrock_k7s41gx",   /* Asrock */ "", "K7S41GX", "1.0", "American Megatrends Inc.", "P1.50", "04/09/2004" },

               { abit_kg7,         "abit_kg7",         "ABIT <<http://www.abit.com.tw>>", "761-686B(KG7)", "Rev:1.0", "Award Software International, Inc.", "6.00 PG", "03/06/2003" },

               { msi_ms_6760,      "msi_ms_6760",      /* MSI MEGA PC 651 */ "MICRO-STAR INTERNATIONAL CO., LTD", "MS-6760", "", "Phoenix Technologies, LTD", "6.00 PG", "07/02/2003" },

               { msi_ms_6728,      "msi_ms_6728",      /* MSI Neo2 FIS2R */ "MICRO-STAR INC.", "MS-6728", "100", "American Megatrends Inc.", "V2.1", "02/19/2004" },

               { asrock_ge_pro,    "asrock_ge_pro",    /* Asrock */ "", "GE PRO(-M)", ".00", "American Megatrends Inc.", "P2.40", "11/04/2003" },

               { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};

int set_biosinfo_defaults(struct biosinfo *b) {
     if (b) {
          memset(b, 0, sizeof(struct biosinfo));
          b->nr_stat = 1;
          b->addr_stat = 0xFF;
          NOBCD(b);
          return 1;
     }
     else return 0;
}

int get_bios_info_by_dmi(struct biosinfo *b) {

     int i;

     /* Set defaults... */
     if (!set_biosinfo_defaults(b))
          return 0;

     if (!__dmi_probe())
          return -1;

     /* Check if we know the mainboard... */
     for(i=0; boards[i].infowriter; i++) {
          if (boards[i].vendor      && (!board_vendor()  || strcmp(boards[i].vendor,      board_vendor() )))
               continue;
          if (boards[i].type        && (!board_type()    || strcmp(boards[i].type,        board_type()   )))
               continue;
          if (boards[i].version     && (!board_version() || strcmp(boards[i].version,     board_version())))
               continue;
          if (boards[i].biosvendor  && (!bios_vendor()   || strcmp(boards[i].biosvendor,  bios_vendor()  )))
               continue;
          if (boards[i].biosversion && (!bios_version()  || strcmp(boards[i].biosversion, bios_version() )))
               continue;
          if (boards[i].biosrelease && (!bios_release()  || strcmp(boards[i].biosrelease, bios_release() )))
               continue;

          /* This is our board... */
          boards[i].infowriter(b);
          return 1;
     }
     return 0;
}



int get_bios_info_by_iw(struct biosinfo *b, const char * iw_name) {

     int i;

     /* Set defaults... */
     if (!set_biosinfo_defaults(b))
          return 0;

     /* Check if we know that infowriter... */
     for(i=0; boards[i].infowriter; i++) {
          if ( strcmp( boards[i].iw_name, iw_name ) )
               continue;

          /* This is our board... */
          boards[i].infowriter(b);
          return 1;
     }
     return 0;
}


