/* $Id: nws_daemon.c,v 1.5 2005/07/08 17:41:17 graziano Exp $ */

#include "config_nws.h"

#include <unistd.h>

#include "diagnostic.h"
#include "osutil.h"
#include "strutil.h"
#include "nws_daemon.h"
#include "host_protocol.h"

/* global lock */
static void *lock = NULL;

/* this is an opaque structure that shouldn't be modified directly */
typedef struct {
	HostTypes type;
	short verbose;
	struct host_cookie me;
	struct host_cookie mem;
	struct host_cookie ns;
	FILE *err;
	FILE *log;
	char pidFile[255 + 1];
	char switches[63 + 1];
	IPAddress addresses[MAX_ADDRESSES];
	unsigned int addressesCount;
	char password[63 + 1];
	int (*exitFunction)(void);
} nws_daemon;

static nws_daemon daemons[10];
static int upTo = 0;


int
InitializeDaemonStructure(	HostTypes type,
				const char *switches,
				int (*exitFunction)(void)) {
	nws_daemon *tmp;
	char *name;
	int ret;

	GetNWSLock(&lock);
	tmp = &daemons[upTo];

	tmp->type = type;
	tmp->exitFunction = exitFunction;
	tmp->verbose = 2;
	tmp->err = tmp->log = stdout;
	tmp->pidFile[0] = '\0';

	/* getopt switches */
	if (switches && strlen(switches) > 0) {
		sprintf(tmp->switches, "%s%s", switches, DAEMON_SWITCHES);
	} else {
		sprintf(tmp->switches, "%s", DAEMON_SWITCHES);
	}

	/* addresses is a pain */
	tmp->addressesCount = IPAddressValues(MyMachineName(),
			&tmp->addresses[0],
			MAX_ADDRESSES);
	if (tmp->addressesCount == 0) {
		ERROR1("Couldn't resolve my name (%s)\n", MyMachineName());
	}

	/* let's get defaults values for me, nameserver and memory */
	Host2Cookie(MyMachineName(), DefaultHostPort(type), &tmp->me);
	
	name = NWSAPI_EnvironmentValue_r("NAME_SERVER", MyMachineName());
	if (name) {
		SAFESTRCPY(tmp->ns.name, name);
		FREE(name);
	}
	Host2Cookie(tmp->ns.name, DefaultHostPort(NAME_SERVER_HOST), &tmp->ns);

	name = NWSAPI_EnvironmentValue_r("SENSOR_MEMORY", MyMachineName());
	if (name) {
		SAFESTRCPY(tmp->mem.name, name);
		FREE(name);
	}
	Host2Cookie(tmp->mem.name, DefaultHostPort(MEMORY_HOST), &tmp->mem);
	upTo++;
	ret = upTo - 1;
	ReleaseNWSLock(&lock);
	
	return (ret);
}

int
DaemonSwitch(	int opt,
		const char *arg,
		int ind) {
	IPAddress address;
	int i, ret;
	const char *c;
	char tmpIP[127+1];
	nws_daemon *state;

	if (ind < 0 || ind >= upTo) {
		ERROR("DaemonSwitch: invalid parameter!\n");
		return 0;
	}

	GetNWSLock(&lock);
	state = &daemons[ind];
	ret = 1;

	switch(opt) {
	case 'n':
		/* let's check the user knows what is doing */
		if (IPAddressValue(arg, &address)) {
			/* save the inet addresses */
			state->addressesCount += IPAddressValues(arg, &state->addresses[state->addressesCount], MAX_ADDRESSES - state->addressesCount);
		} else {
			ERROR1("Unable to resolve '%s': I'll do what you said but expect problems!\n", arg);
		}

		/* overrride the name of the machine: we hope the
		 * user knows what is doing! */
		SAFESTRCPY(state->me.name, arg);

		break;

	case 'a':
		/* let's add this IPs to the list of my addresses */
		for (c = arg; GETTOK(tmpIP, c, ",", &c); ) {
			i = IPAddressValues(tmpIP, &state->addresses[state->addressesCount], MAX_ADDRESSES - state->addressesCount);
			if (i == 0) {
				ERROR1("Unable to convert '%s' into an IP address\n", tmpIP);
			} else {
				state->addressesCount += i;
				/* I want to have these addresses
				 * as first */
				for (; i > 0; i--) {
					address = state->addresses[i - 1];
					state->addresses[i - 1] = state->addresses[state->addressesCount - i];
					state->addresses[state->addressesCount - i] = address;
				}
			}
		}
		break;

	case 'e':
		/* open the error file */
		state->err = fopen(arg, "w");
		if (state->err == NULL) {
			printf("Couldn't open %s!\n", arg);
			exit(1);
		}
		break;

	case 'l':
		/* open the log file */
		state->log = fopen(arg, "w");
		if (state->log == NULL) {
			printf("Couldn't open %s!\n", arg);
			exit(1);
		}
		break;

	case 'i':
		/* let's save our pid */
		SAFESTRCPY(state->pidFile, arg);
		break;

	case 'M':
		Host2Cookie(arg, DefaultHostPort(MEMORY_HOST), &state->mem);
		break;

	case 'N':
		Host2Cookie(arg, DefaultHostPort(NAME_SERVER_HOST), &state->ns);
		break;

	case 'P':
		fprintf(stdout, "Password? ");
		fscanf(stdin, "%s", state->password);
		break;

	case 'p':
		state->me.port = strtol(arg, NULL, 10);
		break;

	case 'v':
		state->verbose = (unsigned short)strtol(arg,NULL,10);
		break;

	case 'V':
		switch (state->type) {
		case MEMORY_HOST:
			printf("nws_memory");
			break;
		case NAME_SERVER_HOST:
			printf("nws_nameserver");
			break;
		case SENSOR_HOST:
			printf("nws_sensor");
			break;
		case PROXY_HOST:
			printf("nws_proxy");
			break;
		default:
			printf("unknown host");
			break;
		}
		printf(" for NWS version %s", VERSION);
#ifdef HAVE_PTHREAD_H
		printf(", with thread support");
#endif
#ifdef WITH_DEBUG
		printf(", with debug support");
#endif
#ifdef EXPERIMENTAL
		printf(", with experimental bits");
#endif
		printf("\n\n");
		exit(0);
		break;

	default:
		ret = 0;
		break;
	}
	ReleaseNWSLock(&lock);

	return ret;
}

const char *
DaemonAllSwitches(int ind) {
	if (ind < 0 || ind >= upTo) {
		FAIL("DaemonAllSwitches: invalid parameter\n");
	}

	return daemons[ind].switches;
}


void
DaemonUsage(	const char *exe,
		const char *name) {
	if (exe == NULL || name == NULL) {
		return;
	}
	printf("\nUsage: %s [OPTIONS]\n", exe);
	printf("%s for the Network Weather Service\n", name);
	printf("Report bugs to <nws@nws.cs.ucsb.edu>.\n");
	printf("\nOPTIONS can be:\n");
	printf("\t-M memory           use memory to store measurements\n");
	printf("\t-N nameserver       register with this nameserver\n");
	printf("\t-e filename         write error messages to filename\n");
	printf("\t-l filename         write info/debug messages to filename\n");        printf("\t-i filename         write pid to filename\n");
	printf("\t-p port             bind to port instead of the default\n");
	printf("\t-a address          use this address as mine (ie multi-homed hosts)\n");
	printf("\t-n name             use name as my hostname\n");
	printf("\t-v level            verbose level (up to 5)\n");
	printf("\t-V                  print version\n");
	printf("\t-h                  this help screen\n");
}

static int
MyExit(void) {
	int i;

	GetNWSLock(&lock);
	for (i = 0; i < upTo; i++) {
		daemons[i].exitFunction();

		if (daemons[i].pidFile[0] != '\0') {
			unlink(daemons[i].pidFile);
		}
	}

	/* only one need to run */
	upTo = 0;
	ReleaseNWSLock(&lock);

	return 1;
}

int
InitializeDaemon(	short local,
			int check_ns,
			int check_mem,
			int ind) {
	FILE *pidFile;
	nws_daemon *state;
	HostInfo info;

	if (ind < 0 || ind >= upTo) {
		FAIL("InitializeDaemon: invalid parameter\n");
	}

	GetNWSLock(&lock);
	state = &daemons[ind];

	/* set the verbosity level first */
	SetDiagnosticLevel(state->verbose, state->err, state->log);

	/* let's do basic checkings on remote servers */
	if (check_mem && state->mem.name[0] != '\0') {
		/* let's check memory first */
		if (!ConnectToHost(&state->mem, NULL)) {
			WARN1("Cannot talk to %s\n", HostCImage(&state->mem));
		} else if (!GetHostInfo(&state->mem, &info, -1)) {
			WARN1("Cannot %s status.\n", HostCImage(&state->mem));
		} else if (info.hostType != MEMORY_HOST) {
			WARN1("%s is not a memory!\n", HostCImage(&state->mem));
		}
	}
	if (check_ns && state->ns.name[0] != '\0') {
		/* now let's check the nameserver */
		if (!ConnectToHost(&state->ns, NULL)) {
			WARN1("Cannot talk to %s\n", HostCImage(&state->ns));
		} else if (!GetHostInfo(&state->ns, &info, -1)) {
			WARN1("Cannot %s status.\n", HostCImage(&state->ns));
		} else if (info.hostType != NAME_SERVER_HOST) {
			WARN1("%s is not a nameserver!\n", HostCImage(&state->ns));
		}
	}

	/* establish the server port */
	if (local) {
		if (!EstablishAnEar(state->me.port, state->me.port, NULL, NULL)) {
			ReleaseNWSLock(&lock);
			FAIL("InitializeDaemon: failed to establish an ear!\n");
		}

		/* now let's set the nameserver */
		if (state->ns.name[0] == '\0') {
			WARN("InitializeDaemon: no nameserver specified?\n");
		} else {
			NWSAPI_UseNameServer(NWSAPI_MakeHostSpec(state->ns.name, state->ns.port));
		}
	} else {
		if (!EstablishHost(HostCImage(&state->me),
				state->type,
				state->addresses,
				state->addressesCount,
				state->me.port,
				state->password,
				&state->ns,
				&state->mem,
				MyExit)) {
			ReleaseNWSLock(&lock);
			FAIL("InitializeDaemon: Unable to establish host: port already in use?\n");
		}
	}

	/* write the pid file after having established the server port */
	if (state->pidFile[0] != '\0') {
		pidFile =  fopen(state->pidFile, "w");
		if (!pidFile) {
			ABORT1("Can't write pidfile %s\n", pidFile);
		}
		fprintf(pidFile, "%d", (int)getpid());
		fclose(pidFile);
	}
	ReleaseNWSLock(&lock);

	/* we are a daemon: let's close stdin */
	fclose(stdin);

	return 1;
}
