package edu.ucsb.cs.nws.nwsprotocol;

import java.io.*;
import java.net.*;


/**
 * The NwsProxy class implements the protocol specific to NWS
 * Proxy -- forecasting requests.
 */


public class NwsProxy extends NwsHost {

  protected static final int GET_FORECASTS = NwsMessage.PROXY_FIRST_MESSAGE;
  protected static final int PROXY_LAST_MESSAGE = NwsMessage.PROXY_LAST_MESSAGE;
  protected static final int GOT_FORECASTS = GET_FORECASTS + 1;
  protected static final int START_FETCHING = GOT_FORECASTS + 1;
  protected static final int START_ACTIVITY_FETCHING = START_FETCHING + 1;
  protected static final int PROXY_ACK = PROXY_LAST_MESSAGE - 1;
  protected static final int PROXY_FAILED = PROXY_LAST_MESSAGE;

  private String my_resource;
  private int howManyHost;


  /** See the constructor for NwsHost. */
  	public NwsProxy(String hostName, int hostPort, boolean keepOpen)
	{
    		super(hostName, hostPort, keepOpen);
  	}


  /** See the constructor for NwsHost. */
  	public NwsProxy(String hostName, int hostPort)
	{
    		super(hostName, hostPort);
  	}


  /** See the constructor for NwsHost. */
  	public NwsProxy(Socket s) {
    		super(s);
  	}



  /**
   * Requests that the proxy generate a series of forecast collections
   */
  	public NwsForecastAPI.ForecastCollection[]
  	getAllForecasts(String resource, String opts, String[] hosts) throws Exception
	{
		byte[][] allBytes = {null, null};
		int i;
		NwsForecastAPI.ForecastCollection[] returnValue;


		if(resource == null) {
			System.err.println("NwsProxy.getAllForecasts: null " +
					   "resource parameter");
			return null;
		}
		my_resource = resource;
		if(hosts == null) {
			System.err.println("NwsProxy.getAllForecasts: null " +
					   "hosts parameter");
			return null;
		}

		/* Build the data payload - first the resource and a NULL,
		   then the options and a NULL, and finally each host
		   separated by a null.
		*/
		allBytes[0] = new CString(resource).toBytes();

		if(opts != null) {
			allBytes[1] = new CString(opts).toBytes();
		}
		else {
			allBytes[1] = new CString(1,"\0").toBytes();
		}
		allBytes[0] = NwsMessage.concatenateBytes(allBytes);

		for(i=0; i < hosts.length; i++) {
			allBytes[1] = new CString(hosts[i]).toBytes();
			allBytes[0] = NwsMessage.concatenateBytes(allBytes);
		}
		howManyHost = hosts.length;
	
		messagePrelude();
		NwsMessage.send(connection, GET_FORECASTS, allBytes[0]);
		returnValue = (NwsForecastAPI.ForecastCollection [])NwsMessage.receive(connection, this);
		messagePostlude();

		return returnValue;
  	}


  /** See NwsMessage. */
  public Object receive(int message,
                        DataInputStream stream,
                        int dataLength) throws Exception {
	boolean interMachine;

	if (message == GOT_FORECASTS) {
		int howMany, z;
		int[] list;
		NwsForecastAPI.ForecastCollection[] coll;

		/* we need an empty ForecastCollection to fill the matrix */
		NwsForecastAPI.Measurement meas = new NwsForecastAPI.Measurement(0.0, 0.0);
		NwsForecastAPI.Forecast[] forecast = new NwsForecastAPI.Forecast[2];
		forecast[0] = new NwsForecastAPI.Forecast(0.0, 0.0, 0.0, -1);
		forecast[1] = new NwsForecastAPI.Forecast(0.0, 0.0, 0.0, -1);

		/* first we receive the # of hosts the proxy has */
		howMany = stream.readInt();
		list = new int[howMany];

		/* then we got the index in respect to the original
		 * listing submitted to the proxy */
		 for (int i = 0; i < howMany; i++) {
		 	list[i] = stream.readInt();
		 }

		 /* and now we can collect the forecasts */
		 if (my_resource.equals(new String("bandwidthTcp")) || my_resource.equals(new String("latencyTcp"))) {
		 	interMachine = true;
			coll = new NwsForecastAPI.ForecastCollection[howManyHost * howManyHost];
			z = howManyHost * howManyHost;
		} else {
		 	interMachine = false;
			z = howManyHost;
			coll = new NwsForecastAPI.ForecastCollection[howManyHost];
		}
		for (int i = 0; i < z; i++) {
			coll[i] = new NwsForecastAPI.ForecastCollection(meas, forecast);
		}

		for(int i = 0; i < howMany; i++) {
			/* j is destination only on a single host
			 * resource */
			if (!interMachine && i > 0) {
				break;
			}

			for(int j = 0; j < howMany; j++) {
				int ciccio;

				/* let's compute the right index */
				if (interMachine && i == j) {
					continue;
				} else if (!interMachine) {
					ciccio = list[j];
				} else {
					ciccio = list[j] + list[i]*howManyHost;
				}

				coll[ciccio] = new NwsForecastAPI.ForecastCollection(stream);
			}
		}
		return coll;
	} else if (message == PROXY_FAILED) {
		System.err.print("Proxy sent NACK!");
		return null;
	} else {
		return super.receive(message, stream, dataLength);
	}
}


  /**
   * The main() method for this class is a small test program that takes a
   */
  public static void main(String[] args) {
	String[] hosts;
	final int MAE_FORECAST =
		NwsForecastAPI.ForecastCollection.MAE_FORECAST;
    	final int MSE_FORECAST = NwsForecastAPI.ForecastCollection.MSE_FORECAST;

	if(args.length <= 2) {
		System.err.print("Usage: NwsProxy <proxyhost> <resource>");
		System.err.println(" <host1> <host2> ... <hostn>");
		System.exit(0);
	}
	hosts = new String[args.length - 2];
	for(int i=2;i<args.length;i++) {
		hosts[i-2] = new String(args[i]);
	}
    try {
      NwsProxy proxy = new NwsProxy(args[0], 8070);
      NwsForecastAPI.ForecastCollection[] forecasts = proxy.getAllForecasts(args[1], null, hosts);

	if(forecasts == null) {
		System.out.println("No results returned.");
		System.exit(0);
	}

	System.out.println("Returning " + forecasts.length + " results");
      	for(int i = 0; i < forecasts.length; i++) {
        	NwsForecastAPI.ForecastCollection forecast = forecasts[i];
        	System.out.println(
          		(int)forecast.measurement.timeStamp + " " +
          		forecast.measurement.measurement + " " +
          		forecast.forecasts[MSE_FORECAST].forecast + " " +
          		forecast.forecasts[MSE_FORECAST].error + " "); /* +
          		methods[forecast.forecasts[MSE_FORECAST].methodUsed].replace(' ', '_'));*/
      		}
    	} catch(Exception x) { 
		System.err.println(x.toString()); 
		x.printStackTrace();
    	}
  }


}
