(* ODBC database interface for mod_caml programs.
 * Copyright (C) 2003-2004 Merjis Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: dbi_odbc.ml,v 1.4 2004/08/03 16:49:41 rwmj Exp $
 *)

(* XXX This code has NEVER even been compiled.  Please let me know if it
 * works / doesn't work for you.
 *)

open Printf

(* XXX Rather naive method of finding the [?] placeholders in the query
 * string. We parse up the query into [ "select name from foo where id = ";
 * "?"; " and bar = "; "?" ]. This doesn't handle naked question-marks within
 * strings properly of course.
 *)
let rec split_query query =
  try
    let i = String.index query '?' in
    let n = String.length query in
    let before, after =
      String.sub query 0 i, String.sub query (i+1) (n-(i+1)) in
    let after_split, count = split_query after in
    (before :: "?" :: after_split), (count+1)
  with
      Not_found -> [query], 0

let escape_string =
  let re1 = Pcre.regexp "'" in		(* Double up any single quotes. *)
  let sub1 = Pcre.subst "''" in
  let re2 = Pcre.regexp "\\\\" in	(* Double up any backslashes. *)
  let sub2 = Pcre.subst "\\\\" in
  fun s ->
    let s = Pcre.replace ~rex:re1 ~itempl:sub1 s in
    let s = Pcre.replace ~rex:re2 ~itempl:sub2 s in
    "'" ^ s ^ "'"			(* Surround with quotes. *)


class statement dbh conn in_transaction original_query =

  (* Split up the query, and calculate the number of placeholders. *)
  let query, nr_args = split_query original_query in

object (self)
  inherit Dbi.statement dbh

  val mutable tuples = None
  val mutable next_tuple = 0
  val mutable types = None

  method execute args =
    if dbh#debug then (
      eprintf "Dbi_odbc: dbh %d: execute %s\n" dbh#id original_query;
      flush stderr
    );

    if dbh#closed then
      failwith "Dbi_odbc: executed called on a closed database handle.";

    if List.length args <> nr_args then
      invalid_arg "Dbi_postgres: execute called with wrong number of args.";

    (* Finish previous statement, if any. *)
    self#finish ();

    (* In transaction? If not we need to issue a BEGIN WORK command. *)
    if not !in_transaction then (
      (* So we don't go into an infinite recursion ... *)
      in_transaction := true;

      let sth = dbh#prepare_cached "begin work" in
      sth#execute []
    );

    (* Substitute the arguments and create the query which we'll send to
     * the database.
     *)
    let args = ref args in
    let query =
      String.concat ""
	(List.map
	   (function
		"?" ->
		  let arg = List.hd !args in
		  args := List.tl !args;
		  (match arg with
		      `Null ->
			"null"
		    | `Int i ->
			string_of_int i
                    | `Float f ->
                        string_of_float f
		    | `String s ->
			escape_string s
		    | `Bool b ->
			if b then "'t'" else "'f'"
		    | _ ->
			failwith "Dbi_odbc: unknown argument type in execute"
		  )
	      | str -> str) query) in

    (* Send the query to the database.
     * XXX error_code ignored?  (Interface seems to suggest it'll throw an
     * exception).
     * XXX How does ODBC quote query parameters?
     *)
    let _, typs, res =
      try
	Ocamlodbc.execute_with_info conn query
      with
	  Ocamlodbc.SQL_Error s -> raise Dbi.SQL_error s in

    types <- typs;
    tuples <- res;
    next_tuple <- 0

  method fetch1 () =
    if dbh#debug then (
      eprintf "Dbi_odbc: dbh %d: fetch1\n" dbh#id;
      flush stderr
    );
    match tuples with
    | None -> failwith "Dbi_odbc.statement#fetch1"
    | Some tuples ->
        begin match tuples with
        | [] -> raise Not_found;
        | tuple :: tl ->
	    tuples <- tl;
	    next_tuple <- next_tuple + 1;

	  (* Convert each field in the tuple according to type. *)
	  (* XXX How does ODBC return NULLs? *)
	  let row =
	    List.map2
	      (fun v ->
		 function
		     Ocamlodbc.SQL_integer | Ocamlodbc.SQL_smallint ->
		       `Int (int_of_string v)
		   | Ocamlodbc.SQL_bit ->
		       `Bool (v = "t")
		   | Ocamlodbc.SQL_float ->
		       `Float (float_of_string v)
		   | _ ->
		       `String v)
	      tuple types in
	  row
        end

  method names =
    match tuples with
    | None -> failwith "Dbi_odbc.statement#names"
    | Some tuples ->
        failwith "Dbi_odbc.statement#names: NOT IMPLEMENTED" (* FIXME *)

  method serial _ =
    (* #serial cannot be implemented for ODBC drivers.  So always
       raise the exception Not_found -- see dbi.mli *)
    raise Not_found

  method finish () =
    if dbh#debug then (
      eprintf "Dbi_odbc: dbh %d: finish %s\n" dbh#id original_query;
      flush stderr
    );
    tuples <- None

end

and connection ?host ?port ?(user = "") ?(password = "") database =

  let conn = Ocamlodbc.connect database user password in

  (* We pass this reference around to the statement class so that all
   * statements belonging to this connection can keep track of our
   * transaction state and issue the appropriate BEGIN WORK command at
   * the right time.
   *)
  let in_transaction = ref false in

object (self)
  inherit Dbi.connection ?host ?port ?user ?password database as super

  method host = ""
  method port = ""
  method user = user
  method password = password
  method database = conn

  method database_type = "odbc"

  method prepare query =
    if self#debug then (
      eprintf "Dbi_odbc: dbh %d: prepare %s\n" self#id query;
      flush stderr
    );

    if self#closed then
      failwith "Dbi_odbc: prepare called on closed database handle.";
    new statement
      (self : #Dbi.connection :> Dbi.connection)
      conn in_transaction query

  method commit () =
    super#commit ();
    let sth = self#prepare_cached "commit work" in
    sth#execute [];
    in_transaction := false

  method rollback () =
    let sth = self#prepare_cached "rollback work" in
    sth#execute [];
    in_transaction := false;
    super#rollback ()

  method close () =
    Connection.finish conn;
    super#close ()

  initializer
    if Connection.status conn = Connection.Bad then
      raise (Dbi.SQL_error (Connection.error_message conn))
end

let connect ?host ?port ?user ?password database =
  new connection ?host ?port ?user ?password database
let close (dbh : connection) = dbh#close ()
let closed (dbh : connection) = dbh#closed
let commit (dbh : connection) = dbh#commit ()
let ping (dbh : connection) = dbh#ping ()
let rollback (dbh : connection) = dbh#rollback ()
