

%  This example reads a volume dataset, extracts two isosurfaces that
%  represent the skin and bone, and then displays them.

vtk_init;

VTK_DATA_ROOT = vtkGetDataRoot();

%  Create the renderer, the render window, and the interactor. The
%  renderer draws into the render window, the interactor enables mouse-
%  and keyboard-based interaction with the scene.
aRenderer = vtkRenderer();
renWin = vtkRenderWindow();
renWin.AddRenderer(aRenderer);
iren = vtkRenderWindowInteractor();
iren.SetRenderWindow(renWin);

%  The following reader is used to read a series of 2D slices (images)
%  that compose the volume. The slice dimensions are set, and the
%  pixel spacing. The data Endianness must also be specified. The reader
%  usese the FilePrefix in combination with the slice number to construct
% d. (In this case the FilePrefix
%  is the root name of the file: quarter.)
v16 = vtkVolume16Reader();
v16.SetDataDimensions(64, 64);
v16.SetDataByteOrderToLittleEndian();
v16.SetFilePrefix(strcat(VTK_DATA_ROOT,"/Data/headsq/quarter"));
v16.SetImageRange(1, 93);
v16.SetDataSpacing(3.2, 3.2, 1.5);

%  An isosurface, or contour value of 500 is known to correspond to the
%  skin of the patient. Once generated, a vtkPolyDataNormals filter is
%  is used to create normals for smooth surface shading during rendering.
%  The triangle stripper is used to create triangle strips 
%  isosurface these render much faster on may systems.
skinExtractor = vtkContourFilter();
skinExtractor.SetInput(v16.GetOutput());
skinExtractor.SetValue(0, 500);
skinNormals = vtkPolyDataNormals();
skinNormals.SetInput(skinExtractor.GetOutput());
skinNormals.SetFeatureAngle(60.0);
skinStripper = vtkStripper();
skinStripper.SetInput(skinNormals.GetOutput());
skinMapper = vtkPolyDataMapper();
skinMapper.SetInput(skinStripper.GetOutput());
skinMapper.ScalarVisibilityOff();
skin = vtkActor();
skin.SetMapper(skinMapper);
skin.GetProperty().SetDiffuseColor(1, .49, .25);
skin.GetProperty().SetSpecular(.3);
skin.GetProperty().SetSpecularPower(20);

%  An isosurface, or contour value of 1150 is known to correspond to the
%  skin of the patient. Once generated, a vtkPolyDataNormals filter is
%  is used to create normals for smooth surface shading during rendering.
%  The triangle stripper is used to create triangle strips 
%  isosurface these render much faster on may systems.
boneExtractor = vtkContourFilter();
boneExtractor.SetInput(v16.GetOutput());
boneExtractor.SetValue(0, 1150);
boneNormals = vtkPolyDataNormals();
boneNormals.SetInput(boneExtractor.GetOutput());
boneNormals.SetFeatureAngle(60.0);
boneStripper = vtkStripper();
boneStripper.SetInput(boneNormals.GetOutput());
boneMapper = vtkPolyDataMapper();
boneMapper.SetInput(boneStripper.GetOutput());
boneMapper.ScalarVisibilityOff();
bone = vtkActor();
bone.SetMapper(boneMapper);
bone.GetProperty().SetDiffuseColor(1, 1, .9412);

%  An outline provides context around the data.
outlineData = vtkOutlineFilter();
outlineData.SetInput(v16.GetOutput());
mapOutline = vtkPolyDataMapper();
mapOutline.SetInput(outlineData.GetOutput());
outline = vtkActor();
outline.SetMapper(mapOutline);
outline.GetProperty().SetColor(0, 0, 0);

%  It is convenient to create an initial view of the data. The FocalPoint
%  and Position form a vector direction. Later on (ResetCamera() method)
%  this vector is used to position the camera to look at the data in
%  this direction.
aCamera = vtkCamera();
aCamera.SetViewUp(0, 0, -1);
aCamera.SetPosition(0, 1, 0);
aCamera.SetFocalPoint(0, 0, 0);
aCamera.ComputeViewPlaneNormal();

%  Actors are added to the renderer. An initial camera view is created.
%  The Dolly() method moves the camera towards the FocalPoint,
%  thereby enlarging the image.
aRenderer.AddActor(outline);
aRenderer.AddActor(skin);
aRenderer.AddActor(bone);
aRenderer.SetActiveCamera(aCamera);
aRenderer.ResetCamera();
aCamera.Dolly(1.5);

%  Set a background color for the renderer and set the size of the
%  render window (expressed in pixels).
aRenderer.SetBackground(1, 1, 1);
renWin.SetSize(640, 480);

%  Note that when camera movement occurs (as it does in the Dolly()
%  method), the clipping planes often need adjusting. Clipping planes
%  consist of two planes: near and far along the view direction. The
%  near plane clips out objects in front of the plane the far plane
%  clips out objects behind the plane. This way only what is drawn
%  between the planes is actually rendered.
aRenderer.ResetCameraClippingRange();

%  Interact with the data.

% renWin.Render();
vtkInitializeInteractor(iren);
