

%  This example demonstrates the conversion of point data to cell data.
%  The conversion is necessary because we want to threshold data based
%  on cell scalar values.

vtk_init;

VTK_DATA_ROOT = vtkGetDataRoot();

%  Read some data with point data attributes. The data is 
%  plastic blow molding process (e.g., to make plastic bottles) and
%  consists of two logical components: a mold and a parison. The
%  parison is the hot plastic that is being molded, and the mold is
%  clamped around the parison to form its shape.
reader = vtkUnstructuredGridReader();
reader.SetFileName(strcat(VTK_DATA_ROOT,"/Data/blow.vtk"));
reader.SetScalarsName("thickness9");
reader.SetVectorsName("displacement9");

%  Convert the point data to cell data. The point data is passed
%  through the filter so it can be warped. The vtkThresholdFilter then
%  thresholds based on cell scalar values and extracts a portion of the
%  parison whose cell scalar values lie between 0.25 and 0.75.
p2c = vtkPointDataToCellData();
p2c.SetInput(reader.GetOutput());
p2c.PassPointDataOn();
warp = vtkWarpVector();
warp.SetInput(p2c.GetUnstructuredGridOutput());
thresh = vtkThreshold();
thresh.SetInput(warp.GetOutput());
thresh.ThresholdBetween(0.25, 0.75);
thresh.SetAttributeModeToUseCellData();

%  This is used to extract the mold 
connect = vtkConnectivityFilter();
connect.SetInput(thresh.GetOutput());
connect.SetExtractionModeToSpecifiedRegions();
connect.AddSpecifiedRegion(0);
connect.AddSpecifiedRegion(1);
moldMapper = vtkDataSetMapper();
moldMapper.SetInput(reader.GetOutput());
moldMapper.ScalarVisibilityOff();
moldActor = vtkActor();
moldActor.SetMapper(moldMapper);
moldActor.GetProperty().SetColor(.2, .2, .2);
moldActor.GetProperty().SetRepresentationToWireframe();

%  The threshold filter has been used to extract the parison.
connect2 = vtkConnectivityFilter();
connect2.SetInput(thresh.GetOutput());
parison = vtkGeometryFilter();
parison.SetInput(connect2.GetOutput());
normals2 = vtkPolyDataNormals();
normals2.SetInput(parison.GetOutput());
normals2.SetFeatureAngle(60);
lut = vtkLookupTable();
lut.SetHueRange(0.0, 0.66667);
parisonMapper = vtkPolyDataMapper();
parisonMapper.SetInput(normals2.GetOutput());
parisonMapper.SetLookupTable(lut);
parisonMapper.SetScalarRange(0.12, 1.0);
parisonActor = vtkActor();
parisonActor.SetMapper(parisonMapper);

%  We generate some contour lines on the parison.
cf = vtkContourFilter();
cf.SetInput(connect2.GetOutput());
cf.SetValue(0, .5);
contourMapper = vtkPolyDataMapper();
contourMapper.SetInput(cf.GetOutput());
contours = vtkActor();
contours.SetMapper(contourMapper);

%  Create graphics stuff
ren = vtkRenderer();
renWin = vtkRenderWindow();
renWin.AddRenderer(ren);
iren = vtkRenderWindowInteractor();
iren.SetRenderWindow(renWin);

%  Add the actors to the renderer, set the background and size
ren.AddActor(moldActor);
ren.AddActor(parisonActor);
ren.AddActor(contours);

ren.GetActiveCamera().Azimuth(60);
ren.GetActiveCamera().Roll(-90);
ren.GetActiveCamera().Dolly(2);
ren.ResetCameraClippingRange();

ren.SetBackground(1, 1, 1);
renWin.SetSize(750, 400);


% renWin.Render();
vtkInitializeInteractor(iren);
