## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {@var{f} =} vtk_trimesh(@var{t},@var{x},@var{y},@var{z},@var{c},[@var{prop},@var{val}])
##
## Displays a meshplot defined by specified triangles.
## @var{t} gives a list of vertex indexes forming the triangles.
## @var{x},@var{y},@var{z} are vectors defining the vertices.  @var{c} is a vector
## of color scalar values for the vertices and may be omitted.
## The vtk_figure object is returned in f, if desired.
## trimesh can be called with 2(3) or 4(5) parameters as
## trimesh(t,[x1 y1 z1; ...; xn, yn, zn],(c)) or
## trimesh(t,[x1; ...; xn],[y1; ...; yn], [z1; ...; zn],(c))
##
## Optional @var{prop},@var{val} arguments can be used to change
## properties of the plot.  Currently, valid properties are
## ";BallRadius;LineRadius;Fancy;".
##
## @end deftypefn
## @seealso{vtk_trisurf, vtk_mesh}

## Author: Dragan Tubic

function f = vtk_trimesh( varargin )

  ## evaluate input arguments
  valid_props = ";BallRadius;LineRadius;Fancy;";
  
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, all_va_args);
  
  if ( no_numerical_params < 2 | no_numerical_params > 5 )
    error("Syntax is trimesh( t, x, y, z, c )");
  end
  
  if ( no_numerical_params == 2 )
    t = nth (varargin,1);
    pts = nth (varargin,2);
    [nr nc] = size(pts);
    if ( nr ~= 3 )
      pts = pts';
      if ( nc ~= 3 )
	error ("p has to be Nx3 matrix!");
      end
    end
    c = pts(:,3);
  elseif ( no_numerical_params == 3 )
    t = nth (varargin,1);
    pts = nth (varargin,2);
    [nr nc] = size(pts);
    if ( nr ~= 3 )
      pts = pts';
      if ( nc ~= 3 )
	error ("p has to be Nx3 matrix!");
      end
    end
    %% Check scalars
    c = nth (varargin,2);
    [nr nc] = size(c);
    if ( nr ~= 1 )
      c = c';
      if ( nc ~= 1 )
	error ("c has to be Nx1 matrix!");
      end
    end
  elseif ( no_numerical_params == 4 )
    t = nth (varargin,1);
    x = nth (varargin,2);
    y = nth (varargin,3);
    z = nth (varargin,4);
    if ( length(x) ~= length(y) | length(x) ~= length(z) )
      error("Lengths of all three coordinate arrays have to be the same.");
    end
    [nr, nc] = size(x);
    if ( nc ~= 1 )
      x = x';
      y = y';
      z = z';
      if ( nr ~= 1 )  
	error ("x has to be Nx1 matrix!");
      end
    end
    pts = [x y z];
    c = z;
  elseif ( no_numerical_params == 5 )
    t = nth (varargin,1);
    x = nth (varargin,2);
    y = nth (varargin,3);
    z = nth (varargin,4);
    c = nth (varargin,5);
    if ( length(x) ~= length(y) | length(x) ~= length(z) | length(x) ~= length(c) )
      error("Lengths of all three coordinate arrays have to be the same.");
    end
    [nr, nc] = size(x);
    if ( nc ~= 1 )
      x = x';
      y = y';
      z = z';
      if ( nr ~= 1 )  
	error ("x has to be Nx1 matrix!");
      end
    end
    pts = [x y z];
  end
  
  if ( no_numerical_params < 4 | no_numerical_params > 6 )
    error("Syntax is trimesh( t, x, y, z, c )");
  end
  
  t = nth (varargin,1);
  x = nth (varargin,2);
  y = nth (varargin,3);
  z = nth (varargin,4);
  
  if ( no_numerical_params == 4 )
    c = max(z) - z;
  end
  
  if ( first_prop_index > 0 )
    properties = struct(varargin{first_prop_index:length(varargin)});
  else
    properties.empty = 1;
  end

  ##
  ##  	if ( line_spec_index > 0 )
  ##  		line_spec = nth (varargin,line_spec_index);
  ##  		[color, marker_type, line_style] = vtk_get_line_spec( line_spec );
  ##  		properties.Color = color;
  ##  	else
  ##  		properties.Color = [1 0 0];
  ##  		marker_type = '*';
  ##  	end


  ## start the real work
  vtk_init;	
  
  %% We'll create the building blocks of polydata including data attributes.
  surface  = vtkPolyData();
  points   = vtkPoints();
  polys    = vtkCellArray();
  scalars  = vtkFloatArray();
  
  %% Load the point, cell, and data attributes.
  coords = vtkFloatArray; 
  coords.SetNumberOfTuples( length(x) );
  coords.SetNumberOfComponents(3);
  pts = [x y z]';
  coords.SetArray( pts(:), 3*length(x), 0 );
  points.SetData(coords);
  
  [nr nc] = size(t);
  if ( nc ~= 3 )
    if ( nr == 3 )
      t=t';
      [nr nc] = size(t);
    else
      error("t has to be a Nx3 matrix");
    endif
  endif
	
  no_tris = length(t);
  t = [ones(no_tris,1)*3 t-1]';
  ptids = vtkIdTypeArray;
  ptids.SetArray( t(:), no_tris*4, 0 );
  polys.SetCells( no_tris, ptids );
  
  ## in VTK, by default red = low values, blue - high values; we reverse that here
  ##c=-c;
  ## manually create color map table instead; a must if later add a scalar bar
  lut = vtkLookupTable();
  lut.SetHueRange(0.66667, 0.0);
  lut.SetNumberOfColors(256);
  lut.SetRampToLinear();
  lut.Build();
  scalars.SetArray( c, length(c) , 0 );
  
  %% We now assign the pieces to the vtkPolyData.
  surface.SetPoints(points);
  surface.SetPolys(polys);
  surface.GetPointData().SetScalars(scalars);

  ## do a simple meshplot if no properties given
  if isfield(properties,"empty")
    %% Now we'll look at it.
    surfaceMapper = vtkPolyDataMapper();
    surfaceMapper.SetInput(surface);
    surfaceMapper.SetLookupTable(lut);            % use manually created color map table
    surfaceMapper.SetScalarRange(min(c),max(c));  % relative scaling
    
    surfaceActor = vtkLODActor();
    surfaceActor.SetMapper(surfaceMapper);
    prop = surfaceActor.GetProperty();
    prop.SetDiffuse(0);
    prop.SetAmbient(1);
    prop.SetRepresentationToWireframe()
    
    f = vtk_figure(0);
    f.renderer.AddActor(surfaceActor);
  else
    ## We will now create a nice looking mesh by wrapping the edges in tubes,
    ## and putting fat spheres at the points.
    extract = vtkExtractEdges();
    extract.SetInput(surface);
    tubes = vtkTubeFilter();
    tubes.SetInput(extract.GetOutput());
    if struct_contains(properties,"LineRadius")
      tube_rad = properties.LineRadius;
    else
      tube_rad = ( max(max(pts)) - min(min(pts)) ) / 250;
    end
    tubes.SetRadius(tube_rad);
    tubes.SetVaryRadiusToVaryRadiusOff();
    tubes.SetNumberOfSides(6);
    mapEdges = vtkPolyDataMapper();
    mapEdges.SetInput(tubes.GetOutput());
    mapEdges.SetLookupTable(lut);            % use manually created color map table
    mapEdges.SetScalarRange(min(c),max(c));   % relative scaling
    
    edgeActor = vtkActor();
    edgeActor.SetMapper(mapEdges);
    
    ball = vtkSphereSource();
    if struct_contains(properties,"BallRadius")
      ball_rad = properties.BallRadius;
    else
      ball_rad = ( max(max(pts)) - min(min(pts)) ) / 100;
    end
    ball.SetRadius(ball_rad);
    ball.SetThetaResolution(12);
    ball.SetPhiResolution(12);
    balls = vtkGlyph3D();
    balls.SetInput(extract.GetOutput());
    balls.ScalingOff();
    balls.SetSource(ball.GetOutput());
    mapBalls = vtkPolyDataMapper();
    mapBalls.SetColorModeToMapScalars();
    mapBalls.SetInput(balls.GetOutput());
    mapBalls.SetLookupTable(lut);            % use manually created color map table
    mapBalls.SetScalarRange(min(c),max(c));   % relative scaling
    ballActor = vtkActor();
    ballActor.SetMapper(mapBalls);
    
    f = vtk_figure(0);
    f.renderer.AddActor(edgeActor);
    f.renderer.AddActor(ballActor);
  endif
  
  vtk_update(f);
  
endfunction
