// Copyright (C) 2006 Bruno Roggeri <mpomme@users.sourceforge.net>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include "ticks.h"
#include "mathutils.h"
#include "text.h"

#define logb(x) log((x))/log(base)

#define mod(x,y) (x)%(y)<0? (x)%(y)+(y) : (x)%(y)

#define LOOKAROUND 0.02

Ticks::Ticks(int direction, ocpl::Handle Parent, double base) {
    left = 0;
    right = 1.;
    range = 1.;
    psize = 1.;
    inverted = false;
    logscale = false;
    
    factors=NULL;

    level = 0;
    _parent = Parent;

    this->direction = direction;
    this->base = base;   // for now we only accept integer bases
}

Ticks::~Ticks()
{
    delete [] factors;
}

void Ticks::SetRange(double left, double right) {
    // Try to conserve previous results
    if(this->left == left && this->right == right) return;
    if(this->left == right && this->right == left) {
        inverted = !inverted;
        return;
    }

    // The left limit must always be smaller than the right one
    // If it isn't the case, swap them and set "inverted"
    if(right-left <0) {
        this->right = left;
        this->left = right;
        this->range = left-right;
        inverted = true;
    } else {
        this->left = left;
        this->right = right;
        this->range = right-left;
        inverted = false;
    }
    // Reset
    ticks.clear();
    level = 0;
}

void Ticks::SetPixelSize(double psize) {
    // Try to conserve previous results
    if(this->psize==fabs(psize)) return;
    this->psize = fabs(psize);
    // Reset
    ticks.clear();
    level = 0;
}

void Ticks::SetLogScale(bool logscale) {
    // Try to conserve previous results
    if(this->logscale == logscale) return;
    this->logscale = logscale;
    // Reset
    ticks.clear();
    level = 0;
}
    
/*
 * Returns the vector of the factors of "base" in "factors"
 * and their number in "nf"
 */
void Ticks::FindBaseFactors() {
    vector<unsigned int> *ifactors = FactorList(static_cast<unsigned int>(base));
    nf = ifactors->size();
    factors = new double[nf];
    for(int i = 0; i < nf; i++) {
        factors[i] = (*ifactors)[i];
    }
    delete ifactors;
}

double Ticks::MaxDT() { 
    double maxdt = 0;
    list<double>::iterator it = ticks.begin();
    double ref = *it;
    if(logscale) {
        for(it++; it != ticks.end(); it++) {
            if((fabs(log(*it) - log(ref))) > maxdt) maxdt = fabs(log(*it) - log(ref));
            ref = *it;
        }
        if(ticks.size() >= 1) {
            double tf = ticks.front();
            double tb = ticks.back();
            if(fabs(log(tf)-log(left)) < fabs(log(right)-log(tb))) {
                ref = fabs(log(right)-log(tb));
            } else {
                ref = fabs(log(tf)-log(left));
            }
            if(ref > maxdt) maxdt = ref;
        }
        maxdt /= log(base);
    } else {
        for(it++; it != ticks.end(); it++) {
            if(fabs(*it - ref) > maxdt) maxdt = fabs(*it - ref);
            ref = *it;
        }
        if(ticks.size() >= 1) {
            double tf = ticks.front();
            double tb = ticks.back();
            if(tf-left < right-tb) {
                ref = right-tb;
            } else {
                ref = tf-left;
            }
            if(ref > maxdt) maxdt = ref;
        }
    }
    _DEBUG_TICKS_PRINT("Largest DT : %g\n", maxdt);
    return maxdt;
}

double Ticks::MinDT() { 
    double mindt = range;
    list<double>::iterator it = ticks.begin();
    double ref = *it;
    if(logscale) {
        for(it++; it != ticks.end(); it++) {
            if((fabs(log(*it) - log(ref))) < mindt) mindt = fabs(log(*it) - log(ref));
            ref = *it;
        }
        if(ticks.size() >= 1) {
            double tf = ticks.front();
            double tb = ticks.back();
            if(fabs(log(tf)-log(left)) > fabs(log(right)-log(tf))) {
                mindt = fabs(log(right)-log(tf));
            } else {
                mindt = fabs(log(tf)-log(left));
            }
        }
        mindt /= log(base);
    } else {
        for(it++; it != ticks.end(); it++) {
            if(fabs(*it - ref) < mindt) mindt = fabs(*it - ref);
            ref = *it;
        }
        if(ticks.size() == 1) {
            double tf = ticks.front();
            if(tf-left > right-tf) {
                mindt = tf-left;
            } else {
                mindt = right-tf;
            }
        }
    }
    _DEBUG_TICKS_PRINT("Smallest DT : %g\n", mindt);
    return mindt;
}

/*
 * Returns true if we should try to add some more ticks
 */
bool Ticks::EvaluateClutter() {
    Text *txt;
    int size = 0;
    int maxsize = 0;
    char *buf = new char[size+1];
    
    // Find the maximum dimension of a label
    double maxw = 0;
    double maxh = 0;

    for(list<double>::iterator it = ticks.begin();
            it != ticks.end(); it++) {
        size = snprintf(buf,maxsize,"%g",*it);
        if(size > maxsize) {
            delete [] buf;
            buf = new char[size + 1];
            maxsize = size;
            sprintf(buf,"%g",*it);
        }

        txt = new Text(_parent,buf,0,0,0);
        if(txt->w() > maxw) maxw = txt->w();
        if(txt->h() > maxh) maxh = txt->h();
        delete txt;
    }

    // Recommend a "ideal" tick spacing based on the maximum label size and a hard
    // limit not to exceed
    double dt;
    double harddt;
    if(direction==0) {
        dt = max(6*maxh, maxw + 3*maxh);
        harddt = max(1.5*maxh, maxw + maxh/2);
    } else {
        dt = 6*maxh;
        harddt = dt/4;
    }
    _DEBUG_TICKS_PRINT("AUTOTICKS recommended dt %g, that's %g\n" , dt, dt*psize);
/*
    // Return true if all ticks are more spaced than the ideal recommendation
    // or if the most spaced ticks are more than 1.5x the ideal recommendation

    return (MinDT() > dt*psize) || 
        ((MaxDT() > 1.5*dt*psize)&&(MinDT()>harddt*psize));
*/
    // Returns true as long as the biggest space is too big
    return MaxDT() > dt*psize;
}

/*
 * Adds dt spaced ticks
 */
void Ticks::AddTicks(double dt) {
    double range = right - left;
    //double l = left - range*LOOKAROUND;
    //double r = right + range*LOOKAROUND;
    double l = left - psize;
    double r = right + psize;
    int t0 = static_cast<int>(ceil(l/dt));
    int tf = static_cast<int>(floor(r/dt)) - t0;
    int i = 0;
    while(i <= tf) {
        ticks.push_back((t0+i)*dt);
        i++;
    }
}

void Ticks::AddTicks() {
    int fexp = static_cast<int>(floor(logb(range)));

    // Finds a convenient dt. The higher the "level" variable is, 
    // the smaller the dt will be

    // If fexp>0, level 0 is dt = base * base^fexp, else it's dt = base
    double dt;
    if(fexp > 0) {
        dt = factors[level%nf] * pow(base,fexp-(level/nf));
    } else {
        dt = factors[level%nf] * pow(base, 0-(level/nf));
    }

    ticks.clear();
    AddTicks(dt);
    // Next time find a smaller dt
    level++;
}

/*
 * Insert a tick between l and r, close to the middle,
 * but as "important" as possible
 */
void Ticks::Between(double l, double r) {
    double logbl = logb(l);
    double logbr = logb(r);
    // middle of the interval on the log scale
    double logbidealt = (logbl + logbr)/2;
    double idealt = pow(base,logbidealt);
    // tick position
    double pt = r + 1; // initiated in an invalid position

    // mantisse
    int m = 1;
    // exponent
    int e;

    // Check if a base^n number is in the interval
    if(floor(logbl)+1 < logbr) { 
        // Find the closest one to the ideal position
        e = static_cast<int>(nearbyint(logbidealt));
        pt = pow(base,e);
        if(pt <=l)
            e++;
        else if(pt >= r)
            e--;
        pt = pow(base,e);
    } 

    // if the position is still invalid 
    // (either because the previous block was skipped, or because it returned
    // a flawed result in pt because of rounding error)
    if(pt >= r || pt <= l) {
        // We're working in a interval that covers only 1 order of magnitude
        //
        // We look for a number in the form m*base^e while respecting the following
        // priorities, in this order :
        // 1. e is an integer as big as possible
        // 2. m is an integer such that the tick is as close to the log-middle as possible

        e = static_cast<int>(ceil(logbl)); // We start with a big exponent (priority 1.)
        do {
            m = static_cast<int>(nearbyint(idealt / pow(base,e))); // priority 2.
            pt = m*pow(base,e);
            if(pt <= l)
                m++;
            else if(pt >= r)
                m--;
            // hack around corner case when the limits are in base^n form,
            // which happens very often
            if(m < 2) 
                m = 2;

            pt = m*pow(base,e);
            e--; 
        } while(pt >= r || pt <= l);
    }
    ticks.push_back(pt);
    _DEBUG_TICKS_PRINT("AUTOTICKS adding %f between %f and %f\n", pt, l, r);
}

void Ticks::AddLogTicks() {
    double range = logb(right)-logb(left);
    // Begin at 5% (in log) on each side of the interval to include important ticks near the boundaries
    //double large_left = left / pow(base,range * LOOKAROUND);
    //double large_right = right * pow(base, range * LOOKAROUND);
    double large_left = left / pow(base, psize);
    double large_right = right * pow(base, psize);
    double a;
    double b;
    if(ticks.empty()) {
        a = large_left;
        b = large_right;
    } else {
        //Find largest space
        list<double>::iterator it;
        double ref = large_left;
        double max = 0;

        for(it=ticks.begin(); it!=ticks.end(); it++) {
            if(logb(*it)-logb(ref) > max) {
                max = logb(*it)-logb(ref);
                a = ref;
                b = *it;
            }
            ref = *it;
        }
        if(logb(large_right)-logb(ref) > max) {
            // max = logb(right)-logb(ref);
            a = ref;
            b = large_right;
        }
    }

    _DEBUG_TICKS_PRINT("AUTOTICKS Between %f,%f\n", a, b);
    Between(a,b);
}

/*
 * Returns the calculated ticks matrix in ticks_mat
 */
void Ticks::GetAutoTicks(Matrix& ticks_mat) {
    _DEBUG_TICKS_PRINT("AUTOTICKS direction %i, nticks %i, pixel size %g\n",
            direction, ticks.size(), psize);
    // Don't recompute if unnecessary
    if(ticks.empty()) {
        if(factors == 0) FindBaseFactors();
        //list<double> backup;
        do {
          //  backup = ticks;
            if(logscale) AddLogTicks();
            else AddTicks();
            ticks.sort();
            ticks.unique();
            _DEBUG_TICKS_PRINT("AUTOTICKS level %i, nticks %i\n", level, ticks.size());
        } while(EvaluateClutter());
        //ticks = backup;
        if(inverted) ticks.reverse();
        level--;
    }



    double *t = new double[ticks.size()];
    int i=0;

    for(list<double>::iterator it = ticks.begin();
            it != ticks.end(); it++) {
        t[i] = *it;
        i++;
    }

    Matrix *tempt = new Matrix(t, ticks.size(), 1);
    ticks_mat = *tempt;
    delete tempt;
}
