/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <cassert>
#include <cmath>
#include <iostream>
#include "Patch.h"
#include "Plugin.h"
#include "Port.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "OSCSender.h"
#include "Connection.h"
#include "InputNode.h"
#include "OutputNode.h"
#include "MidiInNode.h"
#include "Om.h"
using std::cerr; using std::cout; using std::endl;

namespace Om {


Patch::Patch(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size, uint internal_poly) 
: NodeBase(path, poly, parent, srate, buffer_size),
  m_internal_poly(internal_poly),
  m_process_order(NULL),
  m_process(false)
#ifdef HAVE_DSSI
  , m_dssi_events_array(new snd_seq_event_t[1024]),
  m_dssi_events_size(0)
#endif
{
	assert(internal_poly >= 1);

	m_plugin.type(Plugin::Patch);
	m_plugin.lib_path("");
	m_plugin.lib_name("");
	m_plugin.plug_label("om_patch");
	m_plugin.name("Om patch");

	//std::cerr << "Creating patch " << m_name << ", poly = " << poly
	//	<< ", internal poly = " << internal_poly << std::endl;
}


Patch::~Patch()
{
	for (List<Connection*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i) {
		delete (*i);
		delete m_connections.remove(i);
	}

	for (NodeTree::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i)
		delete *i;

	delete m_process_order;

#ifdef HAVE_DSSI
	delete[] m_dssi_events_array;
#endif
}


void
Patch::activate()
{
	NodeBase::activate();

	for (NodeTree::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i)
		(*i)->activate();
}


void
Patch::deactivate()
{
	if (m_activated) {
	
		NodeBase::deactivate();
	
		for (NodeTree::iterator i = m_nodes.begin(); i != m_nodes.end(); ++i)
			if ((*i)->activated())
				(*i)->deactivate();
	}
}


/** Run the patch for the specified number of frames.
 * 
 * Calls all Nodes in the order m_process_order specifies.
 */
inline void
Patch::run(size_t nframes)
{
	if (m_process_order == NULL)
		return;

	// Prepare all ports
	for (List<InputNode*>::iterator i = m_input_nodes.begin(); i != m_input_nodes.end(); ++i)
		(*i)->external_port()->prepare_buffers();
	for (List<OutputNode*>::iterator i = m_output_nodes.begin(); i != m_output_nodes.end(); ++i)
		(*i)->external_port()->prepare_buffers();

	// Run all nodes
	for (List<Node*>::iterator i = m_process_order->begin(); i != m_process_order->end(); ++i) {
		if ((*i) == NULL)
			return;
		else
			(*i)->run(nframes);
	}
}


Port* const
Patch::port(const string& port_name) const
{
	/* This has been changed to look in the node tree for a node repeseting the port,
	 * rather than looking in the ports array.  This is to prevent having to block the event
	 * queue at add-output-node events, which slows down patch loading considerably.
	 *
	 * The AddNodeEvent adds the node to the tree before the process cycle (in the prepare()
	 * method), but doesn't add it to the port list (which Jack reads) until the execute()
	 * function, to prevent a race condition.
	 *
	 * This makes this function much slower, but for the time being it's much better than
	 * blocking the event queue.  This isn't good though, it affects performance _in the
	 * audio thread_ for realtime SetPortValueEvents.
	 */
	
	/*
	for (List<InputNode*>::const_iterator i = m_input_nodes.begin(); i != m_input_nodes.end(); ++i)
		if ((*i)->external_port()->name() == port_name)
			return (*i)->external_port();
	
	for (List<OutputNode*>::const_iterator i = m_output_nodes.begin(); i != m_output_nodes.end(); ++i)
		if ((*i)->external_port()->name() == port_name)
			return (*i)->external_port();
	*/
	
	Node* node = m_nodes.find(port_name);
	if (node != NULL) {
		const Plugin* const pi = node->plugin();
		
		if (pi->type() == Plugin::Internal) {
			if (pi->plug_label() == "control_input" || pi->plug_label() == "audio_input")
				return ((InputNode*)node)->external_port();
			else if (pi->plug_label() == "control_output" || pi->plug_label() == "audio_output")
				return ((OutputNode*)node)->external_port();
		}
	}

	//cerr << "[Patch::port] Did not find port " << port_name << "!" << endl;

	return NULL;
}


/** Returns the number of ports.
 *
 * Needs to override the NodeBase implementation since a Patch's ports are really
 * just it's input and output nodes' ports.
 */
uint
Patch::num_ports() const
{
	return num_in_ports() + num_out_ports();
}


uint
Patch::num_in_ports() const
{
	uint count = 0;

	for (List<InputNode*>::const_iterator i = m_input_nodes.begin(); i != m_input_nodes.end(); ++i)
		if ((*i) != NULL)
			count += (*i)->num_ports();
	
	return count;
}


uint
Patch::num_out_ports() const
{
	uint count = 0;	
	
	for (List<OutputNode*>::const_iterator i = m_output_nodes.begin(); i != m_output_nodes.end(); ++i)
		if ((*i) != NULL)
			count += (*i)->num_ports();
	
	return count;
}


void
Patch::send_creation_messages(lo_address addr) const
{
	om->osc_sender()->send_patch_to(addr, this);
}


void
Patch::send_deletion_messages(lo_address addr) const
{
	om->osc_sender()->send_patch_destruction_to(addr, path());
	
	if (m_parent != NULL) 
		om->osc_sender()->send_node_removal_to(addr, path());
}


// Patch specific stuff


void
Patch::add_node(TreeNode* tn)
{
	assert(tn != NULL);
	assert(tn->node() != NULL);
	assert(tn->node()->poly() == m_internal_poly || tn->node()->poly() == 1);
	
	m_nodes.insert(tn);
}


TreeNode*
Patch::remove_node(TreeNode* tn)
{
#ifndef NDEBUG
	const Node* const orig_node = tn->node();
#endif
	TreeNode* ret = m_nodes.remove(tn);
	assert(ret != NULL);
	assert(ret->node() == orig_node);

	return ret;
}



/** Remove a connection.  Realtime safe.
 */
ListNode<Connection*>* const
Patch::remove_connection(const string& src_port_path, const string& dst_port_path)
{
	bool found = false;
	ListNode<Connection*>* connection = NULL;
	for (List<Connection*>::iterator i = m_connections.begin(); i != m_connections.end(); ++i) {
		if ((*i)->src_port()->path() == src_port_path &&
		    (*i)->dst_port()->path() == dst_port_path) {
			connection = m_connections.remove(i);
			found = true;
		}
	}

	if ( ! found) {
		std::cerr << "WARNING:  [Patch::remove_connection] Connection not found !" << std::endl;
	}

	return connection;
}


} // namespace Om
