/*
 * rtp-session-play.cc --
 *
 *      RTP Session Player
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "rtp-session-play.h"
#include "config.h"
#include <math.h>
#include <errno.h>
#include <string.h>

#include "source.h"
#include "tclcl.h"
#include "media-timer.h"
#include "crypt.h"
#include "timer.h"
#include "ntp-time.h"
#include "session-rtp.h"


DEFINE_OTCL_CLASS(RTPPlaySession, "Session/RTP/Play") {
}

RTPPlaySession::RTPPlaySession() {

}

RTPPlaySession::~RTPPlaySession() {
	send_bye();
	// FIX THIS - Deconstructors aren't being called
}

// There may be a number of local sources during playback, so send a
// report from each of them.  Eventually this should be re-integrated
// into session-rtp.

void RTPPlaySession::send_report(CtrlHandler* ch, int bye, int app)
{
	static u_int8_t buf[2*RTP_MTU];
	Source *ps;

	MTrace(trcArchive | trcVerbose, ("SEND REPORT"));
	Source* s = sm_->localsrc();
	ps = s;
	while (ps != NULL) {
		rtcphdr* rh = (rtcphdr*)buf;
		rh->rh_ssrc = s->srcid();
		int flags = RTP_VERSION << 14;
		timeval now = unixtime();
		int layer = ch - ch_;
		Source::Layer& sl = s->layer(layer);
		sl.lts_ctrl(now);
		int we_sent = 0;
		rtcp_rr* rr;
		/*
		 * If we've sent data since our last sender report send a
		 * new report.  The MediaTimer check is to make sure we still
		 * have a grabber -- if not, we won't be able to interpret the
		 * media timestamps so there's no point in sending an SR.
		 */
		MediaTimer* mt = MediaTimer::instance();
		if (sl.np() != last_np_ && mt) {
			last_np_ = sl.np();
			we_sent = 1;
			flags |= RTCP_PT_SR;
			rtcp_sr* sr = (rtcp_sr*)(rh + 1);
			sr->sr_ntp = ntp64time(now);
			HTONL(sr->sr_ntp.upper);
			HTONL(sr->sr_ntp.lower);
			sr->sr_ts = htonl(mt->ref_ts());
			sr->sr_np = htonl(sl.np());
			sr->sr_nb = htonl(sl.nb());
			rr = (rtcp_rr*)(sr + 1);
		} else {
			flags |= RTCP_PT_RR;
			rr = (rtcp_rr*)(rh + 1);
		}
		int nrr = 0;
		int nsrc = 0;
		/*
		 * we don't want to inflate report interval if user has set
		 * the flag that causes all sources to be 'kept' so we
		 * consider sources 'inactive' if we haven't heard a control
		 * msg from them for ~32 reporting intervals.
		 */
		u_int inactive = u_int(ch->rint() * (32./1000.));
		if (inactive < 2)
			inactive = 2;
		for (Source* sp = sm_->sources(); sp != 0; sp = sp->next_) {
			++nsrc;
			Source::Layer& sl = sp->layer(layer);
			int received = sl.np() - sl.snp();
			if (received == 0) {
				if (u_int(now.tv_sec - sl.lts_ctrl().tv_sec) > inactive)
					--nsrc;
				continue;
			}
			sl.snp(sl.np());
			rr->rr_srcid = sp->srcid();
			int expected = sl.ns() - sl.sns();
			sl.sns(sl.ns());
			u_int32_t v;
			int lost = expected - received;
			if (lost <= 0)
				v = 0;
			else
				/* expected != 0 if lost > 0 */
				v = ((lost << 8) / expected) << 24;
			/* FIXME should saturate on over/underflow */
			v |= (sl.ns() - sl.np()) & 0xffffff;
			rr->rr_loss = htonl(v);
			rr->rr_ehsr = htonl(sl.ehs());
			PacketHandler* handler = (PacketHandler*)sp->handler();
			rr->rr_dv = (handler != 0) ? handler->delvar() : 0;
			rr->rr_lsr = htonl(sl.sts_ctrl());
			if (sl.lts_ctrl().tv_sec == 0)
				rr->rr_dlsr = 0;
			else {
				u_int32_t ntp_now = ntptime(now);
				u_int32_t ntp_then = ntptime(sl.lts_ctrl());
				rr->rr_dlsr = htonl(ntp_now - ntp_then);
			}
			++rr;
			if (++nrr >= 31)
				break;
		}
		flags |= nrr << 8;
		rh->rh_flags = htons(flags);
		int len = (u_char*)rr - buf;
		rh->rh_len = htons((len >> 2) - 1);

		int plen;
		u_char* p = (u_char*)rr;
		if (!bye) {
			plen = build_sdes((rtcphdr*)p, *s);
			p += plen;
			len += plen;
		}
		if (app) {
			plen = build_app((rtcphdr*)p, *s, we_sent);
			p += plen;
			len += plen;
		}
		if (bye) {
			plen += build_bye((rtcphdr*)p, *s);
			len += plen;
		}


		ch->send(buf, len);
		ch->adapt(nsrc, nrr, we_sent);
		ch->sample_size(len);
		if (layer == 0)
			sm_->CheckActiveSources(ch->rint());

		if (s->next_ != NULL) {
			ps = s->next_;
			s = ps;

		}
		else {
			ps=NULL;
		}
	}
}
