/*
 * controller.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1991-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/audio/controller.cc,v 1.30 2002/02/03 03:10:46 lim Exp $";

#include "config.h"
#include "sys-time.h"
#include "audio.h"
#include "ss.h"
#include "controller.h"
#include "encoder.h"
#include "mulaw.h"
#include "tclcl.h"
#include "ntp-time.h"
#include "transducer.h"

#define METER_UPDATE_FREQ 3

/*
 * <otcl> Class AudioController
 * AudioController is the base class of the object that orchestrates
 * the timing loop for audio I/O.  Rather than drive this sub-system
 * of the operating system's clock, we use the audio sampling clock
 * to avoid clock skew (this does not, however, eliminate the
 * problem of skew among audio sources across the network).
 * <p>
 * The public API is all you should need to manipulate
 * the underlying audio controller in new applications.
 * But, if you want to add a new audio device, or
 * modify the existing code, you need to know a bit
 * about the internal audio event model.
 * The general event sequence is as follows:
 * <ul>
 * <li>Audio::dispatch calls the virtual FrameReady() to see if
 * there is a full frame (typically 160 samples) of input data available.
 * If the system's audio driver always returns full frames
 * (e.g., bsd_audio, pc_audio), FrameReady() can simply return
 * 1. But in the usual case the system will signal data available
 * for an arbitrary amount of data so FrameReady() must read
 * the data into a local buffer and only return 1 when it has
 * a full frame.
 * <li>
 * When FrameReady() says there is a full frame available,
 * Audio::dispatch() will call Controller::audio_handle() to
 * consume and process it.
 * <li>
 * The AudioController handles all
 * the data flow between the audio device and the network and is the focus
 * for most of the GUI user actions.  Controller::audio_handle()
 * calls the Audio Read() virtual to get the new frame, then
 * immediately calls the Audio Write() virtual to write a
 * frame's worth of audio data from the net to the audio speaker,
 * then does a bunch of processing associated with silence
 * suppression and the various speakerphone modes that may or
 * may not result in the incoming frame being encoded, encapsulated
 * and sent to the net.
 * <li>
 * When there is audio data available to read, the tk event handler
 * makes a callback to Audio::dispatch().
 * <p>
 * </ul>
 */

class HDController : public Controller {
    public:
	HDController();
	virtual void update(Observable*);
    protected:
	virtual void timeout();
	virtual void audio_handle();
};

/*
 * <otcl> Class AudioController/FullDuplex -superclass AudioController
 * AudioController/FullDuplex specifically handles full duplex
 * devices.
 */
static class ControllerClass : public TclClass {
    public:
	ControllerClass() : TclClass("AudioController/FullDuplex") {}
	TclObject* create(int, const char*const*) {
		return (new Controller);
	}
} controller_class;

/*
 * <otcl> Class AudioController/HalfDuplex -superclass AudioController
 * A controller class specifically for handling half-duplex audio
 * devices like those commonly found in PCs.  It derives its timebase
 * from a 20ms system timer rather than using audio read completions
 * like the normal AudioController/FullDuplex class.
 */
static class HDControllerClass : public TclClass {
    public:
	HDControllerClass() : TclClass("AudioController/HalfDuplex") {}
	TclObject* create(int, const char*const*) {
		return (new HDController);
	}
} hd_controller_class;

Controller::Controller() :
	audio_(0),
	lastnetout_(0),
	lastaudout_(0),
	tsec_(0),
	tusec_(0),
	ostate_(TALK_TAIL + TALK_LEAD),
	loopback_(0),
	test_tone_(0),
	pmeter_(0),
	rmeter_(0),
	talk_thresh_(0),
	echo_thresh_(0),
	echo_suppress_time_(0),
	meter_update_(METER_UPDATE_FREQ),
	active_(0),
	encoder_(0),
	outmax_(AUDIO_FRAMESIZE),
	outlen_(0),
	out_ts_(0),
	out_(0),
        audio_sps(8000)
{
  /*
   * following is timeout interval (in ms) when vat does not have
   * the audio & is running off a system timer instead.  This interval
   * must be the same as the audio read blocksize (AUDIO_FRAMESIZE /
   * audio_sps * 1000 ms) and must be < 1 sec.
   */
  // These are the default settings for 8 bit audio... most of them
  // get reset if you switch to 16 bit audio (look in ::command)...

        frame_time = (AUDIO_FRAMESIZE * 1000 / audio_sps);
#ifdef WIN32
	/* windows95 has trouble with 20ms events */
	timer_interval_ = (frame_time * 4);
#else
	timer_interval_ = (frame_time);
#endif

	bind("echo_thresh_", (int*)&echo_thresh_);
	bind("echo_suppress_time_", (int*)&echo_suppress_time_);
	bind("max_playout_", (int*)&max_playout_);

	as_ = new SampleStream(AUDIO_FRAMESIZE, max_playout_,
			       (TALK_LEAD+1)*AUDIO_FRAMESIZE, 0);
	ns_ = new SampleStream(AUDIO_FRAMESIZE, max_playout_,
			       (TALK_LEAD+1)*AUDIO_FRAMESIZE, 0);

	lastaudout_ = as_->Clock();
	lastnetout_ = ns_->Clock();

	as_->ssthresh(20);
	ns_->ssthresh(20);
}

void Controller::update(Observable*)
{
	if (!audio_->haveaudio()) {
		timeval tv;
		::gettimeofday(&tv, 0);
		tsec_ = tv.tv_sec;
		tusec_ = tv.tv_usec;
		/* Reset the meters and force a redraw. */
		if (pmeter_ != 0) {
		  pmeter_->set(0.);
		}
		if (rmeter_ != 0) {
		  rmeter_->set(0.);
		}
		msched(timer_interval_);
	} else {
		cancel();
	}
}

void Controller::DoAudio()
{
        double rlevel, plevel;
	register int nsmean = ns_->LTMean();

	if (ns_->Max())
		lastnetout_ = as_->Clock();
	ns_->Compute();
	plevel = ns_->Mean();
	mixaudio(*ns_);

	ns_->UpdateAGC();

	if (audio_->RMuted()) {
		rlevel = 0.;
	} else {
		as_->Compute();
		rlevel = as_->Mean();
		/*
		 * Next check is to cut residual echo in speakerphone &
		 * echo-cancel modes: If we haven't been talking recently, have
		 * just sent data to the speaker and the signal we got from the
		 * mike was about the same as the signal we played, assume the
		 * mike signal is echo.  NOTE that nsmean was grabbed before
		 * ns_->Compute() so we are comparing the previous audio output
		 * to the current audio input - this should make it more likely
		 * that we're looking at the echo signal (though under Sun OS
		 * there's still a lot of buffer in the STREAM I/O system).
		 */
		if (audio_->Mode() != Audio::mode_none && !sending() &&
		    (as_->LTMean() - nsmean < echo_thresh_ ||
		     as_->Silent(talk_thresh_)))
			;
		else if (audio_->Mode() != Audio::mode_netmutesmike ||
			 lastnetout_ == 0 ||
			 as_->Clock() - lastnetout_ > echo_suppress_time_) {
			if (Output()) {
				lastnetout_ = 0;
				active_ = 1;
				as_->UpdateAGC();
			}
		}
	}
	/*
	 * Update the meters.  We control the rate with METER_UPDATE_FREQ
	 * to cut down on CPU load from the X window updates.
	 */
	if(--meter_update_ <= 0)
	{
	  if(pmeter_ != 0)
	  {
	    pmeter_->set(plevel);
	  }
	  if(rmeter_ != 0)
	  {
	    rmeter_->set(rlevel);
	  }
	  meter_update_ = METER_UPDATE_FREQ;
	}
}

void Controller::DoTimer()
{
  /* Advance to next audio frame */
  as_->Advance();
  ns_->Advance();
}

/*
 * Called from decoders to mix in the block of samples 'del'
 * samples into the future. This just puts the audio in the
 * sample stream, from which the mixaudio function call will
 * Write it to the audio device.
 */
void Controller::mix_from_net(int del, const u_int8_t* frame, int len)
{
  if (del >= 0)
    ns_->Mix(del, frame, len);
}

// This is called when the audio device's FrameReady() returns 1
void Controller::audio_handle()
{
	DoAudio();
	DoTimer();
}

/*
 * Called when we don't have the audio device.  Normally, our time
 * base comes from the audio device's sample clock, so when we don't
 * have the device open, we revert to timers.
 */
void Controller::timeout()
{
	/*
	 * Use get time of day and keep track of the current hard time
	 * in the tsec_/tusec_ variables.  Tk timers are an unreliable
	 * time base.  We use them to dispatch an event here then call
	 * gettimeofday to see how many times we really should have
	 * been called in the intervening period.
	 */
	timeval tv;
	::gettimeofday(&tv, 0);
	u_int u = (u_int)tv.tv_usec;
	u_int s = (u_int)tv.tv_sec;
	if (s > tsec_ + 3) {
		/*
		 * We're way behind.  Most likely we were suspended and
		 * then resumed.  Instead of trying to catch up, just resync.
		 */
		tusec_ = u;
		tsec_ = s;
	}
	while ((int(tusec_ - u) <= 0 && s == tsec_) || int(tsec_ - s) < 0) {
		DoTimer();
		tusec_ += 1000 * frame_time;
		while (tusec_ >= 1000000) {
			tusec_ -= 1000000;
			++tsec_;
		}
	}
	msched(timer_interval_);
}

/*
 * send the next block of samples to the encoder.  We might coalesce
 * several calls into a larger chunk (depending on the value of outmax_).
 * The point is to stuff more data into a single packet to amortize the
 * packet header overhead (at the cost of increased latency).
 */
void Controller::send_block(u_int32_t ts, u_int8_t* blk, int len)
{
	if (out_ == 0) {
		out_ = blk;
		outlen_ = 0;
		out_ts_ = ts;
	} else if (&out_[outlen_] != blk) {
		/*
		 * frames wrapped in ss buffer -- copy to buf to
		 * keep things contiguous.
		 */
		if (out_ != overflow_) {
			/* first time */
			memcpy(overflow_, out_, outlen_);
			out_ = overflow_;
		}
		/* copy current chunk */
		memcpy(&out_[outlen_], blk, len);
	}
	outlen_ += len;
	if (outlen_ >= outmax_) {
	  encoder_->encode(out_ts_, out_, outlen_);
	  out_ = 0;
	}
}

/*
 * get the current media timestamp
 */
u_int32_t Controller::media_ts()
{
	::gettimeofday(&last_uts_, 0);
	u_int32_t ts = as_->Clock();
	last_mts_ = ts;
	return (ts);
}

u_int32_t Controller::ref_ts()
{
	timeval now;
	::gettimeofday(&now, 0);
	int t = (now.tv_sec - last_uts_.tv_sec) * 8000;
	t += ((now.tv_usec - last_uts_.tv_usec) << 3) / 1000;
	return (last_mts_ + t);
}

/*
 * Check for silence and otherwise send audio frames to the encoder.
 * If this is the start of a talk-spurt, go back TALK_LEAD blocks into
 * the past and send them too (because the silence detector isn't
 * perfect especially near a silence-to-speech transition).  At the
 * end of a talk-spurt, send TALK_TAIL extra blocks because the
 * silence detector is unreliable near a speech-to-silence transition.
 */
int Controller::Output()
{
	register int bs = audio_->GetBlockSize();
	if (audio_->use16bits())
	  bs = bs*2;
	
	if (as_->Silent()) {
		if (ostate_ >= TALK_TAIL) {
			/* between talk spurts */
			if (out_ != 0) {
				/* flush the partial last block */
				send_block(media_ts(), as_->CurBlk(), bs);
				return (1);
			}
			if (ostate_ < TALK_TAIL+TALK_LEAD)
				++ostate_;
			return (0);
		}
		++ostate_;
	} else if (ostate_) {
      	        /*
		 * if start of talk after silence, generate packets for
		 * any leading speech that we might have missed.
		 */
		for (int i = (ostate_ - TALK_TAIL) * bs; i > 0; i -= bs) {
			u_int32_t ts = as_->Clock() - i;
			send_block(ts, as_->BlkBack(i), bs);
		}
		ostate_ = 0;
	}
	send_block(media_ts(), as_->CurBlk(), bs);
	return (1);
}

extern "C" {
extern u_char tonemax[];
extern u_char tone0dBm[];
extern u_char tone6dBm[];
}

void Controller::mixaudio(SampleStream& ss)
{
	/*
	 * The audio driver is ready to give us the next packet.
	 * This serves as our time base.  We do the following:
	 *  - output next chunk to audio driver.
	 *    (We do the output *first* so the rest of our
	 *    processing is overlapped with the real-time audio
	 *    output, otherwise we tend to accumulate estimate
	 *    random delays.)
	 *  - If we're doing echo cancellation, estimate the
	 *    echo resulting from the block just output and mix
	 *    the echo inverse into the *input* sample stream
	 *    at the estimated echo delay.
	 *  - read packet from audio
	 *  - Mix the input data into the input sample stream
	 *    (this is a mix because we might have an inverse
	 *    echo signal or tone that we want summed with the
	 *    input).
	 */
	u_char* blk = 0;

	int blksize = audio_->GetBlockSize();
	// use16bits() returns either 1 or 0
	int blkByteSize = (audio_->use16bits() + 1) * blksize;

	blk = audio_->Read();
	
	if (loopback_)
		ss.Mix(0, blk, blkByteSize);
	if (test_tone_) {
	  if (!audio_->use16bits())
		ss.Mix(0, (test_tone_ == 1) ? tone6dBm :
		          (test_tone_ == 2) ? tone0dBm : tonemax, blkByteSize);
	  else
	    // we could fix this by making linear 16 bit versions of tonemax,
	    // tone0dBm and tone6dBm (which are currently 8 bit mulaw tables.
	    printf("can't do test tones under 16 bit sampling, only 8 bit\n");
	}

	/*
	 * Now write a block of samples to the audio device provided
	 * the following conditions hold:
	 *
	 * (1)	the audio output isn't muted
	 *
	 * (2)	there won't be an echo problem i.e., we're not sending, OR
	 * 	we're in not in the mode where input has priority over output,
	 *	OR the mike is muted.  this last check is not strictly
	 *	necessary but allows the user to immediately hear the
	 *	far end after muting the mike (rather than waiting for
	 *	TALK_TAIL extra audio blocks to drain).
	 *
	 * (3)	the signal to output isn't silence, or we're running
	 *	a loopback test.  i.e., if we're about to write a
	 *	completely silent frame (i.e., no packets at all from
	 *	the network) then don't do it.  This prevents a backlog
	 *	of samples (i.e., a net delay) to build up in the audio
	 *	driver.  Note that this problem is completely independent
	 *	of the silent suppression solution for the outbound path.
	 */
	if (! audio_->PMuted() &&
	    (!sending() || audio_->Mode() != Audio::mode_mikemutesnet ||
	     audio_->RMuted()) &&
	    (loopback_ || test_tone_ || (ss.Max() != 0 && ss.LTMean() != 0))) {
		/*
		 * if we haven't written for a while,
		 * write an extra block to generate a bit
		 * of a backlog between us & the driver.
		 */
		u_int32_t sc = as_->Clock();
		if (u_int(sc - lastaudout_) > (u_int)(4*blkByteSize)) {
			audio_->Write(ss.BlkBack(blkByteSize));
			// printf("writing a back block\n");
		}
		audio_->Write(ss.CurBlk());
		lastaudout_ = sc;
		active_ = 1;
#ifdef notyet
		if (mode == mode_ec && !rmute) {
			int offset = AdjustTime(0);
			u_char ecblk[MAXAUDIOSIZE+4];
			int resid = offset & 3;
			if (resid)
			  ecblk[0] = 0x7f7f7f7f;
			filter->Compute(os, &ecblk[resid],
					blksize);
			as->Mix(offset &~ 3, ecblk, blksize);
		}
#endif
	} else if (lastaudout_ + blkByteSize == as_->Clock() &&
		   audio_->NeedsSilence()) {
	  audio_->Write((u_char *)ss.Zero());
	}
	as_->Mix(0, blk, blkByteSize);
}


HDController::HDController()
{
	timeval tv;
	::gettimeofday(&tv, 0);
	tsec_ = tv.tv_sec;
	tusec_ = tv.tv_usec;
	msched(timer_interval_);
}

void HDController::update(Observable*)
{
	/*
	 * do nothing -- leave the timer running even when
	 * we have the audio
	 */
}

void HDController::timeout()
{
	timeval tv;
	::gettimeofday(&tv, 0);
	u_int u = (u_int)tv.tv_usec;
	u_int s = (u_int)tv.tv_sec;

	if (s > tsec_ + 3) {
	  // We're way behind.  Most likely we were suspended and
	  // then resumed.

	  printf("More than 3 seconds behind\n");
	}
	while ((int(tusec_ - u) <= 0 && s == tsec_) || int(tsec_ - s) < 0) {
		if (audio_->haveaudio())
			if (!(audio_->RMuted() & 1)) {
				if (audio_->FrameReady())
					DoAudio();
			} else
				DoAudio();

		DoTimer();
		tusec_ += 1000 * frame_time;
		while (tusec_ >= 1000000) {
		        tusec_ -= 1000000;
			++tsec_;
		}
	}
	msched(timer_interval_);
}

void HDController::audio_handle()
{
  	printf("HDController::audio_handle() called\n");
	exit(1);
}

int Controller::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		/*
		 * <otcl> AudioController public ntp_time {}
		 * Return the current system time as the middle 32-bit
		 * of a standard 64-bit NTP timestamp.
		 */
		if (strcmp(argv[1], "ntp_time") == 0) {
			sprintf(tcl.buffer(), "%u", ntptime());
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public unix_time {}
		 * Return the number of seconds since Jan 1, 1970 UTC.
		 */
		if (strcmp(argv[1], "unix_time") == 0) {
			sprintf(tcl.buffer(), "%lu", unixtime().tv_sec);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public media-time {}
		 * Return the current time according to the audio
		 * sample clock of the underlying device.
		 * This value is a 32-bit number that wraps
		 * and is not guaranteed to have a well-defined
		 * start value.
		 */
		if (strcmp(argv[1], "media-time") == 0) {
			sprintf(tcl.buffer(), "%u", as_->Clock());
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public active v
		 * Return or set the active status depending
		 * on whether the <i>v</i> argument is present.
		 * The active status is automatically set to 1
		 * whenever the AudioController sends audio
		 * packets to the network.  It is never automatically
		 * cleared.  Instead, the calling tcl script is
		 * responsible for clearing it if it wants to
		 * make use of this hook.  If $v$ is not present,
		 * the current status is returned.
		 */
		if (strcmp(argv[1], "active") == 0) {
			tcl.result(active_ ? "1" : "0");
			return (TCL_OK);
		}
		/* FIXME agc doesn't really work yet. */
		if (strcmp(argv[1], "agc-input") == 0) {
			sprintf(tcl.buffer(), "%d", as_->AGCLevel() / 10 - 10);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "agc-output") == 0) {
			sprintf(tcl.buffer(), "%d", ns_->AGCLevel() / 10 - 10);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
	} else if (argc == 3) {
		/*
		 * <otcl> AudioController public audio o
		 * Install the Audio object named by the <i>o</i>
		 * argument in this controller.  Only one audio
		 * object can be attached at any given time.
		 */
		if (strcmp(argv[1], "audio") == 0) {
			audio_ = (Audio*)TclObject::lookup(argv[2]);
			audio_->attach(this);
			audio_->handler(this);
			update(audio_);
			// get settings from the audio device
			audio_sps = audio_->getSampleRate();
			int samplesPerBlock = audio_->GetBlockSize();
			frame_time = (samplesPerBlock * 1000 / audio_sps);
#ifdef WIN32
			/* windows95 has trouble with 20ms events */
			timer_interval_ = (frame_time * 4);
#else
			timer_interval_ = (frame_time);
#endif
			if(audio_->use16bits()) {
			  // need to make bigger sample streams for 16 bit audio
			  as_ = new SampleStream(samplesPerBlock*2, max_playout_,
			       (TALK_LEAD+1)*samplesPerBlock*2, 0);
			  ns_ = new SampleStream(samplesPerBlock*2, max_playout_,
			       (TALK_LEAD+1)*samplesPerBlock*2, 0);
			  // tell the sample streams that they are 16 bit samples
			  as_->set16bitMode();
			  ns_->set16bitMode();
			  
			  // reset the timers
			  lastaudout_ = as_->Clock();
			  lastnetout_ = ns_->Clock();
			}
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public encoder enc
		 * Install the PCM_Encoder object named by the <i>enc</i>
		 * argument in this controller.  Only one audio
		 * encoder can be attached at any given time.
		 * When a frame of audio samples is ready to be
		 * sent, it is passed on to the ecnoder object
		 * (which in turn is generally connected to a
		 * a network object).  For example, the encoder might
		 * run a compression algorithm on the signal and
		 * encapsulate the result in RTP packets before
		 * sending it on to the network.
		 */
		if (strcmp(argv[1], "encoder") == 0) {
			encoder_ = (PCM_Encoder*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public input-meter meter
		 * Install the Transducer object named by the <i>meter</i>
		 * argument as the input level monitoring device
		 * in this controller.  Only one input-meter
		 * can be attached at any given time.
		 * When audio frame is read from the device, it's
		 * average power level is computed and this value
		 * is conveyed to the input-meter by calling
		 * the Transducer's set method.  A Transducer
		 * is an object that tranforms some series of
		 * real-valued numbers into some UI element.
		 */
		if (strcmp(argv[1], "input-meter") == 0) {
		  // ErikM - 08/18/00 - changed back to use a generic
		  //   transducer base class (and fixed it so it works that
		  //   way)
		  //rmeter_ = (VUMeter*)TclObject::lookup(argv[2]);
		  rmeter_ = (Transducer*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public loopback enable
		 * Arrange for the audio input from the underlying
		 * audio device to be loopbed back in software
		 * to the output.  If <i>enable</i> is true, loopback
		 * is enabled; otherwise, it is disabled.
		 */
		if (strcmp(argv[1], "loopback") == 0) {
			loopback_ = atoi(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public test_tone type
		 * Arrange for an audio test-tone to be generated
		 * to the audio output. <i>type</i> indicates
		 * which test-tone to use and may be one of:
		 * <ul>
		 * <li> none - no test tone
		 * <li> low - 6dBm sine wave
		 * <li> med - 0dBm sine wave
		 * <li> max - full scale sine wave
		 * </ul>
		 */
		if (strcmp(argv[1], "test_tone") == 0) {
			if (strcmp(argv[2], "low") == 0)
				test_tone_ = 1;
			else if (strcmp(argv[2], "med") == 0)
				test_tone_ = 2;
			else if (strcmp(argv[2], "max") == 0)
				test_tone_ = 3;
			else
				test_tone_ = 0;
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public output-meter meter
		 * Install the Transducer object named by the <i>meter</i>
		 * argument as the output level monitoring device
		 * in this controller.  Only one output-meter
		 * can be attached at any given time.
		 * Before each audio frame is written to the encoder object,
		 * it's average power level is computed and this value
		 * is conveyed to the output-meter by calling
		 * the Transducer's set method.  A Transducer
		 * is an object that tranforms some series of
		 * real-valued numbers into some UI element.
		 */
		if (strcmp(argv[1], "output-meter") == 0) {
		  // ErikM - 08/18/00 - changed back to use a generic
		  //   transducer base class (and fixed it so it works that
		  //   way)
		  //pmeter_ = (VUMeter*)TclObject::lookup(argv[2]);
		  pmeter_ = (Transducer*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public silence-thresh level
		 * Set the threshold for the silence suppression algorithm
		 * used by the AudioController to <i>level</i>.
		 * FIXME: need units of level.  Should be probably be dB.
		 */
		if (strcmp(argv[1], "silence-thresh") == 0) {
			int thresh = atoi(argv[2]);
			as_->ssthresh(thresh);
			ns_->ssthresh(thresh);
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public talk-thresh level
		 * Set the amount of additional thresholding that
		 * the controller applies to the silence suppression
		 * when the speaker is not talking.  This means
		 * there's a slight cliff to get over to start
		 * talking, but once the mike is active, it's
		 * harder for it to turn off.  This tends
		 * to do a good job at eliminating noise spikes
		 * (like typing) and artifical drop-outs in mid-sentence.
		 * FIXME: need units of level.  Should be probably be dB.
		 */
		if (strcmp(argv[1], "talk-thresh") == 0) {
			talk_thresh_ = atoi(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public echothresh level
		 * FIXME: need units of level.  Should be probably be dB.
		 */
		if (strcmp(argv[1], "echothresh") == 0) {
			echo_thresh_ = atoi(argv[2]);
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public echodelay units?
		 * FIXME: need units of level.  Should be probably be dB.
		 */
		if (strcmp(argv[1], "echodelay") == 0) {
			echo_suppress_time_ = atoi(argv[2]) / 20 * AUDIO_FRAMESIZE;
			return (TCL_OK);
		}
		/*
		 * <otcl> AudioController public blocks-per-packet n
		 * Set the number of audio frames to include in each packet
		 * to <i>n</i>.
		 */
		if (strcmp(argv[1], "blocks-per-packet") == 0) {
			outmax_ = atoi(argv[2]) * audio_->GetBlockSize();
			//if (audio_->use16bits())
			//  outmax_ = outmax_*2;
			return (TCL_OK);
		}
		/* FIXME agc doesn't work */
		if (strcmp(argv[1], "agc-input") == 0) {
			int level = atoi(argv[2]);
			level = 10 * (level + 10);
			as_->SetAGCLevel(level);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "agc-input-enable") == 0) {
			as_->DoAGC(atoi(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "agc-output") == 0) {
			int level = atoi(argv[2]);
			level = 10 * (level + 10);
			ns_->SetAGCLevel(level);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "agc-output-enable") == 0) {
			ns_->DoAGC(atoi(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "active") == 0) {
			active_ = atoi(argv[2]);
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}
