/*
 * module.h --
 *
 *      Various Module header for video codec
 *
 * Copyright (c) 1995-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/module.h,v 1.16 2002/02/03 03:13:33 lim Exp $
 */

#ifndef mash_module_h
#define mash_module_h

#include "tclcl.h"
#include "config.h"
#include "rtp.h"
#ifdef DMALLOC
#include "dmalloc.h"
#endif

class RTP_BufferPool;
class pktbuf;

//----------------------------------------------------------------------
// Class:
//   Buffer
// Description:
//   This abstract class is one of the most basic class in Mash.  
//   It represents the data that flows from one module to another.
//----------------------------------------------------------------------

class Buffer {
public:
	virtual Buffer* copy() = 0;
	virtual void release();
};


//----------------------------------------------------------------------
// Class:
//   AudioFrame
//
// Superclass:
//   Buffer
//
// Description:
//   Represents a buffer of audio samples.  
//
// Members:
//   ts_  -- timestamp of the samples.
//   bp_  -- a pointer to the audio samples.
//   len_ -- the length of this buffer.
//
// See Also:
//   PCM_Encoder, ADPCMTranscoder, PCMTranscoder
//----------------------------------------------------------------------

struct AudioFrame : public Buffer {
public:
	inline Buffer* copy() { return 0; }//currently not used
	inline AudioFrame(u_int32_t ts, const u_int8_t* bp, int len) :
		ts_(ts), bp_(bp), len_(len) {}
	u_int32_t ts_;
	const u_int8_t* bp_;
	int len_;
};


//----------------------------------------------------------------------
// Class:
//   VideoFrame
//
// Superclass:
//   Buffer
//
// Description:
//   Represents a buffer for a video frame.
//
// Members:
//   ts_  -- timestamp of the samples.
//   bp_  -- a pointer to the audio samples.
//   width_,height_ -- the width and height of this video frame.
//   layer_ -- layer of this frame (for layered coding)
//----------------------------------------------------------------------

class VideoFrame : public Buffer {
    public:
	inline Buffer* copy() { return 0; }//currently not used
	inline VideoFrame(u_int32_t ts, u_int8_t* bp, int w, int h,
			  int layer) :
		ts_(ts), bp_(bp), width_(w), height_(h), layer_(layer) { }
	u_int32_t ts_;
	u_int8_t* bp_;
	int width_;
	int height_;
	int layer_;
};


//----------------------------------------------------------------------
// Class:
//   YuvFrame
//
// Superclass:
//   VideoFrame
//
// Description:
//   Represents a buffer for a video frame with conditional replenishment
//   Vector.
//
// Members:
//   crvec_ -- an array of conditional replenishment flags.
//----------------------------------------------------------------------

class YuvFrame : public VideoFrame {
    public:
	inline YuvFrame(u_int32_t ts, u_int8_t* bp, u_int8_t* crvec,
			int w, int h, int layer = 0) :
		VideoFrame(ts, bp, w, h, layer), crvec_(crvec) {}
	const u_int8_t* crvec_;
};


//----------------------------------------------------------------------
// Class:
//   JpegFrame
//
// Superclass:
//   VideoFrame
//
// Description:
//   Represents a buffer for a JPEG-encoded video frame.  Typically 
//   this frame comes from a hardware capture card.
//
// Members:
//   len_ -- size of the frame in bytes.
//   q_ -- quantization factor.
//   type_ -- type of the frame (as defined in RTP JPEG header).
//----------------------------------------------------------------------

class JpegFrame : public VideoFrame {
    public:
	inline JpegFrame(u_int32_t ts, u_int8_t* bp, int len, int q, int type,
			int w, int h) :
		VideoFrame(ts, bp, w, h, 0), len_(len), q_(q), type_(type) {}
	int len_;
	int q_;
	int type_;
};


//----------------------------------------------------------------------
// Class:
//   DCTFrame
//
// Superclass:
//   VideoFrame
//
// Description:
//   Represents a buffer for a DCT-encoded video frame.  Typically 
//   for H26X family of codec.
//
// Members:
//   q_ -- quantization factor.
//   crvec_ -- conditional replenishment vector.
//----------------------------------------------------------------------

class DCTFrame : public VideoFrame {
    public:
	inline DCTFrame(u_int32_t ts, short* bp, u_int8_t* crv,
		int w, int h, int q = -1) :
		VideoFrame(ts, (u_int8_t*)bp, w, h, 0), crvec_(crv), q_(q) {}

	const u_int8_t *crvec_;
	int q_;			// original q (if applicable)
};


//----------------------------------------------------------------------
// Class:
//   Module
//
// Superclass:
//   TclObject
//
// Description:
//   Module is one of the most basic class in OpenMash.  It represents
//   an object with multiple input ports and a single output port.
//   Multiple modules can be linked together through a member called 
//   target_.  Data is typically pushed into a Module, by calling 
//   recv().  The module processes the data, and pushes it down to 
//   target_ (by calling target_->recv(..)).
//   
//   The concept is similar to "filters" in DirectX and "objects" in 
//   Berkeley CMT.
//
// Members:
//   q_ -- quantization factor.
//   crvec_ -- conditional replenishment vector.
//----------------------------------------------------------------------

class Module : public TclObject {
public:
	Module* next;
	virtual void recv(Buffer*) = 0;
	virtual void sched_exec (Buffer *buf) { recv(buf); }
	int command(int argc, const char*const* argv);
	inline Module* target() { return (target_); }
protected:
	Module();
	Module* target_;
};

//----------------------------------------------------------------------
// Class:
//   SourceModule
//
// Superclass:
//   Module
//
// Description:
//   A SourceModule is a Module without input ports, i.e. no meaningful
//   recv() calls.
//
//----------------------------------------------------------------------

class SourceModule : public Module {
public:
	virtual void recv(Buffer*) {}
};


//----------------------------------------------------------------------
// Class:
//   FrameModule
//
// Superclass:
//   Module
//
// Description:
//   A FrameModule is a Module that deals with frames.  It keeps some 
//   states about previous frames that it processed. The states include
//   width, and height and frame size.
// 
// Members:
//   width_ -- width of the frame in some unit (pixel, blocks..).
//   height_ -- height of the frame in some unit (pixel, blocks..).
//   framesize_ -- precomputed width_*height_.
//
//----------------------------------------------------------------------

class FrameModule : public Module {
protected:
	FrameModule();
	inline void size(int w, int h) {
		width_ = w;
		height_ = h;
		framesize_ = w * h;
	}
	inline int samesize(const VideoFrame* vf) {
		return (vf->width_ == width_ && vf->height_ == height_);
	}
	int width_;
	int height_;
	int framesize_;
};


//----------------------------------------------------------------------
// Class:
//   PacketModule
//
// Superclass:
//   Module
//
// Description:
//   A PacketModule is a Module that deals with packets.  The data
//   passed into a PacketModule through recv() is of type pktbuf*.
// 
//   Important!!!  Subclasses of PacketModule must call pb->release() 
//   on the pktbuf that is passed to recv() if it's a data packet or 
//   if they pass the packet to someone, that someone needs to make
//   make sure pb->release is called.
//----------------------------------------------------------------------

class PacketModule : public Module {
public:
	virtual void recv(Buffer* p) {
		recv((pktbuf*)p);
	}
	virtual void recv(pktbuf* pb) = 0;
};


//----------------------------------------------------------------------
// Class:
//   EncoderModule
//
// Superclass:
//   FrameModule
//
// Description:
//   An EncoderModule is the base class for writing an encoder.  It keeps
//   buffer pool for RTP packets to be sent, and some states about the
//   encoding process (e.g. maximum transfer unit, number of bytes so far).
//
// Members:
//   pool_ -- An RTP buffer pool.
//   mtu_ -- Maximum transfer unit.  This is needed to determine the size
//           of the packet.
//   nb_ -- Number of bytes encoded so far.
//----------------------------------------------------------------------

class EncoderModule : public FrameModule {
public:
	virtual int command(int argc, const char*const* argv);
	inline u_int32_t nb() const { return nb_; }
protected:
	EncoderModule();
	RTP_BufferPool* pool_;
	int mtu_;
	u_int32_t nb_;
};


//----------------------------------------------------------------------
// Class:
//   MultiInputManager
//
// Description:
//   A pure abstract class that defines two methods for recving packets.
//   recv_data() and recv_ctrl().
//
//----------------------------------------------------------------------

class MultiInputManager {
public:
	virtual void recv_data(pktbuf*) = 0;
	virtual void recv_ctrl(pktbuf*) = 0;
};


//----------------------------------------------------------------------
// Class:
//   DataInputHandler
//
// Superclass:
//   PacketModule
//
// Description:
//   A DataInputHandler is a PacketModule that treats all packets received
//   as DataPacket.
// 
// See Also:
//   ControlInputHandler.
//----------------------------------------------------------------------

class DataInputHandler : public PacketModule {
public:
	DataInputHandler() : manager_(0) {}
	inline void manager(MultiInputManager* m) { manager_ = m; }
	virtual void recv(pktbuf* pb) {
		manager_->recv_data(pb);
	}
protected:
	MultiInputManager* manager_;
};


//----------------------------------------------------------------------
// Class:
//   ControlInputHandler
//
// Superclass:
//   DataInputHandler
//
// Description:
//   A ControlInputHandler is a PacketModule that treats all packets 
//   received as control packet.
// 
// See Also:
//   DataInputHandler.
//----------------------------------------------------------------------

class ControlInputHandler : public DataInputHandler {
public:
	virtual void recv(pktbuf* pb) {
		manager_->recv_ctrl(pb);
	}
};

#endif
