/*
 * fca-pkt.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <tclcl.h>
#include "config.h"
#include "misc/all-types.h"
#include "misc/debug.h"
#include "misc/nethost.h"
#include "fc.h"
#include "fca-pkt.h"


/*#define FCA_VERSION 1

#define PKT_FLOOR_REQUEST        1
#define PKT_FLOOR_CANCEL         2
#define PKT_FLOOR_RELEASE        3
#define PKT_GRANT_UPDATE         4
#define PKT_QUEUE_UPDATE         5
#define PKT_MODERATOR_SA         6
#define PKT_PARTICIPANT_SA       7
#define PKT_PARTICIPANT_RREQ     8
#define PKT_MODERATOR_RREQ       9
#define PKT_PARTICIPANT_RREPLY   10
#define PKT_MODERATOR_RREPLY     11


class FCA_Packet : public TclObject {
public:
  static FCA_Packet *New();
  static void Delete(FCA_Packet *packet);

  FCA_Packet() { }

  Bool Extract(u_char *&pb, int &len);
  Bool ExtractFloorRequest (u_char *&pb, int &len);
  Bool ExtractFloorCancel  (u_char *&pb, int &len);
  Bool ExtractFloorRelease (u_char *&pb, int &len);
  Bool ExtractModeratorSA  (u_char *&pb, int &len);
  Bool ExtractParticipantSA(u_char *&pb, int &len);
  Bool ExtractModeratorUpdate  (u_char *&pb, int &len, u_int32_t pktType);
  Bool ExtractGrantUpdateHelper(u_char *&pb, int &len);
  Bool ExtractQueueUpdateHelper(u_char *&pb, int &len);

  Bool Packetize(u_char *pb, int &len);
  Bool PacketizeFloorRequest (u_char *&pb, int &len);
  Bool PacketizeFloorCancel  (u_char *&pb, int &len);
  Bool PacketizeFloorRelease (u_char *&pb, int &len);
  Bool PacketizeModeratorSA  (u_char *&pb, int &len);
  Bool PacketizeParticipantSA(u_char *&pb, int &len);
  Bool PacketizeModeratorUpdate  (u_char *&pb, int &len, u_int32_t pktType);
  Bool PacketizeGrantUpdateHelper(u_char *&pb, int &len);
  Bool PacketizeQueueUpdateHelper(u_char *&pb, int &len);

private:
  void SetVar(const char *varname, const char *value);
  void SetVar(const char *varname, u_int32_t value);
  void SetVar(const char *varname, u_int16_t value);

  void LAppendVar(const char *varname, const char *value);
  void LAppendVar(const char *varname, u_int16_t value);

  const char *GetVar(const char *varname);
  void GetVar(const char *varname, u_int32_t &value);
  void GetVar(const char *varname, u_int16_t &value);
  void GetVar(const char *varname, SrcId &value);
  void DGetVar(const char *varname, char *&value);

  const char *LIndex(const char *varname, int index);
  void LIndex(const char *varname, int index, u_int16_t &value);
  void LIndex(const char *varname, int index, FCA_Packet *&value);

  inline int roundoff(int value) {
    return ((value % 4 == 0) ? value : (value/4 + 1) * 4);
  }

  inline void incr(u_char *&pb, int &len, int howMuch) {
    pb  += howMuch;
    len -= howMuch;
  }
};*/


static char *pktTypeStrings[] = {
    "",
    "PKT_FLOOR_REQUEST",
    "PKT_FLOOR_CANCEL",
    "PKT_FLOOR_RELEASE",
    "PKT_GRANT_UPDATE",
    "PKT_QUEUE_UPDATE",
    "PKT_MODERATOR_SA",
    "PKT_PARTICIPANT_SA",
    "PKT_PARTICIPANT_RREQ",
    "PKT_MODERATOR_RREQ",
    "PKT_PARTICIPANT_RREPLY",
    "PKT_MODERATOR_RREPLY",
    "PKT_OBSOLETE"
};



static class FCA_PacketClass : public TclClass {
public:
    FCA_PacketClass() : TclClass("FCA_Packet") { }
    ~FCA_PacketClass() { }
    TclObject *create(int /*argc*/, const char*const* /*argv*/) {
	return new FCA_Packet;
    }
} fca_PacketClass;





/*
 * Format of the FloorControl packet header.
 * The transport level header.
 */
struct Pkt_FCA_Hdr {
    u_int16_t version;		/* version number */
    u_int16_t pktType;            /* the type of packet */
};


/*
 * Packet header contains (paran contain the OTcl instvar name for the field):
 *   Version: 2 bytes (not stored)
 *   Packet Type: 2 bytes (pktType)
 */

Bool
FCA_Packet::Extract(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_FCA_Hdr))) {
	Debug(dbgFCA, ("Warning: FCA packet is too small (%d bytes); ignoring",
		       len));
	return FALSE;
    }

    Pkt_FCA_Hdr *pHdr = (Pkt_FCA_Hdr*)pb;
    u_int16_t version = net2host(pHdr->version);
    if (version!=FCA_VERSION) {
	Debug(dbgFCA, ("Warning: Version mismatch: expected %d, got %d",
		       FCA_VERSION, version));
	return FALSE;
    }

    u_int16_t pktType = net2host(pHdr->pktType);
    this->SetVar("pktType", pktTypeStrings[pktType]); // <---

    incr(pb, len, sizeof(Pkt_FCA_Hdr));


    switch (pktType) {
    case PKT_FLOOR_REQUEST:
	return ExtractFloorRequest(pb, len);

    case PKT_FLOOR_CANCEL:
	return ExtractFloorCancel(pb, len);

    case PKT_FLOOR_RELEASE:
	return ExtractFloorRelease(pb, len);

    case PKT_GRANT_UPDATE:
	return ExtractModeratorUpdate(pb, len, pktType);

    case PKT_QUEUE_UPDATE:
	return ExtractModeratorUpdate(pb, len, pktType);

    case PKT_MODERATOR_SA:
	return ExtractModeratorSA(pb, len);

    case PKT_PARTICIPANT_SA:
	return ExtractParticipantSA(pb, len);

    case PKT_PARTICIPANT_RREQ:
	return ExtractParticipantRreq(pb, len);

    case PKT_MODERATOR_RREQ:
	return ExtractModeratorRreq(pb, len);

    case PKT_PARTICIPANT_RREPLY:
	return ExtractParticipantRreply(pb, len);

    case PKT_MODERATOR_RREPLY:
	return ExtractModeratorRreply(pb, len);

    default:
	Debug(dbgFCA, ("Invalid packet type; ignoring it"));
	return FALSE;
    }
}


Bool
FCA_Packet::Packetize(u_char *pb, int &len)
{
    u_char *origPB = pb;
    Bool returnValue;
    if (len < int(sizeof(Pkt_FCA_Hdr))) {
	Debug(dbgFCA, ("Warning: FCA packet is too small (%d bytes); ignoring",
		       len));
	return FALSE;
    }

    const char *pktTypeStr = this->GetVar("pktType");
    u_int16_t pktType = 0;
    for (int i=0; i < int(sizeof(pktTypeStrings)/sizeof(char*)); i++) {
	if (strcmp(pktTypeStr, pktTypeStrings[i])==0) {
	    pktType = i;
	    break;
	}
    }

    Pkt_FCA_Hdr *pHdr = (Pkt_FCA_Hdr*) pb;
    pHdr->version = host2net((u_int16_t)FCA_VERSION);
    pHdr->pktType = host2net(pktType);

    incr(pb, len, sizeof(Pkt_FCA_Hdr));

    switch (pktType) {
    case PKT_FLOOR_REQUEST:
	returnValue = PacketizeFloorRequest(pb, len);
	break;

    case PKT_FLOOR_CANCEL:
	returnValue = PacketizeFloorCancel(pb, len);
	break;

    case PKT_FLOOR_RELEASE:
	returnValue = PacketizeFloorRelease(pb, len);
	break;

    case PKT_GRANT_UPDATE:
	returnValue = PacketizeModeratorUpdate(pb, len, pktType);
	break;

    case PKT_QUEUE_UPDATE:
	returnValue = PacketizeModeratorUpdate(pb, len, pktType);
	break;

    case PKT_MODERATOR_SA:
	returnValue = PacketizeModeratorSA(pb, len);
	break;

    case PKT_PARTICIPANT_SA:
	returnValue = PacketizeParticipantSA(pb, len);
	break;

    case PKT_PARTICIPANT_RREQ:
	return PacketizeParticipantRreq(pb, len);

    case PKT_MODERATOR_RREQ:
	return PacketizeModeratorRreq(pb, len);

    case PKT_PARTICIPANT_RREPLY:
	return PacketizeParticipantRreply(pb, len);

    case PKT_MODERATOR_RREPLY:
	return PacketizeModeratorRreply(pb, len);

    default:
	Debug(dbgFCA, ("Invalid packet type; ignoring it"));
	return FALSE;
    }
    if (returnValue==TRUE) {
	len = pb - origPB;
    }
    return returnValue;
}



/*
 * PKT_FLOORREQUEST:
 *   Request ID: 4 bytes (requestId)
 *   Comment: some multiple of 4 bytes (comment)
 *   number of floors: 4 bytes (numFloors)
 *   floor type: sequence of 1 bytes (floors - Tcl list)
 *   (extra padding required to get total # of floor types to be multiple of 4)
 */
Bool
FCA_Packet::ExtractFloorRequest(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t requestId = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("requestId", requestId); // <---
    Debug (dbgFCA, ("requestId = %lu", requestId));
    if (len==0) {
	this->SetVar("comment", "");	// <---
    } else {
        this->SetVar("comment", (char*)pb); // <---
        Debug (dbgFCA, ("comment = %s", (char*)pb));

	incr(pb, len, roundoff(strlen((char*)pb) + 1));
    }
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t numFloors = net2host(*(u_int32_t*)pb);
    this->SetVar("numFloors", numFloors); // <---
    Debug (dbgFCA, ("numFloors = %lu", numFloors));
    incr(pb, len, sizeof(u_int32_t));
    if (len < roundoff(numFloors)) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    this->SetVar("floors", ""); // <---  (in case numFloors is 0)
    for (int i=0; i < int(numFloors); i++) {
	this->LAppendVar("floors", (u_int16_t) (*(pb+i))); // <---
	Debug (dbgFCA, ("floor = %lu", (*(pb+i))));
    }
    incr(pb, len, roundoff(numFloors));
    return TRUE;
}


Bool
FCA_Packet::PacketizeFloorRequest(u_char *&pb, int &len)
{
    u_int32_t requestId, numFloors;
    char *comment;

    this->GetVar("requestId", requestId);
    this->DGetVar("comment", comment);
    this->GetVar("numFloors", numFloors);

    if (len < int(sizeof(u_int32_t) + roundoff(strlen(comment)+1)
		  + sizeof(u_int32_t) + numFloors * sizeof(Byte)) ) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    *((u_int32_t*) pb) = host2net(requestId); // <---
    incr(pb, len, sizeof(u_int32_t));

    strcpy((char*)pb, comment);		// <---
    incr(pb, len, roundoff(strlen(comment)+1));
    delete [] comment;

    *((u_int32_t*) pb) = host2net(numFloors); // <---
    incr(pb, len, sizeof(u_int32_t));

    u_int16_t floorType;
    for (int i=0; i < int(numFloors); i++) {
	this->LIndex("floors", i, floorType);
	*(pb+i) = (u_char) floorType; // <---
    }
    incr(pb, len, roundoff(numFloors));
    return TRUE;
}


/*
 * PKT_FLOORCANCEL:
 *   Sequence no: 4 bytes (seqno)
 *   Request ID:  4 bytes (requestId)
 */
Bool
FCA_Packet::ExtractFloorCancel(u_char *&pb, int &len)
{
    if (len < int(2 * sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t seqno = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("seqno", seqno); // <---

    u_int32_t requestId = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("requestId", requestId); // <---

    return TRUE;
}


Bool
FCA_Packet::PacketizeFloorCancel(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t uint;
    this->GetVar("seqno", uint);
    *((u_int32_t*) pb) = host2net(uint); // <---
    incr(pb, len, sizeof(u_int32_t));

    this->GetVar("requestId", uint);
    *((u_int32_t*) pb) = host2net(uint); // <---
    incr(pb, len, sizeof(u_int32_t));

    return TRUE;
}


struct Pkt_FloorRelease {
    u_int32_t seqno;
    u_int32_t grantSeqno;
    Byte      floorType;
    Byte      floorInstance;
    u_int16_t padding;
};


/*
 * PKT_FLOORRELEASE:
 *    struct Pkt_FloorRelease {
 *        u_int32_t seqno;
 *        u_int32_t grantSeqno;
 *        Byte      floorType;
 *        Byte      floorInstance;
 *        u_int16_t padding;
 *    };
 */
Bool
FCA_Packet::ExtractFloorRelease(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_FloorRelease))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    Pkt_FloorRelease *pRelease = (Pkt_FloorRelease*) pb;
    incr(pb, len, sizeof(Pkt_FloorRelease));

    this->SetVar("seqno", net2host(pRelease->seqno)); // <---
    this->SetVar("grantSeqno", net2host(pRelease->grantSeqno)); // <---
    this->SetVar("floorType", (u_int16_t) pRelease->floorType); // <---
    this->SetVar("floorInstance", (u_int16_t) pRelease->floorInstance); // <---
    return TRUE;
}


Bool
FCA_Packet::PacketizeFloorRelease(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_FloorRelease))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int16_t tmp;
    Pkt_FloorRelease *pUpdate = (Pkt_FloorRelease*) pb;
    this->GetVar("seqno", pUpdate->seqno); // <---
    this->GetVar("grantSeqno", pUpdate->grantSeqno); // <---
    this->GetVar("floorType", tmp);
    pUpdate->floorType = (Byte) tmp; // <---
    this->GetVar("floorInstance", tmp);
    pUpdate->floorInstance = (Byte) tmp; // <---
    incr(pb, len, sizeof(Pkt_FloorRelease));

    pUpdate->seqno        = host2net(pUpdate->seqno); // <---
    pUpdate->grantSeqno   = host2net(pUpdate->grantSeqno); // <---
    pUpdate->floorType    = host2net(pUpdate->floorType); // <---
    pUpdate->floorInstance= host2net(pUpdate->floorInstance); // <---
    return TRUE;
}





struct Pkt_ModeratorUpdate {
    u_int32_t moderatorState;
    u_int32_t numUpdates;
};


struct Pkt_GrantUpdate {
    SrcId     srcId;
    u_int32_t grantSeqno;
    u_int32_t requestId;
    Byte      floorType;
    Byte      floorInstance;
    u_int16_t padding;
};


struct Pkt_QueueUpdate {
    SrcId     srcId;
    u_int32_t requestId;
    Byte      isAdd;
    Byte      padding[3];
};


/*
 * PKT_MODERATOR_UPDATE:
 *    struct Pkt_ModeratorUpdate {
 *      u_int32_t moderatorState;       (moderatorState)
 *      u_int32_t numUpdates;           (numUpdates)
 *    };
 *
 *   a number of Pkt_GrantUpdate/Pkt_QueueUpdate's
 *   (pktUpdates - Tcl list of pkt objects)
 */
Bool
FCA_Packet::ExtractModeratorUpdate(u_char *&pb, int &len, u_int32_t pktType)
{
    Debug(dbgFCA, ("Got %d bytes in moderator update pkt", len));
    if (len < int(sizeof(Pkt_ModeratorUpdate))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    Pkt_ModeratorUpdate *pUpdate = (Pkt_ModeratorUpdate*) pb;
    u_int32_t numUpdates = net2host(pUpdate->numUpdates);
    this->SetVar("moderatorState", net2host(pUpdate->moderatorState)); // <---
    this->SetVar("numUpdates", numUpdates); // <---
    incr(pb, len, sizeof(Pkt_ModeratorUpdate));

    FCA_Packet *updatePkt;
    for (int i=0; i < int(numUpdates); i++) {
	updatePkt = FCA_Packet::New();
	switch (pktType) {
	case PKT_GRANT_UPDATE:
	    if (updatePkt->ExtractGrantUpdateHelper(pb, len)==FALSE) // <---
		return FALSE;
	    break;

	case PKT_QUEUE_UPDATE:
	    if (updatePkt->ExtractQueueUpdateHelper(pb, len)==FALSE) // <---
		return FALSE;
	    break;
	}
	this->LAppendVar("pktUpdates", updatePkt->name()); // <---
    }
    return TRUE;
}


Bool
FCA_Packet::PacketizeModeratorUpdate(u_char *&pb, int &len, u_int32_t pktType)
{
    if (len < int(sizeof(Pkt_ModeratorUpdate))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t moderatorState;
    this->GetVar("moderatorState", moderatorState);
    u_int32_t numUpdates;
    this->GetVar("numUpdates", numUpdates);

    Pkt_ModeratorUpdate *pUpdate = (Pkt_ModeratorUpdate*) pb;
    pUpdate->moderatorState = host2net(moderatorState); // <---
    pUpdate->numUpdates = host2net(numUpdates); // <---
    incr(pb, len, sizeof(Pkt_ModeratorUpdate));

    if (len < int(numUpdates * sizeof(Pkt_GrantUpdate))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    FCA_Packet *updatePkt;
    for (int i=0; i < int(numUpdates); i++) {
	this->LIndex("pktUpdates", i, updatePkt);

	switch (pktType) {
	case PKT_GRANT_UPDATE:
	    if (updatePkt->PacketizeGrantUpdateHelper(pb, len)==FALSE) // <---
		return FALSE;
	    break;

	case PKT_QUEUE_UPDATE:
	    if (updatePkt->PacketizeQueueUpdateHelper(pb, len)==FALSE) // <---
		return FALSE;
	    break;
	}
    }
    return TRUE;
}


/*
 * struct Pkt_GrantUpdate {
 *   SrcId     srcId;          (srcId)
 *   u_int32_t grantSeqno;     (grantSeqno)
 *   u_int32_t requestId;      (requestId)
 *   Byte      floorType;      (floorType)
 *   Byte      floorInstance;  (floorInstance)
 *   u_int16_t padding;
 * };
 */
Bool
FCA_Packet::ExtractGrantUpdateHelper(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_GrantUpdate))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    Pkt_GrantUpdate *pUpdate = (Pkt_GrantUpdate*) pb;
    incr(pb, len, sizeof(Pkt_GrantUpdate));
    SrcId sid  = pUpdate->srcId;
    sid.ss_uid = net2host(sid.ss_uid);

    this->SetVar("srcId", (const char*) sid); // <---
    this->SetVar("grantSeqno", net2host(pUpdate->grantSeqno)); // <---
    this->SetVar("requestId", net2host(pUpdate->requestId)); // <---
    this->SetVar("floorType", (u_int16_t) pUpdate->floorType); // <---
    this->SetVar("floorInstance", (u_int16_t) pUpdate->floorInstance); // <---
    //this->SetVar("isGrant", (u_int16_t) pUpdate->isGrant); // <---
    return TRUE;
}



Bool
FCA_Packet::PacketizeGrantUpdateHelper(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_GrantUpdate))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int16_t tmp;
    Pkt_GrantUpdate *pUpdate = (Pkt_GrantUpdate*) pb;
    this->GetVar("srcId", pUpdate->srcId); // <---
    this->GetVar("grantSeqno", pUpdate->grantSeqno); // <---
    this->GetVar("requestId", pUpdate->requestId); // <---
    this->GetVar("floorType", tmp);
    pUpdate->floorType = (Byte) tmp; // <---
    this->GetVar("floorInstance", tmp);
    pUpdate->floorInstance = (Byte) tmp; // <---
    /*this->GetVar("isGrant", tmp);
    pUpdate->isGrant = (Byte) tmp; // <---*/

    incr(pb, len, sizeof(Pkt_GrantUpdate));

    pUpdate->srcId.ss_uid = host2net(pUpdate->srcId.ss_uid); // <---
    pUpdate->grantSeqno   = host2net(pUpdate->grantSeqno); // <---
    pUpdate->requestId    = host2net(pUpdate->requestId);	// <---
    pUpdate->floorType    = host2net(pUpdate->floorType);	// <---
    pUpdate->floorInstance= host2net(pUpdate->floorInstance); // <---
    //pUpdate->isGrant      = host2net(pUpdate->isGrant); // <---
    return TRUE;
}



/*
 * PKT_QUEUEUPDATE
 *   struct Pkt_QueueUpdate {
 *     SrcId     srcID;               (srcId)
 *     u_int32_t requestId;           (requestId)
 *     Byte      isAdd;               (isAdd)  // isAdd=1 => I'm adding this
 *                                             // entry; 0 => deleting
 *     Byte      padding[3];
 *   };
 */
Bool
FCA_Packet::ExtractQueueUpdateHelper(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_QueueUpdate))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    Pkt_QueueUpdate *pUpdate = (Pkt_QueueUpdate*) pb;
    SrcId sid = pUpdate->srcId;
    sid.ss_uid = net2host(sid.ss_uid);

    this->SetVar("srcId", (const char*) sid); // <---
    this->SetVar("requestId", net2host(pUpdate->requestId)); // <---
    this->SetVar("isAdd", (u_int16_t) net2host(pUpdate->isAdd)); // <---

    incr(pb, len, sizeof(Pkt_QueueUpdate));
    return TRUE;
}


Bool
FCA_Packet::PacketizeQueueUpdateHelper(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_QueueUpdate))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int16_t tmpIsAdd;
    Pkt_QueueUpdate *pUpdate = (Pkt_QueueUpdate*) pb;
    this->GetVar("srcId", pUpdate->srcId); // <---
    this->GetVar("requestId", pUpdate->requestId); // <---
    this->GetVar("isAdd", tmpIsAdd); // <---

    pUpdate->srcId.ss_uid   = host2net(pUpdate->srcId.ss_uid); // <---
    pUpdate->requestId      = host2net(pUpdate->requestId); // <---
    pUpdate->isAdd          = host2net((Byte)tmpIsAdd); // <---

    incr(pb, len, sizeof(Pkt_QueueUpdate));
    return TRUE;
}


/*
 * PKT_MODERATOR_SA:
 *   current moderator state: 4 bytes (currentState)
 *   followed by PKT_PARTICIPANT_SA
 */
Bool
FCA_Packet::ExtractModeratorSA(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t currentState = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("currentState", currentState); // <---
    return this->ExtractParticipantSA(pb, len); // <--
}


Bool
FCA_Packet::PacketizeModeratorSA(u_char *&pb, int &len)
{
    u_int32_t currentState;
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    this->GetVar("currentState", currentState);
    currentState = host2net(currentState);
    *((u_int32_t*)pb) = currentState; // <---
    incr(pb, len, sizeof(u_int32_t));
    return this->PacketizeParticipantSA(pb, len);	// <---
}


struct Pkt_ParticipantSA {
    u_int32_t maxRequestId;
    u_int32_t maxImportantSeqno; /* cancel or release message */
};


/*
 * PKT_PARTICIPANT_SA:
 *    struct Pkt_ParticipantSA {
 *      u_int32_t maxRequestId;       (maxRequestId)
 *      u_int32_t maxImportantSeqno;  (maxImportantSeqno)
 *    };
 */
Bool
FCA_Packet::ExtractParticipantSA(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_ParticipantSA))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    Pkt_ParticipantSA *pSA = (Pkt_ParticipantSA*) pb;
    incr(pb, len, sizeof(Pkt_ParticipantSA));

    this->SetVar("maxRequestId", net2host(pSA->maxRequestId)); // <---
    this->SetVar("maxImportantSeqno", net2host(pSA->maxImportantSeqno)); //<---
    return TRUE;
}


Bool
FCA_Packet::PacketizeParticipantSA(u_char *&pb, int &len)
{
    if (len < int(sizeof(Pkt_ParticipantSA))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    Pkt_ParticipantSA *pSA = (Pkt_ParticipantSA*) pb;
    incr(pb, len, sizeof(Pkt_ParticipantSA));

    this->GetVar("maxRequestId", pSA->maxRequestId); // <---
    this->GetVar("maxImportantSeqno", pSA->maxImportantSeqno); // <---
    pSA->maxRequestId = host2net(pSA->maxRequestId); // <---
    pSA->maxImportantSeqno = host2net(pSA->maxImportantSeqno); // <---
    return TRUE;
}



struct Pkt_ParticipantRreq {
    u_int32_t isImportant; // isImportant=0 => floor_request
    u_int32_t start;
    u_int32_t end;
};


/*
 * PKT_PARTICIPANT_RREQ:
 *    number of requests: 4 bytes   (numRequests)
 *    followed by a number of Pkt_Participant_Rreq's (list)
 *        (each element in "list" is a triple of <isImportant, start, end>)
 */
Bool
FCA_Packet::ExtractParticipantRreq(u_char *&pb, int &len)
{

    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t numRequests = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("numRequests", numRequests); // <---
    Debug(dbgFCA, ("ExtractParticipantRreq, numRequests = %d", numRequests));
    if (len < int(numRequests * sizeof(Pkt_ParticipantRreq))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
    }

    this->SetVar("list", ""); // <---
    static char buffer[256];
    Pkt_ParticipantRreq *pRreq;
    pRreq = (Pkt_ParticipantRreq*) pb;
    incr(pb, len, numRequests * sizeof(Pkt_ParticipantRreq));

    for (int i=0; i < int(numRequests); i++, pRreq++) {
	sprintf(buffer, "%lu %lu %lu",
		(unsigned long) net2host(pRreq->isImportant),
		(unsigned long) net2host(pRreq->start),
		(unsigned long) net2host(pRreq->end));
	this->LAppendVar("list", buffer); // <---
	  Debug(dbgFCA, ("ExtractParticipantRreq, list buffer: %s",buffer));

    }
    return TRUE;
}


Bool
FCA_Packet::PacketizeParticipantRreq(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t numRequests;
    this->GetVar("numRequests", numRequests);
    Debug(dbgFCA, ("PacketizeParticipantRreq, numRequests = %d", numRequests));
    *((u_int32_t*) pb) = host2net(numRequests); // <---
    incr(pb, len, sizeof(u_int32_t));

    if (len < int(numRequests * sizeof(Pkt_ParticipantRreq))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
    }

    Pkt_ParticipantRreq *pRreq;
    pRreq = (Pkt_ParticipantRreq*) pb;
    incr(pb, len, numRequests * sizeof(Pkt_ParticipantRreq));
    const char *buffer;

    for (int i=0; i < int(numRequests); i++, pRreq++) {
	Debug(dbgFCA, ("About to call LIndex"));
	buffer = this->LIndex("list", i);
	Debug(dbgFCA, ("PacketizeParticipantRreq, list buffer: %s",buffer));
	pRreq->isImportant = 0;
	pRreq->start = 0;
	pRreq->end = 0;
	sscanf(buffer, "%lu %lu %lu", (unsigned long *) &pRreq->isImportant,
	       (unsigned long *) &pRreq->start,
	       (unsigned long *) &pRreq->end);
	Debug(dbgFCA, ("PacketizeParticipantRreq: list %lu %lu %lu", pRreq->isImportant, pRreq->start, pRreq->end));
	pRreq->isImportant = host2net(pRreq->isImportant);
	pRreq->start       = host2net(pRreq->start);
	pRreq->end         = host2net(pRreq->end);
	Debug(dbgFCA, ("PacketizeParticipantRreq: list %lu %lu %lu", pRreq->isImportant, pRreq->start, pRreq->end));
    }
    return TRUE;
}


/*
 * PKT_MODERATOR_RREQ:
 *    requestor's idea of the moderator's state #: 4 bytes   (moderatorState)
 */
Bool
FCA_Packet::ExtractModeratorRreq(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t moderatorState = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("moderatorState", moderatorState); // <---
    return TRUE;
}


Bool
FCA_Packet::PacketizeModeratorRreq(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t moderatorState;
    this->GetVar("moderatorState", moderatorState);
    *((u_int32_t*) pb) = host2net(moderatorState); // <---
    incr(pb, len, sizeof(u_int32_t));
    return TRUE;
}



/*
 * PKT_PARTICIPANT_RREPLY:
 *   number of sub-packets: 4 bytes  (numReplies)
 *   followed by a bunch of PKT_FLOOR_REQUEST, PKT_FLOOR_CANCEL,
 *   PKT_FLOOR_RELEASE, or PKT_OBSOLETE (pktReplies)
 *   (each packet contains an initial 4-byte pktType entry in addition to the
 *    usual stuff)
 */
Bool
FCA_Packet::ExtractParticipantRreply(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t numReplies = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("numReplies", numReplies); // <---
    Debug(dbgFCA, ("ExtractParticipantRreply, numReplies (%lu)",
		   numReplies));

    u_int32_t pktType;
    FCA_Packet *pkt;
    this->SetVar("pktReplies", "");
    for (int i=0; i < int(numReplies); i++) {
	if (len < int(sizeof(u_int32_t))) {
	    Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	    return FALSE;
	}
	Debug(dbgFCA, ("about to new the packet"));
	pkt = FCA_Packet::New();
	pktType = net2host(*(u_int32_t*)pb);
	incr(pb, len, sizeof(u_int32_t));
	pkt->SetVar("pktType", pktTypeStrings[pktType]); // <---
	Debug(dbgFCA, ("ExtractParticipantRreply, pktType (%lu)", pktType));
	switch(pktType) {
	case PKT_FLOOR_REQUEST:
	    Debug (dbgFCA, ("it is a floor request"));
	    if (pkt->ExtractFloorRequest(pb, len)==FALSE) {
		Debug (dbgFCA, ("ExtractParticipantRreply, "
				"unable to extract floor request"));
		FCA_Packet::Delete(pkt);
		return FALSE;
	    }
	    break;

	case PKT_FLOOR_CANCEL:
	    if (pkt->ExtractFloorCancel(pb, len)==FALSE) {
		FCA_Packet::Delete(pkt);
		return FALSE;
	    }
	    break;

	case PKT_FLOOR_RELEASE:
	    if (pkt->ExtractFloorRelease(pb, len)==FALSE) {
		FCA_Packet::Delete(pkt);
		return FALSE;
	    }
	    break;

	case PKT_OBSOLETE:
	    if (pkt->ExtractObsolete(pb, len)==FALSE) {
		FCA_Packet::Delete(pkt);
		return FALSE;
	    }
	    break;

	default:
	    Debug (dbgFCA, ("PacketizeParticipantRreply, wrong pktType"));
	}
	this->LAppendVar("pktReplies", pkt->name());
    }
    return TRUE;
}


Bool
FCA_Packet::PacketizeParticipantRreply(u_char *&pb, int &len)
{
    Debug (dbgFCA, ("PacketizeParticipantRreply, "
		    "should be called from fill_ADU"));
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t numReplies;
    this->GetVar("numReplies", numReplies);
    Debug (dbgFCA, ("PacketizeParticipantRreply, numReplies = %lu\n",
		    numReplies));
    *((u_int32_t*) pb) = host2net(numReplies); // <---
    incr(pb, len, sizeof(u_int32_t));

    //u_int32_t pktType = 0;
    FCA_Packet *pkt;

    for (int i=0; i < int(numReplies); i++) {
	this->LIndex("pktReplies", i, pkt);


	const char *pktTypeStr = pkt->GetVar("pktType");
	u_int32_t pktType = 0;
	for (int j=0; j < int(sizeof(pktTypeStrings)/sizeof(char*)); j++) {
	  if (strcmp(pktTypeStr, pktTypeStrings[j])==0) {
	    Debug (dbgFCA, ("pktType j = %lu, string == %s",
			    j, pktTypeStrings[j]));
	    pktType = j;
	    break;
	  }
	}


	//pkt->GetVar("pktType", pktType);
	Debug (dbgFCA, ("PacketizeParticipantRreply, pktType = %lu\n",
			pktType));
	*((u_int32_t*) pb) = host2net(pktType); // <---
	Debug (dbgFCA, ("PacketizeParticipantRreply, "
			"after converting to net, pktType = %lu\n",
			*((u_int32_t*) pb)));
	incr(pb, len, sizeof(u_int32_t));

	switch (pktType) {
	case PKT_FLOOR_REQUEST:
            Debug (dbgFCA,("PacketizeParticipantRreply, it is floor request"));
	    if (pkt->PacketizeFloorRequest(pb, len)==FALSE) return FALSE;
	    break;

	case PKT_FLOOR_CANCEL:
	    if (pkt->PacketizeFloorCancel(pb, len)==FALSE) return FALSE;
	    break;

	case PKT_FLOOR_RELEASE:
	    if (pkt->PacketizeFloorRelease(pb, len)==FALSE) return FALSE;
	    break;

	case PKT_OBSOLETE:
	    if (pkt->PacketizeObsolete(pb, len)==FALSE) return FALSE;
	    break;

	default:
	    Debug (dbgFCA, ("PacketizeParticipantRreply, wrong pktType"));
	}
    }
    return TRUE;
}


/*
 * PKT_OBSOLETE:
 *   seqno: 4 bytes (seqno)
 */
Bool
FCA_Packet::ExtractObsolete(u_char *&pb, int &len)
{
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    u_int32_t seqno = net2host(*(u_int32_t*)pb);
    incr(pb, len, sizeof(u_int32_t));
    this->SetVar("seqno", seqno); // <---
    return TRUE;
}


Bool
FCA_Packet::PacketizeObsolete(u_char *&pb, int &len)
{
    u_int32_t seqno;
    if (len < int(sizeof(u_int32_t))) {
	Debug(dbgFCA, ("Error in packet; too few bytes: %d", len));
	return FALSE;
    }

    this->GetVar("seqno", seqno);
    seqno = host2net(seqno);
    *((u_int32_t*)pb) = seqno; // <---
    incr(pb, len, sizeof(u_int32_t));
    return TRUE;
}









/*
 * PKT_MODERATOR_RREPLY:
 *   a PKT_GRANT_UPDATE (pktGrantUpdate) followed by
 *   a PKT_QUEUE_UPDATE (pktQueueUpdate)
 */
Bool
FCA_Packet::ExtractModeratorRreply(u_char *&pb, int &len)
{
    FCA_Packet *pktGrant, *pktQueue;

    pktGrant = FCA_Packet::New();
    if (pktGrant->ExtractModeratorUpdate(pb, len, PKT_GRANT_UPDATE)==FALSE) {
	FCA_Packet::Delete(pktGrant);
	return FALSE;
    }

    pktQueue = FCA_Packet::New();
    if (pktQueue->ExtractModeratorUpdate(pb, len, PKT_QUEUE_UPDATE)==FALSE) {
	FCA_Packet::Delete(pktGrant);
	FCA_Packet::Delete(pktQueue);
	return FALSE;
    }
    this->LAppendVar("pktGrantUpdate", pktGrant->name());
    this->LAppendVar("pktQueueUpdate", pktQueue->name());
    return TRUE;
}


Bool
FCA_Packet::PacketizeModeratorRreply(u_char *&pb, int &len)
{
    FCA_Packet *pktGrant, *pktQueue;
    this->LIndex("pktGrantUpdate", 0, pktGrant);
    this->LIndex("pktQueueUpdate", 0, pktQueue);

    if (pktGrant->PacketizeModeratorUpdate(pb, len, PKT_GRANT_UPDATE)==FALSE) {
	return FALSE;
    }

    if (pktQueue->PacketizeModeratorUpdate(pb, len, PKT_QUEUE_UPDATE)==FALSE) {
	return FALSE;
    }
    return TRUE;
}











#ifdef OLD
Bool
Packetizer::ExtractHeader(u_char *&pb, int &len, SrcId &sid)
{
    if (len < int(sizeof(Pkt_FCA_Hdr))) {
	Debug(dbgFCA, ("Warning: FCA packet is too small (%d bytes); ignoring",
		       len));
	return FALSE;
    }
    Pkt_FCA_Hdr *pHdr = (Pkt_FCA_Hdr*) pb;
    u_int16_t version = net2host(pHdr->version);
    if (version!=FCA_VERSION) {
	Debug(dbgFCA, ("Warning: Version mismatch: expected %d, got %d",
		       FCA_VERSION, version));
	return FALSE;
    }
    sid = pHdr->srcId;
    sid.ss_uid = net2host(sid.ss_uid);
    // sid.ss_addr is in network format; leave it that way!

    pb  += sizeof(Pkt_FCA_Hdr);
    len -= sizeof(Pkt_FCA_Hdr);
    return TRUE;
}


Bool
Packetizer::ExtractRReq(u_char *&pb, int &len, FCA_Packet *pkt)
{
    SrcId sid;
    if (len < sizeof(Pkt_FCA_RReq)) {
	Debug(dbgFCA, ("Invalid repair-request packet: expected %d bytes, got %d",
		       sizeof(Pkt_FCA_RReq), len));
	return FALSE;
    }

    Pkt_FCA_RReq *pRreq = (Pkt_FCA_RReq*) pb;
    sid = pRreq->srcId;
    sid.ss_uid = net2host(sid.ss_uid);

    this->SetVar("srcid",   (const char*)sid);
    this->SetVar("type",    net2host(pRreq->rreqType));
    this->SetVar("startid", net2host(pRreq->startRequestId));
    this->SetVar("endid",   net2host(pRreq->endRequestId));

    pb  += sizeof(Pkt_FCA_RReq);
    len -= sizeof(Pkt_FCA_RReq);
    return TRUE;
}


Bool
Packetizer::ExtractRReply(u_char *&pb, int &len, FCA_Packet *pkt)
{
    SrcId sid;
    if (len < sizeof(Pkt_FCA_RReplyHdr)) {
	Debug(dbgFCA, ("Invalid repair-reply packet: expected %d bytes, got %d",
		       sizeof(Pkt_FCA_RReply), len));
	return FALSE;
    }

    Pkt_FCA_RReplyHdr *pRReplyHdr = (Pkt_FCA_RReplyHdr*) pb;
    pb  += sizeof(Pkt_FCA_RReplyHdr);
    len -= sizeof(Pkt_FCA_RReplyHdr);

    sid = pRReplyHdr->srcId;
    sid.ss_uid = net2host(sid.ss_uid);
    type = net2host(pRReplyHdr->rreplyType);

    pkt = packetizer_->NewPacket();
    this->SetVar("srcid", (const char*)sid);
    this->SetVar("type",  type);
    switch(type) {
    case rreqFloorCancel:
	if (len < sizeof(u_int32_t)) {
	    Debug(dbgFCA, ("Invalid repair-reply packet: expected %d bytes, got %d",
			   sizeof(Pkt_FCA_RReply), len));
	    return FALSE;
	}
	numReplies = net2host(*(u_int32_t*)pb);
	this->SetVar("numReplies", numReplies);
	pb  += sizeof(u_int32_t);
	len -= sizeof(u_int32_t);

	for (idx=0; idx < numReplies; idx++) {
	    if (ExtractFloorRequest(pb, len ,pkt)==FALSE) return FALSE;
	}
	return TRUE;

    case rreqFloorRequest:
	if (len < sizeof(u_int32_t)) {
	    Debug(dbgFCA, ("Invalid repair-reply packet: expected %d bytes, got %d",
			   sizeof(Pkt_FCA_RReply), len));
	    return FALSE;
	}
	numReplies = net2host(*(u_int32_t*)pb);
	this->SetVar("numReplies", numReplies);
	pb  += sizeof(u_int32_t);
	len -= sizeof(u_int32_t);

	for (idx=0; idx < numReplies; idx++) {
	    if (ExtractFloorRequest(pb, len ,pkt)==FALSE) return FALSE;
	}
	return TRUE;


    case rreqModeratorState:

    };
}
#endif




FCA_Packet *
FCA_Packet::New()
{
    Tcl::instance().evalf("new FCA_Packet");
    return (FCA_Packet*) Tcl::instance().lookup(Tcl::instance().result());
}


void
FCA_Packet::Delete(FCA_Packet *packet)
{
    Tcl::instance().evalf("delete {%s}", packet->name());
}


void
FCA_Packet::SetVar(const char *varname, const char *value)
{
    Tcl::instance().evalf("{%s} set {%s} {%s}", name(), varname, value);
}


void
FCA_Packet::SetVar(const char *varname, u_int32_t value)
{
    Tcl::instance().evalf("{%s} set {%s} {%lu}", name(), varname,
			  (unsigned long)value);
}


void
FCA_Packet::SetVar(const char *varname, u_int16_t value)
{
    Tcl::instance().evalf("{%s} set {%s} {%u}", name(), varname,
			  (unsigned)value);
}


void
FCA_Packet::LAppendVar(const char *varname, const char *value)
{
    Tcl::instance().evalf("{%s} lappend {%s} {%s}", name(), varname, value);
}


void
FCA_Packet::LAppendVar(const char *varname, u_int16_t value)
{
    Tcl::instance().evalf("{%s} lappend {%s} {%lx}", name(), varname,
			  (unsigned long) value);
}


const char *
FCA_Packet::GetVar(const char *varname)
{
    Tcl::instance().evalf("{%s} set {%s}", name(), varname);
    return Tcl::instance().result();
}


void
FCA_Packet::DGetVar(const char *varname, char *&value)
{
    const char *result = GetVar(varname);
    value = new char [strlen(result)+1];
    strcpy(value, result);
}


void
FCA_Packet::GetVar(const char *varname, u_int32_t &value)
{
    long unsigned tmp;
    sscanf(GetVar(varname), "%lu", &tmp);
    value = (u_int32_t) tmp;
}



void
FCA_Packet::GetVar(const char *varname, u_int16_t &value)
{
    long unsigned tmp;
    sscanf(GetVar(varname), "%lu", &tmp);
    value = (u_int16_t) tmp;
}



void
FCA_Packet::GetVar(const char *varname, SrcId &value)
{
    value = GetVar(varname);
}


const char *
FCA_Packet::LIndex(const char *varname, int index)
{
    Debug(dbgFCA, ("Evaling {%s} lindex {%s} {%d}", name(), varname, index));
    Tcl::instance().evalf("{%s} lindex {%s} {%d}", name(), varname, index);
    return Tcl::instance().result();
}


void
FCA_Packet::LIndex(const char *varname, int index, u_int16_t &value)
{
    long unsigned tmp;
    sscanf(LIndex(varname, index), "%lu", &tmp);
    value = (u_int16_t) tmp;
}


void
FCA_Packet::LIndex(const char *varname, int index, FCA_Packet *&value)
{
    value = (FCA_Packet*) Tcl::instance().lookup(LIndex(varname, index));
}







/*void
FCA_Packet::StrAppendVar(const char *varname, const char *value)
{
Tcl::instance().evalf("%s set %s \"[%s set]%s\"", name(), varname, name(),
value);
}*/

