/*
 * mb-basepage.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-basepage.cc,v 1.23 2002/02/03 03:16:30 lim Exp $
 */

#include "mb/mb-obj.h"
#include "mb-cmd.h"
#include "mb/mb-nethost.h"

int MBPageObject::stepSize_=1000;

MBPageObject::MBPageObject(MBBaseMgr *pMgr, const PageId &pgid)
	: targetTime_(cMBTimeAny), pMgr_(pMgr), szName_(NULL),
	  id_(pgid), pRequest_(NULL), maxCmdSeqno_(0),
	  topReceived_(0), lastSent_(0), aCmds_(stepSize_)
{
}


MBPageObject::~MBPageObject()
{
	delete [] szName_;
	for (u_long l=0; l<aCmds_.GetSize(); l++) delete aCmds_[l];
}



// schedule the next repair request if neccessary
void MBPageObject::SchedRequest(const SrcId &srcId)
{
	// conside sending out request only if a request is not already
	// pending
	if (pRequest_) return;

	// REVIEW: optimizations for local page?
	ulong i=topReceived_+1;
	while (aCmds_[i] && i <= getMaxSeqno()) i++;
	topReceived_ = i-1;

	if (topReceived_ < getMaxSeqno()) {
		assert(pRequest_==NULL);
		while (!aCmds_[i] && i <= getMaxSeqno()) i++;
		i--;
		pRequest_ = new MBRequest(srcId, this, topReceived_+1, i);
		MTrace(trcMB|trcVerbose,
		       ("new request for %lu to %lu", topReceived_+1, i));
		pMgr_->SchedRequest(pRequest_, srcId);
	}
}




// notify the page about an incoming request
// sidReqSrc is the origin of the request
// sidReqTgt is the source that the request is asking data for
void
MBPageObject::HandleRequest(const SrcId &sidReqSrc, const SrcId &sidReqTgt,
                            ulong snStart, ulong snEnd)
{
	/*
	 * if we have a pending request that overlaps this one, ignore this
	 * one. since we won't have the data
	 */
	if (pRequest_ && pRequest_->getStart() <= snEnd
	    && snStart <= pRequest_->getEnd()) {
		// should propably backoff the timer?
		pRequest_->backoff();
		return;
	}

	// get the block that we have
	while (snStart <= snEnd && aCmds_[snStart]==NULL) ++snStart;
	if (snStart > snEnd) return;  // don't any operations
	assert(snStart <= snEnd && aCmds_[snStart] && "algo check");

	// we have at least one command to send out
	ulong i=snStart;
	while (i<=snEnd && aCmds_[i]!=NULL) ++i;
	snEnd = i-1;
	assert(snStart<=snEnd && "algo check");
	/*
	 * if we have a pending repair that overlaps this one, ignore
	 * REVIEW!: should combine them to the extend of max ADU size
	 */
	for (ListIndex idx=lReplies_.getFirst();
	     !lReplies_.IsDone(idx);
	     idx = lReplies_.getNext(idx)) {
		if (lReplies_.getData(idx)->OverlapOrig(snStart,snEnd))
			return;
	}

	// Schedule the reply otherwise
	MBReply* pNewReply = new MBReply(sidReqTgt, this, snStart, snEnd);
	lReplies_.InsertAtHead(pNewReply);
	pMgr_->SchedReply(pNewReply, sidReqSrc);
}



// takes a range of sequence numbers and reduce it so that
// the snStart=first absent command and snEnd = last absent command
// within the range
void
MBPageObject::UpdateRequest(ulong& snStart, ulong& snEnd)
{
	while (aCmds_[snStart] && snStart<=snEnd) snStart++;
	while (aCmds_[snEnd] && snEnd>=snStart) snEnd--;
}


//
// - Cancel THE request
// - we take in the pointer now for sanity check
// - to ignore new requests after this one, we call ignore() to
//   cancel the request after 3*d.
//
// REVIEW: also we might want multiple out-standing requests in future?
//
void
MBPageObject::CancelRequest(MBRequest* pRequest)
{
	assert(pRequest==pRequest_);   // sanity check
	if (pRequest) {
		pRequest_->cancel();
		/* note: we do not and should not delete pRequest_, this the
		 *     cancellation might have occurred as part of an upcall
		 *     calling cancel will result in self destruction when
		 *     timer fires
		 */
		pRequest_ = 0;
	}
}

void
MBPageObject::CancelReply(MBReply* pReply)
{
	Trace(EXCESSIVE, ("#### CancelReply (%p)", pReply));
	lReplies_.Remove(pReply);
	pReply->cancel();
}

/* Function; suppress replies */
/* Assumption: called when a reply for the page arrives */
/* Operations:
   - reduce the range of overlapping replies
   - if there is no overlapping replies, we create a dummy one to ignore
     further replies */
void
MBPageObject::SuppressReplies(const SrcId& origSrc,
                              ulong snStart, ulong snEnd)
{
	Trace(EXCESSIVE, ("#### SuppressReplies"));
	ListIndex idx, next;
	MBReply *pReply;
	Bool foundOverlap = FALSE;

	idx = lReplies_.getFirst();
	while (!lReplies_.IsDone(idx)) {
		next   = lReplies_.getNext(idx);
		pReply = lReplies_.getData(idx);
		if (pReply->OverlapOrig(snStart, snEnd)) {
			foundOverlap = TRUE;
			if (pReply->Shrunk(snStart, snEnd)==0) {
				// range is null after shrinking
				lReplies_.Remove(idx);
				// destruction happens on next timeout 3d
				// later
				pReply->reply_done();
			}
			// REVIEW: what to do with overlapping timers
#if 0
			else {
			// overlaps partially only, so backoff the timer
			        pReply->Backoff();
			}
#endif
		}
		idx = next;
	}
	if (!foundOverlap) {
		// we did not find an overlap, so have to install a dummy
		// reply to ignore further requests
		MBLOG(("dummy rpy (%d, %d) to %s",
		       snStart, snEnd, intoa(origSrc.ss_addr)));
		MBReply* pNewReply = new MBReply(origSrc, this, snStart,
						 snEnd);
		lReplies_.InsertAtHead(pNewReply);
		// dummy call to set the source and target of the request
		SrcId nullSrc;
		pMgr_->SchedReply(pNewReply, nullSrc);
		pNewReply->reply_done(); // so that no actual reply will be
		                         // sent
	}
}

// MBPageObject::AddCmd --
//    Remember a new command to the page
//    If (overwrite = 0) then don't overwrite any previously
//      added command with the same sequence number
//    Note: setting overwrite to 1 can be dangerous since there might
//          be other states associated with the command
//
void
MBPageObject::AddCmd(MBCmd *pCmd, int overwrite/*=0*/)
{
	MTrace(trcMB|trcExcessive,
	       ("Adding command # %d (maxCmdSeqno_=%d)", pCmd->getSeqno(),
		maxCmdSeqno_));
	ulong seqno = pCmd->getSeqno();
	if ((aCmds_[seqno] != NULL) && !overwrite && aCmds_[seqno]!=pCmd) {
		MTrace(trcAll,("pCmd[]=%x; seq=%ld",
			       aCmds_[seqno], seqno));
		MTrace(trcAll,("duplicate sequence #!"));
		return;
	}
	/* overwrite is 1 or we are adding the same command ptr */
	if (aCmds_[seqno] != pCmd) {
		delete aCmds_[seqno];
		aCmds_[seqno] = pCmd;
	}
	if (seqno > maxCmdSeqno_) {
		maxCmdSeqno_ = seqno;
	}
}

void
MBPageObject::Defer(MBCmd *pCmd)
{
	// add to the deferred list in ascending order of seqNo
	if (deferredList_.IsEmpty()) {
		// add at the head of the list
		deferredList_.InsertAtHead(pCmd);
	}
	else if (deferredList_.PeekAtHead()->getSeqno() > pCmd->getSeqno()) {
		// add at the head of the list
		deferredList_.InsertAtHead(pCmd);
	}
	else if (deferredList_.PeekAtTail()->getSeqno() < pCmd->getSeqno()) {
		// add at the tail of the list
		deferredList_.InsertAtTail(pCmd);
	}
	else {
		// insert somewhere in the middle of the list
		ListIndex idx;
		for (idx=deferredList_.getFirst();
		     !deferredList_.IsDone(idx);
		     idx=deferredList_.getNext(idx))
		{
			if (deferredList_.getData(idx)->getSeqno()
			    > pCmd->getSeqno()) {
				deferredList_.InsertBefore(idx, pCmd);
				break;
			}
		}
	}
}

void
MBPageObject::FlushDeferred(MBBaseRcvr *pRcvr)
{
	ListIndex idx, next;
	MBCmd *pCmd;
	if (deferredList_.IsEmpty()) return;

	/* trim leading cmds that have completed */
	idx = deferredList_.getFirst();
	while (!deferredList_.IsDone(idx)) {
		next = deferredList_.getNext(idx);
		pCmd = deferredList_.getData(idx);
		if (!pCmd->Incomplete(this)) {
			pRcvr->handleCmd(pCmd, this, cMBTimeNegInf,
					 targetTime());
			deferredList_.Remove(idx);
		}
		idx = next;
	}
}

void
MBPageObject::UpdateDeferred(const MBTime& t)
{
	ListIndex idx, next;
	MBCmd *pCmd;
	if (deferredList_.IsEmpty()) return;

	/* trim leading cmds that have completed */
	idx = deferredList_.getFirst();
	while (!deferredList_.IsDone(idx)) {
		next = deferredList_.getNext(idx);
		pCmd = deferredList_.getData(idx);
		if (!pCmd->activeAt(t)) {
			deferredList_.Remove(idx);
		}
		idx = next;
	}
}


// returns whether the page has more data than indicated by the update
Bool
MBPageObject::Update(const SrcId &srcId, ulong maxSn)
{
	if (maxCmdSeqno_ <= maxSn) {
		maxCmdSeqno_ = maxSn;
		// time to ask for new data
		if (pMgr_->isVisible(getId())) {
			// ask for new data only if the page is currently
			// visible
			SchedRequest(srcId);
		}
		return FALSE;
	}
	// we maxCmdSeqno_ > maxSn => the page has more data!
	return TRUE;
}

//
// - Function: update the request state and schedule the next request if
//   neccessary
//
void
MBPageObject::UpdateRequest(const SrcId &srcId, ulong startSn, ulong endSn)
{
	if (pRequest_) {
		// there is already a pending request

		ulong rqStart = pRequest_->getStart();
		ulong rqEnd   = pRequest_->getEnd();

		// check if there is an overlap
		if (pRequest_->CheckOverlap(startSn, endSn)) {
			// an overlapping range came in; suppress the pending
			// request
			pRequest_->backoff();
			assert(rqStart==topReceived_+1);

			if (startSn <= rqStart) {
				// I got a few commands; update topReceived_
				// and the start of the repair-request

				assert(!topReceived_||aCmds_[topReceived_]);
				// REVIEW: should be able to skip the block
				// that just came in

				ulong i = topReceived_+1;
				while (aCmds_[i]) i++;
				i--;
				/* i now holds the command top most command before
				 * a lost */
				topReceived_ = i;
				pRequest_->SetStart(topReceived_+1);
			}

			if (pRequest_->CheckActive()) {
				// there are still some commands that haven't
				// arrived
				if (endSn >= rqEnd) {
					// shrink the end of the request if
					// required
					ulong i = endSn;
					while (aCmds_[i]
					       && i >= topReceived_) i--;
					/* i corresponds to the first hole before endSn */
					pRequest_->SetEnd(i);
				}
				if (!pRequest_->CheckActive()) {
					// oh! we've got all the commands we
					// were waiting for;
					// cancel this request
					CancelRequest(pRequest_);
					SchedRequest(srcId);
				}
			} else {
				// the request is no longer active; we've
				// received all the commands
				CancelRequest(pRequest_);
				SchedRequest(srcId);
			}
		}
	}
	else {
		// don't have a request pending, check whether should
		// send new requests
		if (startSn==topReceived_+1) {
			ulong i;
#ifdef MB_DEBUG
			for (i=topReceived_+1; i<=endSn && aCmds_[i]; ) i++;
#endif // MB_DEBUG
			i = endSn;
			while (aCmds_[i]) i++;
			topReceived_ = i-1;
		}
		assert(topReceived_<=getMaxSeqno());
		if (topReceived_ < getMaxSeqno()) {
			SchedRequest(srcId);
		}
	}
}




// returns max seqno filled (if possible), otherwise 0
// fill in from snStart as much as possible unitl snEnd
ulong
MBPageObject::FillReply(int& outlen, Byte *pb, u_int len,
			ulong snStart, ulong snEnd)
{
	outlen=0;
	if (snEnd > getMaxSeqno()) {
		Trace(ALL,("Error: requesting more than what we have"));
		assert(FALSE);
	}
	// bail out if we could not even send one packet
	if (len < sizeof(Pkt_PageHdr) + aCmds_[snStart]->getPacketLen()) {
		return 0;
	}
	// skip space for commandgroup hdr
	Byte* pbCurr = pb+sizeof(Pkt_PageHdr);
	Byte* pbLimit = pb+len;
	ulong nextSeqno = snStart;
	// fill in until the next packet is too large to fit in
	for (; nextSeqno <= snEnd &&
		     pbCurr + aCmds_[nextSeqno]->getPacketLen() <= pbLimit;
	     nextSeqno++)
	{
		assert(aCmds_[nextSeqno]);
		pbCurr=aCmds_[nextSeqno]->Packetize(pbCurr);
	}
	assert(nextSeqno>snStart); // inserted at least one packet
	// save the header
	Pkt_PageHdr* phdr=(Pkt_PageHdr*) pb;
	host2net(id_, phdr->pd_page);
	phdr->pd_sseq = host2net(snStart);
	phdr->pd_eseq = host2net(nextSeqno-1);

	assert(pbCurr - pb <= (int)len);       // sanity check
	outlen = pbCurr - pb;

	char *szPgId = PgId2Str(id_);
	Trace(ALL,("reply (%lu-%lu) %d b (pg:%s)", snStart, nextSeqno-1,
		   outlen, szPgId));
	delete[] szPgId;

	return (nextSeqno-1);             // last packet sent
}


void
MBPageObject::FillStatus(Pkt_PgStatus& pktStat)
{
	MTrace(trcMB|trcExcessive,
	       ("#### Status for Page: (%u@%s):%lx: maxSeqno: %d",
		getId().sid.ss_uid,intoa(getId().sid.ss_addr),
		getId().uid, getMaxSeqno()));
	host2net(getId(),pktStat.pageid);
	// for local pages, we want to send out the high watermark of
	//   the sent pkts
	// for remote pages, we want to send out the max sn we received
	if (lastSent_) {
		pktStat.maxSn = host2net(lastSent_);
	} else {
		pktStat.maxSn = host2net(getMaxSeqno());
	}
}



// returns the number of bytes filled in
int
MBPageObject::NextADU(Byte *pb, u_int len, int& nextSize)
{
	Trace(EXCESSIVE,("Invoking MBPageObject::NextADU for page %d",
			 getId().uid));
	u_int maxSN = getMaxSeqno();
	if (lastSent_ >= maxSN) {
		assert(lastSent_==maxSN &&
		       "should never send more than we have");
		return 0;           // nothing else to send
	}

	/*
	 * Loop thru until we get enough packets up to len
	 */

	// skip space for commandgroup hdr
	Byte* pbCurr=pb+sizeof(Pkt_PageHdr);
	Byte* pbLimit=pb+len;
	ulong nextSeqno = lastSent_+1;

	// bail out if we could not even send one packet
	if (pbCurr + aCmds_[nextSeqno]->getPacketLen() > pbLimit) {
		if (nextSeqno <= maxSN) {
			nextSize = aCmds_[nextSeqno]->getPacketLen() +
				sizeof(Pkt_PageHdr);
		}
		return 0;
	}

	// fill in until the next packet is too large to fit in
	for (; nextSeqno <= maxSN &&
		     pbCurr + aCmds_[nextSeqno]->getPacketLen() <= pbLimit;
	     nextSeqno++) {
		pbCurr=aCmds_[nextSeqno]->Packetize(pbCurr);
	}
	assert(nextSeqno>lastSent_+1); // inserted at least one packet

	// save the header
	Pkt_PageHdr* phdr=(Pkt_PageHdr*) pb;
	host2net(id_, phdr->pd_page);
	phdr->pd_sseq = host2net(lastSent_+1);
	phdr->pd_eseq = host2net(nextSeqno-1);
	MBLOG(("s dta %u %u sz %u", lastSent_+1, nextSeqno-1, pbCurr-pb));
	assert(pbCurr - pb <= (int)len);       // sanity check
	lastSent_ = nextSeqno-1;      // raise the high water mark
	if (nextSeqno <= maxSN) {
		nextSize = aCmds_[nextSeqno]->getPacketLen() +
			sizeof(Pkt_PageHdr);
	}
	return (pbCurr - pb);
}


/* Return the number of bytes that it wants to send out next */
int
MBPageObject::RequestSendAmount()
{
	if ( lastSent_ < getMaxSeqno() )
		return sizeof(Pkt_PageHdr)
			+ aCmds_[lastSent_+1]->getPacketLen();
	else
		return 0;
}

#ifdef MB_DEBUG
void
MBPageObject::Dump(int dumptype, Tcl_Obj* pObj) {
	if (dumptype==1) DumpTS(pObj);
}


void
MBPageObject::DumpTS(Tcl_Obj* pObj)
{
	PageId pgId=getId();
	char szTmp[200];
	sprintf(szTmp, "\nPage: (%d%s):%lx MaxSN=%lu\n",
	      pgId.sid.ss_uid, intoa(pgId.sid.ss_addr), pgId.uid,
	      getMaxSeqno());
	Tcl_AppendToObj(pObj, szTmp, -1);
	for (ulong i=getMaxSeqno(); i>=1; i--) {
		if (aCmds_[i])
			aCmds_[i]->DumpTS(pObj);
		else {
			sprintf(szTmp, "- - - %lu is EMPTY - - - -\n", i);
			Tcl_AppendToObj(pObj, szTmp, -1);
		}
	}
	if (!deferredList_.IsEmpty()) {
		sprintf(szTmp, "--- DeferList ---\n");
		Tcl_AppendToObj(pObj, szTmp, -1);
		sprintf(szTmp,"head: %lx tail: %lx\n",
			(unsigned long) deferredList_.PeekAtHead(),
			(unsigned long) deferredList_.PeekAtTail());
		Tcl_AppendToObj(pObj, szTmp, -1);

		ListIndex idx;
		for (idx=deferredList_.getFirst();
		     !deferredList_.IsDone(idx);
		     idx=deferredList_.getNext(idx))
		{
			aCmds_[deferredList_.getData(idx)->getSeqno()]
				->DumpTS(pObj);
		}
	}
}

void DumpTS();
void DumpCanvas();
#endif // MB_DEBUG

