/*
 * mbv2-item.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-item.cc,v 1.18 2002/02/03 03:17:08 lim Exp $";
#endif


#include "mbv2-item.h"
#include "mtrace.h"


#define OFFSET(type,field) (size_t)(&((type*)(NULL))->field)


static char *arrowType2String[] = {
	"none",
	"first",
	"last",
	"both"
};


MBv2Item:: ~MBv2Item()
{
	if (points_) delete [] points_;
}


MBv2ItemSpec
MBv2LineItem::specs_[] = {
	{ MBv2Item::set_int,   "-width", OFFSET(MBv2LineItem, width_    ) },
	{ MBv2Item::set_arrow, "-arrow", OFFSET(MBv2LineItem, arrowType_) },
	{ MBv2Item::set_color, "-fill",  OFFSET(MBv2LineItem, color_    ) },
	{ NULL, NULL, 0 }
};


MBv2ItemSpec
MBv2PolyItem::specs_[] = {
	{ MBv2Item::set_int,   "-width",   OFFSET(MBv2PolyItem, width_  ) },
	{ MBv2Item::set_color, "-fill",    OFFSET(MBv2PolyItem, fill_   ) },
	{ MBv2Item::set_color, "-outline", OFFSET(MBv2PolyItem, outline_) },
	{ NULL, NULL, 0 }
};


MBv2ItemSpec
MBv2TextItem::specs_[] = {
	{ MBv2Item::set_string, "-text",   OFFSET(MBv2TextItem, text_ )   },
	{ MBv2Item::set_string, "-font",   OFFSET(MBv2TextItem, font_ )   },
	{ MBv2Item::set_justify,"-justify",OFFSET(MBv2TextItem, justify_) },
	{ MBv2Item::set_color,  "-fill",   OFFSET(MBv2TextItem, color_)   },
	{ NULL, NULL, 0 }
};


MBv2ItemSpec
MBv2ImageItem::specs_[] = {
	{ MBv2Item::set_string, "-file",  OFFSET(MBv2ImageItem, filename_ ) },
	{ MBv2Item::set_string, "-image", OFFSET(MBv2ImageItem, imagename_) },
	{ NULL, NULL, 0 }
};


MBv2Item *
MBv2Item::create(MBv2ItemType type)
{
	switch(type) {
	case MBv2Line:  return new MBv2LineItem();
	case MBv2Text:  return new MBv2TextItem();
	case MBv2Rect:  return new MBv2PolyItem(type);
	case MBv2Oval:  return new MBv2PolyItem(type);
	case MBv2Image: return new MBv2ImageItem();
	default:
		MTrace(trcMB, ("received unknown item type %d", type));
		return NULL;
	}
}


MBv2Item *
MBv2Item::create(MBv2ItemType type, const Byte *pb, int len)
{
	MBv2Item *item = create(type);
	if (!item) return NULL;

	if (len < (int)sizeof(ADU_Coords)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_Coords)));
		delete item;
		return NULL;
	}
	ADU_Coords *coords = (ADU_Coords*)pb;
	item->numPoints_ = ntoh(coords->numPoints);
	pb += sizeof(ADU_Coords);
	len-= sizeof(ADU_Coords);

	if (len < (int)(sizeof(MBv2Point) * item->numPoints_)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(MBv2Point) * item->numPoints_));
		delete item;
		return NULL;
	}

	if (item->numPoints_) {
		item->points_ = new MBv2Point [item->numPoints_];
		MBv2Point *point = (MBv2Point*)pb;
		for (int i=0; i < item->numPoints_; i++, point++) {
			item->points_[i].x = ntoh(point->x);
			item->points_[i].y = ntoh(point->y);
		}

		pb += item->numPoints_ * sizeof(MBv2Point);
		len-= item->numPoints_ * sizeof(MBv2Point);
	}

	if (!item->extract(pb, len)) {
		delete item;
		return NULL;
	}
	return item;
}


Byte *
MBv2Item::packetize(Byte *pb)
{
	ADU_Coords *coords = (ADU_Coords*)pb;
	coords->numPoints = hton(numPoints_);
	for (int i=0; i < numPoints_; i++) {
		coords->points[i].x = hton(points_[i].x);
		coords->points[i].y = hton(points_[i].y);
	}
	return (Byte*)(pb + sizeof(ADU_Coords) + numPoints_*sizeof(MBv2Point));
}


void
MBv2Item::copy_points(MBv2Item *item)
{
	numPoints_ = item->numPoints_;
	// replace the pointers with copies
	points_ = new MBv2Point [numPoints_];
	memcpy(points_, item->points_, sizeof(MBv2Point)*numPoints_);
}


void
MBv2Item::set_int(void *dest, const char *value)
{
	*((int*)dest) = strtoul(value, NULL, 10);
}


void
MBv2Item::set_arrow(void *dest, const char *value)
{
	for (MBv2ArrowType i=MBv2ArrowNone; i < MBv2ArrowInvalid;
	     i= (MBv2ArrowType)(int(i) + 1)) {
		if (strcmp(arrowType2String[i], value)==0) {
			*((MBv2ArrowType*)dest) = i;
			return;
		}
	}
	*((MBv2ArrowType*)dest) = MBv2ArrowNone;
}


void
MBv2Item::set_justify(void *dest, const char *value)
{
	MBv2JustifyType j = MBv2JustifyLeft;
	if (strcmp(value, "right")==0) j = MBv2JustifyRight;
	else if (strcmp(value, "center")==0) j = MBv2JustifyCenter;
	*((MBv2JustifyType*)dest) = j;
}


void
MBv2Item::set_string(void *dest, const char *value)
{
	char *copy = new char [strlen(value)+1];
	strcpy(copy, value);
	*((char**)dest) = copy;
}


inline u_int16_t
convert_hex(const char *hex, int digits)
{
	u_int16_t val = 0, d;
	for (int i=0; i < digits; i++) {
		d = hex[i];
		if (d >= '0' && d <= '9') d -= '0';
		else if (d >= 'A' && d <= 'F') d -= ('A' - 10);
		else if (d >= 'a' && d <= 'f') d -= ('a' - 10);
		else return 0;
		val = (val << 4) | d;
	}
	return val;
}


void
MBv2Item::set_color(void *dest, const char *value)
{
	MBv2Color c;
	if (strcmp(value, "none")==0) {
		c.r = c.g = c.b = 0;
		c.a = 1;
	} else {
		// the color is always in the format #RRRRGGGGBBBB
		c.a = 0;
		c.r = convert_hex(value+1, 4);
		c.g = convert_hex(value+5, 4);
		c.b = convert_hex(value+9, 4);
	}

	*((MBv2Color*)dest) = c;
}


Bool
MBv2Item::set_properties(int propsc, const char * const *propsv)
{
	// we should have an even number of specs
	if (propsc % 2) propsc--;

	MBv2ItemSpec *specs = get_property_specs(), *spec;
	const char *option, *value;
	for (int i=0; i < propsc; i+=2) {
		option = propsv[i];
		value  = propsv[i+1];
		if (!option || !value) continue;
		for (spec=specs; spec->function!=NULL; spec++) {
			if (strcmp(option, spec->option)==0) {
				(*spec->function)(((char*)this)+spec->offset,
						  value);
				break;
			}
		}
	}

	return TRUE;
}


Bool
MBv2LineItem::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_LineItem)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_LineItem)));
		return FALSE;
	}
	ADU_LineItem *adu = (ADU_LineItem *) pb;
	width_     = ntoh(adu->width);
	arrowType_ = ntoh(adu->arrow);
	color_     = ntoh(adu->color);
	return TRUE;
}


Byte *
MBv2LineItem::packetize(Byte *pb)
{
	pb = MBv2Item::packetize(pb);
	ADU_LineItem *adu = (ADU_LineItem *) pb;
	adu->width = hton((u_int16_t)width_);
	adu->arrow = hton((u_int16_t)arrowType_);
	adu->color = hton(color_);
	return (Byte*)(adu+1);
}


Bool
MBv2LineItem::get_properties(Tcl_Interp *interp, Tcl_Obj *props)
{
	if (!append_property(interp, props, "-width", Tcl_NewIntObj(width_)))
		return FALSE;
	if (!append_color(interp, props, "-fill", color_))
		return FALSE;
	char *atype = ((arrowType_ >= MBv2ArrowNone &&
		       arrowType_ < MBv2ArrowInvalid) ?
		       arrowType2String[arrowType_]   :
		       arrowType2String[MBv2ArrowNone]);
	if (!append_property(interp,props,"-arrow",Tcl_NewStringObj(atype,-1)))
		return FALSE;
	if (!append_property(interp, props, "-joinstyle",
			     Tcl_NewStringObj("round", -1)))
		return FALSE;
	if (!append_property(interp, props, "-capstyle",
			     Tcl_NewStringObj("round", -1)))
		return FALSE;
	return TRUE;
}


Bool
MBv2PolyItem::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_PolyItem)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_PolyItem)));
		return FALSE;
	}
	ADU_PolyItem *adu = (ADU_PolyItem *) pb;
	width_   = ntoh(adu->width);
	fill_    = ntoh(adu->fill);
	outline_ = ntoh(adu->outline);
	return TRUE;
}


Byte *
MBv2PolyItem::packetize(Byte *pb)
{
	pb = MBv2Item::packetize(pb);
	ADU_PolyItem *adu = (ADU_PolyItem *) pb;
	adu->width   = hton((u_int16_t)width_);
	adu->fill    = hton(fill_);
	adu->outline = hton(outline_);
	return (Byte*)(adu+1);
}


Bool
MBv2PolyItem::get_properties(Tcl_Interp *interp, Tcl_Obj *props)
{
	if (!append_property(interp, props, "-width", Tcl_NewIntObj(width_)))
		return FALSE;
	if (!append_color(interp, props, "-fill", fill_))
		return FALSE;
	if (!append_color(interp, props, "-outline", outline_))
		return FALSE;
	return TRUE;
}


Bool
MBv2TextItem::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_TextItem)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_TextItem)));
		return FALSE;
	}
	ADU_TextItem *adu = (ADU_TextItem *) pb;
	color_   = ntoh(adu->color);
	justify_ = (MBv2JustifyType)ntoh(adu->justify);
	pb += sizeof(ADU_TextItem);
	len-= sizeof(ADU_TextItem);

	// find the terminating '\0'
	int i;
	for (i=0; i<len; i++)
		if (pb[0]=='\0') break;

	font_ = new char [i+1];
	if (i>0) memcpy(font_, pb, i);
	font_[i] = '\0';
	return TRUE;
}


Byte *
MBv2TextItem::packetize(Byte *pb)
{
	pb = MBv2Item::packetize(pb);
	ADU_TextItem *adu = (ADU_TextItem *) pb;
	adu->color   = hton(color_);
	adu->justify = hton((u_int32_t)justify_);
	strcpy((char*)pb+sizeof(ADU_TextItem), font_);
	return (Byte*)(pb + sizeof(ADU_TextItem) + strlen(font_) + 1);
}


MBv2Item *
MBv2TextItem::make_copy()
{
	MBv2TextItem *item = new MBv2TextItem;
	*item = *this;
	if (text_) {
		item->text_ = new char [strlen(text_)+1];
		strcpy(item->text_, text_);
		item->max_chars_ = strlen(text_);
	}
	if (font_) {
		item->font_ = new char [strlen(font_)+1];
		strcpy(item->font_, font_);
	}

	item->copy_points(this);
	return item;
}


Bool
MBv2TextItem::get_properties(Tcl_Interp *interp, Tcl_Obj *props)
{
	if (!append_property(interp, props, "-font",
			     Tcl_NewStringObj(font_, -1)))
		return FALSE;
	if (!append_property(interp, props, "-text",
			     Tcl_NewStringObj((text_ ? text_ : (char*)""), -1)))
		return FALSE;
	if (!append_color(interp, props, "-fill", color_))
		return FALSE;

	char *justify = ((justify_==MBv2JustifyCenter) ? (char*)"center" :
			((justify_==MBv2JustifyRight) ? (char*)"right" : (char*)"left"));
	char *anchor = ((justify_==MBv2JustifyCenter) ? (char*)"n" :
			((justify_==MBv2JustifyRight) ? (char*)"ne" : (char*)"nw"));

	if (!append_property(interp, props, "-justify",
			     Tcl_NewStringObj(justify, -1)))
		return FALSE;
	if (!append_property(interp, props, "-anchor",
			     Tcl_NewStringObj(anchor, -1)))
		return FALSE;
	return TRUE;
}


Bool
MBv2FileItem::extract(const Byte *pb, int len)
{
	// try a global proc called MBv2FileItem_create_file
	// this proc should return a 2-element list {chan fname}
	// if that doesn't work, try coming up with our own filename
	Tcl_Channel chan=NULL;
	Tcl_Interp *interp = Tcl::instance().interp();
	if (Tcl_GlobalEval(interp, "MBv2FileItem_create_file")==TCL_OK) {
		Tcl_Obj *result = Tcl_GetObjResult(interp),*chanObj,*fnameObj;
		if (Tcl_ListObjIndex(interp, result, 0, &chanObj )==TCL_OK &&
		    Tcl_ListObjIndex(interp, result, 0, &fnameObj)==TCL_OK) {
			chan = Tcl_GetChannel(interp, Tcl_GetStringFromObj(
				chanObj, NULL), NULL);
			if (chan) {
				filename(Tcl_GetStringFromObj(fnameObj,NULL));
			}
		}
	}

	if (!chan) {
		// create a random file name
		static int fileCnt_=1;
		char fname[100];

#ifdef WIN32
		sprintf(fname, "C:/temp/mb-%d.tmp", fileCnt_++);
#else
		sprintf(fname, "/tmp/mb-%ld-%d.tmp", (long) getpid(), fileCnt_++);
#endif
		chan = Tcl_OpenFileChannel(interp, fname, "w", 0644);
		if (!chan) {
			MTrace(trcMB, ("could not open file %s: %s",
				       fname, Tcl_GetStringResult(interp)));
			return FALSE;
		}
		filename(fname);
	}

	// first set the translation mode to binary
	Tcl_SetChannelOption(interp, chan, "-translation", "binary");

	if (Tcl_Write(chan, (char*)pb, len)!=len) {
		MTrace(trcMB, ("could not write to file %s: %s",
			       filename(), Tcl_GetStringResult(interp)));
		Tcl_Close(interp, chan);
		return FALSE;
	}

	Tcl_Close(interp, chan);
	return TRUE;
}


Byte *
MBv2FileItem::packetize(Byte *pb)
{
	pb = MBv2Item::packetize(pb);

	Tcl_Channel chan=NULL;
	Tcl_Interp *interp = Tcl::instance().interp();
	chan = Tcl_OpenFileChannel(interp, filename_, "r", 0644);
	if (!chan) {
		MTrace(trcMB, ("could not open file %s: %s",
			       filename_, Tcl_GetStringResult(interp)));
		return NULL;
	}

	// first set the translation mode to binary
	Tcl_SetChannelOption(interp, chan, "-translation", "binary");
	// adu_size() must already have been called on this object
	// so we know the size
	if (filesize_ <= 0) return NULL;
	if (Tcl_Read(chan, (char*)pb, filesize_) <= 0) {
		MTrace(trcMB, ("could not read from file %s: %s",
			       filename_, Tcl_GetStringResult(interp)));
		Tcl_Close(interp, chan);
		return NULL;
	}

	return (Byte*)(pb + filesize_);
}


int
MBv2FileItem::adu_size()
{
	if (filesize_ < 0) {
		// figure out the file size
		Tcl_Interp *interp = Tcl::instance().interp();
		Tcl_Obj *cmd = Tcl_NewStringObj("file size", -1);
		Tcl_IncrRefCount(cmd);
		if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj(
			filename_, -1))==TCL_ERROR ||
		    Tcl_GlobalEvalObj(interp, cmd)==TCL_ERROR)
			filesize_ = 0;
		else {
			int s;
			if (Tcl_GetIntFromObj(interp, Tcl_GetObjResult(interp),
					      &s)==TCL_ERROR)
				filesize_ = 0;
			else
				filesize_ = s;
		}
		Tcl_DecrRefCount(cmd);
	}

	return MBv2Item::adu_size() + filesize_;
}


Bool
MBv2ImageItem::get_properties(Tcl_Interp *interp, Tcl_Obj *props)
{
	create_image();
	if (!append_property(interp, props, "-image",
			     Tcl_NewStringObj(imagename_, -1)))
		return FALSE;
	if (!append_property(interp, props, "-anchor",
			     Tcl_NewStringObj("nw",-1)))
		return FALSE;
	return TRUE;

}


Bool
MBv2ImageItem::create_image()
{
	if (imagename_) return TRUE;

	Tcl_Obj *cmd;
	Tcl_Interp *interp = Tcl::instance().interp();
	cmd = Tcl_NewStringObj("image create photo -file", -1);
	Tcl_IncrRefCount(cmd);
	if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj(
		(char*)filename(),-1))==TCL_ERROR) {
		Tcl_DecrRefCount(cmd);
		MTrace(trcMB, ("tcl error: %s",
			       Tcl_GetStringResult(interp)));
		return FALSE;
	}
	if (Tcl_GlobalEvalObj(interp, cmd)==TCL_ERROR) {
		Tcl_DecrRefCount(cmd);
		MTrace(trcMB, ("tcl error: %s",
			       Tcl_GetStringResult(interp)));
		return FALSE;
	}

	imagename(Tcl_GetStringResult(interp));
	Tcl_DecrRefCount(cmd);

	return TRUE;
}


Bool
MBv2ImageItem::delete_image()
{
	if (!imagename_) return TRUE;

	Tcl_Obj *cmd;
	Tcl_Interp *interp = Tcl::instance().interp();
	cmd = Tcl_NewStringObj("image delete", -1);
	Tcl_IncrRefCount(cmd);
	if (Tcl_ListObjAppendElement(interp, cmd, Tcl_NewStringObj(
		imagename_, -1))==TCL_ERROR) {
		Tcl_DecrRefCount(cmd);
		MTrace(trcMB, ("tcl error: %s",
			       Tcl_GetStringResult(interp)));
		return FALSE;
	}
	if (Tcl_GlobalEvalObj(interp, cmd)==TCL_ERROR) {
		Tcl_DecrRefCount(cmd);
		MTrace(trcMB, ("tcl error: %s",
			       Tcl_GetStringResult(interp)));
		return FALSE;
	}

	delete [] imagename_;
	imagename_ = NULL;
	Tcl_DecrRefCount(cmd);
	return TRUE;
}


MBv2Item *
MBv2ImageItem::make_copy()
{
	MBv2ImageItem *item = new MBv2ImageItem;
	*item = *this;
	if (filename_) {
		item->filename_ = new char [strlen(filename_)+1];
		strcpy(item->filename_, filename_);
	}
	if (imagename_) {
		item->imagename_ = new char [strlen(imagename_)+1];
		strcpy(item->imagename_, imagename_);
	}

	item->copy_points(this);
	return item;
}

