/*
 * mbv2-sndr.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-sndr.cc,v 1.13 2002/02/03 03:17:08 lim Exp $";
#endif

#include <sys/time.h>

#include "mbv2-obj.h"
#include "mbv2-canv.h"
#include "mbv2-cmd.h"
#include <tk.h>


class MBv2Sender : public MBv2Source {
public:
	MBv2Sender();
	virtual ~MBv2Sender();
	int app_info(int argc, const char * const *argv);
	int create_item    (int argc, const char * const* argv);
	int create_group   (int argc, const char * const* argv);
	int delete_item    (int argc, const char * const* argv);
	int copy_item      (int argc, const char * const* argv);
	int move_item      (int argc, const char * const* argv);
	int edit_text_item (int argc, const char * const* argv);
	int insert_chars   (int argc, const char * const* argv);
	int undo_item      (int argc, const char * const* argv);
	int new_page       (int argc, const char * const* argv);
	int switch_page    (int argc, const char * const* argv);
	int get_canvas_item(int argc, const char * const* argv);
	int get_cmdid      (int argc, const char * const* argv);
	int sanitize_color (int argc, const char * const* argv);

private:
	MBv2Page *create_page(MBv2PageId pageid) {
		if (pageid.pageid == -1) {
			const int *sid = srm_get_source_id(srm_source());
			for (int i=0; i<4; i++) pageid.srcid[i] = sid[i];
			pageid.pageid= nextPageId_++;
		}
		return MBv2Source::create_page(pageid);
	}

	MBv2CmdId delete_item(MBv2Page *page, MBv2CmdId cmdId);
	MBv2CmdId undelete_item(MBv2Page *page, MBv2CmdId cmdId);
	MBv2CmdId move_item(MBv2Page *page, MBv2CmdId cmdId,
			    int32_t dx, int32_t dy);
	MBv2CmdId unedit_text_item(MBv2Page *page, MBv2CmdId cmdId,
				   MBv2CmdId cmd_to_undo);

	MBv2Page *extract_page(int &argc, const char* const *&argv,
			       Bool should_create);
	Bool extract_pageid(int &argc, const char * const *&argv,
			    MBv2PageId &pageid);

	MBv2Page *currentPage_;
	MBv2PageId currentPageId_;
	int nextPageId_;
};


DEFINE_OTCL_CLASS(MBv2Sender, "MBv2Sender") {
	INSTPROC_PUBLIC(app_info);
	INSTPROC_PUBLIC(create_item);
	INSTPROC_PUBLIC(create_group);
	INSTPROC_PUBLIC(delete_item);
	INSTPROC_PUBLIC(copy_item);
	INSTPROC_PUBLIC(move_item);
	INSTPROC_PUBLIC(edit_text_item);
	INSTPROC_PUBLIC(insert_chars);
	INSTPROC_PUBLIC(undo_item);
	INSTPROC_PUBLIC(new_page);
	INSTPROC_PUBLIC(switch_page);
	INSTPROC_PUBLIC(get_canvas_item);
	INSTPROC_PUBLIC(get_cmdid);
	INSTPROC_PUBLIC(sanitize_color);
}


MBv2Sender::MBv2Sender()
	: MBv2Source(), currentPage_(NULL), currentPageId_(), nextPageId_(0)
{
	currentPageId_.pageid = -1;
}


MBv2Sender::~MBv2Sender()
{
}


int
MBv2Sender::app_info(int argc, const char * const *argv)
{
	const char *info;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(info);
	END_PARSE_ARGS;

	srm_set_app_info(srm_source(), (const unsigned char*)info,
			 strlen(info)+1);
	return TCL_OK;
}


MBv2Page *
MBv2Sender::extract_page(int &argc, const char* const *&argv,
			 Bool should_create)
{
	MBv2PageId pageid;

	if (!extract_pageid(argc, argv, pageid)) {
		pageid = currentPageId_;
	}

	if (pageid.pageid < 0) {
		Tcl::instance().result("invalid pageid encountered");
		return NULL;
	}

	// FIXME: when we have epochs, this should find a page in the current
	// epoch
	MBv2Page *page = find_page(pageid);
	if (!page) {
		if (should_create) {
			// we must create a page
			page = create_page(pageid);
		}

		if (!page) {
			Tcl::instance().result("could not find page");
			return NULL;
		}
	}

	if (pageid==currentPageId_) currentPage_ = page;
	return page;
}


Bool
MBv2Sender::extract_pageid(int &argc, const char * const *&argv,
			   MBv2PageId &pageid)
{
	if (argc > 3) {
		if (strcmp(argv[2], "-page")==0) {
			pageid = MBv2PageId(argv[3]);
			argc -= 2;
			argv += 2;
			return TRUE;
		}
	}
	return FALSE;
}


int
MBv2Sender::get_canvas_item(int argc, const char * const* argv)
{
	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 0);
	if (!page) return TCL_ERROR;

	if (argc < 3) {
		Tcl::instance().result("no cmd id");
		return TCL_ERROR;
	}
	MBv2CmdId cmdId = (MBv2CmdId) strtoul(argv[2], NULL, 10);
	MBv2CanvId canvId = page->get_canvas_item(cmdId);
	if (canvId != MBv2InvalidCanvId)
		Tcl::instance().result(Tcl_NewIntObj(canvId));
	else
		Tcl::instance().result(Tcl_NewObj());
	return TCL_OK;
}


int
MBv2Sender::create_item(int argc, const char * const* argv)
{
	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 1);
	if (!page) {
		// may be there is no page right now!
		// we should try creating one
		if (currentPageId_.pageid < 0) {
			// create a new page anyway
			page = currentPage_ = create_page(currentPageId_);
		}

		if (!page) {
			Tcl::instance().result("error creating new page");
			return TCL_ERROR;
		}

		// reset the result
		Tcl::instance().result("");
	}

	if (argc < 3) {
		Tcl::instance().result("no type argument");
		return TCL_ERROR;
	}
	MBv2ItemType type = MBv2TkCanvas::item_type(argv[2]);
	if (type==MBv2InvalidItem) {
		Tcl::instance().resultf("invalid item type %s", argv[2]);
		return TCL_ERROR;
	}

	argc -= 3;
	argv += 3;
	// extract the coordinates (there should be an even number)
	int numPoints=0;
	while (numPoints < argc) {
		const char *p = argv[numPoints];
		if (*p == '-') p++;
		while ((*p >= '0' && *p <= '9') || *p == '.') p++;
		if (*p != '\0') break;
		numPoints++;
	}
	if (numPoints % 2) {
		Tcl::instance().result("odd number of coordinate values");
		return TCL_ERROR;
	}

	numPoints /= 2;
	if (!numPoints) {
		Tcl::instance().result("no coordinates");
		return TCL_ERROR;
	}

	MBv2Point *points = new MBv2Point[numPoints];
	MBv2TkCanvas *canv = (MBv2TkCanvas*)(page->canvas());
	double x, y;
	for (int i=0; i<numPoints; i++) {
		// scale the points first
		sscanf(argv[i*2],   "%lg", &x);
		sscanf(argv[i*2+1], "%lg", &y);
		canv->imap(x, y);
		points[i].x = (int32_t) x;
		points[i].y = (int32_t) y;
	}

	MBv2Item *item = MBv2Item::create(type);
	item->points(points, numPoints);
	if (!item->set_properties(argc-numPoints*2, argv+numPoints*2)) {
		// this will delete the points array as well
		delete item;
		Tcl::instance().result("could not set item properties");
		return TCL_ERROR;
	}

	// the item's been successfully created
	// create the associated MBv2CmdCreate
	MBv2Cmd* cmd = new MBv2CmdCreate(current_time(), item);
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdCreate");
		return TCL_ERROR;
	}

	Tcl::instance().result(Tcl_NewIntObj(cmd->id()));
	return TCL_OK;
}


int
MBv2Sender::create_group(int argc, const char * const* argv)
{
	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 1);
	if (!page) return TCL_ERROR;

	if (argc < 3) {
		Tcl::instance().result("no type argument");
		return TCL_ERROR;
	}
	MBv2ItemType type = MBv2TkCanvas::item_type(argv[2]);
	if (type!=MBv2Line && type!=MBv2Text) {
		Tcl::instance().resultf("invalid item type %s", argv[2]);
		return TCL_ERROR;
	}

	argc -= 3;
	argv += 3;

	// extract the start and end cmd id's
	if (argc < 2) {
		Tcl::instance().result("missing start/end cmd id's");
		return TCL_ERROR;
	}

	MBv2CmdId start, end;
	start = (MBv2CmdId) strtoul(argv[0], NULL, 10);
	end   = (MBv2CmdId) strtoul(argv[1], NULL, 10);

	MBv2Cmd *cmd = MBv2CmdGroup::create(type, current_time(), start, end);
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdGroup");
		return TCL_ERROR;
	}

	Tcl::instance().result(Tcl_NewIntObj(cmd->id()));
	return TCL_OK;
}


int
MBv2Sender::delete_item(int argc, const char * const* argv)
{
	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 0);
	if (!page) return TCL_ERROR;

	if (argc < 3) {
		Tcl::instance().result("no cmd id");
		return TCL_ERROR;
	}
	MBv2CmdId cmdId = (MBv2CmdId) strtoul(argv[2], NULL, 10), newId;

	if ( (newId = delete_item(page, cmdId)) == MBv2InvalidCmdId) {
		return TCL_ERROR;
	}

	Tcl::instance().result(Tcl_NewIntObj(newId));
	return TCL_OK;
}


MBv2CmdId
MBv2Sender::delete_item(MBv2Page *page, MBv2CmdId cmdId)
{
	MBv2Cmd *cmd = new MBv2CmdDelete(current_time(), cmdId);
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdDelete");
		return MBv2InvalidCmdId;
	}

	return cmd->id();
}


int
MBv2Sender::copy_item(int argc, const char * const* argv)
{
	/* this has the following syntax:
	 *     $sndr copy_item ?-page $pageid? cmdIdToCopy ?pageIdofCmd?
	 */

	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, ((argc > 5) ? 1 : 0));
	if (!page) return TCL_ERROR;

	if (argc < 3) {
		Tcl::instance().result("no cmd id");
		return TCL_ERROR;
	}
	MBv2CmdId cmdId = (MBv2CmdId) strtoul(argv[2], NULL, 10);
	u_int32_t cid;
	if (argc > 3) {
		MBv2Page *p = find_page(MBv2PageId(argv[3]));
		if (!p) {
			Tcl::instance().result("no page to copy from");
			return TCL_ERROR;
		}
		cid = p->cid();
	} else {
		cid = page->cid();
	}

	MBv2Cmd *cmd = new MBv2CmdCopy(MBv2Copy, current_time(), cid, cmdId);
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdCopy");
		return TCL_ERROR;
	}

	Tcl::instance().result(Tcl_NewIntObj(cmd->id()));
	return TCL_OK;
}


int
MBv2Sender::edit_text_item(int argc, const char * const* argv)
{
	/* this has the following syntax:
	 *     $sndr edit_text_item ?-page $pageid? cmdIdToEdit
	 */

	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 0);
	if (!page) return TCL_ERROR;

	if (argc < 3) {
		Tcl::instance().result("no cmd id");
		return TCL_ERROR;
	}
	MBv2CmdId cmdId = (MBv2CmdId) strtoul(argv[2], NULL, 10);

	MBv2Cmd *cmd = new MBv2CmdCopy(MBv2EditText, current_time(),
				       page->cid(), cmdId);
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdCopy "
				       "(EditText)");
		return TCL_ERROR;
	}

	Tcl::instance().result(Tcl_NewIntObj(cmd->id()));
	return TCL_OK;
}


MBv2CmdId
MBv2Sender::unedit_text_item(MBv2Page *page, MBv2CmdId cmdId,
			     MBv2CmdId cmd_to_undo)
{
	MBv2Cmd *cmd=page->cmd(cmd_to_undo), *target=NULL;
	MBv2Page *targetPage=NULL;

	if (!cmd || !cmd->target(page, targetPage, target)) {
		Tcl::instance().result("could not find target for "
				       "unedit_text");
		return MBv2InvalidCmdId;
	}
	if (page != targetPage) {
		Tcl::instance().result("target page is not the same");
		return MBv2InvalidCmdId;
	}
	MBv2Item *item = target->make_copy(targetPage);
	if (!item || item->type()!=MBv2Text) {
		Tcl::instance().result("item must be a text object");
		if (item) delete item;
		return MBv2InvalidCmdId;
	}

	cmd = new MBv2CmdCopy(MBv2EditText, current_time(), page->cid(),cmdId);
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2UneditText");
		return MBv2InvalidCmdId;
	}

	cmdId = cmd->id();
	cmd = new MBv2CmdChars(current_time(), cmdId, MBv2CmdChars_ReplaceText,
			       ((MBv2TextItem*)item)->get_text());
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2UneditText");
		return MBv2InvalidCmdId;
	}

	cmd = MBv2CmdGroup::create(MBv2Text, current_time(), cmdId, cmd->id());
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2UneditText");
		return MBv2InvalidCmdId;
	}

	return cmd->id();
}


MBv2CmdId
MBv2Sender::undelete_item(MBv2Page *page, MBv2CmdId deleteCmdId)
{
	MBv2Cmd *cmd;
	cmd = page->cmd(deleteCmdId);
	if (!cmd || cmd->type()!=MBv2Delete) {
		Tcl::instance().resultf("could not find cmd %u (or not "
					"MBv2CmdDelete)", deleteCmdId);
		return MBv2InvalidCmdId;
	}

	cmd = new MBv2CmdCopy(MBv2Undelete, current_time(),
			      page->cid(),((MBv2CmdDelete*)cmd)->target_cmd());
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdUndelete");
		return MBv2InvalidCmdId;
	}

	return cmd->id();
}


int
MBv2Sender::move_item(int argc, const char * const* argv)
{
	/* this has the following syntax:
	 *     $sndr move_item ?-page $pageid? cmdIdToMove dx dy
	 */

	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 0);
	if (!page) return TCL_ERROR;

	if (argc != 5) {
		Tcl::instance().result("invalid number of arguments");
		return TCL_ERROR;
	}
	MBv2CmdId cmdId = (MBv2CmdId) strtoul(argv[2], NULL, 10), newId;
	double dx = (int32_t) strtoul(argv[3], NULL, 10);
	double dy = (int32_t) strtoul(argv[4], NULL, 10);

	// scale the points first
	MBv2TkCanvas *canv = (MBv2TkCanvas*)(page->canvas());
	canv->imap(dx, dy);

	if ( (newId = move_item(page, cmdId, (int32_t)dx, (int32_t)dy))
	     == MBv2InvalidCmdId) {
		return TCL_ERROR;
	}

	Tcl::instance().result(Tcl_NewIntObj(newId));
	return TCL_OK;
}


MBv2CmdId
MBv2Sender::move_item(MBv2Page *page, MBv2CmdId cmdId, int32_t dx, int32_t dy)
{
	MBv2Cmd *cmd = new MBv2CmdMove(current_time(), cmdId, dx, dy);
	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdMove");
		return MBv2InvalidCmdId;
	}

	return cmd->id();
}


int
MBv2Sender::insert_chars(int argc, const char * const* argv)
{
	/* this has the following syntax:
	 *     $sndr insert_chars ?-page $pageid? cmdId index chars
	 */

	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 0);
	if (!page) return TCL_ERROR;

	if (argc < 3) {
		Tcl::instance().result("no cmd id");
		return TCL_ERROR;
	}
	MBv2CmdId cmdId = (MBv2CmdId) strtoul(argv[2], NULL, 10);

	if (argc < 4) {
		Tcl::instance().result("no index");
		return TCL_ERROR;
	}
	int index = (int)strtoul(argv[3], NULL, 10);

	if (argc < 5) {
		Tcl::instance().result("no chars");
		return TCL_ERROR;
	}
	const char *chars = argv[4];


	MBv2Cmd *cmd;
	if (chars[0] != '\0' && chars[1] == '\0') {
		// this is a single character
		if (*chars == '\b' && index == 0) {
			// we can't backspace if we are already at 0
			Tcl::instance().result("");
			return TCL_OK;
		}

		cmd = new MBv2CmdChar (current_time(), cmdId, index, *chars);
	} else {
		cmd = new MBv2CmdChars(current_time(), cmdId, index, chars);
	}

	if (!dispatch(page, cmd)) {
		Tcl::instance().result("could not dispatch MBv2CmdChar(s)");
		return TCL_ERROR;
	}

	Tcl::instance().result(Tcl_NewIntObj(cmd->id()));
	return TCL_OK;
}


int
MBv2Sender::new_page(int argc, const char * const* argv)
{
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;

	currentPageId_.pageid = -1;
	currentPage_  = create_page(currentPageId_);
	if (!currentPage_) {
		Tcl::instance().result("could not create new page");
		return TCL_ERROR;
	}

	currentPageId_= currentPage_->id();

	char pageidStr[48];
	currentPageId_.to_string(pageidStr);
	Tcl::instance().resultf("%s", pageidStr);
	return TCL_OK;
}


int
MBv2Sender::switch_page(int argc, const char * const *argv)
{
	if (!extract_pageid(argc, argv, currentPageId_)) {
		// we must have the -page argument
		Tcl::instance().result("missing -page argument in "
				       "MBv2Sender::switch_page");
		return TCL_ERROR;
	}
	if (argc > 2) {
		Tcl::instance().result("too many arguments in "
				       "MBv2Sender::switch_page");
		return TCL_ERROR;
	}

	return TCL_OK;
}


int
MBv2Sender::sanitize_color(int argc, const char * const *argv)
{
	const char *color;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(color);
	END_PARSE_ARGS;

	if (strcmp(color, "none")==0) {
		Tcl::instance().resultf("%s", color);
		return TCL_OK;
	}

	Tcl_Interp *interp = Tcl::instance().interp();
	Tk_Uid uid = Tk_GetUid(color);
	Tk_Window tkwin = Tk_MainWindow(interp);
        XColor *xcolor = Tk_GetColor(interp, tkwin, uid);
        if (xcolor==NULL) {
		Tcl::instance().resultf("bad color %s", color);
                return TCL_ERROR;
        }
	Tcl::instance().resultf("#%04x%04x%04x", xcolor->red, xcolor->green,
				xcolor->blue);
        Tk_FreeColor(xcolor);
	return TCL_OK;
}


int
MBv2Sender::get_cmdid(int argc, const char * const* argv)
{
	/* this has the following syntax:
	 *     $sndr get_cmdid ?-page $pageid? canvId
	 */

	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 0);
	if (!page) return TCL_ERROR;

	if (argc != 3) {
		Tcl::instance().result("invalid number of arguments");
		return TCL_ERROR;
	}
	MBv2CanvId canvId = (MBv2CanvId) strtoul(argv[2], NULL, 10);
	MBv2CmdId cmdId = page->get_cmdid(canvId);
	if (cmdId != MBv2InvalidCmdId)
		Tcl::instance().result(Tcl_NewIntObj(cmdId));
	else
		Tcl::instance().result("");
	return TCL_OK;
}


int
MBv2Sender::undo_item(int argc, const char * const* argv)
{
	/* this has the following syntax:
	 *     $sndr undo_item ?-page $pageid? cmdIdToUndo
	 */

	// extract out the optional page argument
	MBv2Page *page = extract_page(argc, argv, 0);
	if (!page) return TCL_ERROR;

	if (argc < 3) {
		Tcl::instance().result("no cmd id");
		return TCL_ERROR;
	}
	MBv2CmdId cmdId = (MBv2CmdId) strtoul(argv[2], NULL, 10),
		latestId, nextUndo, newCmd;
	MBv2Cmd *cmd = page->cmd(cmdId), *latest;
	if (!cmd) {
		Tcl::instance().resultf("could not find cmd %u", cmdId);
		return TCL_ERROR;
	}

	latestId = page->tag2cmd(cmd->tag());
	latest = page->cmd(latestId);
	if (!latest) {
		Tcl::instance().resultf("could not find latest cmd %u",
					latestId);
		return TCL_ERROR;
	}

	switch (cmd->type()) {
	case MBv2Create:
	case MBv2Copy:
	case MBv2Undelete:
		if (latest->type()==MBv2Delete) {
			Tcl::instance().result("latest command can't be "
					       "MBv2Delete");
			return TCL_ERROR;
		}
		newCmd = delete_item(page, latestId);
		// assumes cmdIds start with 0
		nextUndo = (u_int32_t(cmdId) > 0 ?
			    MBv2CmdId(u_int32_t(cmdId) - 1) :MBv2InvalidCmdId);
		break;

	case MBv2Group:
		if (latest->type()==MBv2Delete) {
			Tcl::instance().result("latest command can't be "
					       "MBv2Delete");
			return TCL_ERROR;
		}

		// check if this group is that of an EditText or not
		cmdId = ((MBv2CmdGroup*)cmd)->start_cmdid();
		if (page->cmd(cmdId)->type()==MBv2EditText) {
			newCmd = unedit_text_item(page, latestId, cmdId);
		} else {
			// delete the old object
			newCmd = delete_item(page, latestId);
		}

		// assumes cmdIds start with 0
		nextUndo = (u_int32_t(cmdId) > 0 ?
			    MBv2CmdId(u_int32_t(cmdId) - 1) :MBv2InvalidCmdId);
		break;

	case MBv2Delete:
		if (latest->type()!=MBv2Delete) {
			Tcl::instance().result("latest command must be "
					       "MBv2Delete");
			return TCL_ERROR;
		}
		newCmd = undelete_item(page, latestId);
		// assumes cmdIds start with 0
		nextUndo = (u_int32_t(cmdId) > 0 ?
			    MBv2CmdId(u_int32_t(cmdId) - 1) :MBv2InvalidCmdId);
		break;

	case MBv2Move:
		if (latest->type()==MBv2Delete) {
			Tcl::instance().result("latest command can't be "
					       "MBv2Delete");
			return TCL_ERROR;
		}
		newCmd = move_item(page, latestId, -((MBv2CmdMove*)cmd)->dx(),
				    -((MBv2CmdMove*)cmd)->dy());
		// assumes cmdIds start with 0
		nextUndo = (u_int32_t(cmdId) > 0 ?
			    MBv2CmdId(u_int32_t(cmdId) - 1) :MBv2InvalidCmdId);
		break;

	case MBv2Char:
	case MBv2Chars:
	default:
		Tcl::instance().resultf("invalid cmd type", cmd->type());
		return TCL_ERROR;
	}

	if (newCmd==MBv2InvalidCmdId) return TCL_ERROR;

	Tcl_Obj *result = Tcl_NewObj();
	if (nextUndo==MBv2InvalidCmdId) {
		Tcl_ListObjAppendElement(Tcl::instance().interp(), result,
					 Tcl_NewObj());
	} else {
		Tcl_ListObjAppendElement(Tcl::instance().interp(), result,
					 Tcl_NewIntObj(nextUndo));
	}

	Tcl_ListObjAppendElement(Tcl::instance().interp(), result,
				 Tcl_NewIntObj(newCmd));
	Tcl::instance().result(result);
	return TCL_OK;
}
