/*
 * mplug-config.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "mplug-config.h"
#include <ctype.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "mplug-compat.h"


MPlugConfiguration::MPlugConfiguration()
{
	InitHashTable(&ht_, HASH_STRING_KEYS);

	// look for mplugrc in one of the following places:
	//     ~/.mash/                (personal)
	//     $MASH_HOME/lib          (system-wide?)

	// first load the systemwide configuration (if we can find it)
	LocateAndReadFile("MASH_HOME", DEFAULT_MASH_HOME,
			  "lib/prefs-mplug");

	// now read the personal file
	LocateAndReadFile("HOME", DEFAULT_HOME, ".mash/prefs-mplug");
}


MPlugConfiguration::MPlugConfiguration(const char *filename)
{
	InitHashTable(&ht_, HASH_STRING_KEYS);
	if (filename && Read(filename)==FALSE) {
		MPlug_Output("Error reading configuration file '%s': \n%s\n",
			     filename, strerror(errno));
	}
}


MPlugConfiguration::~MPlugConfiguration()
{
	Clear();
	DeleteHashTable(&ht_);
}


void
MPlugConfiguration::Clear()
{
	HashEntry *entry;
	HashSearch search;
	char *string;

	while( (entry = FirstHashEntry(&ht_, &search)) ) {
		string = (char*)GetHashValue(entry);
		delete [] string;
		DeleteHashEntry(entry);
	}
}


void
MPlugConfiguration::LocateAndReadFile(const char *environVar,
				      const char *defaultEnvironVal,
				      const char *suffix)
{
	const char *environVal = getenv(environVar);
	if (!environVal) environVal = defaultEnvironVal;
	char *filename = ::MPlug_JoinFilename(environVal, suffix);
	LOG(("MPlugConfiguration::LocateAndReadFile trying to read %s... ",
	     filename));

	if (MPlug_IsReadable(filename)) {
		// we can access the file
		if (Read(filename)==FALSE) {
			LOG(("failed\n"));
			MPlug_Output("Error reading configuration file '%s': "
				     "\n%s\n", filename, strerror(errno));
		}
		else LOG(("succeeded\n"));
	}
	else LOG(("failed\n"));
	delete [] filename;
}


NPBool
MPlugConfiguration::Read(const char *filename)
{
	const int MAX_LINE_SIZE=512;
	char line[MAX_LINE_SIZE], *ptr, *key, *end, *value;
	FILE *fp;

	fp = fopen(filename, "r");
	if (fp==NULL) return FALSE;

	while (fgets(line, MAX_LINE_SIZE, fp)!=NULL) {
		// FIXME max line size is assumed to be MAX_LINE_SIZE!

		// each line must be of the form "key: value"
		// comments start with ! or #
		ptr = line;

		// ignore leading whitespace
		while(isspace(*ptr)) ptr++;
		if (*ptr=='!' || *ptr=='#') {
			// this is a comment; ignore it
			continue;
		}

		// extract the option key
		key = ptr;
		while(!isspace(*ptr) && *ptr!=':' && *ptr!='\0') ptr++;
		if (ptr==key || *ptr=='\0') {
			// invalid line; ignore it
			continue;
		}
		end = ptr;

		// ignore whitespace
		while(isspace(*ptr)) ptr++;
		if (*ptr!=':') {
			// invalid line; ignore it
			continue;
		}
		*end = '\0';

		ptr++;
		// ignore whitespace
		while(isspace(*ptr)) ptr++;

		// extract the option value
		value = ptr;

		// ignore trailing whitespace
		ptr = value + strlen(value) - 1;
		while(ptr >= value && isspace(*ptr)) ptr--;
		*(ptr+1) = '\0';

		if (Insert(key, value)==FALSE) return FALSE;
	}

	return TRUE;
}


NPBool
MPlugConfiguration::Insert(const char *key, const char *value)
{
	int newEntry = 0;
	HashEntry *entry = CreateHashEntry(&ht_, key, &newEntry);
	char *string;

	if (newEntry) {
		string = new char [strlen(value)+1];
		strcpy(string, value);
		SetHashValue(entry, string);
	} else {
		char *oldValue = (char*) GetHashValue(entry);
		/*int len = strlen(oldValue);
		string = new char [len + strlen(value)+1];
		strcpy(string, oldValue);
		strcpy(string + len, value);*/

		string = new char [strlen(value)+1];
		strcpy(string, value);
		SetHashValue(entry, string);
		delete [] oldValue;
	}

	return TRUE;
}


const char *
MPlugConfiguration::GetValue(const char *key)
{
	LOG(("Trying to find %s in MPlugConfiguration\n", key));
	HashEntry *entry = FindHashEntry(&ht_, key);
	LOG(("Found hash entry: %p\n", entry));
	if (entry) return (const char*)GetHashValue(entry);
	else return NULL;
}
