/*
 * module-temp_combine.cc --
 *
 *      TemporalCombineModule
 *
 *      Class definition for temporal combiner object. Does temporal
 *      reconstruction using only RTP header info, thus is type independent.
 *      Marker bit is assumed to indicate last packet for given timestamp.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "tclcl.h"
#include "pktbuf.h"
#include "codec/module.h"
#include "rtp/media-timer.h"

#define KPATEL_DEBUG

#ifdef KPATEL_DEBUG
#include "logger.h"
extern LogService logger;
#endif


class TemporalCombineModule : public PacketModule, public Timer {
public:
  TemporalCombineModule() : PacketModule(), Timer(), ssrc_(0), last_ts_(0),
    seq_no_(0), last_sent_flag_(0),
    first_time_flag_(1), processing_q_flag_(0), est_ts_delta_(0),
    ts_offset_(0), large_adjust_factor_(0.3), small_adjust_factor_(0.7),
    expected_tolerance_factor_(1.25), catch_up_factor_(1.00),
    dup_last_packet_(0), marked_packet_(0), unmarked_packet_(0),
    late_packet_(0), early_expected_(0), early_queued_(0), late_valid_(0) {
      bind("dup_last_", (int*)&dup_last_packet_);
      bind("marked_", (int*)&marked_packet_);
      bind("unmarked_", (int*)&unmarked_packet_);
      bind("late_", (int*)&late_packet_);
      bind("early_expected_", (int*)&early_expected_);
      bind("early_queued_", (int*)&early_queued_);
      bind("late_valid_", (int*)&late_valid_);
      bind("q_length_", (int*)&q_.length_);
  };

  void recv(pktbuf *pb);
  virtual void timeout();
  virtual int command(int argc, const char*const* argv);

protected:
  void schedule_q();

  void q_recv(pktbuf *pb);

  u_int32_t ssrc_;                /* Source id for resulting packet stream. */

  u_int32_t last_ts_;             /* Timestamp of last emitted packet.      */

  u_int16_t seq_no_;              /* Next packet sequence number to use.    */

  int last_sent_flag_;            /* Indicates whether frame marker bit of
				   * last packet was set. */

  int first_time_flag_;           /* Indicates whether next packet will be
				   * first packet processed. */

  int processing_q_flag_;         /* Indicates whether packet being procecessed
				   * actually came from buffer queue and is
				   * not a new packet. */

  /*
   * PktBufQueue
   *
   *    Internal class definition for queue of packets waiting to be
   *    transmitted.
   */

  class PktBufQueue {
  public:
    PktBufQueue () : head_(0), length_(0) {};
    pktbuf *head_;
    int length_;
    int insert (pktbuf *pb);
    inline pktbuf *get_head() {
      pktbuf *ret;
      if (head_ == 0) return 0;
      ret = head_;
      head_ = ret->next;
      ret->next = 0;
      length_--;
      return ret;
    }
  } q_;                           /* Queue of buffered packets. */

  u_int32_t est_ts_delta_;        /* Estimate of interframe timestamp
				   * difference. */

  void frame_advance(pktbuf *pb);

  MediaTimer mt_;                 /* Media clock. */

  int ts_offset_;                 /* Current delay estimate/target */

  double large_adjust_factor_;    /* Factor used to adjust delay estimate
				   * when frames are dropped. */

  double small_adjust_factor_;    /* Factor used to adjust delay estimage
				   * when frames are sent early. */

  double expected_tolerance_factor_; /* Factor used to determine if
				      * timestamp could be from next expected
				      * frame. */

  double catch_up_factor_;        /* Factor to determine if frame is too
				   * early (must be buffered) or elgible
				   * to be sent now */

  /* Stat variables */

  int dup_last_packet_;
  int marked_packet_;
  int unmarked_packet_;
  int late_packet_;
  int early_expected_;
  int early_queued_;
  int late_valid_;
};

static class TemporalCombineModuleClass : public TclClass {
public:
  TemporalCombineModuleClass() : TclClass("Module/Combine/Temporal") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new TemporalCombineModule());
  }
} combine_temporal_;

int
TemporalCombineModule::command(int argc, const char*const* argv)
{
  if (argc == 2) {
    if (strcmp(argv[1], "reset_stats") == 0) {
      dup_last_packet_ = 0;
      marked_packet_ = 0;
      unmarked_packet_ = 0;
      late_packet_ = 0;
      early_expected_ = 0;
      early_queued_ = 0;
      late_valid_ = 0;
      return TCL_OK;
    }
  }
  if (argc == 3) {
    if (strcmp(argv[1], "ssrc") == 0) {
      u_int32_t new_ssrc;

      new_ssrc = atoi(argv[2]);
      ssrc_ = new_ssrc;
      return TCL_OK;
    }
    if (strcmp(argv[1], "small_factor") == 0) {
      double new_small_factor;

      new_small_factor = atof(argv[2]);
      if (new_small_factor < 0.0) {
	new_small_factor = 0.0;
      } else if (new_small_factor > 1.0) {
	new_small_factor = 1.0;
      }
      small_adjust_factor_ = new_small_factor;
      return TCL_OK;
    }
    if (strcmp(argv[1], "large_factor") == 0) {
      double new_large_factor;

      new_large_factor = atof(argv[2]);
      if (new_large_factor < 0.0) {
	new_large_factor = 0.0;
      } else if (new_large_factor > 1.0) {
	new_large_factor = 1.0;
      }
      large_adjust_factor_ = new_large_factor;
      return TCL_OK;
    }
    if (strcmp(argv[1], "expected_tolerance_factor") == 0) {
      double new_expected_tolerance_factor;

      new_expected_tolerance_factor = atof(argv[2]);
      expected_tolerance_factor_ = new_expected_tolerance_factor;
      return TCL_OK;
    }
    if (strcmp(argv[1], "catch_up_factor") == 0) {
      double new_catch_up_factor;

      new_catch_up_factor = atof(argv[2]);
      catch_up_factor_ = new_catch_up_factor;
      return TCL_OK;
    }
  }
  return (PacketModule::command(argc, argv));
}

void
TemporalCombineModule::frame_advance(pktbuf *pb)
{
  rtphdr *rh = (rtphdr *)pb->data;
  u_int32_t ts = ntohl(rh->rh_ts);

  est_ts_delta_ = (0.5 * est_ts_delta_) + (0.5 * (ts - last_ts_));
  last_ts_ = ts;
  if (rh->rh_flags & htons(RTP_M)) {
    last_sent_flag_ = 1;
  } else {
    last_sent_flag_ = 0;
  }

  if (target_) {
    rh->rh_seqno = htons(seq_no_++);
    rh->rh_ssrc = htonl(ssrc_);
    target_->recv(pb);
  } else {
    pb->release();
  }
  return;
}

int
TemporalCombineModule::PktBufQueue::insert(pktbuf *pb) {
  rtphdr *rh = (rtphdr *)pb->data;
  u_int32_t ts = ntohl(rh->rh_ts);
  pktbuf *qidx, *prev_buf;
  rtphdr *nrh;
  u_int32_t nts;

  length_++;
  if (head_ == 0) {
    head_ = pb;
    head_->next = 0;
    return 1;
  } else {
    prev_buf = 0;
    qidx = head_;
    while (qidx != 0) {
      nrh = (rtphdr *)qidx->data;
      nts = ntohl(nrh->rh_ts);

      if (ts < nts)
	break;
      else if (nts == ts) {
	if (ntohs(rh->rh_seqno) < ntohs(nrh->rh_seqno))
	  break;
      }
      prev_buf = qidx;
      qidx = qidx->next;
    }
    if (prev_buf == 0) {
      pb->next = head_;
      head_ = pb;
      return 1;
    } else {
      prev_buf->next = pb;
      pb->next = qidx;
      return 0;
    }
  }
  /* Shouldn't ever make it here. */
  return 0;
}

void
TemporalCombineModule::schedule_q() {
  if (!processing_q_flag_) {
    if (q_.head_ != 0) {
      pktbuf *pb = q_.head_;
      rtphdr *rh = (rtphdr *) pb->dp;

      u_int32_t now = mt_.media_ts();

      int delay = (((int) (ntohl(rh->rh_ts) - now)) - ts_offset_ - ((int) (est_ts_delta_ * catch_up_factor_))) / 90;
      if (delay < 1) {
	delay = 1;
      }

      msched(delay);
    }
  }
}

void
TemporalCombineModule::timeout()
{
  pktbuf *pb = q_.get_head();

  if (pb != 0) {
    q_recv(pb);
  }
  schedule_q();
  return;
}

void
TemporalCombineModule::q_recv(pktbuf *pb) {
  processing_q_flag_++;
  recv(pb);
  processing_q_flag_--;
}

void
TemporalCombineModule::recv(pktbuf *pb) {
  rtphdr *rh = (rtphdr *)pb->data;
  u_int32_t ts = ntohl(rh->rh_ts);
  u_int32_t now;

  // TOTAL HACK, must fix fundamental problem behind this:
  if (!processing_q_flag_) {
    //    rh->rh_ts = htonl(ts + ntohl(rh->rh_ssrc) % 13);
  }

  now = mt_.media_ts();

  if (first_time_flag_) {
    first_time_flag_ = 0;
    last_ts_ = ts;
    ts_offset_ = ts - now;
  }

  if (ts == last_ts_) {
    if (rh->rh_flags & htons(RTP_M)) {
      if (last_sent_flag_) {
#ifdef KPATEL_DEBUG
	logger.gen_log("DupLastPacket");
#endif
	dup_last_packet_++;
	pb->release();
	schedule_q();
	return;
      } else {
	last_sent_flag_ = 1;
	marked_packet_++;

	if (target_) {
#ifdef KPATEL_DEBUG
	  logger.gen_log("Sent_Marked_Packet");
#endif
	  rh->rh_seqno = htons(seq_no_++);
	  rh->rh_ssrc = htonl(ssrc_);
	  target_->recv(pb);
	} else
	  pb->release();

	pb = q_.get_head();
	if (pb != 0) {
	  q_recv(pb);
	}
	schedule_q();
	return;
      }
    } else {
      unmarked_packet_++;

      if (target_) {
#ifdef KPATEL_DEBUG
	logger.gen_log("Sent_Normal_Packet");
#endif
	rh->rh_seqno = htons(seq_no_++);
	rh->rh_ssrc = htonl(ssrc_);
	target_->recv(pb);
      } else
	pb->release();

      pb = q_.get_head();
      if (pb != 0) {
	q_recv(pb);
      }
      schedule_q();
      return;
    }
  }
  else if (ts < last_ts_) {
    late_packet_++;

#ifdef KPATEL_DEBUG
    logger.gen_log("LatePacket");
#endif
    pb->release();

    int sample_ts_offset = (ts - now);

    if (!processing_q_flag_) {
      ts_offset_ = (int) (((1.0 - large_adjust_factor_) *
			   ((double) ts_offset_) +
			   large_adjust_factor_ *
			   ((double) sample_ts_offset)));
    }

    schedule_q();
    return;

  } else if (ts > last_ts_) {
    int sample_expect_delta = ((int) (now - ts)) + ts_offset_;

    if (sample_expect_delta < 0) {
      /* I'm ahead of my time. */
      if (sample_expect_delta > -1.0 * est_ts_delta_ * catch_up_factor_) {
	if (last_sent_flag_) {
	  /* But within the catchup factor. */
	  if (!q_.insert(pb)) {
	    pb = q_.get_head();
	    q_recv(pb);
	  } else {
	    early_expected_++;
	    ts_offset_ = (int) ((1.0 - small_adjust_factor_) *
				((double) ts_offset_)
				+
				small_adjust_factor_ *
				((int) (ts - now)));
	    pb = q_.get_head();
#ifdef KPATEL_DEBUG
	    logger.gen_log("EarlyExpected");
#endif
	    frame_advance(pb);
	  }
	  schedule_q();
	  return;
	}
      }
#ifdef KPATEL_DEBUG
      logger.gen_log("EarlyQueued");
#endif
      q_.insert(pb);
      early_queued_++;
      schedule_q();
      return;
    } else {
      /* I'm behind, adjust a bit and advance regardless. */
      if (!q_.insert(pb)) {
	pb = q_.get_head();
	q_recv(pb);
      } else {
	late_valid_++;
	ts_offset_ = (int)
	  ((1.0 - small_adjust_factor_) * ((double) ts_offset_) +
	   small_adjust_factor_ * ((int) (ts - now)));
	pb = q_.get_head();
#ifdef KPATEL_DEBUG
	logger.gen_log("LateValid");
#endif
	frame_advance(pb);
      }
      schedule_q();
      return;
    }
  }
  schedule_q();
  return;
}

