# agent-rtp.tcl --
#
#       Source/RTP, MediaAgent and RTPAgent object definitions.
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/net/agent-rtp.tcl,v 1.74 2002/02/03 04:28:05 lim Exp $


# make sure we have network utilities
import NetworkManager Observable mashutils Configuration

#
# Default value for C++ instance variables.
#
Source/RTP set reportLoss_ 0
Session/RTP set nb_ 0
Session/RTP set nf_ 0
Session/RTP set np_ 0
Session/RTP set loopback_ 1

Source/RTP set badsesslen_ 0
Source/RTP set badsessver_ 0
Source/RTP set badsessopt_ 0
Source/RTP set badsdes_ 0
Source/RTP set badbye_ 0
SourceLayer/RTP set nchan_ 1

Session/RTP set badversion_ 0
Session/RTP set badoptions_ 0
Session/RTP set badfmt_ 0
Session/RTP set badext_ 0
Session/RTP set nrunt_ 0

#FIXME
Session/RTP set loopbackLayer_ 1000

Source/RTP public layer-stat which {
	$self instvar layers_
	set s 0
	foreach l $layers_ {
		set s [expr $s + [$l set $which]]
	}
	return $s
}

Source/RTP public ns {} {
	$self instvar layers_
	set s 0
	foreach l $layers_ {
		set s [expr $s + [$l set cs_] - [$l set fs_]]
	}
	return $s
}

Source/RTP public missing {} {
	$self instvar layers_
	set s 0
	foreach l $layers_ {
		set nm [expr [$l set cs_] - [$l set fs_] - [$l set np_]]
		if { $nm > 0 } {
			set s [expr $s + $nm]
		}
	}
	return $s
}

# FIXME
Source/RTP instproc is_mixer {} {
	return [expr [$self srcid] != [$self ssrc]]
}

SourceLayer/RTP set nrunt_ 0
SourceLayer/RTP set ndup_ 0
SourceLayer/RTP set fs_ 0
SourceLayer/RTP set cs_ 0
SourceLayer/RTP set np_ 0
SourceLayer/RTP set nf_ 0
SourceLayer/RTP set nb_ 0
SourceLayer/RTP set nm_ 0
SourceLayer/RTP set ntp_ts_sec_ 0
SourceLayer/RTP set ntp_ts_fsec_ 0
SourceLayer/RTP set mts_ 0
SourceLayer/RTP set ref_ntp_sec_ 0
SourceLayer/RTP set ref_ntp_fsec_ 0
SourceLayer/RTP set ref_mts_ 0


Source/RTP public init { sm srcid ssrc addr } {
	$self next $srcid $ssrc $addr
	$self set sm_ $sm
	$self instvar layers_
	set k 0
	set report 0
	if { [$sm info vars network_] != "" } {
		set net [$sm set network_]
		set n [$net set nchan_]
		set report [$net usingRLM]
	} else {
		set n [SourceLayer/RTP set nchan_]
	}
	while { $k < $n } {
		set l [new SourceLayer/RTP]
		lappend layers_ $l
		$self layer $k $l
#puts stderr "installed layer $k ($l)"
		incr k
	}

	$self set reportLoss_ $report
}


Source/RTP public destroy {} {
	$self instvar sm_

	# delete all the layers associated to this source (SourceLayer/RTP objects)
	foreach layer [$self set layers_] {
		$layer destroy
	}

	# if the Source/RTP object is associated to a data handler (i.e., if the 
	# source to which the object is associated is sending data), deactivate 
	# the corresponding decoder
	if {[$self data-handler] != ""} {
		$self deactivate
	}

	# unregister and delete the source in the SourceManager
	$self unregister
	$sm_ delete $self;

	# destroy the object
	$self next
}



#
# Return the best name for the given source, based on
# the possibly limited information we have.
#
Source/RTP public getid {} {
	set name [$self sdes name]
	if { $name == "" } {
		set name [$self sdes cname]
		if { $name == "" } {
			set name [$self addr]
		}
	}
	return $name
}

#
# Return a the RTP format name as a string for the RTP
# payload format of the media stream underlying this source
# object.
#
Source/RTP public format_name {} {
	$self instvar sm_
	return [$sm_ rtp_type [$self format]]
}

#
# The <u>MediaAgent</u> class is the core programming layer
# for all ``media agents'' that manipulate media over the network.
# These agents are relatively large and complex objects,
# but are used by most applications in a fairly straightforward
# manner.  The idea is to give the MASH programmer
# a fairly course-grained programming object that relieves the
# burden of managing video capture devices, multicast sockets,
# audio processing code, etc. into a high-level, easy-to-use API.
# The effort can then be spent figuring out how to cleanly
# integrate video, audio, or whiteboard objects into the
# application under design.
# <p>
# MediaAgents are ``bare processing engines''.  They do not create
# or rely upon a user-interface.  Instead, the script that creates
# the object manipulates it directly and if a user-interface is desired,
# that script must create and bind the UI to the agent.
# The protocol for communicating between the agent and the UI
# follows the MASH Observer model.
# In fact, since the observer abstraction is completely general,
# any object can attach itself to a MediaAgent as
# an observer, not just UIs.
#
# <p>
# <i>Note: This is an abstract base class. Do not create objects directly
# of this class</i>
#
Class MediaAgent -superclass {SourceManager Observable}

#
# Arrange for each of the methods on an RTP source object to be
# re-directed to the source-manager and in turn dispatched
# to each of the observers.  This arrangement allows subclasses
# to override this default behavior (e.g., the VideoAgent class
# catches activate so it can create a decoder).
#
foreach method "unregister activate deactivate \
		trigger_media \
		trigger_format \
		trigger_sdes \
		trigger_idle \
		trigger_sr \
		notify" {
	Source/RTP public $method {args} \
		"\$self instvar sm_ ; eval \$sm_ $method \$self \$args"
	MediaAgent public $method src "\$self notify_observers $method \$src"
}

MediaAgent public init {} {
	$self next
	$self set sources_ ""
}

MediaAgent public destroy {} {
	$self instvar sources_

	# destroy all the Source/RTP objects
	foreach src $sources_ {
		$src destroy;
	}

	$self next
}


MediaAgent public active_list {} {
	$self instvar active_
	if ![info exists active_] {
		return ""
	}
	return [array names active_]
}

#
# Override some of the above SourceManager methods...
#

MediaAgent public activate src {
	$self instvar active_
	set active_($src) 1
	$self notify_observers activate $src
}

MediaAgent public deactivate src {
	$self instvar active_
	unset active_($src)
	$self notify_observers deactivate $src
}

MediaAgent public unregister src {
	$self notify_observers unregister $src
	$self instvar sources_
	set k [lsearch -exact $sources_ $src]
	set sources_ [lreplace $sources_ $k $k]
}

#
# Attach an observer to this agent.
#
MediaAgent public attach o {
	$self attach_observer $o
#
# FIXME local_ is in sources_ list
#	$o register $local_
	$self instvar sources_ active_
	foreach s $sources_ {
		$o update register $s
		if [info exists active_($s)] {
			$o update activate $s
			$s enable_trigger
		}
	}
}

#
# Detach an observer from this agent.
#
MediaAgent public detach o {
	$self detach_observer $o
	$self instvar sources_ active_
	foreach s $sources_ {
		if [info exists active_($s)] {
			$o update deactivate $s
		}
		$o update unregister $s
	}
}

#
# Called from C++ when a new RTP source is seen (and validated).
#
MediaAgent public create-source { srcid ssrc addr srcsess } {
	set s [new Source/RTP $self $srcid $ssrc $addr]
	$s set session_ $srcsess
	$self instvar sources_
	lappend sources_ $s
	return $s
}

#
# This class implements the machinery of an RTP session.
# It is pure mechanism; no user-interface etc.
# Instead it is manipulated by user interfaces or
# other agents that define policy.
#
Class RTPAgent -superclass MediaAgent -configuration {
	mtu 1024
	loopback 0
	siteDropTime "300"
}

RTPAgent public init {ab {callback {}} } {
	$self next
	$self instvar session_ mtu_ callback_
	if { $callback!={} } { set callback_ $callback }
	set session_ [$self create_session]
	$session_ sm $self
	$session_ buffer-pool [new BufferPool]

	if { $ab != "" } {
		$self reset $ab
	}
	set mtu_ [$self get_option mtu]

	#FIXME
	global V
	set V(sm) $self
}

RTPAgent public destroy {} {
	$self instvar session_ network_

	# exit the M/C session so all the other sources realize we're leaving
	$session_ exit;
	delete $session_

	# clean up the network infrastructure
	delete $network_

	$self next
}

RTPAgent public reset_spec spec {
	set ab [new AddressBlock $spec]
	$self reset $ab
	delete $ab
}

#
# Reset the address specifier of the underlying session in question.
# This allows the network objects to be dynamically reconfigured to
# facilitate higher-level control protocols that might instruct an
# application to switch multicast sessions or speak to a different
# unicast destination. FIXME should only reset the piece that matters.
#
RTPAgent public reset ab {
	$self instvar network_ session_ sources_

	# check the ab is really an AddressBlock object
	if {([catch {$ab info class}]) || ([$ab info class] != "AddressBlock")} {
		# maybe a spec was introduced instead of an AddressBlock object
		$self reset_spec $ab
		return
	}


	# It is important to create the new objects before destroying the 
	# old ones (NetworkManager and Source/RTP) so that, in the case 
	# the agent is transmitting, the encoder has a place to deliver 
	# its packets


	# copy all the old object handlers to destroy them at the end
	if [info exists network_] {
		set old_network $network_
	}
	set old_sources $sources_

	# inform the other session members that we're leaving (send RTP BYE message)
	$session_ exit

	# create the new network object
	set network_ [new NetworkManager $ab $session_ $self]


	#FIXME
	$self app_loopback 1
	$self net_loopback [$self get_option loopback]

	set key [$self get_option sessionKey]
	if { $key != "" } {
		$network_ install-key $key
	}

	# create the local rtp-source object
	$self mk_local_source

	#FIXME currently session object understands only one bandwidth limit
	# FIXME get this into b/s eventually.
	$session_ max-bandwidth [expr [$ab set maxbw_(0)]/1000.]

	# FIXME Eventually use 'attach' mechanism.
	$self instvar callback_
	if [info exists callback_] {
	    eval $callback_ [list $ab]
	} else {
	    catch {
			set a [Application instance]
			if [catch {$a reset $ab rtp $self}] {
				$a reset $ab
			}
		}
	}

	# clean up the old object

	# destroy all the Source/RTP objects
	foreach src $old_sources {
		$src destroy;
	}

	# destroy the old NetworkManager object
	if [info exists old_network] {
		delete $old_network
	}

}

RTPAgent private notify {src layer} {
	$self instvar network_
	if ![$network_ usingRLM] { return }
	$network_ notify-loss $src $layer
}

#
# Return a list of names and values of the statistics
# for the RTP session as a whole.
#
RTPAgent public stats {} {
	set s [$self set session_]
	return " \
		Bad-RTP-version [$s set badversion_] \
		Bad-RTPv1-options [$s set badoptions_] \
		Bad-Payload-Format [$s set badfmt_] \
		Bad-RTP-Extension [$s set badext_] \
		Runts [$s set nrunt_]"
	#FIXME should report counters with bad crypt operations
}

#
# Create the local source object and do proper initialization
# of its various parameters and fields like sdes info.
#
RTPAgent private mk_local_source {} {
	$self instvar network_ session_ local_

	set net [$network_ data-net 0]

	# choose the initial RTP srcid
	set a [$net addr]
	set srcid [$session_ random-srcid $a]
	# create the local source object
	set src [$self create-local $srcid [$net interface]]
	set local_ $src
	$self notify_observers register $local_

	set cname [$self get_option cname]
	if { $cname == "" } {
		set interface [$net interface]
		if { $interface == "0.0.0.0" } {
			# this happens under solaris
			set interface [$session_ local-addr-heuristic]
		}
		set cname [user_heuristic]@$interface
	}
	$src sdes name [$self get_option rtpName]
	$src sdes email [$self get_option rtpEmail]
	# We use the loc field for distributed recording, setting it to
	# be the closest local AS1 platform
	$src sdes loc [$self get_option rtpLoc]
	$src sdes cname $cname

	set tool [Application name]\-[version]

	global tcl_platform
	if {[info exists tcl_platform(os)] && $tcl_platform(os) != "" && \
			$tcl_platform(os) != "unix"} {
		set p $tcl_platform(os)
		if {$tcl_platform(osVersion) != ""} {
			set p $p-$tcl_platform(osVersion)
		}
		if {$tcl_platform(machine) != ""} {
			set p $p-$tcl_platform(machine)
		}
		set tool "$tool/$p"
	}
	$src sdes tool $tool

	return $src
}

#
# Return true iff an underlying network object has been
# created and attached to this RTP agent.
#
RTPAgent public have_network {} {
	$self instvar network_
	return [info exists network_]
}

#
# Return true iff the underlying object that represents
# the local source has been created and attached to this RTP agent.
#
RTPAgent public have_localsrc {} {
	$self instvar local_
	return [info exists local_]
}

#
# Instruct the media-agent to perform encryption and decryption of all
# data sent to or received from the network.  The format of the key is
# as specified by the SDP specification.  FIXME put format here.
#
RTPAgent public install-key key {
	$self instvar network_
	if [info exists network_] {
		$network_ install-key $key
	}
}

#
# Return the network object that underlies the RTP session,
# or return "none" if it does not yet exist.
#
RTPAgent public network {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	return [$network_ data-net 0]
}

#
# Return the network address of the underlying
# communication session, or the string "none"
# if the underlying network object hasn't yet
# been created.
#
RTPAgent public session-addr {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	return [[$self network] addr]
}

#
# Return the network port number of the underlying
# communication session, or the string "none"
# if the underlying network object hasn't yet
# been created.  FIXME why do we have three port methods?
#
RTPAgent public session-port {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	return [[$self network] port]
}

#
# Return the inbound port number of the underlying
# communication session, or the string "none"
# if the underlying network object hasn't yet
# been created.
#
RTPAgent public session-rport {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	return [[$self network] rport]
}

#
# Return the outbound port number of the underlying
# communication session, or the string "none"
# if the underlying network object hasn't yet
# been created.
#
RTPAgent public session-sport {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	return [[$self network] sport]
}

#
# Return the RTP SRCID of the object that represents
# the local source, or the string "none"
# if the underlying network object hasn't yet
# been created.
#
RTPAgent public get_local_srcid {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	$self instvar local_
	return [$local_ srcid]
}

#
# Return the name of an object that sinks the locally generated
# RTP packet stream and in turns sends it out over the network.
#
RTPAgent public get_transmitter {} {
	return [$self set session_]
}

#
# Return the time-to-live values used by the underlying
# communication session, or the string "none"
# if the underlying network object hasn't yet
# been created.  This value is undefined if the
# underlying session is not a multicast address.
#
RTPAgent public session-ttl {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	return [[$self network] ttl]
}

#
# Return the time-to-live values used by the underlying
# communication session, or the string "none"
# if the underlying network object hasn't yet
# been created.  This value is undefined if the
# underlying session is not a multicast address.
#
RTPAgent public local-name {} {
	$self instvar network_
	if ![info exists network_] {
		return none
	}
	$self instvar local_
	return [$local_ sdes name]
}

#
# Set an RTP SDES string for the local source
# to the indicate value.  <i>which</i> is the SDES
# string type and <i>value</i> is the desired string.
# Example usage:
# <pre>
#	$o set_local_sdes email mccanne@cs.berkeley.edu
# </pre>
#
RTPAgent public set_local_sdes { which value } {
	$self instvar local_
	$local_ sdes $which $value
}

#
# Turn off encryption.
#
RTPAgent public crypt_clear {} {
	if [info exists network_] {
		$network_ crypt_clear
	}
}

#
# Gracefully tear down the underlying session. Set or query the media
# associated with this session. Please note that the media is
# automatically set by the init method of this class. Programmers
# should not invoke this method directly to set the media.
#
RTPAgent public shutdown {} {
	$self instvar session_
	$session_ exit
}

#
# Set the maximum number of network channels used by the
# session to <i>n</i>.  Multiple channels are used
# when transmitting layered media streams (e.g., the PVH
# video coding format).  See
# <a href=http://www.cs.berkeley.edu/~mccanne/phd-work>McCanne's
# PhD work</a> for more info on layered video transmission.
#
RTPAgent public set_maxchannel n {}

#
# Set the network bandwidth used by the underlying media transmission
# protocols to <i>bps</i> bits per second.  This rate constraint is
# met in a media-specific fashion.  Currently, most all of the video
# codecs simply adjust the inter-frame spacing to meet the bit-rate
# budget.
#
RTPAgent public set-bandwidth bps {
	[$self set session_] data-bandwidth $bps
}

#
# Set the multicast loopback flag on the underlying session
# according to the <i>sense</i> argument.  If true, then
# packets are looped back according to the semantics of
# the underlying operating system's loopback flag.
# This can have unpredictable results if you don't know
# exactly what you're doing.  Most of the mash applications
# assume that packets are not looped back and will take
# unpredictable action if they see their own traffic looped back.
# However, unless you set filter_own to 0, the app will never
# see the traffic.
#
RTPAgent public net_loopback enable {
	$self instvar network_
	$network_ loopback $enable
}

RTPAgent public app_loopback enable {
	$self instvar session_
	$session_ set loopback_ $enable
}

RTPAgent public set-bandwidth bps {
	[$self set session_] data-bandwidth $bps
}
