# pipeline.tcl --
#
#       Video Pipelines and Video Taps, include from video capture devices
#   The VideoPipeline class provides a coarse-grained interface that
#   abstracts away all the details of the compression and encooding of the
#   user's video for transmission over the network.
#   A VideoTap is the core object that does video capture;
#   it provides an easy-to-use encapsulation of the
#   lower-level VideoCapture objects and Device class hooks.
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/video/pipeline.tcl,v 1.45 2002/02/11 02:28:56 lim Exp $


import Configuration
import RTP/Video

#FIXME goes elsewhere
#Module/VideoEncoder/Pixel/H263+ instproc frame-format {} "return cif"
#Module/VideoEncoder/Pixel/H263 instproc frame-format {} "return cif"
Module/VideoEncoder/Pixel/PVH set pt_ 0
Module/VideoEncoder set nb_ 0
Module/Framer/JPEG set nb_ 0

#
# The Device object is a simple helper object that stores query-able
# device attributes for video hardware.  The overall goal is
# to be able to query information on the OTcl class without
# creating an object instance.  This is a bit tricky and the
# code below is somewhat subtle.  The basic algorithm is to
# call Device::register_class for each class that supports
# hardware video capture, assistance, or output.  This method
# then creates the trivial "nickname" and "attributes" methods
# on the class as well as a set of less trivial methods that
# are re-directed to the Device object but with the first
# argument being the fully-qualified class.
#
# For example, once VideoCapture/Meteor is registered, then queries
# can be carried out on the class itself:
#
#	[VideoCapture/Meteor nickname] -> "Meteor"
#	[VideoCapture/Meteor get_attribute ports] -> "RCA S-Video etc..."
#
Object Device
Device proc register_class c {
	$c proc nickname {} {
		return [$self set nickname_]
	}
	$c proc attributes {} {
		return [$self set attributes_]
	}
	foreach method "get_attribute supports" {
		$c proc $method args "eval Device $method $c \$args"
	}
}

Device proc get_attribute { cl attr } {
	$cl instvar attributes_
	set k [lsearch -exact $attributes_ $attr]
	if { $k >= 0 } {
		incr k
		return [lindex $attributes_ $k]
	}
	return ""
}

Device proc supports { cl key item } {
	set itemList [$self get_attribute $cl $key]
	if { $item == "*" } {
		if { $itemList == "" } {
			return 0
		} else {
			return 1
		}
	} else {
		return [inList $item $itemList]
	}
}

#
# If there are video capture objects available, then set up the
# necessary class procs so that the attributes and nicknames can
# be queried in a clean manner without actually opening the device.
# FIXME need to check for assistors and output-devices too.
#
if [TclObject is-class VideoCapture] {
	foreach c [VideoCapture info subclass] {
		Device register_class $c
	}
}

#
# Simple helper procedure that returns true iff
# <i>item</i> is an element of the list <i>L</i>.
#
proc inList { item L } {
	return [expr [lsearch -exact $L $item] >= 0]
}

#
# A VideoTap is the core object that does video capture;
# it provides an easy-to-use encapsulation of the
# lower-level VideoCapture objects and Device class hooks.
# <p>
# The video capture objects do rate control by looking
# at the absolute byte count produce by the target encoder
# object before and after coding the frame.
#
Class VideoTap

VideoTap public init {} {
	$self next
	$self instvar device_ running_ fps_ bps_ decimate_
	set device_ ""
	set running_ 0
	set fps_ 8
	set bps_ 128000
	set decimate_ 2
}

VideoTap public target { target encoder } {
	$self instvar grabber_
	$grabber_ encoder $encoder
	$grabber_ target $target
}

#
# Return true iff the underlying video capture sub-system
# is running, i.e., that frames are actively being captured,
# compressed, and sent to the target module (e.g., over the net).
#
VideoTap public running {} {
	return [$self set running_]
}

#
# Return a list of the available (i.e., compiled-in)
# modules of supported video codecs.
#
VideoTap public input_devices {} {
	if ![TclObject is-class VideoCapture] {
		return ""
	}
	return [VideoCapture info subclass]
}

#
# Release the underlying capture device if it
# has been created.  Otherwise, do nothing.
#
VideoTap public release {} {
	$self instvar grabber_
	if [info exists grabber_] {
		$self close
	}
	$self instvar norm_
	if [info exists norm_] {
		unset norm_
	}
}

#
# Close the currently select video capture device
# and deallocate the compression pipeline.
# If the device is active, it will be gracefully shutdown
# and deallocated.  If any capture windows we created
# to perform the capture, they will be deallocated.
#
VideoTap public close {} {
	$self instvar grabber_ capwin_
	$self stop

	if [info exists grabber_] {
		delete $grabber_
		unset grabber_
	}

	#FIXME
	if [info exists capwin_] {
		# delete the C++ object, then destrory the tk window
		delete $capwin_
		destroy [winfo toplevel capwin_]
		unset capwin_
	}
}

#
# If the compression pipeline is running, halt it.
# Though halted, no resources are actually de-allocated;
# thus, the capture process can be efficiently resumed
# by calling VideoTap::start.  This method has no
# effect when called while the device is already stopped
# or non-existant.
#
VideoTap public stop {} {
	$self instvar running_ grabber_ capwin_
	if $running_ {
		$grabber_ send 0
		set running_ 0
		if [info exists capwin_] {
			wm withdraw [winfo toplevel $capwin_]
		}
	}
}

#
# If the compression pipeline is not running, start it.
# The underlying capture device is created
# and initialized if necessary.  The capture process
# continues indefinitely, until VideoTap::stop
# or VideoTap::release are called.
#
VideoTap public start {} {
	$self instvar running_ grabber_ capwin_
	if !$running_ {
		if ![info exists grabber_] {
#			puts strerr "Crash here?"
			return "VideoTap::start: device is not openned"
		}
		if [info exists capwin_] {
			wm deiconify [winfo toplevel $capwin_]
			update idletasks
		}
		$grabber_ send 1
		set running_ 1
	}
	return ""
}

#
# Set the background ``fill rate'' to <i>v</i> blocks per second.
# The fill rate determines how many blocks of each video frame
# are sent as background fill.  Each frame a few of these blocks
# are coded in left-to-right scan at very high quality.  Thus,
# static backgrounds (like video-capture slides) eventually fill
# in at high quality.
#
VideoTap public fillrate v {
	$self instvar grabber_
	if [info exists grabber_] {
		$grabber_ fillrate $v
	}
}

#
# Open the underlying video hardware...
#
VideoTap public open { device videoType } {
	$self instvar grabber_ device_ capwin_ fps_ bps_ decimate_ port_
# puts "entered VideoTap::open (device=$device, videoType=$videoType)"
	if [info exists grabber_] {
		$self close
	}
#	puts "device assigned"
	set device_ $device
	set grabber_ [new $device_ $videoType]

	#
	# special cases
	#
	if { $grabber_ == "" && $videoType == "411" } {
		# try cif instead of 411
		set grabber_ [new $device_ cif]
	}
	if { $grabber_ == "" } {
		# shouldn't happen
		$self fatal "couldn't set up [$device nickname] grabber for $videoType"
	}
	set error [$grabber_ status]
	if { $error < 0 } {
		$self close
		if { $error == -2 } {
			#FIXME hack: this happens when we tried to open
			# a jvideo with a non-jpeg format
			return "Can't use jvideo with $format_ format"
		}
		return "can't open [$device_ nickname] capture device"
	}

	#
	# Splice in the titler object if present.
	# (Titlers are enabled via an external agent over the
	#  coordination bus.)
#	Titler insert $grabber_

	if { [$grabber_ need-capwin] && ![info exists capwin_] } {
		#
		# we need to create a window for input devices that
		# require capturing to the frame buffer.  create but
		# don't map it until after we call "$grabber decimate"
		# to specify it's size
		#
		toplevel .capture -class Vic
		wm title .capture "Video Capture Window"
		$grabber_ create-capwin .capture.video
		set capwin_ .capture.video
		pack .capture.video

		# capture window shouldn't be covered
		bind .capture <Visibility> "raise .capture"
	}
	$grabber_ fps $fps_
	$grabber_ bps $bps_
	$grabber_ decimate $decimate_
	if [info exists port_] {
		$grabber_ port $port_
	}
	$self instvar norm_
	if [info exists norm_] {
		$grabber_ norm $norm_
	}

	return ""
}

#
# FIXME - passes command onto underlying grabber... not sure
# we should do this this way.
#
VideoTap instproc grabber args {
	$self instvar grabber_
	if [info exists grabber_] {
		eval $grabber_ $args
	}
}

#
# Set the bit rate of the underlying session to <i>v</i> bits/second.
# This controls the target bit rate as well as the rate assumed
# by the session for its control bandwidth target.  (RTCP uses
# at most 5% of the data rate limit for control traffic.)
#
VideoTap instproc set_bps v {
	$self instvar grabber_ bps_
	set bps_ $v
	if [info exists grabber_] {
		$grabber_ bps $v
	}
}

#
# Set the maximum frame rate of the underlying video capture process
# to <i>v</i> frames/second.  The operational rate may be lower because
# of the bit-rate constraint.  If the transmission is frame-rate limited,
# then the packets are spaced according to the bit-rate constraint.
# For example, if the bit-rate is 100 kb/s, the frame is 10 f/s,
# each frame is 5 kb, and a packet is 1kb, then the 5 packets that
# comprise a frame will be sent 10ms apart for a total of 50ms,
# even though the frames are spaces 100ms apart.
#
VideoTap instproc set_fps v {
	$self instvar grabber_ fps_
	set fps_ $v
	if [info exists grabber_] {
		$grabber_ fps $v
	}
}

#
# Set the spatial decimation factor for the video capture
# process to a factor of <i>v</i> both horizontally and vertically.
# For example, if the native capture size is 640x480
# and the decimation factor is set to 2, then the coded
# stream will be 320x240.
#
VideoTap instproc set_decimate v {
	$self instvar grabber_ decimate_
	set decimate_ $v
	if [info exists grabber_] {
		$grabber_ decimate $v
	}
}

#
# Set the input port for video capture to <i>p</i>, which must be
# one of the names in the selected device's port attribute.
#
VideoTap instproc set_port p {
	$self instvar grabber_ port_
	set port_ $p
	if [info exists grabber_] {
		$grabber_ port $p
	}
}

#
# Set the input norm for the current port for video capture to <i>n</i>,
# which must be one of the names in the selected device's norm attribute.
#
VideoTap instproc set_norm n {
	$self instvar grabber_ norm_
	set norm_ $n
	if [info exists grabber_] {
		$grabber_ norm $n
	}
}

#
# The VideoPipeline class provides a coarse-grained interface that
# abstracts away all the details of the compression and encooding of the
# user's video for transmission over the network.
#
Class VideoPipeline -superclass RTP/Video -configuration {
	mtu 1024
}

#
# FIXME comment.  tap is a videotap.
# <i>session</i> is must be an object that simply supports
# two methods: get_local_srcid, which returns the RTP SRCID
# to use for the local transmitter, and get_transmitter, which
# returns the name of PacketModule to serve as the target
# of the video compressor's output.
#
VideoPipeline public init session {
	$self next
	$self instvar format_ tap_ session_ quality_ initialized_
	set tap_ [new VideoTap]
	set session_ $session
	set format_ ""
	set quality_ 10
	set initialized_ 0
}

VideoPipeline public destroy {} {
	$self instvar tap_ bufferPool_

	# release the device (delete encoder y video capture device)
	$self release_device

	# delete the BufferPool/RTP object if it exists
	if [info exists bufferPool_] {
		$bufferPool_ destroy
	}

	# delete the underlying VideoTap
	$tap_ destroy

	$self next
}

#
# FIXME fix this... should just expose videotap.
#
VideoPipeline public set_decimate { v } {
	return [[$self set tap_] set_decimate $v]
}

VideoPipeline public running {} {
	return [[$self set tap_] running]
}

VideoPipeline public input_devices {} {
	return [[$self set tap_] input_devices]
}

VideoPipeline public set_bps {{args {}}} {
	$self instvar tap_
	if {[llength $args] == 0} {
		return [$tap_ set bps_]
	}
	return [eval $tap_ set_bps $args]
}

VideoPipeline public set_fps args {
	return [eval [$self set tap_] set_fps $args]
}

VideoPipeline public start args {
	$self instvar initialized_ device_ format_

	if {!$initialized_} {
		# FIXME - need to put the is-hardware-encode check back in once 
		# it gets fixed
		#	if { [$device_ is-hardware-encode] } {
		# }
		if { 0 } {

			set err [$self hwopen $device_ $format_]
		} else {
			set err [$self open $device_ $format_]
		}
		if {$err == ""} {
			set initialized_ 1
		} else {
			return $err
		}
	}
	return [eval [$self set tap_] start $args]
}

VideoPipeline public stop args {
	return [eval [$self set tap_] stop $args]
}

VideoPipeline public set_port args {
	return [eval [$self set tap_] set_port $args]
}

VideoPipeline public set_norm args {
	return [eval [$self set tap_] set_norm $args]
}

VideoPipeline public fillrate args {
	return [eval [$self set tap_] fillrate $args]
}

VideoPipeline public hardware args {
	return [eval [$self set tap_] grabber $args]
}

#
# Given a video capture device <i>d</i>, return a list of
# all the possible video formats that we are capable of
# generating by splicing together arbitrary compression pipelines.
# Note that this is not just the native formats supported by
# the device but any format that we are capable of generating
# in software from a native format.  You might think that
# they should be any of the supported codec formats, but because
# we don't necessarily support software encoding of every
# format (e.g., software MPEG is not currently supported),
# this method may return a subset of the set of all possible formats.
#
VideoPipeline public available_formats device {
	set sizes [$device get_attribute size]
	set formats [$device get_attribute format]
	set fmtList ""
	if [inList 422 $formats] {
		set fmtList "$fmtList nv nvdct cellb jpeg"
	}
	if [inList 411 $formats] {
		set fmtList "$fmtList pvh"
	}
	if [inList cif $sizes] {
		set fmtList "$fmtList h261 h263+ h263"
#change: take out H.263 for a while.
#		set fmtList "$fmtList h261"
	}
	if [inList jpeg $formats] {
		set fmtList "$fmtList jpeg"
		if [$self yesno [$self get_option useJPEGforH261]] {
			set fmtList "$fmtList h261"
		}
	}
	return $fmtList
}

#
# Release the currently select video capture device.
# If the device is active, it will be gracefully shutdown
# and deallocated.  If any capture windows we created
# to perform the capture, they will be deallocated.
#
VideoPipeline public release_device {} {
	$self instvar tap_ initialized_
	if $initialized_ {
		$self close
	}
}

#
# Close the currently select video capture device
# and deallocate the compression pipeline.
# If the device is active, it will be gracefully shutdown
# and deallocated.  If any capture windows we created
# to perform the capture, they will be deallocated.
#
VideoPipeline public close {} {
	$self instvar encoder_ tap_ initialized_
	$tap_ release
	set initialized_ 0
	if [info exists encoder_] {
		delete $encoder_
		unset encoder_
	}
}

#
# Choose a video compression <i>format</i> for the specified <i>device</i>.
# May be called while the device is active or is not selected.
# If active, the underlying compression pipeline is gracefully
# reconfigued.  If not active, the new format will be used
# when that device is next selected.
# Returns empty string if all goes well, otherwise returns
# the error message from openning the device.
#
VideoPipeline public select { device format } {
#puts "VideoPipeline::select device=$device format=$format"
	#
	# Remember whether we were running.
	#
	$self instvar tap_ device_ format_ initialized_
	set running [$tap_ running]
	#
	# Simply close and re-open the device to change the selection.
	# FIXME - we should check if the device is the same
	# and avoid the device shutdown because on some systems
	# this is very slow (e.g., solaris xil)
	#
	set err ""
	if $initialized_ {
		if {($device_!=$device)||($format_!=$format)} {
			$self close
			set err [$self open $device $format]
		}
	}
#	puts "device assigned in videopipeline"
	set device_ $device
#puts [$device -hue]
	set format_ $format
	#
	# Restart if necessary
	#
	if $running {
		if {$err != ""} {return $err}
		$self start
	}
}

#
# Create and return encoder object that is capable of encoding
# the video stream type indicated by <i>fmt</i>.  Perform any
# format-specific intialization required.
#
VideoPipeline private create_encoder fmt {
	set pt [$self rtp_fmt_number $fmt]
	if { $fmt == "nvdct" } {
		set encoder [new Module/VideoEncoder/Pixel/NV]
		$encoder use-dct 1
	} else {
		set fmt [$self classmap $fmt]
		set encoder [new Module/VideoEncoder/Pixel/$fmt]
	}
	if { $encoder == "" } {
		# should never happen
		$self fatal "cannot allocate $fmt encoder"
	}
	if {$pt >= 0} {
		$encoder set pt_ $pt
	}
	return $encoder
}

#
# Open the video capture device and initialize the compression
# pipeline.  Assumes the format has already been specified
# from select.  Uses a number of heuristics combined with
# cues from configuration options to construct the best
# arrangement of objects to carry out the desired operation.
# For example, hardware compression is used when available,
# and capture windows are created when the video capture
# hardware is built into the frame buffer (e.g., parallax).
# Returns a null string upon success; otherwise, returns
# a string describing the error encountered.
#
VideoPipeline public open { device format } {
	set useJPEGforH261 [$self yesno [$self get_option useJPEGforH261]]
	$self instvar tap_ encoder_ format_ bufferPool_ session_ quality_
	$tap_ release
	# puts "Entered VideoPipeline::open(device=$device, format=$format)"

	set format_ $format

	set DF [$device get_attribute format]
	set DS [$device get_attribute size]

	# first, see if device can produce bitstream directly
	# if not, try a transcoder
	# if that doesn't work, try anything else

	if [inList $format_ $DF] {
		set encoder_ [$self create_encoder $format_]
		set grabtarget $encoder_
		set grabq ""
	} elseif { $format_ == "h261" && [inList jpeg $DF] && \
			$useJPEGforH261 } {
		# just jpeg->h261 at this time
		set transcoder [new transcoder/jpeg/dct]
		set encoder_ [new Module/VideoEncoder/DCT/H261]
		$transcoder target $encoder_
		set grabtarget $transcoder
		set grabq "70"
	} elseif { [inList $format_ [$self available_formats $device] ] } {
		set encoder_ [$self create_encoder $format_]
		set grabtarget $encoder_
		set grabq ""
	}

	$encoder_ mtu [$self get_option mtu]

	# configure the BufferPool/RTP object with the session's srcid
	if ![info exists bufferPool_] {
		set bufferPool_ [new BufferPool/RTP]
	}
	# configure the BufferPool/RTP object with the session's srcid
	$bufferPool_ srcid [$session_ get_local_srcid]

	# inform the encoder about the BufferPool/RTP it has to get packets from
	$encoder_ buffer-pool $bufferPool_

	# link the encoder to the session
	$encoder_ target [$session_ get_transmitter]

	set ff [$grabtarget frame-format]
	set err [$tap_ open $device $ff]
	if { $err != "" } {
		#FIXME memory leak?
		return $err
	}


	$tap_ target $grabtarget $encoder_

	#
	# Splice in the titler object if present.
	# (Titlers are enabled via an external agent over the
	#  coordination bus.)
#FIXME not yet
#	Titler insert $grabber_

#	if { $grabq != "" } {
#		$grabber_ q $grabq
#	}
	$self set_quality $quality_

	return ""
}

#
# LAR: this code isn't working so I'm commenting it out.
# the call on setq_$format isn't right see "_" at end, so
# it always returns 0 and just returns a random string
#
# Set the ``quality factor'' for the underlying
# video compression algorithm in use.  <i>q</i> is
# a format-independent factor between 1 and 100, where
# lower numbers indicate lower quality.  Each format
# type translates this generic number into one or
# more codec parameters that impact quality (by
# trading off bit rate).  Returns the codec-specific
# quality value.
#
VideoPipeline public set_quality q {
    $self instvar format_ quality_
    set quality_ $q
    if { [catch "$self setq_$format_ $q" val] == 0 } {
        return $val         
    }
    return -1
}


#
# Switches the VideoPipeline session
#
VideoPipeline public switch_session session {
	$self instvar tap_ encoder_ bufferPool_ session_ running_

	# get sure the VideoTap is stopped while we change the session
	if [$tap_ set running_] {
		set restart 1
		$self stop

	} else {
		set restart 0
	}

	# change the session
	set session_ $session

	# configure the BufferPool/RTP object with the session's srcid
	if ![info exists bufferPool_] {
		set bufferPool_ [new BufferPool/RTP]
	}
	# configure the BufferPool/RTP object with the session's srcid
	$bufferPool_ srcid [$session_ get_local_srcid]

	# link the encoder to the session
	$encoder_ target [$session_ get_transmitter]

	if {$restart == 1} {
		$self start
	}
}


#
# Format specific routine to map generic quality <i>value</i>
# into actions that program the underlying JPEG codec.
#
VideoPipeline private setq_jpeg value {
	incr value
	if { $value > 95 } {
		set value 95
	} elseif { $value < 5 } {
		set value 5
	}
#	$self instvar grabber_
#	if [info exists grabber_] {
#		$grabber_ q $value
#	}

#FIXME this doesn't work for hardware jpeg

	$self instvar encoder_
	if [info exists encoder_] {
		$encoder_ q $value
	}
	return $value
}

#
# Format specific routine to map generic quality <i>value</i>
# into actions that program the underlying H.261 codec.
#
VideoPipeline private setq_h261 value {
	set value [expr int((1 - $value / 100.) * 29) + 1]
	$self instvar encoder_
	if [info exists encoder_] {
		$encoder_ q $value
	}
	return $value
}

#
# Format specific for H.263+
# Paul
VideoPipeline private setq_h263+ value {
	set value [expr int((1 - $value / 100.) * 29) + 1]
	$self instvar encoder_
	if [info exists encoder_] {
		$encoder_ q $value
	}
	return $value
}

#
# Format specific for H.263
# Paul
VideoPipeline private setq_h263 value {
	set value [expr int((1 - $value / 100.) * 29) + 1]
	$self instvar encoder_
	if [info exists encoder_] {
		$encoder_ q $value
	}
	return $value
}

#
# Format specific routine to map generic quality <i>value</i>
# into actions that program the underlying NV codec.
#
VideoPipeline private setq_nv value {
	set value [expr (100 - $value) / 10]
	$self instvar encoder_
	if [info exists encoder_] {
		$encoder_ q $value
	}
	return $value
}

set pvh_shmap { 0 1 2 1 }
set pvh_shs {
	{ lum-dct 0 5-1--11- }
	{ lum-dct 1 ---5111- }
	{ lum-dct 2 --51-11- }
	{ lum-sbc 0 ----4--2 }
	{ lum-sbc 1 ----4--2 }
	{ lum-sbc 2 ----4--2 }
	{ chm     0 -5---1-- }
	{ chm     1 ---5-1-- }
	{ chm     2 --5--1-- }
}

#
# Format specific routine to map generic quality <i>value</i>
# into actions that program the underlying PVH codec.
#
VideoPipeline private setq_pvh value {
	$self instvar encoder_
	if ![info exists encoder_] {
		return -1
	}
	#FIXME ignore value and just set up the bit allocation
	#FIXME should have variable strategies here
	global pvh_shmap pvh_shs
	set n [llength $pvh_shmap]
	set i 0
	while { $i < $n } {
		$encoder_ shmap $i [lindex $pvh_shmap $i]
		incr i
	}
	set i 0
	foreach tuple $pvh_shs {
		set compID [lindex $tuple 0]
		set shID [lindex $tuple 1]
		set pattern [lindex $tuple 2]
		$encoder_ comp $compID $shID $pattern
	}
	#FIXME
	return -1
}


#
# Send a signal to the encoder to ask them to send an intra frame
#
VideoPipeline private send_full_intra_frame { } {
	$self instvar tap_
	$tap_ send_full_intra_frame
}
VideoTap private send_full_intra_frame { } {
	$self instvar grabber_
	$grabber_ send_full_intra_frame
}
