/*
 * rtp-qm-indexer.cc --
 *
 *      Receives archived audio packets and finds the corresponding mic
 *      index at the time of recording.  Used to synchronize mic inputs with
 *      audio packets during replay.
 *
 * Copyright (c) 2000-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "tclcl.h"
#include "rtp-qm-indexer.h"

#include "qm-files.h"

static class RTPQMIndexerClass : public TclClass {
public:
	RTPQMIndexerClass() : TclClass("Module/RTPQMIndexer") {}
	TclObject* create(int argc, const char*const* argv) {
		return(new RTPQMIndexer(argc, argv));
	}
} rtp_qm_indexer_class;

/**********************************************************/

RTPQMIndexer::RTPQMIndexer(int argc, const char* const* argv)
{
//  printf("RTPQMIndexer::RTPQMIndexer: called; this=0x%x\n", (unsigned int)this);
  struct QMHeader hdr;

  infile = NULL;
  firstTs = 0;
  lastTs = 0;
  callback[0] = '\0';

  if(argc > 4)
  {
    if(argv[4][0] != '\0')
    {
      infile = fopen(argv[4], "r");
      if(infile == NULL)
      {
	fprintf(stderr, "unable to open timestamps file\n");
      }
      else
      {
	fread(&hdr, sizeof(struct QMHeader), 1, infile);
	if(hdr.postProcessed)
	{
	  firstTs = hdr.firstTs;
	  lastTs = hdr.lastTs;
	}
	else
	{
	  fprintf(stderr, "Warning:  rtp qm indexer file is not post processed\n");
	}
      }
    }
  }

  if(argc > 5)
  {
    if(strlen(argv[5]) > CALLBACK_LENGTH)
    {
      fprintf(stderr, "RTPQMIndexer::RTPQMIndexer: Error, callback must be less than %d characters.  Will not use callback\n", CALLBACK_LENGTH);
    }
    else
    {
      if(argv[5][0] != '\0')
      {
	sprintf(callback, argv[5]);
      }
    }
  }
}


RTPQMIndexer::~RTPQMIndexer()
{
  if(infile != NULL)
  {
    fclose(infile);
  }
}

void RTPQMIndexer::recv(pktbuf* pb)
{
  int seqno;
  int ts;
  int index;

  rtphdr* rh = (rtphdr*)pb->data;
  seqno = ntohs(rh->rh_seqno);
  ts = ntohl(rh->rh_ts);

  // do the lookup in the file for this sequence number and check if there
  //    is a corresponding index number
  index = getIndex(ts);

  if(index >= 0)
  {
    if(callback[0] != '\0')
    {
      Tcl& tcl = Tcl::instance();
      tcl.evalf("%s %d", callback, index);
    }
  }

  // important!!  RTP data packets must be released
  pb->release();
}

int RTPQMIndexer::getIndex(u_int32_t ts)
{
  int offset;
  struct QMPacketInfo info;

  if(ts >= firstTs && ts <= lastTs)
  {
    // the index is guaranteed to be there since it's a post-processed file
    //   however, index may be -1 which means there was no packet with this
    //   timestamp received by the QM (this can happen if the recorder gets
    //   a packet that QM does not)
    offset = (ts - firstTs) / TS_STEP;
//    printf("ts = %u, firstTs = %u, offset = %d\n", ts, firstTs, offset);
    offset = (offset * sizeof(struct QMPacketInfo))+ sizeof(struct QMHeader);
    fseek(infile, offset, SEEK_SET);
    fread(&info, sizeof(struct QMPacketInfo), 1, infile);
//    printf("ts = %u, index = %u\n", ts, info.index);
    return(info.index);
  }
  else
  {
    fprintf(stderr, "Got a TS out of range of timestamps file: %u\n", ts);
  }
  return(-1);
}

