/*
 * Copyright (c) 1997 Carnegie Mellon University. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is hereby granted (including for commercial or
 * for-profit use), provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative
 * works, or modified versions, and any portions thereof, and that
 * both notices appear in supporting documentation, and that credit
 * is given to Carnegie Mellon University in all publications reporting
 * on direct or indirect use of this code or its derivatives.
 *
 * THIS SOFTWARE IS EXPERIMENTAL AND IS KNOWN TO HAVE BUGS, SOME OF
 * WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON PROVIDES THIS
 * SOFTWARE IN ITS ``AS IS'' CONDITION, AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * Carnegie Mellon encourages (but does not require) users of this
 * software to return any improvements or extensions that they make,
 * and to grant Carnegie Mellon the rights to redistribute these
 * changes without encumbrance.
 */

#ifndef lint
static const char sccsID[] = "Revision: %W%        %G%     Hui Zhang";
#endif

#include <stdio.h>
#include <sys/time.h>
#include <math.h>

#include "pktbuf.h"
#include "hfsc_queue.h"
#include "time.h"
#include "ntp-time.h"
#include "hfsc_err.h"
#include "hfsc_def.h"
#include "hfsc_decl.h"
#include "hfsc_fun.h"

#define DEBUG_HFSC 
#ifdef DEBUG_HFSC
FILE *debug_fp;
#endif

/* defines the amount of LOG entries in HFSC_LOG, 10 means 1/10 of entries */
#define LOG_FACTOR 10

/* extern double clock; */
static const char HierarchyFile[] = "hierarchy_input"; /* input file */
/* extern struct Channel *Channel[]; */
 

/* used for randomization, for logging purpose */
int rand2(void)
{
	static unsigned long int next = 1;
	next = next * 1103515245 + 12345;
	return (unsigned int)(next/65536);
}


/************************************************************************
 *
 *  NAME: coarseClock
 *
 *  DESCRIPTION: 
 *    Returns clock in 1/COARSE_FACTOR milliseconds scale.  So
 *    assuming COARSE_FACTOR is 100, the function returns the clock 
 *    in 10 microseconds (10usec)
 *
 *  NOTE:
 *    mult(x) function is replaced by macro defined in hfsc_fun.h
 *
 ************************************************************************/


inline double gettime()
{
	timeval tv;
	::gettimeofday(&tv, 0);
	double t = 1e5 * tv.tv_sec + tv.tv_usec/10;
	printf("Time returned is: %f\n", t);
	return (t);
}

int coarseClock() 
{
	/* debugging*/
	static int my_clock;
	//	static i=0;

	/* test clock wrapping
	 * WRAP = 0;  normal case
	 * WRAP = 2143000000;  from positive to negative
	 * WRAP = -3000000; from negative to positive
	 */
	static int WRAP = 0;
	my_clock = (int) (gettime() * COARSE_FACTOR) + WRAP; 

	return my_clock; 
}



HFSCQueue::HFSCQueue(ClassItem *root)
{
	data = (HFSCStruct *) newHFSCData(root);
}


void HFSCQueue::insert(SRMv2_pktbuf *pb)
{
	unsigned char id = pb->tcid;
	ClassItem   *ps = getClassItem(hfscRoot(), id);

	this->length++;

	insert_at_tail(obj_to_lst((char *) pb), ps->dataQueue);
	if (!ps->nBussy) {
		changeSessionStatus(data, ps, TRUE);
		/* compute the new eligible time and deadline */
		ps->e = getRTSC_x(ps->pEligible, mult(ps->cWork));
		ps->d = getRTSC_x(ps->pDeadline, mult(ps->cWork + pb->len));
		ps->pGuarantReq->e = ps->e;
		ps->pGuarantReq->d = ps->d;
		enqueueReq2(data->pReqQueue, ps->pGuarantReq, coarseClock());
	}
}


SRMv2_pktbuf *HFSCQueue::remove()
{
	register lstptr  headEntry;
	pkt_str *dataQueue;
	SRMv2_pktbuf *pkt = 0;
	
	HFSCStruct *phData = this->data;
      
	ReqItem    *preq;
	ClassItem  *ps; 
	u_int      len;
	
	if ((preq = dequeueFirstReq2(phData->pReqQueue, coarseClock()))) {
		
		ps = (ClassItem *) preq->pClass;
		
		/* delete the packet from the data queue */
		if ((dataQueue = (pkt_str *)(ps->dataQueue)) == NULL)
			hfsc_panic("HFSCDelete");
		
		if ((headEntry = dataQueue->head)) {
			pkt = (SRMv2_pktbuf*) headEntry->udata;
			length--;
			pop_list(dataQueue);
			
			/* yc-a: the following code solves cWork wrap around problem, by
			 * reseting pEligible and pDeadline sc, and then set cWork to 0
			 */
			/* can be optimized */
			if ((int)ps->cWork > (int)MAX_WORK) {
				cutRTSC(ps->pEligible, mult(ps->cWork));
				cutRTSC(ps->pDeadline, mult(ps->cWork));
				ps->cWork = 0;
			} 
			len = pkt->len;
			ps->cWork     += len;
			/* phData->cWork += pkt->pkt_len; */
			free(headEntry);
			
			/* update the virtual time and the total amount of work in hierarchy */ 
			updateHierarchy(ps, len);
			
			/* yc: debugging info */
			
			
			if (ps->dataQueue->head) {
				/* compute the deadline of the next packet */
				SRMv2_pktbuf *pktb = (SRMv2_pktbuf *) ps->dataQueue->head->udata;
				len = pktb->len;
				
				ps->e = getRTSC_x(ps->pEligible, mult(ps->cWork));
				ps->d = getRTSC_x(ps->pDeadline, mult(ps->cWork + len));
				ps->pGuarantReq->e = ps->e;
				ps->pGuarantReq->d = ps->d;
				enqueueReq2(phData->pReqQueue, ps->pGuarantReq, coarseClock());
			} else 
				
				/* the session becomes passive */
				changeSessionStatus(phData, ps, FALSE); 
			
		} /* else 
		     hfsc_panic(ErrMsg[ERR_EMPTY_QUEUE]); */
	} else {
		
		/* get the session in the hierarchy with the minimum vt */
		if (!(ps = selSesByVT(phData->proot))) 
			return NULL;
		
		/* delete the packet from the data queue */
		if ((dataQueue = (pkt_str *)(ps->dataQueue)) == NULL)
			hfsc_panic("HFSCDelete");
		
		if ((headEntry = dataQueue->head)) {
			
			/* ps->dataQueue->length--; */
			pkt = (struct SRMv2_pktbuf *)headEntry->udata;
			length--;
			pop_list(dataQueue);
			len = pkt->len;
			free(headEntry);
			
			/* update the virtual time and the total amount of work in hierarchy */ 
			updateHierarchy(ps, len);
			
			/* yc: debugging info */
      
			if (ps->dataQueue->head) {
				/* compute the deadline of the next packet */
				len = ((SRMv2_pktbuf *) ps->dataQueue->head->udata)->len;

			  /* compute the new eligible time and the new deadline */
				dequeueReq2(phData->pReqQueue, ps->pGuarantReq, 
					    coarseClock());
				ps->e = getRTSC_x(ps->pEligible, mult(ps->cWork));
				ps->d = getRTSC_x(ps->pDeadline, mult(ps->cWork + len));
				ps->pGuarantReq->e = ps->e;
				ps->pGuarantReq->d = ps->d;
				enqueueReq2(phData->pReqQueue, ps->pGuarantReq, 
					    coarseClock());

			} else {

				dequeueReq2(phData->pReqQueue, ps->pGuarantReq, 
					    coarseClock());
				/* update the bussy flag in the hierarchy */
				changeSessionStatus(phData, ps, FALSE); 
			}
		} /* else 
		     hfsc_panic(ErrMsg[ERR_EMPTY_QUEUE]); */
	}

	return pkt;
}

ClassItem *HFSCQueue::hfscRoot()
{
	return data->proot;
}
