/*
 * coordbus.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "iohandler.h"
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include "inet.h"


#define CB_MULTICAST_ADDR  "224.255.222.239"
#define CB_MULTICAST_PORT  57007
#define CB_MTU             512


class CoordinationBus : public IOHandler {
public:
	CoordinationBus() : rsock_(-1), ssock_(-1) { }
	virtual ~CoordinationBus() { close(); }

	int open(int argc, const char * const * argv);
	int close(int argc, const char * const * argv);
	int transmit(int argc, const char * const * argv);

private:
	virtual void dispatch(int);
	void close();
	int create_sender(u_int32_t address, int port, int ttl);
	int create_receiver(u_int32_t address, int port, int ttl);
	int rsock_, ssock_;
	unsigned char buffer_[CB_MTU+1];
};


int
CoordinationBus::open(int argc, const char * const * argv)
{
	const char *mode;
	int channel, ttl;
	u_int32_t address;
	int port;
	
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(channel);
	ARG(ttl);
	ARG(mode);
	
	address = inet_addr(CB_MULTICAST_ADDR);
	if (address == (u_int32_t) -1) {
		Tcl::instance().resultf("invalid multicast address: %s",
			    CB_MULTICAST_ADDR);
		return TCL_ERROR;
	}
	port = ntohs(CB_MULTICAST_PORT + channel);

	if (strcmp(mode, "readonly")==0 || strcmp(mode, "readwrite")==0) {
		if (create_receiver(address, port, ttl)==TCL_ERROR)
			return TCL_ERROR;
	}
	
	if (strcmp(mode, "writeonly")==0 || strcmp(mode, "readwrite")==0) {
		if (create_sender(address, port, ttl)==TCL_ERROR) {
			if (rsock_ >= 0) ::close(rsock_);
			rsock_ = -1;
			return TCL_ERROR;
		}
	}

	if (rsock_ >= 0) link(rsock_, TCL_READABLE);
	return TCL_OK;
}


int
CoordinationBus::close(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;

	close();
	return TCL_OK;
}


void
CoordinationBus::close()
{
	if (rsock_ >= 0) {
		unlink();
		::close(rsock_);
		rsock_ = -1;
	}

	if (ssock_ >= 0) {
		::close(ssock_);
		ssock_ = -1;
	}
}


int
CoordinationBus::transmit(int argc, const char * const *argv)
{
	const char *packet;
	int bytes;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(packet);
	END_PARSE_ARGS;

	if (ssock_ >= 0) {
		bytes = send(ssock_, packet, strlen(packet), 0);
		if (bytes < 0) {
			Tcl::instance().resultf("CoordinationBus::transmit: "
						"send error: %s",
						strerror(errno));
			return TCL_ERROR;
		}
	} else {
		Tcl::instance().result("CoordinationBus::transmit: "
				       "bus does not have write permissions");
		return TCL_ERROR;
	}
	
	return TCL_OK;
}


/*
 *   address: the multicast group address of the channel(in network format)<BR>
 *   port: the multicast port of the channel (in network format)<BR>
 *   ttl: the ttl to be used (in host format) ttl=0 =&gt; local scope <BR>
 */
int
CoordinationBus::create_receiver(u_int32_t address, int port, int ttl)
{
	Tcl &tcl = Tcl::instance();
	
#ifdef IP_ADD_MEMBERSHIP
	rsock_ = socket(AF_INET, SOCK_DGRAM, 0);
	if (rsock_ < 0) {
		tcl.resultf("CoordinationBus::create_receiver socket(): %s",
			    strerror(errno));
		return TCL_ERROR;
	}
	int on = 1;
	if (setsockopt(rsock_, SOL_SOCKET, SO_REUSEADDR, (char *)&on,
		       sizeof(on)) < 0) {
		tcl.resultf("CoordinationBus::create_receiver "
			    "SO_REUSEADDR: %s", strerror(errno));
		return TCL_ERROR;
	}
#ifdef SO_REUSEPORT
	on = 1;
	if (setsockopt(rsock_, SOL_SOCKET, SO_REUSEPORT, (char *)&on,
		       sizeof(on)) < 0) {
		tcl.resultf("CoordinationBus::create_receiver "
			    "SO_REUSEPORT: %s", strerror(errno));
		return TCL_ERROR;
	}
#endif
	sockaddr_in sin;
	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = port;
	sin.sin_addr.s_addr = address;
	if (::bind(rsock_, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		sin.sin_addr.s_addr = INADDR_ANY;
		if (::bind(rsock_, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
			::close(rsock_);
			rsock_ = -1;
			tcl.resultf("CoordinationBus::create_receiver "
				    "bind(): %s", strerror(errno));
			return TCL_ERROR;
		}
	}

	struct ip_mreq mr;
	mr.imr_multiaddr.s_addr = address;
	if (ttl==0) {
		// this is a local channel; use the loopback interface
		mr.imr_interface.s_addr = htonl(INADDR_LOOPBACK);
	}
	else {
		// this is a network channel; use a real interface
		mr.imr_interface.s_addr = INADDR_ANY;
	}

	if (setsockopt(rsock_, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
		       (char *)&mr, sizeof(mr)) < 0) {
		if (ttl==0) {
			// probably, the loopback interface didn't succeed;
			// try using a real interface instead
			mr.imr_interface.s_addr = INADDR_ANY;
			if (setsockopt(rsock_, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
				       (char *)&mr, sizeof(mr)) < 0) {
				// host probably doesn't have multicast
				::close(rsock_);
				rsock_ = -1;
				tcl.resultf("CoordinationBus::create_receiver "
					    "ADD_MEMBERSHIP: %s",
					    strerror(errno));
				return TCL_ERROR;
			}
		}
		else {
			/*
			 * host probably doesn't have multicast
			 */
			::close(rsock_);
			rsock_ = -1;
			tcl.resultf("CoordinationBus::create_receiver "
				    "ADD_MEMBERSHIP: %s", strerror(errno));
			return TCL_ERROR;
		}
	}

	return TCL_OK;
#else 
	tcl.resultf("CoordinationBus::create_receiver: not compiled with "
		    "multicast support");
	return TCL_ERROR;
#endif // IP_ADD_MEMBERSHIP
}


/*
 *   address: the multicast group address of the channel(in network format)<BR>
 *   port: the multicast port of the channel (in network format)<BR>
 *   ttl: the ttl to be used (in host format) ttl=0 =&gt; local scope<BR>
 */
int
CoordinationBus::create_sender(u_int32_t address, int port, int ttl)
{
	Tcl &tcl = Tcl::instance();
	
#ifdef IP_ADD_MEMBERSHIP
	sockaddr_in sin;
  
	ssock_ = socket(AF_INET, SOCK_DGRAM, 0);
	if (ssock_ < 0) {
		tcl.resultf("CoordinationBus::create_sender socket(): %s",
			    strerror(errno));
		return TCL_ERROR;
	}
	sin.sin_port = 0;
	if (ttl==0) {
		sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	}
	else {
		sin.sin_addr.s_addr = INADDR_ANY;
	}
	if (::bind(ssock_, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		sin.sin_addr.s_addr = INADDR_ANY;
		if (::bind(ssock_, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
			tcl.resultf("CoordinationBus::create_sender bind(): "
				    "%s", strerror(errno));
			return TCL_ERROR;
		}
	}
	sin.sin_family = AF_INET;
	sin.sin_port = port;
	sin.sin_addr.s_addr = address;
	if (connect(ssock_, (struct sockaddr*)&sin, sizeof(sin)) < 0) {
		tcl.resultf("CoordinationBus::create_sender connect(): %s",
			    strerror(errno));
		return TCL_ERROR;
	}
	char ttl_ = (char) ttl;
	if (setsockopt(ssock_, IPPROTO_IP, IP_MULTICAST_TTL,
		       &ttl_, 1) < 0) {
		tcl.resultf("CoordinationBus::create_sender IP_MULTICAST_TTL: "
			    "%s", strerror(errno));
		return TCL_ERROR;
	}

	u_int32_t addr;
	if (ttl==0)
		addr = htonl(INADDR_LOOPBACK);
	else
		addr = INADDR_ANY;

	if (setsockopt(ssock_, IPPROTO_IP, IP_MULTICAST_IF,
		       (char *)&addr, 4) < 0) {
		if (ttl==0) {
			addr = INADDR_ANY;
			if (setsockopt(ssock_, IPPROTO_IP, IP_MULTICAST_IF,
				       (char *)&addr, 4) < 0) {
				tcl.resultf("CoordinationBus::create_sender "
					    "IP_MULTICAST_IF: %s",
					    strerror(errno));
				return TCL_ERROR;
			}
		}
		else {
			tcl.resultf("CoordinationBus::create_sender "
				    "IP_MULTICAST_IF: %s", strerror(errno));
			return TCL_ERROR;
		}
	}

#ifdef ultrix
	char on = 1;
	if (setsockopt(ssock_, IPPROTO_IP, IP_MULTICAST_LOOP,
		       &on, 1) < 0) {
		tcl.resultf("CoordinationBus::create_sender "
			    "IP_MULTICAST_LOOP: %s", strerror(errno));
		return TCL_ERROR;
	}
#endif

	return TCL_OK;
#else
	tcl.resultf("CoordinationBus::create_sender: not compiled with "
		    "multicast support");
	return TCL_ERROR;
#endif // IP_ADD_MEMBERSHIP
}


void 
CoordinationBus::dispatch(int)
{
	int bytes;

	bytes = recv(rsock_, buffer_, CB_MTU, 0);
	if (bytes < 0) return;

	buffer_[bytes] = '\0';
	Tcl::instance().evalf("{%s} dispatch {%s}", name(), buffer_);
}
