/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAttribute.cpp - abstract SAML attribute class

   Scott Cantor
   2/23/02
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLAttributeFactory* SAMLAttribute::m_factory=NULL;

SAMLAttributeFactory* SAMLAttribute::setFactory(SAMLAttributeFactory* factory)
{
    SAMLConfig::getConfig().saml_lock();
    SAMLAttributeFactory* ret=m_factory;
    m_factory=factory;
    SAMLConfig::getConfig().saml_unlock();
    return ret;
}

SAMLAttribute* SAMLAttribute::getInstance(DOMElement* e)
{
    if (!e)
        throw MalformedException(SAMLException::RESPONDER,"SAMLAttribute::getInstance given an empty DOM");

    SAMLAttribute* a=NULL;
    SAMLConfig::getConfig().saml_lock();
    SAMLAttributeFactory* f=m_factory;
    SAMLConfig::getConfig().saml_unlock();
    return (f) ? f(e) : new SAMLAttribute(e);
}

SAMLAttribute* SAMLAttribute::getInstance(istream& in)
{
    XML::Parser p;
    XML::StreamInputSource src(in);
    Wrapper4InputSource dsrc(&src,false);
    DOMDocument* doc=p.parse(dsrc);
    try
    {
        SAMLAttribute* a=getInstance(doc->getDocumentElement());
        a->setDocument(doc);
        return a;
    }
    catch(...)
    {
        doc->release();
        throw;
    }
}

SAMLAttribute::SAMLAttribute(
    const XMLCh* name, const XMLCh* ns, const saml::QName* type, long lifetime, const Iterator<const XMLCh*>& values
    ) : m_name(XML::assign(name)), m_namespace(XML::assign(ns)), m_lifetime(lifetime)
{
    RTTI(SAMLAttribute);
    m_type=type ? new saml::QName(*type) : NULL;
    while (values.hasNext())
        m_values.push_back(XML::assign(values.next()));
}

SAMLAttribute::SAMLAttribute(DOMElement* e, bool processDOM) : m_name(NULL), m_namespace(NULL), m_type(NULL), m_lifetime(0)
{
    RTTI(SAMLAttribute);
    if (processDOM)
        fromDOM(e);
}

SAMLAttribute::SAMLAttribute(istream& in, bool processDOM) : SAMLObject(in), m_name(NULL), m_namespace(NULL), m_type(NULL)
{
    RTTI(SAMLAttribute);
    if (processDOM)
        fromDOM(m_document->getDocumentElement());
}

SAMLAttribute::~SAMLAttribute()
{
    if (m_bOwnStrings) {
        XMLString::release(&m_name);
        XMLString::release(&m_namespace);
        for (vector<const XMLCh*>::iterator i=m_values.begin(); i!=m_values.end(); i++) {
            XMLCh* p = const_cast<XMLCh*>(*i);
            XMLString::release(&p);
        }
    }
    delete m_type;
}

void SAMLAttribute::ownStrings()
{
    if (!m_bOwnStrings) {
        m_name=XML::assign(m_name);
        m_namespace=XML::assign(m_namespace);
        for (vector<const XMLCh*>::iterator i=m_values.begin(); i!=m_values.end(); i++)
            (*i)=XML::assign(*i);
        m_bOwnStrings=true;
    }
}

void SAMLAttribute::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);
    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAML_NS,L(Attribute)))
        throw MalformedException("SAMLAttribute::fromDOM() missing saml:Attribute element at root");

    m_name=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(AttributeName)));
    m_namespace=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(AttributeNamespace)));

    e=XML::getFirstChildElement(e,XML::SAML_NS,L(AttributeValue));
    while (e) {
        if (!m_type)
            m_type=saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type));
        valueFromDOM(e);
        e=XML::getNextSiblingElement(e,XML::SAML_NS,L(AttributeValue));
    }
    
    checkValidity();
}

void SAMLAttribute::valueFromDOM(DOMElement* e)
{
    const XMLCh* ptr=NULL;
    DOMNode* n=e->getFirstChild();
    if (n && n->getNodeType()==DOMNode::TEXT_NODE && (ptr=n->getNodeValue()))
        m_values.push_back(ptr);
    else {
#ifdef _DEBUG
        saml::NDC ndc("valueFromDOM");
#endif
        SAML_log.warn("skipping AttributeValue without a single, non-empty text node");
        m_values.push_back(&chNull);
    }
}

void SAMLAttribute::setName(const XMLCh* name)
{
    if (XML::isEmpty(name))
        throw MalformedException("name cannot be null or empty");
    
    if (m_bOwnStrings)
        XMLString::release(&m_name);
    else {
        m_name=NULL;
        ownStrings();
    }
    m_name=XML::assign(name);
    setDirty();
}

void SAMLAttribute::setNamespace(const XMLCh* ns)
{
    if (XML::isEmpty(ns))
        throw MalformedException("name cannot be null or empty");
    
    if (m_bOwnStrings)
        XMLString::release(&m_namespace);
    else {
        m_namespace=NULL;
        ownStrings();
    }
    m_namespace=XML::assign(ns);
    setDirty();
}

XMLCh* SAMLAttribute::computeTypeDecl(DOMElement* e) const
{
    static const XMLCh sep[]={chColon, chNull};
    XMLCh* xsitype=NULL;
    e->removeAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,typens));
    if (m_type) {
        const XMLCh* prefix=NULL;
        if (!XMLString::compareString(XML::XSD_NS,m_type->getNamespaceURI()))
            prefix=L(xsd);
        else {
            e->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,typens),m_type->getNamespaceURI());
            prefix=L(typens);
        }
        xsitype=new XMLCh[XMLString::stringLen(prefix) + XMLString::stringLen(m_type->getLocalName()) + 2];
        xsitype[0]=chNull;
        XMLString::catString(xsitype,prefix);
        XMLString::catString(xsitype,sep);
        XMLString::catString(xsitype,m_type->getLocalName());
    }
    return xsitype;
}

void SAMLAttribute::setType(const QName* type)
{
    delete m_type;
    if (type)
        m_type=new QName(*type);
    ownStrings();
    setDirty();
}

void SAMLAttribute::setLifetime(long lifetime)
{
    m_lifetime=lifetime;
}

DOMNodeList* SAMLAttribute::getValueElements() const
{
    return (!m_bDirty && m_root) ?
        static_cast<DOMElement*>(m_root)->getElementsByTagNameNS(XML::SAML_NS,L(AttributeValue)) : NULL;
}

void SAMLAttribute::setValues(const Iterator<const XMLCh*>& values)
{
    while (m_values.size())
        removeValue(0);
    while (values.hasNext())
        addValue(values.next());
}

void SAMLAttribute::addValue(const XMLCh* value)
{
    if (!value)
        value=&chNull;

    ownStrings();

    m_values.push_back(XML::assign(value));
    if (!m_sbValues.empty()) {
        char* temp=toUTF8(value);
        m_sbValues.push_back(temp);
        delete[] temp;
    }
    setDirty();
}

void SAMLAttribute::removeValue(unsigned long index)
{
    if (m_bOwnStrings) {
        XMLCh* p=const_cast<XMLCh*>(m_values[index]);
        XMLString::release(&p);
    }
    m_values.erase(m_values.begin()+index);
    if (!m_sbValues.empty())
        m_sbValues.erase(m_sbValues.begin()+index);
    ownStrings();
    setDirty();
}

DOMElement* SAMLAttribute::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* a = doc->createElementNS(XML::SAML_NS, L(Attribute));
    if (xmlns)
        a->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    a->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsi),XML::XSI_NS);
    a->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsd),XML::XSD_NS);
    return a;
}

void SAMLAttribute::valueToDOM(unsigned int index, DOMElement* e) const
{
    if (!XML::isEmpty(m_values[index]))
        e->appendChild(e->getOwnerDocument()->createTextNode(m_values[index]));
}

DOMNode* SAMLAttribute::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc,xmlns);
    DOMElement* a=static_cast<DOMElement*>(m_root);
    
    if (m_bDirty) {
        a->setAttributeNS(NULL,L(AttributeName),m_name);
        a->setAttributeNS(NULL,L(AttributeNamespace),m_namespace);

        XMLCh* xsitype=computeTypeDecl(a);
        for (unsigned int i=0; i<m_values.size(); i++) {
            DOMElement* v = a->getOwnerDocument()->createElementNS(XML::SAML_NS,L(AttributeValue));
            if (xsitype)
                v->setAttributeNS(XML::XSI_NS, L_QNAME(xsi,type), xsitype);
            valueToDOM(i,v);
            a->appendChild(v);
        }
        delete[] xsitype;
    
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(a,XML::SAML_NS);
        DECLARE_NAMESPACE(a,xsi,XML::XSI_NS);
        DECLARE_NAMESPACE(a,xsd,XML::XSD_NS);
    }

    return m_root;
}

Iterator<string> SAMLAttribute::getSingleByteValues() const
{
    if (m_sbValues.empty()) {
        for (vector<const XMLCh*>::const_iterator i=m_values.begin(); i!=m_values.end(); i++) {
            char* temp=toUTF8(*i);
            if (temp)
                m_sbValues.push_back(temp);
            delete[] temp;
        }
    }
    return m_sbValues;
}

void SAMLAttribute::checkValidity() const
{
    if (XML::isEmpty(m_name) || XML::isEmpty(m_namespace) || m_values.empty())
        throw MalformedException("Attribute invalid, requires name and namespace, and at least one value");
}

SAMLObject* SAMLAttribute::clone() const
{
    return new SAMLAttribute(m_name,m_namespace,m_type,m_lifetime,m_values);
}

