/*
** Copyright (C) 2000 Idan Shoham <idan@m-tech.ab.ca>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include "project.h"
#include "reporter.h"

static NETWORK_CELL **network_cols;
static int numX;
static int numY;


/* construct start and finish markers */
void Reporter::constructStartFinish( Project *project )
{
    mStartTask = new TASK("", "START", 0);
    mStartTask->setStart(0);
    mStartTask->nx = 0;
    mStartTask->ny = project->networkStartY();
    
    mFinishTask = new TASK("", "FINISH", 0);
    mFinishTask->setStart(0);
    mFinishTask->nx = numX - 1;
    mFinishTask->ny = project->networkFinishY();
}

    
/* return 1 if nx not assigned yet but all dependencies have been assigned */
static int canAssignX(TASK *task)
{
    if ( task->nx != -1)
	return 0;			/* already assigned */

    if ( task->isVacation() )
	return 0;			/* vacations aren't assigned */

    for ( TASK::PTRLIST::const_iterator pt = task->begin_depends(); 
	  pt != task->end_depends(); 
	  pt++ )
	if ( (*pt)->nx == -1 )
	    return 0;

    return 1;
}


/* set value of nx to be 1 more than child values */
static void assignX(TASK *task)
{
    /* the "start" task will be in column 0 so all true tasks must be
       in column 1 or greater */
    int max_depend = 0;
    for ( TASK::PTRLIST::const_iterator pt = task->begin_depends(); 
	  pt != task->end_depends(); 
	  pt++ )
	if ( (*pt)->nx > max_depend )
	    max_depend = (*pt)->nx;

    task->nx = max_depend + 1;
}


/* find maximum X value */
int Reporter::maximumX(Project *project)
{
    int maxX = -1;
    for ( Project::TPLCI pt = project->beginTaskList() ; 
	  pt != project->endTaskList() ; 
	  pt++ )
    {
	if ( (*pt)->isVacation() )
	    continue;
	
	if ( (*pt)->nx > maxX )
	    maxX = (*pt)->nx;
    }
    if ( maxX == -1 )
	Error("Can't find maxX in maximuX()");

    return maxX;
}


static int nextAvailableY(NETWORK_CELL *clist, int y)
{
    NETWORK_CELL *pc;
    int ny;

    ny = y;
    while (1)
    {
	ny++;
	for ( pc = clist; pc != NULL; pc = pc->next )
	    if ( pc->task->ny == ny )
		break;
	if ( pc == NULL )
	    return ny;
    }
    return 0;				/* not reached */
}


/* assign Y values */
void Reporter::assignY(Project *project)
{
    int y;
    int x;
    NETWORK_CELL *pc;
    NETWORK_CELL *cell;

    /* find maximum X values */
    numX = maximumX(project);

    // printf("maxX = %d\n", numX);

    numX++;				/* convert to number of x vals */

    numX++;				/* to allow for finish in last col */

    if ((network_cols = (NETWORK_CELL **)malloc(sizeof(NETWORK_CELL *) * numX)) == NULL)
	Error("assignY: Couldn't alloc network_cols");
    for ( x = 0; x < numX ; x++ )
	network_cols[x] = NULL;

    /* build up an index of cells by column */
    for ( Project::TPLCI pt = project->beginTaskList() ; 
	  pt != project->endTaskList() ; 
	  pt++ )
    {
	if ( (*pt)->isVacation() )
	    continue;

	x = (*pt)->nx;

	if ( ( cell = (NETWORK_CELL *)malloc(sizeof(NETWORK_CELL))) == NULL)
	    Error("assignY: Can't alloc network cell");
	cell->task = (*pt);
	cell->isSpace = 0;
	cell->prev = NULL;
	cell->next = NULL;

	if ( network_cols[x] == NULL )
	    network_cols[x] = cell;
	else
	{
	    for ( pc = network_cols[x]; pc->next != NULL; pc = pc->next )
		;
	    cell->prev = pc;
	    pc->next = cell;
	}
    }
    
    /* assign Y values on first come basis, skipping those assigned */
    for ( x = 0; x < numX; x++ )
    {
	y = nextAvailableY(network_cols[x], -1);
	for ( pc = network_cols[x]; pc != NULL; pc = pc->next )
	    if ( pc->task->ny == -1 )
	    {
		pc->task->ny = y;
		y = nextAvailableY(network_cols[x], y);
	    }
    }

    /* find number of Y values */
    numY = 0;
    for ( x = 0; x < numX; x++ )
	for ( pc = network_cols[x]; pc != NULL; pc = pc->next )
	    if ( pc->task->ny > numY )
		numY = pc->task->ny;
    numY++;
}


FILE *Reporter::openNetworkDiagram( Project *project, 
				    const char *filename, 
				    int llx, int lly, int urx, int ury)
{ 
    FILE *f;

    if ( ( f = fopen(filename, "w") ) == NULL )
    {
	Error("Cannot open file [%s]",filename);
    }

    fprintf(f, "%%!\n");
    fprintf(f, "%%%%BoundingBox: %d %d %d %d\n",llx,lly,urx,ury);

    fprintf(f, "/OUTLINE\n");
    fprintf(f, "{\n");
    fprintf(f, "  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
    fprintf(f, "  x1 y1 moveto\n");
    fprintf(f, "  x2 y1 lineto\n");
    fprintf(f, "  x2 y2 lineto\n");
    fprintf(f, "  x1 y2 lineto\n");
    fprintf(f, "  closepath\n");
    fprintf(f, "  0 setgray\n");
    fprintf(f, "  stroke\n");
    fprintf(f, "} def\n");

    fprintf(f, "/BACKGROUND\n");
    fprintf(f, "{\n");
    fprintf(f, "  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
    fprintf(f, "  x1 y1 moveto\n");
    fprintf(f, "  x2 y1 lineto\n");
    fprintf(f, "  x2 y2 lineto\n");
    fprintf(f, "  x1 y2 lineto\n");
    fprintf(f, "  closepath\n");
    fprintf(f, "  %f setgray\n", tg_white);
    fprintf(f, "  fill\n");
    fprintf(f, "} def\n");

    fprintf(f, "/%s findfont\n", pc_fontname1); 
    fprintf(f, "dup length dict begin\n"); 
    fprintf(f, " { 1 index /FID ne\n"); 
    fprintf(f, " {def}\n"); 
    fprintf(f, " {pop pop}\n"); 
    fprintf(f, " ifelse\n"); 
    fprintf(f, " } forall\n"); 
    fprintf(f, " /Encoding ISOLatin1Encoding def\n"); 
    fprintf(f, " currentdict\n"); 
    fprintf(f, "end\n"); 
    fprintf(f, "/%s-ISOLatin1 exch definefont pop\n", pc_fontname1); 
    fprintf(f, "/%s findfont %f scalefont setfont\n",
	    pc_fontname1, pc_fontsize1 );

    fprintf(f, "0 setgray\n");
    fprintf(f, "0.01 setlinewidth\n");

    return f;
}


NETWORK_CELL *Reporter::findCellInColumn(int x, const char *id)
{
    NETWORK_CELL *cell;

    for ( cell = network_cols[x] ; cell != NULL ; cell = cell->next )
    {
	if ( cell->isSpace )
		continue;
	if ( strcmp( cell->task->id(), id ) == 0 )
	    return cell;
    }

    Error("Asked for cell (%s) in col (%d) but not there", id, x );
    return NULL;			/* only to keep compiler happy */
}


void Reporter::write_milestones( Project *project,
				 FILE *fp, int ax, int bx, int ay, int by)
{
    for ( Project::MPLCI ml = project->beginMilestoneList() ; 
	  ml != project->endMilestoneList() ; 
	  ml++ )
    {
	MILESTONE * m = *ml;
	TASK * t = m->critical();

	int px = ax * t->nx + bx + pc_width;
	int py = ay * t->ny + by + pc_height;

	fprintf(fp, "%d %d moveto ", px + pc_height / 8, py + pc_height / 4);
	fprintf(fp, "%d %d lineto ", px, py);
	fprintf(fp, "%d %d lineto ", px - pc_height / 8, py + pc_height / 4);
	fprintf(fp, "%d %d lineto ", px + pc_width / 3, py + pc_height / 4);
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px - pc_height / 8,
		py + pc_height / 4 + pc_textup );
	char buf[1024];
	if (milestone_ids)
	    sprintf(buf, "%s %s", m->id(), m->name());
	else
	    sprintf(buf, "%s", m->name());
	fprintf(fp, "(%s) show\n", buf);
    }
}

	

void Reporter::write_chart( Project *project, 
			    int start, int finish, const char *filename)
{
    int minX;
    int maxX;
    int minY;
    int maxY;
    int sizeX;
    int sizeY;
    int x;
    NETWORK_CELL *cell;
    int px, py;				/* coords in postscript */
    int ax, bx, ay, by;
    int llx, lly, urx, ury;
    FILE *fp;
    char buf[1024];
    int pxd, pyd;
    NETWORK_CELL *pc;
    int c0, c1, c2;
    int r0, r1, r2, r3;
    double dTextHeight;

    /* work out the size */
    if ( start > 0 )
	minX = start;
    else
	minX = 0;
    if ( finish < numX )
	maxX = finish;
    else
	maxX = numX - 1;
    sizeX = maxX - minX + 1;
    minY = numY;
    maxY = 0;
    for ( x = minX ; x <= maxX ; x++)
	for ( cell = network_cols[x] ; cell != NULL ; cell = cell->next )
	{
	    if ( cell->task->ny < minY )
		minY = cell->task->ny;
	    if ( cell->task->ny > maxY )
		maxY = cell->task->ny;
	}
    sizeY = maxY - minY + 1;

//    printf("(minX,maxX,sizeX)=(%d,%d,%d)\n", minX, maxX, sizeX);
//    printf("(minY,maxY,sizeY)=(%d,%d,%d)\n", minY, maxY, sizeY);


    /* work out map from (x,y) to postscript coords of bottom left corner */
    ax = pc_width + pc_space;
    bx = -ax * minX;
    ay = -( pc_height + pc_space );
    by = ( pc_height + pc_space ) * maxY;
    
    /* work out bounding box */
    llx = 0;
    lly = -1;				/* to allow for line width */
    urx = ax * maxX + bx + pc_width + 1; /* +1 to allow for line width */
    ury = ay * minY + by + pc_height;
    
    /* allow for potential milestone marks on top row */
    urx += pc_width / 3;
    ury += pc_height / 2;

    fp = openNetworkDiagram(project, filename, llx, lly, urx, ury);

    /* do all of the joins first */
    for ( x = minX ; x <= maxX ; x++)
    {
	px = ax * x + bx;
	for ( cell = network_cols[x] ; cell != NULL ; cell = cell->next )
	{
	    if ( cell->isSpace )
		continue;
	    
	    py = ay * cell->task->ny + by;

	    if ( cell->task->begin_depends() == cell->task->end_depends() )
	    {
		if ( minX == 0 )
		{
		    pxd = ax * mStartTask->nx + bx;
		    pyd = ay * mStartTask->ny + by;
		    
		    fprintf(fp, "%d %d moveto\n", px, py + pc_height / 2 );
		    fprintf(fp, "%d %d lineto\n",
			    pxd + pc_width, pyd + pc_height / 2 );
		    fprintf(fp, "stroke\n" );
		}
	    }
	    else
	    {
		for ( TASK::PTRLIST::const_iterator pt = cell->task->begin_depends(); 
		      pt != cell->task->end_depends(); 
		      pt++ )
		{
		    pc = findCellInColumn( (*pt)->nx , (*pt)->id() );
		    pxd = ax * pc->task->nx + bx;
		    pyd = ay * pc->task->ny + by;
		    
		    fprintf(fp, "%d %d moveto\n", px, py + pc_height / 2 );
		    fprintf(fp, "%d %d lineto\n",
			    pxd + pc_width, pyd + pc_height / 2 );
		    fprintf(fp, "stroke\n" );
		}
	    }
	}
    }
    if ( maxX == numX - 1 )
    {
	px = ax * maxX + bx;
	py = ay * mFinishTask->ny + by;
	for ( x = minX ; x <= maxX ; x++)
	{
	    for ( cell = network_cols[x] ; cell != NULL ; cell = cell->next )
	    {
		if ( cell->isSpace )
		    continue;

		if ( cell->task->begin_follows() == cell->task->end_follows() )
		{
		    pxd = ax * cell->task->nx + bx;
		    pyd = ay * cell->task->ny + by;
		    
		    fprintf(fp, "%d %d moveto\n", px, py + pc_height / 2 );
		    fprintf(fp, "%d %d lineto\n",
			    pxd + pc_width, pyd + pc_height / 2 );
		    fprintf(fp, "stroke\n" );
		}
	    }
	}
    }

    
    c0 = 0;
    c1 = pc_width / 3;
    c2 = 2 * pc_width / 3;
    dTextHeight = 0.25 * ( pc_height - 7 * pc_textup );
    r0 = 0;
    r1 = 2 * pc_textup + (int)dTextHeight;
    r2 = 3 * pc_textup + (int)( 2.0 * dTextHeight );
    r3 = 5 * pc_textup + (int)( 3.0 * dTextHeight );

    fprintf(fp,"1.0 setlinewidth\n");

    /* do the start marker */
    if ( minX == 0 )
    {
	x = 0;
	px = ax *  x + bx;
	py = ay * mStartTask->ny + by;
	fprintf(fp, "%d %d %d %d BACKGROUND\n",
		px, px + pc_width, py, py + pc_height);

	fprintf(fp, "%d %d %d %d OUTLINE\n",
		px, px + pc_width, py, py + pc_height);

	fprintf(fp, "%d %d moveto ", px, py + r1 );
	fprintf(fp, "%d %d lineto ", px + pc_width, py + r1 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px, py + r3 );
	fprintf(fp, "%d %d lineto ", px + pc_width, py + r3 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c1, py );
	fprintf(fp, "%d %d lineto ", px + c1, py + r1 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c1, py + r3 );
	fprintf(fp, "%d %d lineto ", px + c1, py + pc_height );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c2, py );
	fprintf(fp, "%d %d lineto ", px + c2, py + r1 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c2, py + r3 );
	fprintf(fp, "%d %d lineto ", px + c2, py + pc_height );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c0 + pc_textin, py + r2 + pc_textup );
	fprintf(fp, "(START) show\n");

	fprintf(fp, "%d %d moveto ", px + c0 + pc_textin, py + r3 + pc_textup );
	fprintf(fp, "(%s) show\n", project->sStartDay());
    }
    
    for ( x = minX ; x <= maxX ; x++)
    {
	px = ax * x + bx;
	for ( cell = network_cols[x] ; cell != NULL ; cell = cell->next )
	{
	    if ( cell->isSpace )
		continue;
	    
	    py = ay * cell->task->ny + by;

	    fprintf(fp, "%d %d %d %d BACKGROUND\n",
		    px, px + pc_width, py, py + pc_height);

	    fprintf(fp, "%d %d %d %d OUTLINE\n",
		    px, px + pc_width, py, py + pc_height);

	    fprintf(fp, "%d %d moveto ", px, py + r1 );
	    fprintf(fp, "%d %d lineto ", px + pc_width, py + r1 );
	    fprintf(fp, "stroke\n");
	    
	    fprintf(fp, "%d %d moveto ", px, py + r3 );
	    fprintf(fp, "%d %d lineto ", px + pc_width, py + r3 );
	    fprintf(fp, "stroke\n");
	    
	    fprintf(fp, "%d %d moveto ", px + c1, py );
	    fprintf(fp, "%d %d lineto ", px + c1, py + r1 );
	    fprintf(fp, "stroke\n");
	    
	    fprintf(fp, "%d %d moveto ", px + c1, py + r3 );
	    fprintf(fp, "%d %d lineto ", px + c1, py + pc_height );
	    fprintf(fp, "stroke\n");
	    
	    fprintf(fp, "%d %d moveto ", px + c2, py );
	    fprintf(fp, "%d %d lineto ", px + c2, py + r1 );
	    fprintf(fp, "stroke\n");
	    
	    fprintf(fp, "%d %d moveto ", px + c2, py + r3 );
	    fprintf(fp, "%d %d lineto ", px + c2, py + pc_height );
	    fprintf(fp, "stroke\n");
	    
	    fprintf(fp, "%d %d moveto ", px + c0 + pc_textin, py + r2 + pc_textup );
	    if (task_ids)
		sprintf(buf, "%s %s", cell->task->id(), cell->task->name());
	    else
		sprintf(buf, "%s", cell->task->name());
	    fprintf(fp, "(%s) show\n", buf);

	    fprintf(fp, "%d %d moveto ", px + c0 + pc_textin, py + r1 + pc_textup );
	    fprintf(fp, "(%s) show\n", cell->task->assigned()->id());

	    fprintf(fp, "%d %d moveto ", px + c1 + pc_textin,
		    py + r3 + pc_textup );
	    sprintf(buf, "%10d", cell->task->fullduration());
	    fprintf(fp, "(%s) show\n", buf);
	    
	    fprintf(fp, "%d %d moveto ", px + c0 + pc_textin, py + r3 + pc_textup );
	    fprintf(fp, "(%s) show\n", project->sDays(cell->task->start()));
	    
	    fprintf(fp, "%d %d moveto ", px + c2 + pc_textin, py + r3 + pc_textup );
	    fprintf(fp, "(%s) show\n", project->sDays(cell->task->finish()));

	    fprintf(fp, "%d %d moveto ", px + c0 + pc_textin, py + r0 + pc_textup );
	    fprintf(fp, "(%s) show\n", project->sDays(cell->task->lstart()));
	    
	    fprintf(fp, "%d %d moveto ", px + c1 + pc_textin,
		    py + r0 + pc_textup );
	    sprintf(buf, "%10d", cell->task->slack());
	    fprintf(fp, "(%s) show\n", buf);
	    
	    fprintf(fp, "%d %d moveto ", px + c2 + pc_textin, py + r0 + pc_textup );
	    fprintf(fp, "(%s) show\n", project->sDays(cell->task->lfinish()));
	}
    }

    /* do the finish marker */
    if ( maxX == numX - 1 )
    {
	x = numX - 1;
	px = ax *  x + bx;
	py = ay * mFinishTask->ny + by;
	fprintf(fp, "%d %d %d %d BACKGROUND\n",
		px, px + pc_width, py, py + pc_height);

	fprintf(fp, "%d %d %d %d OUTLINE\n",
		px, px + pc_width, py, py + pc_height);

	fprintf(fp, "%d %d moveto ", px, py + r1 );
	fprintf(fp, "%d %d lineto ", px + pc_width, py + r1 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px, py + r3 );
	fprintf(fp, "%d %d lineto ", px + pc_width, py + r3 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c1, py );
	fprintf(fp, "%d %d lineto ", px + c1, py + r1 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c1, py + r3 );
	fprintf(fp, "%d %d lineto ", px + c1, py + pc_height );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c2, py );
	fprintf(fp, "%d %d lineto ", px + c2, py + r1 );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c2, py + r3 );
	fprintf(fp, "%d %d lineto ", px + c2, py + pc_height );
	fprintf(fp, "stroke\n");
	    
	fprintf(fp, "%d %d moveto ", px + c0 + pc_textin, py + r2 + pc_textup );
	fprintf(fp, "(FINISH) show\n");

	fprintf(fp, "%d %d moveto ", px + c2 + pc_textin, py + r3 + pc_textup );
	fprintf(fp, "(%s) show\n", project->sFinishDay());
    }
    
    /* do the milestones */
    write_milestones(project, fp, ax, bx, ay, by);

    fprintf(fp, "showpage\n");
    fclose(fp);
}


void Reporter::NetworkDiagram(Project *project, 
			      int start, int finish, const char *filename)
{
    TASK *task;

    /* go through assigning x values as max of children + 1 */
    do
    {
	task = NULL;
	for ( Project::TPLCI pt = project->beginTaskList() ; 
	      pt != project->endTaskList() ; 
	      pt++ )
	    if ( canAssignX(*pt) )
	    {
		task = (*pt);
		break;
	    }

	if ( task != NULL )
	    assignX(task);
    }
    while ( task != NULL );


    /* now check that they are all done */
    for ( Project::TPLCI pt = project->beginTaskList() ; 
	  pt != project->endTaskList() ; 
	  pt++ )
    {
	if ( (*pt)->isVacation() )
	    continue;
	if ( (*pt)->nx == -1 )
	    Error("NetworkDiagram: task %s still has nx == -1", 
		  (*pt)->id());
    }

    /* assign y values */
    assignY(project);

    /* construct start and finish markers */
    constructStartFinish(project);

    /* do the chart */
    write_chart(project, start, finish, filename);

//    for ( x = 0 ; x < numX ; x++ )
//	for ( cell = network_cols[x] ; cell != NULL ; cell = cell->next )
//	    printf("%s -> %d,%d\n", cell->task->id, x, cell->task->ny);
}

