
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <rrd.h>

#include <debug/memory.h>
#include <debug/log.h>

#include "mkdirhier.h"
#include "router.h"
#include "bprintf.h"
#include "db.h"

#define NB_OF(x) (sizeof (x) / sizeof ((x)[0]))

/*
 * Create an RRD database for each interface in the list. Return NULL if
 * successful, the error message otherwise.
 */
const char *db_create (const char *path,const router_t *router)
{
   char *end,*filename;
   char *argv[] =
	 {
		"create",
		NULL,
		"DS:ifInOctets:COUNTER:600:U:U",
		"DS:ifOutOctets:COUNTER:600:U:U",
		"DS:ifInDiscards:COUNTER:600:U:U",
		"DS:ifOutDiscards:COUNTER:600:U:U",
		"DS:ifInErrors:COUNTER:600:U:U",
		"DS:ifOutErrors:COUNTER:600:U:U",
		"DS:ifSpeed:GAUGE:600:U:U",
		"DS:ifAdminStatus:GAUGE:600:U:U",
		"DS:ifOperStatus:GAUGE:600:U:U",
		"RRA:AVERAGE:0.5:1:600",
		"RRA:AVERAGE:0.5:6:700",
		"RRA:AVERAGE:0.5:24:775",
		"RRA:AVERAGE:0.5:288:797",
		"RRA:MAX:0.5:1:600",
		"RRA:MAX:0.5:6:700",
		"RRA:MAX:0.5:24:775",
		"RRA:MAX:0.5:288:797",
		"RRA:LAST:0.5:1:600",
		"RRA:LAST:0.5:6:700",
		"RRA:LAST:0.5:24:775",
		"RRA:LAST:0.5:288:797"
	 };
   struct stat sb;
   interface_t *tmp;

   if ((filename = (char *) mem_alloc ((strlen (path) + strlen (router->hostname) + 24) * sizeof (char))) == NULL)
	 {
		rrd_set_error ("out of memory");
		return (rrd_get_error ());
	 }

   argv[1] = filename;

   strcpy (filename,path);
   if (*filename != '\0') strcat (filename,"/");
   strcat (filename,router->hostname);
   if (*filename != '\0') strcat (filename,"/");
   end = filename + strlen (filename);

   mkdirhier (filename);

   for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
	 {
		sprintf (end,"%d.rrd",tmp->idx);

		if (stat (filename,&sb) < 0)
		  {
			 log_printf (LOG_VERBOSE,"creating RRD database %s\n",filename);
			 rrd_clear_error ();
			 optind = opterr = 0;
			 if (rrd_create (NB_OF (argv),argv) < 0)
			   {
				  mem_free (filename);
				  return (rrd_get_error ());
			   }
		  }
	 }

   mem_free (filename);

   return (NULL);
}

/*
 * Update the RRD database associated with each interface in the list.
 * Return NULL if successful, the error message otherwise.
 */
const char *db_update (const char *path,const router_t *router)
{
   char str[256],*filename,*end,*argv[3];
   interface_t *tmp;

   if ((filename = (char *) mem_alloc ((strlen (path) + strlen (router->hostname) + 24) * sizeof (char))) == NULL)
	 {
		rrd_set_error ("out of memory");
		return (rrd_get_error ());
	 }

   strcpy (filename,path);
   if (*filename != '\0') strcat (filename,"/");
   strcat (filename,router->hostname);
   if (*filename != '\0') strcat (filename,"/");
   end = filename + strlen (filename);

   argv[0] = "update";
   argv[1] = filename;
   argv[2] = str;

   for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
	 {
		sprintf (end,"%d.rrd",tmp->idx);

		/* counters & speed gauge */
		if (tmp->valid && tmp->status[IF_OPER_STATUS] == IF_STATUS_UP && tmp->status[IF_ADMIN_STATUS] == IF_STATUS_UP)
		  {
			 sprintf (str,
					  "N:%" PRIu32 ":%" PRIu32 ":%" PRIu32 ":%" PRIu32 ":%" PRIu32 ":%" PRIu32 ":%" PRIu32,
					  tmp->octets[IF_PACKET_IN],
					  tmp->octets[IF_PACKET_OUT],
					  tmp->discards[IF_PACKET_IN],
					  tmp->discards[IF_PACKET_OUT],
					  tmp->errors[IF_PACKET_IN],
					  tmp->errors[IF_PACKET_OUT],
					  tmp->speed);
		  }
		else strcpy (str,"N:U:U:U:U:U:U:U");

		/* interface status */
		if (tmp->valid)
		  {
			 sprintf (str,
					  "%s:%d:%d",
					  str,
					  tmp->status[IF_ADMIN_STATUS],
					  tmp->status[IF_OPER_STATUS]);
		  }
		else strcat (str,":U:U");

		rrd_clear_error ();
		log_printf (LOG_DEBUG,"updating database %s: %s\n",argv[1],argv[2]);
		optind = opterr = 0;
		if (rrd_update (3,argv) < 0)
		  {
			 mem_free (filename);
			 return (rrd_get_error ());
		  }
	 }

   mem_free (filename);

   return (NULL);
}

/*
 * Generate I/O graph from specified RRD database. Return NULL
 * if successful, the error message otherwise.
 */
const char *db_graph_io (int *width,int *height,const char *pngfile,const char *database)
{
   char *a,*b,*c,**print,now[64],title[128],*argv[] =
	 {
		"graph",
		(char *) pngfile,
		"--imgformat", "PNG",
		"--end", now,
		"--vertical-label", "bits per second",
		"--width", "540",
		"--height", "200",
		"--title", title,
		NULL,
		NULL,
		NULL,
		"CDEF:InBits=InBytes,8,*",
		"CDEF:OutBits=OutBytes,8,*",
		"CDEF:Unknown=InBytes,OutBytes,+,UN,INF,0,IF",
		"CDEF:OutOfRange=InBits,UN,0,InBits,IF,Speed,GT,INF,0,IF,OutBits,UN,0,OutBits,IF,Speed,GT,INF,0,IF,+",
		"CDEF:InBitsLimited=InBits,0,Speed,LIMIT",
		"CDEF:OutBitsLimited=OutBits,0,Speed,LIMIT",
		"AREA:InBitsLimited#00cc00:Inbound",
		"GPRINT:InBitsLimited:LAST: Current\\: %8.2lf %sbps",
		"GPRINT:InBitsLimited:AVERAGE:Average\\: %8.2lf %sbps",
		"GPRINT:InBitsLimited:MAX:Maximum\\: %8.2lf %sbps\\n",
		"LINE3:OutBitsLimited#0000ff:Outbound",
		"GPRINT:OutBitsLimited:LAST:Current\\: %8.2lf %sbps",
		"GPRINT:OutBitsLimited:AVERAGE:Average\\: %8.2lf %sbps",
		"GPRINT:OutBitsLimited:MAX:Maximum\\: %8.2lf %sbps\\n",
		"AREA:Unknown#cccccc:Unknown",
		"STACK:OutOfRange#ff0000:Out Of Range\\n"
	 };
   time_t curtime = time (NULL) - 305;
   int i;

   sprintf (now,"%lu",curtime);
   if (strftime (title,sizeof (title),"%a %b %e %T %Z %Y",localtime (&curtime)) < 0)
	 {
		rrd_set_error ("strftime failed");
		return (rrd_get_error ());
	 }

   if ((a = bprintf ("DEF:InBytes=%s:ifInOctets:AVERAGE",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		return (rrd_get_error ());
	 }

   if ((b = bprintf ("DEF:OutBytes=%s:ifOutOctets:AVERAGE",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		mem_free (a);
		return (rrd_get_error ());
	 }

   if ((c = bprintf ("DEF:Speed=%s:ifSpeed:LAST",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		mem_free (a);
		mem_free (b);
		return (rrd_get_error ());
	 }

   argv[14] = a;
   argv[15] = b;
   argv[16] = c;

   log_printf (LOG_NOISY,"rrd_graph\n");
   for (i = 0; i < NB_OF (argv); i++) log_printf (LOG_NOISY,"   argv[%2d]: %s\n",i,argv[i]);

   rrd_clear_error ();
   optind = opterr = 0;
   if (rrd_graph (NB_OF (argv),argv,&print,width,height) < 0)
	 {
		mem_free (a);
		mem_free (b);
		mem_free (c);
		return (rrd_get_error ());
	 }

   if (print)
	 {
		int i;
		for (i = 0; print[i] != NULL; i++) free (print[i]);
		free (print);
	 }

   mem_free (a);
   mem_free (b);
   mem_free (c);

   return (NULL);
}

/*
 * Generate Alerts graph from specified RRD database. Return NULL
 * if successful, the error message otherwise.
 */
const char *db_graph_alerts (int *width,int *height,const char *pngfile,const char *database)
{
   char *a,*b,*c,*d,**print,now[64],title[128],*argv[] =
	 {
		"graph",
		(char *) pngfile,
		"--imgformat", "PNG",
		"--end", now,
		"--vertical-label", "packets",
		"--width", "540",
		"--height", "200",
		"--title", title,
		NULL,
		NULL,
		NULL,
		NULL,
		"CDEF:Unknown=InDiscards,OutDiscards,InErrors,OutErrors,+,+,+,UN,INF,0,IF",
		"AREA:InErrors#777700:Inbound errors",
		"GPRINT:InErrors:LAST:    Current\\: %8.2lf %s",
		"GPRINT:InErrors:AVERAGE:Average\\: %8.2lf %s",
		"GPRINT:InErrors:MAX:Maximum\\: %8.2lf %s\\n",
		"AREA:InDiscards#be5b9d:Inbound discarded",
		"GPRINT:InDiscards:LAST: Current\\: %8.2lf %s",
		"GPRINT:InDiscards:AVERAGE:Average\\: %8.2lf %s",
		"GPRINT:InDiscards:MAX:Maximum\\: %8.2lf %s\\n",
		"LINE3:OutErrors#1e8da0:Outbound errors",
		"GPRINT:OutErrors:LAST:   Current\\: %8.2lf %s",
		"GPRINT:OutErrors:AVERAGE:Average\\: %8.2lf %s",
		"GPRINT:OutErrors:MAX:Maximum\\: %8.2lf %s\\n",
		"LINE3:OutDiscards#000000:Outbound discarded",
		"GPRINT:OutDiscards:LAST:Current\\: %8.2lf %s",
		"GPRINT:OutDiscards:AVERAGE:Average\\: %8.2lf %s",
		"GPRINT:OutDiscards:MAX:Maximum\\: %8.2lf %s\\n",
		"AREA:Unknown#cccccc:Unknown\\n"
	 };
   time_t curtime = time (NULL) - 305;
   int i;

   sprintf (now,"%lu",curtime);
   if (strftime (title,sizeof (title),"%a %b %e %T %Z %Y",localtime (&curtime)) < 0)
	 {
		rrd_set_error ("strftime failed");
		return (rrd_get_error ());
	 }

   if ((a = bprintf ("DEF:InDiscards=%s:ifInDiscards:AVERAGE",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		return (rrd_get_error ());
	 }

   if ((b = bprintf ("DEF:OutDiscards=%s:ifOutDiscards:AVERAGE",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		mem_free (a);
		return (rrd_get_error ());
	 }

   if ((c = bprintf ("DEF:InErrors=%s:ifInErrors:AVERAGE",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		mem_free (a);
		mem_free (b);
		return (rrd_get_error ());
	 }

   if ((d = bprintf ("DEF:OutErrors=%s:ifOutErrors:AVERAGE",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		mem_free (a);
		mem_free (b);
		mem_free (c);
		return (rrd_get_error ());
	 }

   argv[14] = a;
   argv[15] = b;
   argv[16] = c;
   argv[17] = d;

   log_printf (LOG_NOISY,"rrd_graph\n");
   for (i = 0; i < NB_OF (argv); i++) log_printf (LOG_NOISY,"   argv[%2d]: %s\n",i,argv[i]);

   rrd_clear_error ();
   optind = opterr = 0;
   if (rrd_graph (NB_OF (argv),argv,&print,width,height) < 0)
	 {
		mem_free (a);
		mem_free (b);
		mem_free (c);
		mem_free (d);
		return (rrd_get_error ());
	 }

   if (print)
	 {
		int i;
		for (i = 0; print[i] != NULL; i++) free (print[i]);
		free (print);
	 }

   mem_free (a);
   mem_free (b);
   mem_free (c);
   mem_free (d);

   return (NULL);
}

/*
 * Generate Status graph from specified RRD database. Return NULL
 * if successful, the error message otherwise.
 */
const char *db_graph_status (int *width,int *height,const char *pngfile,const char *database)
{
   char *a,*b,**print,now[64],title[128],*argv[] =
	 {
		"graph",
		(char *) pngfile,
		"--imgformat", "PNG",
		"--end", now,
		"--vertical-label", "",
		"--width", "540",
		"--height", "200",
		"--title", title,
		NULL,
		NULL,
		"CDEF:Unknown=AdminStatus,OperStatus,+,UN,INF,0,IF",
		"CDEF:AdminStatusBoolean=AdminStatus,1,EQ",
		"CDEF:OperStatusBoolean=OperStatus,1,EQ",
		"AREA:OperStatusBoolean#005b54:Operational status\\n",
		"LINE3:AdminStatusBoolean#ff00ec:Administrative status\\n",
		"AREA:Unknown#cccccc:Unknown\\n"
	 };
   time_t curtime = time (NULL) - 305;
   int i;

   sprintf (now,"%lu",curtime);
   if (strftime (title,sizeof (title),"%a %b %e %T %Z %Y",localtime (&curtime)) < 0)
	 {
		rrd_set_error ("strftime failed");
		return (rrd_get_error ());
	 }

   if ((a = bprintf ("DEF:AdminStatus=%s:ifAdminStatus:LAST",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		return (rrd_get_error ());
	 }

   if ((b = bprintf ("DEF:OperStatus=%s:ifOperStatus:LAST",database)) == NULL)
	 {
		rrd_set_error ("out of memory");
		mem_free (a);
		return (rrd_get_error ());
	 }

   argv[14] = a;
   argv[15] = b;

   log_printf (LOG_NOISY,"rrd_graph\n");
   for (i = 0; i < NB_OF (argv); i++) log_printf (LOG_NOISY,"   argv[%2d]: %s\n",i,argv[i]);

   rrd_clear_error ();
   optind = opterr = 0;
   if (rrd_graph (NB_OF (argv),argv,&print,width,height) < 0)
	 {
		mem_free (a);
		mem_free (b);
		return (rrd_get_error ());
	 }

   if (print)
	 {
		int i;
		for (i = 0; print[i] != NULL; i++) free (print[i]);
		free (print);
	 }

   mem_free (a);
   mem_free (b);

   return (NULL);
}

