
/******************************************************************************
**
**  Copyright (C) 2006 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    cmp_filter.c
**  Date:    September 17, 2003
**
**  Author:  Brian Wotring
**
**  Purpose: structure representing a comparison filter object
**           used to filter out deltas from the logs.
**
*****************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include <string.h>


osi_bool print_cmp_filter( OSI_CMP_FILTER *filter )
{
    if( ( filter == NULL ) || ( strlen(filter->exclude) == 0 ) )
    {
        return FALSE;
    }

    /* skip the comment. */

    if ( strcmp( filter->exclude, CMP_FILTER_COMMENT ) == 0 )
    {
        return FALSE;
    }

    fprintf( stdout, "%s", filter->exclude );
    return TRUE;
}

unsigned int pack_cmp_filter( OSI_CMP_FILTER *filter, char *buffer,
                                 int buffer_size )
{
    char *temp = buffer;

    if( ( filter == NULL ) || ( buffer == NULL ) || ( buffer_size <= 0 ) )
    {
        return 0;
    }

    memcpy( temp, filter->exclude, sizeof( filter->exclude) );
    temp += sizeof( filter->exclude);

    return ( temp - buffer );
}

void unpack_cmp_filter( OSI_CMP_FILTER *filter, char *buffer,
                        int buffer_size )
{
    char *temp = buffer;

    if( ( filter == NULL ) || ( buffer == NULL ) || ( buffer_size <= 0 ) )
    {
        return;
    }

    memcpy( filter->exclude, temp, sizeof( filter->exclude ) );
}


int write_cmp_filters_to_file( osi_list filters, const char *path )
{
    FILE *file = NULL;
    osi_list head;

    if ( filters == NULL || path == NULL )
    {
        return -1;
    }
    
    file = osi_fopen( path, "w", 0600 );
    
    if ( file == NULL )
    {
        return -1;
    }

    head = list_get_first_node( filters );

    while( head )
    {
        OSI_CMP_FILTER *filter;

        if( ( filter = (OSI_CMP_FILTER *)head->data ) != NULL )
        {
            fwrite( filter->exclude, sizeof(char),
                    strlen( filter->exclude ), file );

            fwrite( "\n", sizeof(char), 1, file );
        }

        head = list_get_next_node( head );
    }

    fclose( file );
    return OSI_OK;
}

osi_list read_cmp_filters_from_file( const char *path )
{
    FILE *file = NULL;
    osi_list filters = NULL;

    char temp_line[MAX_PATH_LENGTH];
    char *line;

    if ( path == NULL )
    {
        return filters;
    }

    file = osi_fopen( path, "r", 0600 );
    
    if ( file == NULL )
    {
        return filters;
    }

    filters = list_new();

    for(;;)
    {
        line = fgets( temp_line, sizeof( temp_line ), file );

        if( line  == NULL )
        {
            break;
        }

        line = trim_white_space( line );

        if( line != NULL )
        {
            OSI_CMP_FILTER *filter = osi_malloc( sizeof(OSI_CMP_FILTER) );
            osi_strlcpy( filter->exclude, line, sizeof( filter->exclude ) );

            list_add( filters, filter );
        }
    }
    
    fclose( file );
    return filters;
}
