/* pam-keyring-tool.c - utility for manipulating keyrings.

   Copyright (C) 2004 W. Michael Petullo

   The Gnome Keyring Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Keyring Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Author: W. Michael Petullo <pam-keyring-tool@flyn.org>
*/

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gnome-keyring.h>

/* ============================ PASSWORD_SIZE =============================== */
const size_t PASSWORD_SIZE = 1024;

char *getpass(const char *);


/* ============================ stdin_getpass () ============================= */
/* SIDE AFFECTS: at most size - 1 characters are read from stream and placed
 *               in stdinpass; terminating carriage return in s is removed
 * OUTPUT: copy of stdinpass
 */

static char *stdin_getpass(void)
{
	char stdinpass[PASSWORD_SIZE];
	size_t len;

        /*
         * if no error is reported from fgets() and string at least contains
         * the newline that ends the password, then replace the newline with
         * a null terminator.
         */
        if ( fgets(stdinpass, PASSWORD_SIZE, stdin) != NULL) {
                if ((len = strlen(stdinpass)) > 0) {
                        if(stdinpass[len-1] == '\n')
                                stdinpass[len - 1] = 0;
                }
        }
        return g_strdup(stdinpass);
}

/* ============================ get_pass () ============================= */
/* INPUT: pointer to a password prompt, and a gboolean for use stdin option 
 * SIDE AFFECT: pass is filled with an authentication token
 * OUTPUT: if error NULL else pointer to pass
 */

static gchar *get_pass(const gchar *prompt, gboolean opt_use_stdin)
{
	char *pass = NULL;
		
        if (opt_use_stdin) {
			pass = stdin_getpass();
        } else {
			pass = getpass(prompt);
        }
        return (pass);
}

/* ============================ pam_keyring_create () =============================== */
/* INPUT: name, the name of the keyring to create
 * SIDE AFFECT: the keyring has been created
 * OUTPUT: GnomeKeyringResult
 */
static GnomeKeyringResult pam_keyring_create(gchar *keyring, gchar *password)
{
	GnomeKeyringResult result;
	
	assert(keyring != NULL);

	if (password == NULL) {
		g_fprintf(stderr, "pam-keyring-tool: error reading password\n");
		result = GNOME_KEYRING_RESULT_BAD_ARGUMENTS;
		goto _return;
	}
	
	result = gnome_keyring_create_sync(keyring, password);

_return:
	return result;
}

/* ============================ pam_keyring_set_default () =========================== */
/* INPUT: name, the name of the keyring to set as the default
 * SIDE AFFECT: the keyring has been set to the default
 * OUTPUT: GnomeKeyringResult
 */
static GnomeKeyringResult pam_keyring_set_default(gchar *keyring)
{
	GnomeKeyringResult result;

	assert(keyring != NULL);

	result = gnome_keyring_set_default_keyring_sync(keyring);

	return result;
}

/* ============================ pam_keyring_get_default () =========================== */
/* SIDE AFFECT: the name of the keyring is set to the default that is returned
 * OUTPUT: GnomeKeyringResult
 */
static GnomeKeyringResult pam_keyring_get_default(gchar **keyring)
{
	char *name;
	GnomeKeyringResult result;

	result = gnome_keyring_get_default_keyring_sync(&name);

	if (result == GNOME_KEYRING_RESULT_OK) {
		*keyring = g_strdup(name);
		g_free(name);
	}

	return result;
}

/* ============================ pam_keyring_unlock () =============================== */
/* INPUT: name, the name of the keyring to unlock, boolean of how to get password
 * SIDE AFFECT: the keyring is unlocked
 * OUTPUT: GnomeKeyringResult
 */
static GnomeKeyringResult pam_keyring_unlock(char *keyring, gboolean opt_use_stdin)
{
	GnomeKeyringResult result;
	char *password = NULL;

	assert(keyring != NULL);

	password = get_pass( "Password:", opt_use_stdin);

	if (password == NULL) {
		g_fprintf(stderr, "pam-keyring-tool: error reading authtok\n");
		result = GNOME_KEYRING_RESULT_IO_ERROR;
		goto _return;
	}
	
	result = gnome_keyring_unlock_sync(keyring, password);
	
	if ( result == GNOME_KEYRING_RESULT_NO_SUCH_KEYRING ) {
		result = pam_keyring_create(keyring, password);
	}

	memset(password, 0x0, sizeof(password));
_return:
	return result;
}
	
	
static gboolean opt_unlock_keyring = FALSE;
static gboolean opt_getdefault_keyring = FALSE;
static gboolean opt_use_stdin = FALSE;
static char *keyring = NULL;

GOptionEntry entries[] = {
	{ "unlock", 'u', 0, G_OPTION_ARG_NONE, &opt_unlock_keyring, "Unlock Keyring", NULL },
	{ "get-default", 'g', 0, G_OPTION_ARG_NONE, &opt_getdefault_keyring, "Get Default Keyring", NULL },
	{ "use-stdin", 's', 0, G_OPTION_ARG_NONE, &opt_use_stdin, "Use stdin for Password Prompt", NULL },
	{ "keyring", 0, 0, G_OPTION_ARG_STRING, &keyring, "Name of Keyring", "name" },
	{ NULL }
};

int main(int argc, char *argv[], char *env[])
{
	gchar *error = NULL;
	GOptionContext *ctx;

	ctx = g_option_context_new("");
	
	g_option_context_add_main_entries(ctx, entries, NULL);
	g_option_context_parse(ctx, &argc, &argv, NULL);
	g_option_context_free(ctx);
	
	if (opt_getdefault_keyring || !keyring) {
		if (pam_keyring_get_default(&keyring) != GNOME_KEYRING_RESULT_OK) {
			error = "error retrieving default keyring";
			goto _error;
		}
		if (opt_getdefault_keyring) {
			g_fprintf(stdout, "keyring: %s\n", keyring);
			goto _exit;
		}
		if ( !keyring) {
			keyring = "default";
			pam_keyring_set_default(keyring);
		}
	}
	
	if (opt_unlock_keyring && ! opt_getdefault_keyring) {
		if (pam_keyring_unlock(keyring, opt_use_stdin) != GNOME_KEYRING_RESULT_OK) {
			error = g_strdup_printf("error unlocking the %s keyring", keyring);
			goto _error;
		}
		goto _exit;
	}
	
_error:

	if (error) { 
		g_fprintf(stderr, "pam-keyring-tool: %s\n", error);
	} else {
		g_fprintf(stderr, "pam-keyring-tool: only one keyring action my be specified on the commandline\n");
	}
	exit(EXIT_FAILURE);
	
_exit:
	exit(EXIT_SUCCESS);
}
