#ifdef ZLIB

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <glib.h>

#include "CompressFilter.h"
#include "Connection.h"
#include "TeloptFilter.h"

// Max. outbuf size for a single decompression
// (output larger than this will be handled in several decompressions; this
// does not limit maximum final outbuf size)
#define BUF_SIZE 16384

/* zlib helpers */

static void *zlib_alloc(void *opaque, unsigned int items, unsigned int size)
{
    return calloc(items, size);
}

static void zlib_free(void *opaque, void *address)
{
    free(address);
}

CompressFilter::CompressFilter(Connection * s) 
    : Filter(s, "Compression")
{
    stream.zalloc = zlib_alloc;
    stream.zfree = zlib_free;
    stream.opaque = NULL;

    if (inflateInit(&stream) != Z_OK)
	abort();
}

CompressFilter::~CompressFilter()
{
    inflateEnd(&stream);
}

bool CompressFilter::process(Buffer &out)
{
    int outsize = BUF_SIZE;
    bool made_progress = true;

    while (input.getLength() && made_progress)
    {
	// Do some decompression.
	Bytef *out_next = (Bytef*)out.prepare_space(outsize);    
	Bytef *in_next = (Bytef*)input.getText();
	
	stream.next_in = in_next;
	stream.avail_in = input.getLength();
	stream.next_out = out_next;
	stream.avail_out = outsize;

	int status = inflate(&stream, Z_PARTIAL_FLUSH);
	
	if (status == Z_OK || status == Z_STREAM_END)
	{
	    // Decompressed ok. Update buffers.
	    made_progress = false;
    
	    if (stream.next_in != in_next)
	    {
		input.strip(stream.next_in - in_next);
		made_progress = true;
	    }
	    if (stream.next_out != out_next)
	    {
		out.consume_space(stream.next_out - out_next);
		made_progress = true;
	    }
	    
	    // Maybe this was the end of compression.
	    if (status == Z_STREAM_END) {
		// Commit suicide
		conn->getSocket()->inputFilters.removeFilter(this);

                // Tell TeloptFilter that compression is off
                TeloptFilter *telnetFilter = static_cast<TeloptFilter*>
                  (conn->getSocket()->inputFilters.findFilter("TelnetOptions"));

                if (telnetFilter->get_local(Telnet::COMPRESS)) {
                    printf( "Turning off COMPRESS\n" );
                    telnetFilter->set_local(Telnet::COMPRESS,
                                            TeloptFilter::OFF);
                    telnetFilter->set_remote(Telnet::COMPRESS,
                                            TeloptFilter::OFF);
                }

                if (telnetFilter->get_local(Telnet::COMPRESS2)) {
                    printf( "Turning off COMPRESS2\n" );
                    telnetFilter->set_local(Telnet::COMPRESS2,
                                            TeloptFilter::OFF);
                    telnetFilter->set_remote(Telnet::COMPRESS2,
                                            TeloptFilter::OFF);
                }

                return false;
	    }
	}
	else if (status != Z_BUF_ERROR) // Z_BUF_ERROR is transient
	{
	    // Some sort of compression error.
	    conn->getSocket()->inputFilters.removeFilter(this);
	    return false;
	}
    }

    return true;
}

#endif
