#include "papaya/system.h"

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <string.h>
#include <stdlib.h>

#include <gdk/gdkkeysyms.h>

#include "mudclient.h"
#include "callbacks.h"
#include "Prefs.h"
#include "support.h"
#include "libusers.h"

#include "GTKTwoWindow.h"
#include "PluginHandler.h"
#include "PreferenceHandler.h"

extern Prefs * globalPreferences;
extern BaseWindow * mainWindow;
extern PluginHandler * phandler;
extern PreferenceHandler * preferenceHandler;

//@@
#include "GTKTwoVT.h"

GTKTwoWindow::GTKTwoWindow() {
  mud_menu = NULL;
  window = NULL;
  createInterface();
}

GTKTwoWindow::~GTKTwoWindow() {
  int width, height;

  width = window->allocation.width;
  height = window->allocation.height;

  globalPreferences->setPreferenceInteger("WindowX", width);
  globalPreferences->setPreferenceInteger("WindowY", height);

  globalPreferences->save((MUD *)NULL);

  //  gtk_widget_hide(window);
  //  gtk_widget_destroy(window);

  window = NULL;
}

extern "C" gint on_gtktwowindow_ok_button_clicked(GtkWidget * button, gpointer data) {
  GladeXML * xml = (GladeXML *)data;
  GtkWidget * widget = glade_xml_get_widget(xml, "quit_dialog");
  gtk_widget_destroy(widget);
  g_object_unref(data);
  gtk_main_quit();
  return 1;
}

extern "C" gint on_gtktwowindow_cancel_button_clicked(GtkWidget * button, gpointer data) {
  GladeXML * xml = (GladeXML *)data;
  GtkWidget * widget = glade_xml_get_widget(xml, "quit_dialog");
  gtk_widget_destroy(widget);
  g_object_unref(data);
  return 1;
}

int papaya_exit() {

  // If the user is connected to a host, ask them to confirm.
  if (mainWindow->getCurrentConnection()) {
    char buf[1024];
    snprintf(buf, 1024, "%s/share/papaya/quit_dialog.glade", getPrefix());
    GladeXML * xml = glade_xml_new(buf , NULL, NULL);
    glade_xml_signal_connect_data(xml, "on_gtktwovt_ok_button_clicked", GTK_SIGNAL_FUNC(on_gtktwowindow_ok_button_clicked), xml);
    glade_xml_signal_connect_data(xml, "on_gtktwovt_cancel_button_clicked", GTK_SIGNAL_FUNC(on_gtktwowindow_cancel_button_clicked), xml);
    return 1;
  }
  
  gtk_main_quit();
  return 1;
}

gchar * gtk_two_window_xlate(const gchar * input, void * data) {
  return (gchar *)_(input);
}

void on_preferences_button_clicked(GtkButton * menu_item, gpointer data, gint i) {
  Connection * conn = mainWindow->getCurrentConnection();
  if (conn)      
    preferenceHandler->createWindow(conn->getMUD());
  else
    preferenceHandler->createWindow(NULL);
}


static GtkItemFactoryEntry menu_items[] = {
  { _("/_File"),            NULL,         0,                     0, "<Branch>" },
  { _("/File/tearoff1"),    NULL,         NULL,                  0, "<Tearoff>" },
  { _("/File/Restart Papaya"), NULL,      (GtkItemFactoryCallback)on_copyover_button_clicked, 0, "<StockItem>", GTK_STOCK_REFRESH },
  { _("/File/sep1"),        NULL,         NULL,                  0, "<Separator>" },
  { _("/File/_Quit"),       "<control>Q", (GtkItemFactoryCallback)papaya_exit,       0, "<StockItem>", GTK_STOCK_QUIT },

  { _("/_Connection"),      NULL,         0, 0, "<Branch>" },
  { _("/Connection/tearoff1"), NULL, NULL, 0, "<Tearoff>" },
  { _("/Connection/Connection Wizard..."), "<control>W", (GtkItemFactoryCallback)on_connection_wizard_button_clicked, 0 },
  { _("/Connection/Connect to..."), "<control>C", (GtkItemFactoryCallback)on_connect_button_clicked, 0 },
  { _("/Connection/Disconnect"), "<control>D", (GtkItemFactoryCallback)on_disconnect_button_clicked, 0 },
  { _("/Connection/Reconnect"), "<control>R", (GtkItemFactoryCallback)on_reconnect_button_clicked, 0},
  { _("/Connection/MUD List..."), "<control>L", (GtkItemFactoryCallback)on_mud_list_clicked, 0},

  { _("/_Settings"), NULL, 0, 0, "<Branch>" },
  { _("/Settings/tearoff1"), NULL, NULL, 0, "<Tearoff>" },
  { _("/Settings/_Preferences..."), NULL, (GtkItemFactoryCallback)on_preferences_button_clicked, 0, "<StockItem>", GTK_STOCK_PREFERENCES },
  { _("/Settings/Aliases and Triggers..."), NULL, (GtkItemFactoryCallback)on_entities_button_clicked, 0 },

  { _("/_Plugins"), NULL, 0, 0, "<Branch>" },
  { _("/Plugins/Installed Plugins..."), NULL, (GtkItemFactoryCallback)on_help_plugins_clicked, 0 },

  { _("/_Help"), NULL, 0, 0, "<LastBranch>" },
  { _("/Help/tearoff1"), NULL, NULL, 0, "<Tearoff>" },
  { _("/Help/User Guide"), NULL, (GtkItemFactoryCallback)on_help_userguide_clicked, 0 },
  { _("/Help/_About"), NULL, (GtkItemFactoryCallback)on_about_clicked, 0 },
};

void GTKTwoWindow::createInterface() {

  // Create the main application window
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  g_object_set_data(G_OBJECT(window), "window", window);
  gtk_window_set_resizable(GTK_WINDOW(window), true);

  setTitleBar("Not Connected");
 
  /* NULL window variable when window is closed */
  //  g_signal_connect (G_OBJECT (window), "destroy",
  //                    G_CALLBACK (gtk_widget_destroyed),
  //                    &window);

  // Create a table to stick everything into
  GtkWidget * table = gtk_table_new(1, 4, FALSE);
  gtk_container_add(GTK_CONTAINER(window), table);
  //  g_object_set_data_full(G_OBJECT(window), "window", table,
  //                         (GDestroyNotify) g_object_unref);
  gtk_widget_show(table);
  
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

  accel_group = gtk_accel_group_new ();
  item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", 
                                       accel_group);

  gtk_item_factory_set_translate_func(item_factory, gtk_two_window_xlate, NULL, NULL);

  /* This function generates the menu items. Pass the item factory,
     the number of items in the array, the array itself, and any
     callback data for the the menu items. */
  gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);

#ifndef WIN32
  /* Attach the new accelerator group to the window. */
  gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);
#endif

  gtk_table_attach (GTK_TABLE (table),
                    gtk_item_factory_get_widget (item_factory, "<main>"),
                    0, 1,
                    0, 1,
                    (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions)0,
                    0, 0);

  gtk_widget_show(gtk_item_factory_get_widget(item_factory, "<main>"));

  // Create a toolbar
  toolbar = gtk_toolbar_new ();
  gtk_toolbar_set_icon_size (GTK_TOOLBAR (toolbar),
                             GTK_ICON_SIZE_SMALL_TOOLBAR);

  if (globalPreferences->getPreferenceBoolean("UseToolbar"))
    gtk_widget_show(toolbar);
  //  g_object_set_data_full(G_OBJECT(window), "toolbar", toolbar,
  //                         (GDestroyNotify) g_object_unref);

  gtk_toolbar_insert_stock (GTK_TOOLBAR (toolbar),
                            GTK_STOCK_PREFERENCES,
                            _("Edit Preferences"),
                            NULL,
                            G_CALLBACK (on_preferences_button_clicked),
                            window, /* user data for callback */
                            -1);  /* -1 means "append" */

  gtk_table_attach (GTK_TABLE (table),
                    toolbar,
                    0, 1,
                    1, 2,
                    (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions)(GTK_FILL),
                    0, 0);

  splash_update(_("Loading Papaya logo"));

  background_pixmap = create_pixmap (window, "papaya-logo-newbie.xpm");
  if (background_pixmap) {

    gtk_widget_ref (background_pixmap);
  //  g_object_set_data_full (G_OBJECT (window), "background_pixmap", background_pixmap,
  //			  (GDestroyNotify) g_object_unref);
    gtk_widget_show (background_pixmap);
  }


  // attach to the table.
  gtk_table_attach(GTK_TABLE(table),
                   background_pixmap,
                   0, 1,
                   2, 3,
                   (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
                   (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
                   0, 0);

  notebook = gtk_notebook_new ();
  //  g_object_set_data_full (G_OBJECT (window), "notebook", notebook,
  //			  (GDestroyNotify) g_object_unref);
  GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(notebook), GTK_CAN_FOCUS);
  
  // attach to the table
  gtk_table_attach(GTK_TABLE(table),
                   notebook,
                   0, 1,
                   2, 3,
                   (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
                   (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
                   0, 0);

  // Connect the notebook signals
  g_signal_connect_data(G_OBJECT(notebook), "switch-page",
			GTK_SIGNAL_FUNC(on_notebook_page_switch),
			NULL, NULL, (GConnectFlags)0);

  // Connect the window events
  g_signal_connect_data(G_OBJECT(window), "key_press_event",
			GTK_SIGNAL_FUNC(on_window_keypress),
			NULL, NULL, (GConnectFlags)0);
  g_signal_connect_data(G_OBJECT(window), "delete_event",
			GTK_SIGNAL_FUNC(gtk_main_quit),
			NULL, NULL, (GConnectFlags)0);

  gtk_window_set_default_size(GTK_WINDOW(window), globalPreferences->getPreferenceInteger("WindowX"), globalPreferences->getPreferenceInteger("WindowY"));

}


GtkItemFactory * GTKTwoWindow::getItemFactory() {

	return item_factory;

}


void GTKTwoWindow::display() {
  gtk_widget_show(window);
}

void GTKTwoWindow::setToolbarVisible(bool t) {
  if (t)
    gtk_widget_show(toolbar);
  else
    gtk_widget_hide(toolbar);
}

/**
 * Add a connection to the main interface.
 */

void GTKTwoWindow::addConnection(Connection * c) {
  int n = 1;
  char name[1024];


  // Figure out if this connection needs a different name...
  gtk_widget_hide(background_pixmap);
  gtk_widget_show(notebook);

  sprintf(name, "%s", c->getName());

  while (findConnection(name)) {
    sprintf(name, "%s (%d)", c->getName(), n++);
  }

  c->setName(name);

  // Show the box containing the VT and plugin tray.
  gtk_widget_show(c->getVT()->getMainWindow());
  GtkWidget * label = gtk_label_new(c->getName());
  gtk_widget_show(label);
 
  // @@ Stick this page into the notebook -- this causes the font to be "lost"
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), c->getVT()->getMainWindow(), label);

  // Insert this page into the list of connections.
  c->setNext(first_connection);
  first_connection = c;

  // Let us bring this to the front of the notebook.
  int page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), c->getVT()->getMainWindow());
  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
  setTitleBar(c->getMUD()->getName());

  // @@ NASTY HACK - reload preferences here, as gtk_notebook_append_page mangles
  // @@ the Terminal's fonts on some GTK/VTE combinations.  (2.2.1/0.10.25 for sure)
  c->getVT()->prefsUpdated();
}

/**
 * Remove a connection from the main interface.
 */

void GTKTwoWindow::removeConnection(Connection * c) {
  // Find which page this connection's VT is situated on, and delete it.
  gint page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), c->getVT()->getMainWindow());

  if (page == -1)
    return; // Already been removed.

  gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);

  if (numPages() == 0) {
    gtk_widget_hide(notebook);
    gtk_widget_show(background_pixmap);
  }

  if (first_connection == c) {
    first_connection = c->getNext();
    return;
  }
  
  for (Connection * tmp = first_connection; tmp; tmp = tmp->getNext()) {
    if (tmp->getNext() == c) {
      tmp->setNext(c->getNext());
      c->setNext((Connection *)NULL);
      return;
    }
  }

}

/**
 * Returns the Connection object associated with the currently open page.
 */

Connection * GTKTwoWindow::getCurrentConnection() {

  gint page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));

  if (page == -1)
    return NULL;

  GtkWidget * label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page));


  const gchar * text;
  text = gtk_label_get_text(GTK_LABEL(label));

  // Search through all the connection objects until we find a host with
  // this name.

  if (!strcmp(text, "mudFTP"))
    return NULL;

  for (Connection * tmp = first_connection; tmp; tmp = tmp->getNext()) {
    if (!strcmp(tmp->getName(), text)) {
      return tmp;
    }
  }

  return (Connection *)NULL;
}

/**
 * Highlights a the tab red if new text appears and this is not the current
 * connection.
 */

void GTKTwoWindow::nextText(Connection * c) {

  // Figure out which page this connection belongs to...
  int page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), c->getVT()->getMainWindow());
  int curr_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
  
  if (page != curr_page) {
    
    GtkWidget * label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page));

    GtkStyle * style = gtk_style_copy(label->style);
    GdkColor red = { 0, 0xffff, 0x0000, 0x0000 };
    gdk_color_alloc(gdk_colormap_get_system(), &red);
    style->fg[0] = red;
    
    gtk_widget_set_style(label, style);
    g_object_unref(G_OBJECT(style));
  }
}


/**
 * Called when the user changes notebook pages to set the focus.
 */

void GTKTwoWindow::pageSwitch(int p) {
  GtkWidget * label;

  // Figure out which page we're on...
  
  label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), p));
  gtk_widget_set_style(GTK_WIDGET(label), NULL);

  // Set focus.
  for (Connection * tmp = first_connection; tmp; tmp = tmp->getNext()) {
    // Find which page this connection's VT is situated on, and grab focus.
    guint page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), tmp->getVT()->getMainWindow());
    if (page == p) {
      tmp->getVT()->focus();
      setTitleBar(tmp->getMUD()->getName());
    }
  }
  
  phandler->pageSwitched();

}

int GTKTwoWindow::numPages() {
  int i = 0;

  while (1) {
    if (!gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), i)) {
      return i;
    }
    i++;
  }
  return i;
}

void GTKTwoWindow::switchPage(int page) {

  if (!gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page))
    return;

  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);

  for (Connection * tmp = first_connection; tmp; tmp = tmp->getNext()) {
    // Find which page this connection's VT is situated on, and delete it.
    gint p = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), tmp->getVT()->getMainWindow());
    if (page == p) {
      tmp->getVT()->focus();
      setTitleBar(tmp->getMUD()->getName());
    }
  }

}

void menu_add_menu(GtkWidget *, GtkWidget **, char *, bool);

void GTKTwoWindow::nextPage() {
  gint page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
  
  if (page < 0)
    return;

  page++;

  if (page >= numPages())
    page = 0;

  gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);

  for (Connection * tmp = first_connection; tmp; tmp = tmp->getNext()) {
    // Grab focus in the input box.
    gint p = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), tmp->getVT()->getMainWindow());
    if (page == p) {
      tmp->getVT()->focus();
      setTitleBar(tmp->getMUD()->getName());
    }
  }
}

int get_x_id();

void GTKTwoWindow::swallow() {
#if 0

#ifndef WIN32
  int id = get_x_id();
  int parent_id = -1;
  if (id == -1)
    return;

  // Create a socket widget and make it visible.
  GtkSocket * socket = (GtkSocket *)gtk_socket_new();
  gtk_widget_show (GTK_WIDGET(socket));

  // Check if we're trying to steal papaya.
  socket->plug_window = gdk_window_lookup(id);
  /*
  if (socket->plug_window && socket->plug_window->user_data) {
    new Message("Error", _("Papaya cannot swallow itself."), NULL);
    gtk_widget_hide(GTK_WIDGET(socket));
    gtk_widget_destroy(GTK_WIDGET(socket));
    return;
  }
  */

  GtkWidget * label = gtk_label_new(_("Swallowed Program"));
  gtk_widget_show(label);

  // Stick it in the notebook.
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(socket), label);

  // Figure out the ID of the child

  // This is the ID of the parent window
  parent_id = GDK_WINDOW_XWINDOW (GTK_WIDGET(socket)->window);

  // Note that this does NOT work for enlightenment, but will work for
  // fvwm and sawmill.  It seems to be an enlightenment bug.

  gtk_socket_add_id(GTK_SOCKET(socket), id);
#endif
#endif
}

void GTKTwoWindow::closeWindow() {

  // Figure which is the active notebook page, and what its title is.
  gint page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
  GtkWidget * label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page));
  

  const gchar * text;
  text = gtk_label_get_text(GTK_LABEL(label));

  if (!strcmp(text, _("Swallowed Program")) || !strcmp(text, "mudFTP")) {
    // Remove this page.
    gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);
    return;
  }

  // Deleting this object should be enough to remove it neatly...

  Connection * conn = getCurrentConnection();
  if (!conn || conn->isClosed())
    return;
  delete getCurrentConnection();
}

GtkWidget * GTKTwoWindow::addPageForApp(char * name) {
#ifndef WIN32

	GtkWidget *socket = gtk_socket_new();
  gtk_widget_show (socket);
  GtkWidget * label = gtk_label_new(name);
  gtk_widget_show(label);

  gtk_widget_hide(background_pixmap);
  gtk_widget_show(notebook);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), socket, label);
  /* The following call is only necessary if one of
   * the ancestors of the socket is not yet visible.
   */
  gtk_widget_realize (socket);
  return socket;
#else

  return NULL;

#endif

}

void GTKTwoWindow::addPage(char * name, GtkWidget * frame) {

  GtkWidget * label = gtk_label_new(name);
  gtk_widget_show(label);

  gtk_widget_hide(background_pixmap);
  gtk_widget_show(notebook);

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);
}

void GTKTwoWindow::addCustomMenu(char * name, GtkItemFactoryCallback cb) {
//  mudmenu_add_menu(name, cb);
}

void GTKTwoWindow::getDimensions(int * x, int * y) {
  // These values don't seem to include the window decorations on WIN32
  *x = window->allocation.width;
  *y = window->allocation.height;
}

void GTKTwoWindow::setTitleBar(char * text) {
  char buf[1024];

  snprintf(buf, 1024, _("%s - Papaya"), text);
  gtk_window_set_title (GTK_WINDOW (window), buf);
}

void GTKTwoWindow::setIcon(char * text) {
}
