#include "LoginWizard.h"

// All plugins must include this, even on UNIX platforms where .sos are sane.
#include "Win32PluginAPI.cpp"

static LoginWizardPlugin * loginWizard;

// Configure the minor and major versions of this plugin.
#define MAJOR "1"
#define MINOR "0"

extern "C" G_MODULE_EXPORT void plugin_init(plugin_address_table_t * pat) {
  plugin_address_table_init(pat);
  loginWizard = new LoginWizardPlugin();
}

extern "C" G_MODULE_EXPORT void plugin_cleanup(void) {
  delete loginWizard;
}

extern "C" G_MODULE_EXPORT char * plugin_query_name() {
  return "LoginWizard";
}

extern "C" G_MODULE_EXPORT char * plugin_query_description() {
  return "";
}

extern "C" G_MODULE_EXPORT char * plugin_query_major() {
  return MAJOR;
}

extern "C" G_MODULE_EXPORT char * plugin_query_minor() {
  return MINOR;
}

extern "C" G_MODULE_EXPORT gint on_login_wizard_ok_button_clicked(GtkWidget * button, gpointer data) {
  loginWizard->okClicked(data);
}

extern "C" G_MODULE_EXPORT gint on_login_wizard_cancel_button_clicked(GtkWidget * button, gpointer data) {
  loginWizard->cancelClicked(data);
}

static char * escape(char * input) {
  static char output[16384];
  char * output_ptr = output;
  while (*input != '\0') {
    switch (*input) {
    case '(':
    case ')':
    case '[':
    case ']':
    case '?':
    case '.':
      *output_ptr++ = '\\';
      
      break;
    }

    *output_ptr++ = *input++;
  }

  *output_ptr = '\0';

  return output;
}

LoginWizardPlugin::LoginWizardPlugin() {
  version = 1.0;
  name = strdup("LoginWizardPlugin");

  // This MUST be done after all variable initialisation as it can cause
  // onEvent(Event *, Connection *) to be called.
  register_plugin(this, VERSION);
  plugin_handler_add_input_filter(get_plugin_handler(), this);
  plugin_handler_add_output_filter(get_plugin_handler(), this);
  plugin_handler_add_prompt_filter(get_plugin_handler(), this);
}

LoginWizardPlugin::~LoginWizardPlugin() {

  unregister_plugin(this);

  free(name);

  // @@ Nuke any items in the loginDataList.
}

// Uncomment this if you want to parse input from the user
void G_MODULE_EXPORT LoginWizardPlugin::input(Connection * c, char * input) {

  if (strlen(input) <= 2)
    return;

  struct loginData * data = find_data(c);
  if (!data)
    return;

  if (!data->username) {
    data->username = strdup(input);
    return;
  }

  if (!data->password) {
    data->password = strdup(input);


    printf ("Username string: %s\n"
	    "Username: %s\n"
	    "Password string: %s\n"
	    "Password: %s\n",
	    data->usernameString, data->username, data->passwordString, data->password);

    char buf[1024];
    snprintf(buf, 1024, "%s/share/papaya/loginwizard.glade", get_prefix());
    data->xml = glade_xml_new(buf, NULL, NULL);

    glade_xml_signal_connect_data(data->xml, "on_login_wizard_ok_button_clicked", GTK_SIGNAL_FUNC(on_login_wizard_ok_button_clicked), c);
    glade_xml_signal_connect_data(data->xml, "on_login_wizard_cancel_button_clicked", GTK_SIGNAL_FUNC(on_login_wizard_cancel_button_clicked), c);


    GtkWidget * password_entry = glade_xml_get_widget(data->xml, "login_password_entry");
    GtkWidget * password_trigger_entry = glade_xml_get_widget(data->xml, "login_password_trigger_entry");
    GtkWidget * username_entry = glade_xml_get_widget(data->xml, "login_username_entry");
    GtkWidget * username_trigger_entry = glade_xml_get_widget(data->xml, "login_username_trigger_entry");
    
    gtk_entry_set_text(GTK_ENTRY(username_entry), data->username);
    gtk_entry_set_text(GTK_ENTRY(username_trigger_entry), data->usernameString);
    gtk_entry_set_text(GTK_ENTRY(password_entry), data->password);
    gtk_entry_set_text(GTK_ENTRY(password_trigger_entry), data->passwordString);
    
    return;
  }
}

// Uncomment this if you want to parse output from the MUD
void G_MODULE_EXPORT LoginWizardPlugin::output(Connection * c, char * input) {
  struct loginData * data = find_data(c);
  if (!data)
    return;

  if (!data->username && strlen(input) > 2) {
    free(data->usernameString);
    data->usernameString = strdup(input);
    return;
  }

  if (!data->password && strlen(input) > 2) {
    free(data->passwordString);
    data->passwordString = strdup(input);
    return;
  }

}

// Uncomment this if you want to parse incoming prompts
void G_MODULE_EXPORT LoginWizardPlugin::prompt(Connection * c, char * input) {
  struct loginData * data = find_data(c);
  if (!data)
    return;

  if (!data->username && strlen(input) > 2) {
    free(data->usernameString);
    data->usernameString = strdup(input);
    return;
  }

  if (!data->password && strlen(input) > 2) {
    free(data->passwordString);
    data->passwordString = strdup(input);
    return;
  }

}

// Uncomment this if you want to respond to "EvConnect" and "EvDisconnect"
// events.
void G_MODULE_EXPORT LoginWizardPlugin::onEvent(Event * event, Connection * c) {
  
  if (event_get_type(event) == EvConnect) {
    
    MUD * mud = connection_get_mud(c);

    if (mud_get_login_name(mud) && strlen(mud_get_login_name(mud)) > 0)
      return;

    if (mud_get_password(mud) && strlen(mud_get_password(mud)) > 0)
      return;

    Prefs * p = connection_query_preferences(c);
    char * pref = preferences_get_preference(p, "loginwizard");
    if (!strcmp(pref, "1"))
      return;

    struct loginData * data = find_data(c);
    if (!data) {
      data = (struct loginData *)malloc(sizeof(struct loginData));
      memset(data, 0, sizeof(struct loginData));
      data->connection = c;

      // Add it to the list.
      add_data(data);
    }

    if (data->usernameString)
      free(data->usernameString);
    if (data->passwordString)
      free(data->passwordString);
    if (data->username)
      free(data->username);
    if (data->password)
      free(data->password);
    memset(data, 0, sizeof(struct loginData));

    return;
  }

  if (event_get_type(event) == EvDisconnect) {
    struct loginData * data = find_data(c);
    if (data) {
      if (data->usernameString)
	free(data->usernameString);
      if (data->passwordString)
	free(data->passwordString);
      if (data->username)
	free(data->username);
      if (data->password)
	free(data->password);
      if (data->xml) {
	g_object_unref(data->xml);
	data->xml = NULL;
      }

      remove_data(data);
    }
    return;
  }
  
}

void LoginWizardPlugin::okClicked(gpointer d) {

  Connection * c = (Connection *)d;

  struct loginData * data = find_data(c);
  if (!data || !data->xml)
    return;

  GtkWidget * password_entry = glade_xml_get_widget(data->xml, "login_password_entry");
  GtkWidget * password_trigger_entry = glade_xml_get_widget(data->xml, "login_password_trigger_entry");
  GtkWidget * username_entry = glade_xml_get_widget(data->xml, "login_username_entry");
  GtkWidget * username_trigger_entry = glade_xml_get_widget(data->xml, "login_username_trigger_entry");

  const gchar * username, * username_trigger;
  const gchar * password, * password_trigger;

  username = gtk_entry_get_text(GTK_ENTRY(username_entry));
  username_trigger = gtk_entry_get_text(GTK_ENTRY(username_trigger_entry));
  password = gtk_entry_get_text(GTK_ENTRY(password_entry));
  password_trigger = gtk_entry_get_text(GTK_ENTRY(password_trigger_entry));

  MUD * mud = connection_get_mud(c);
  mud_set_password(mud, (gchar *)password);
  mud_set_login_name(mud, escape((gchar *)username));
  mud_set_login_trigger(mud, escape((gchar *)username_trigger));
  mud_set_password_trigger(mud, (gchar *)password_trigger);

  gtk_widget_hide(glade_xml_get_widget(data->xml, "login_wizard_window"));
  g_object_unref(data->xml);
  data->xml = NULL;
  if (data->usernameString)
    free(data->usernameString);
  if (data->passwordString)
    free(data->passwordString);
  if (data->username)
    free(data->username);
  if (data->password)
    free(data->password);

  remove_data(data);
  return;
}

void LoginWizardPlugin::cancelClicked(gpointer d) {

  Connection * c = (Connection *)d;
  struct loginData * data = find_data(c);
  if (!data || !data->xml)
    return;

  GtkWidget * checkbox = glade_xml_get_widget(data->xml, "login_never_checkbox");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox))) {
    Prefs * p = connection_query_preferences(c);
    preferences_set_preference(p, "loginwizard", "1");
  }

  gtk_widget_hide(glade_xml_get_widget(data->xml, "login_wizard_window"));
  g_object_unref(data->xml);
  data->xml = NULL;
  if (data->usernameString)
    free(data->usernameString);
  if (data->passwordString)
    free(data->passwordString);
  if (data->username)
    free(data->username);
  if (data->password)
    free(data->password);

  remove_data(data);
}

static int LoginCmp(struct loginData * l1, struct loginData *l2) {
  return (l1 < l2);
}

void LoginWizardPlugin::remove_data(struct loginData * data) {
  LoginDataList::iterator i = std::lower_bound(loginList.begin(),
                                               loginList.end(),
                                               data,
                                               LoginCmp);
  if (i == loginList.end() || (*i) != data)
    return;

  loginList.erase(i);
}

struct loginData * LoginWizardPlugin::find_data(Connection * connection) {
  for (LoginDataList::iterator i = loginList.begin(); i != loginList.end(); i++)
    if ((*i)->connection == connection)
      return (*i);
  return NULL;
}

void LoginWizardPlugin::add_data(struct loginData * data) {
    LoginDataList::iterator i = std::lower_bound(loginList.begin(),
                                                 loginList.end(),
                                                 data,
                                                 LoginCmp);
    
    loginList.insert(i, data);
}
