
/*  Pcsx - Pc Psx Emulator
 *  Copyright (C) 1999-2002  Pcsx Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <dlfcn.h>
#include <sys/stat.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <signal.h>
#include <sys/time.h>

#include "Linux.h"
#include "plugins.h"
#include "Sio.h"

enum {
	RESET_REQD_NO,
	RESET_REQD
}; /* Needs reset */

enum {
	PSX_TYPE_NTSC,
	PSX_TYPE_PAL
}; /* PSX Type */

extern int UseGui;
long LoadCdBios;

PSEgetLibType		PSE_getLibType = NULL;
PSEgetLibVersion	PSE_getLibVersion = NULL;
PSEgetLibName		PSE_getLibName = NULL;

// Helper Functions
void UpdatePluginsBIOS();
void UpdatePluginsBIOS_UpdateGUI(GladeXML *xml);
void ScanPlugins(gchar*);
void FindNetPlugin(GladeXML *xml);

void OnNet_Conf(GtkWidget *widget, gpointer user_data);
void OnNet_Test(GtkWidget *widget, gpointer user_data);
void OnNet_About(GtkWidget *widget, gpointer user_data);

// Functions Callbacks
/*make them getting the right params, will be used later*/
void OnFile_RunExe();
void OnFile_RunCd();
void OnFile_RunCdBios();
void OnEmu_Run();
void OnEmu_Reset();
void OnConf_Gpu();
void OnConf_Spu();
void OnConf_Cdr();
void OnConf_Pads();
void OnConf_Mcds();
void OnConf_Cpu();
void OnConf_Conf();
void OnConf_Net();
void OnHelp_Help();
void OnHelp_About();
void OnDestroy();
void OnFile_Exit();

void OnStates_Load1();
void OnStates_Load2();
void OnStates_Load3();
void OnStates_Load4();
void OnStates_Load5();
void OnStates_LoadOther();
void OnStates_Save1();
void OnStates_Save2();
void OnStates_Save3();
void OnStates_Save4();
void OnStates_Save5();
void OnStates_SaveOther();

int CheckPlugins();

GtkWidget *Window = NULL;
GtkWidget *ConfDlg;
GtkWidget *AboutDlg;
GtkWidget *FileSel;

GtkAccelGroup *AccelGroup;

GtkWidget *controlwidget;

#define DEFAULT_MEM_CARD_1 "/.pcsx/memcards/card1.mcr"
#define DEFAULT_MEM_CARD_2 "/.pcsx/memcards/card2.mcr"
#define PLUGINS_DIR "/.pcsx/plugins/"
#define PLUGINS_CFG_DIR "/.pcsx/plugins/cfg/"
#define BIOS_DIR "/.pcsx/bios/"

#define FindComboText(combo,list,conf) \
	if (strlen(conf) > 0) { \
		int i; \
		for (i=2;i<255;i+=2) { \
			if (!strcmp(conf, list[i-2])) { \
				gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i/2-1); \
				break; \
			} \
		} \
	}

#define GetComboText(combo,list,conf) \
	{ \
		int row; \
		row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo)); \
		strcpy(conf, (char *) list[row*2]); \
	}

int Slots[5] = { -1, -1, -1, -1, -1 };

void ResetMenuSlots(GladeXML *xml) {
	GtkWidget *Item;
	
	char str[256];
	int i;

	for (i = 0; i < 5; i++) {
		sprintf(str, "GtkMenuItem_LoadSlot%d", i+1);
		Item = glade_xml_get_widget (xml, str);
		if (Slots[i] == -1) 
			gtk_widget_set_sensitive(Item, FALSE);
		else
			gtk_widget_set_sensitive(Item, TRUE);
	}
}

void UpdateMenuSlots() {
	gchar *str;
    char SStateFile[64];
	int i;

	for (i = 0; i < 5; i++) {
        // Ryan: no giant home dirs, kthx

		str = g_strdup (getenv("HOME"));
		
		sprintf(SStateFile, "/.pcsx/sstates/%10.10s.%3.3d", CdromLabel, i);
        str = g_strconcat (str, SStateFile, NULL);
        /* str will now be ~/.pcsx/sstates/identifier */
		Slots[i] = CheckState(str);
	}
	
	g_free (str);
}

void StartGui() {
	GladeXML *xml;
	GtkWidget *widget;
	
	struct stat buf;
	char dotdir[MAXPATHLEN];

	if (Window != NULL) {
		gtk_window_present (GTK_WINDOW (Window));
		return;
	}

	xml = glade_xml_new(PACKAGE_DATA_DIR "pcsx.glade2", "MainWindow", NULL);

	if (!xml)
	{
		g_warning("We could not load the interface!");
		return;
	}

	Window = glade_xml_get_widget(xml, "MainWindow");
	gtk_window_set_title(GTK_WINDOW(Window), "PCSX");
	gtk_window_set_icon_from_file(GTK_WINDOW(Window), PIXMAPDIR "pcsx-icon.png", NULL);
	gtk_window_set_default_icon_from_file(PIXMAPDIR "pcsx-icon.png", NULL);
	ResetMenuSlots(xml);
	
	/* Set up callbacks */
	g_signal_connect_data(GTK_OBJECT(Window), "delete-event",
			GTK_SIGNAL_FUNC(OnDestroy), xml, (GClosureNotify)g_object_unref, G_CONNECT_AFTER);

	/* File menu */	
	widget = glade_xml_get_widget(xml, "RunCd");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnFile_RunCd), NULL, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "RunCdBios");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnFile_RunCdBios), NULL, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "RunExe");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnFile_RunExe), NULL, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "exit2");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnFile_Exit), NULL, NULL, G_CONNECT_AFTER);
	
	/* States */
	widget = glade_xml_get_widget(xml, "GtkMenuItem_LoadSlot1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Load1), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_LoadSlot2");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Load2), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_LoadSlot3");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Load3), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_LoadSlot4");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Load4), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_LoadSlot5");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Load5), NULL, NULL, G_CONNECT_AFTER);	
	widget = glade_xml_get_widget(xml, "other1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_LoadOther), NULL, NULL, G_CONNECT_AFTER);			

	widget = glade_xml_get_widget(xml, "GtkMenuItem_SaveSlot1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Save1), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_SaveSlot2");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Save2), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_SaveSlot3");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Save3), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_SaveSlot4");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Save4), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "GtkMenuItem_SaveSlot5");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_Save5), NULL, NULL, G_CONNECT_AFTER);	
	widget = glade_xml_get_widget(xml, "other2");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnStates_SaveOther), NULL, NULL, G_CONNECT_AFTER);

	/* Emulation menu */
	widget = glade_xml_get_widget(xml, "run1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnEmu_Run), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "reset1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnEmu_Reset), NULL, NULL, G_CONNECT_AFTER);
	
	/* Configuration menu */
	widget = glade_xml_get_widget(xml, "plugins_bios");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Conf), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "graphics1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Gpu), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "sound1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Spu), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "cd-rom1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Cdr), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "controllers1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Pads), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "netplay1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Net), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "cpu1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Cpu), NULL, NULL, G_CONNECT_AFTER);
	widget = glade_xml_get_widget(xml, "memory_cards1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnConf_Mcds), NULL, NULL, G_CONNECT_AFTER);

	/* Help menu */
	widget = glade_xml_get_widget(xml, "about_pcsx1");
	g_signal_connect_data(GTK_OBJECT(widget), "activate",
			GTK_SIGNAL_FUNC(OnHelp_About), NULL, NULL, G_CONNECT_AFTER);

	gtk_main();
}

void RunGui() {
	StartGui();
}

int destroy=0;

void OnDestroy() {
	if (!destroy) OnFile_Exit();
}

void ConfigurePlugins() {
	if (!UseGui) {
		/* Should log to console */
		/* How do we get here if we're not running the GUI? */
		return;
	}
	
	OnConf_Conf();
}

void ConfigureMemcards() {
	OnConf_Mcds();
}

void OnFile_RunExe() {
	if (CheckPlugins() == 1) { 
		ConfigurePlugins();
		return;
	}
	
    FileSel = gtk_file_chooser_dialog_new (_("Select PSX EXE File"),
			NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
            NULL);

	/* ADB Set this to the config object and retain it - maybe LastUsedDir */
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (FileSel), getenv("HOME"));

    if (gtk_dialog_run (GTK_DIALOG (FileSel)) == GTK_RESPONSE_ACCEPT) {
        gchar *file;
        //char exe[MAXPATHLEN];

        /* TODO Need to validate the file */

        file = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileSel));
        //strcpy(exe, File);
        gtk_widget_destroy(FileSel);
        destroy=1;
        gtk_widget_destroy(Window);
        destroy=0;
        gtk_main_quit();
        while (gtk_events_pending()) gtk_main_iteration();
        if (OpenPlugins() == -1) { SysRunGui(); return; }
        SysReset();
        needreset = RESET_REQD_NO;
        Load(file);
        psxCpu->Execute();
		g_free (file);
    }
    else
        gtk_widget_destroy(FileSel);
}

void OnFile_RunCd() {
	if (CheckPlugins() == 1) { 
		ConfigurePlugins();
		return;
	}
	// this stops a segfault when a plugin file changes after being initially loaded
	ReleasePlugins();
	LoadPlugins();
	LoadCdBios = 0;
	destroy=1;
	gtk_widget_destroy(Window);
	destroy=0;
	gtk_main_quit();
	while (gtk_events_pending()) gtk_main_iteration();
	if (OpenPlugins() == -1) { SysRunGui(); return; }
	SysReset();
	needreset = RESET_REQD_NO;
	CheckCdrom();
	if (LoadCdrom() == -1) {
		ClosePlugins();
		SysMessage(_("Could not load CD-ROM!\n"));
		return;
	}
	LoadCdrom();
	psxCpu->Execute();
}

void OnFile_RunCdBios() {
	if (CheckPlugins() == 1) { 
		ConfigurePlugins();
		return;
	}
	// this stops a segfault when a plugin file changes after being initially loaded
	ReleasePlugins();
	LoadPlugins();
	LoadCdBios = 1;
	destroy=1;
	gtk_widget_destroy(Window);
	destroy=0;
	gtk_main_quit();
	while (gtk_events_pending()) gtk_main_iteration();
	if (OpenPlugins() == -1) { SysRunGui(); return; }
	SysReset();
	needreset = 0;
	CheckCdrom();
	psxCpu->Execute();
}

void OnFile_Exit() {
	DIR *dir;
	struct dirent *ent;
	void *Handle;
	char plugin[MAXPATHLEN];
	gchar *dotdir;

	dotdir = g_strdup (getenv("HOME"));
	dotdir = g_strconcat (dotdir, PLUGINS_DIR, NULL);
	
	// with this the problem with plugins that are linked with the pthread
	// library is solved

	dir = opendir(dotdir);
	if (dir != NULL) {
		while ((ent = readdir(dir)) != NULL) {
			sprintf (plugin, "%s%s", dotdir, ent->d_name);

			if (strstr(plugin, ".so") == NULL)
				continue;
			Handle = dlopen(plugin, RTLD_NOW);
			if (Handle == NULL)
				continue;
		}
	}
	g_free (dotdir);

	bind_textdomain_codeset(PACKAGE_NAME, "");
	if (UseGui)
		gtk_main_quit();
	SysClose();
	if (UseGui)
		gtk_exit (0);
	else
		exit(0);
}

void States_Load(int num) {
	gchar *Text;
    char SStateFile[64];
	int ret;

	destroy=1;
	gtk_widget_destroy(Window);
	destroy=0;
	gtk_main_quit();
	while (gtk_events_pending()) gtk_main_iteration();
	if (OpenPlugins() == -1) { SysRunGui(); return; }
	SysReset();
	needreset = 0;

	Text = g_strdup (getenv("HOME"));
	Text = g_strconcat (Text, "/.pcsx/", NULL);
	
	sprintf(SStateFile, "sstates/%10.10s.%3.3d", CdromLabel, num);
    g_strconcat(Text, SStateFile, NULL);

	ret = LoadState(Text);
	if (ret == 0)
		 sprintf(Text, _("Loaded state %d."), num+1);
	else sprintf(Text, _("Error loading state %d!"), num+1);
	
	GPU_displayText(Text);

	psxCpu->Execute();
	
	g_free (Text);
}

void States_Save(int num) {
	gchar *Text;
    char SStateFile[64];
	int ret;

	destroy=1;
	gtk_widget_destroy(Window);
	destroy=0;
	gtk_main_quit();
	while (gtk_events_pending()) gtk_main_iteration();
	if (OpenPlugins() == -1) { SysRunGui(); return; }
	if (needreset) {
		SysReset();
		needreset = 0;
	}
	GPU_updateLace();

	Text = g_strdup (getenv("HOME"));
	Text = g_strconcat (Text, "/.pcsx/", NULL);
	
	sprintf(SStateFile, "sstates/%10.10s.%3.3d", CdromLabel, num);
	Text = g_strconcat (Text, SStateFile, NULL);
	
	GPU_freeze(2, (GPUFreeze_t *)&num);
	ret = SaveState(Text);
	if (ret == 0)
		 sprintf(Text, _("Saved state %d."), num+1);
	else sprintf(Text, _("Error saving state %d!"), num+1);
	
	GPU_displayText(Text);

	psxCpu->Execute();
	
	g_free (Text);
}

void OnStates_Load1() { States_Load(0); } 
void OnStates_Load2() { States_Load(1); } 
void OnStates_Load3() { States_Load(2); } 
void OnStates_Load4() { States_Load(3); } 
void OnStates_Load5() { States_Load(4); } 

void OnStates_LoadOther() {
    gchar *SStateFile;

	SStateFile = g_strdup (getenv("HOME"));
	SStateFile = g_strconcat (SStateFile, "/.pcsx/sstates/", NULL);

    FileSel = gtk_file_chooser_dialog_new(_("Select State File"), NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
            NULL);
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (FileSel), SStateFile);

    if (gtk_dialog_run (GTK_DIALOG (FileSel)) == GTK_RESPONSE_ACCEPT) {
        //const gchar *File;
		gchar *filename;
        //char str[MAXPATHLEN];
        char Text[MAXPATHLEN+20];
        int ret;

        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (FileSel));

        gtk_widget_destroy(FileSel);
        destroy=1;
        gtk_widget_destroy(Window);
        destroy=0;
        gtk_main_quit();
		while (gtk_events_pending()) gtk_main_iteration();
		if (OpenPlugins() == -1) { SysRunGui(); return; }
		SysReset();
		needreset = 0;

		ret = LoadState(filename);
		if (ret == 0)
			 sprintf(Text, _("Loaded state: %s"), filename);
		else sprintf(Text, _("Error loading state: %s"), filename);
		GPU_displayText(Text);

		psxCpu->Execute();
		
		g_free (filename);
	}
		else gtk_widget_destroy (FileSel);
	
		g_free (SStateFile);
} 

void OnStates_Save1() { States_Save(0); } 
void OnStates_Save2() { States_Save(1); } 
void OnStates_Save3() { States_Save(2); } 
void OnStates_Save4() { States_Save(3); } 
void OnStates_Save5() { States_Save(4); } 

void OnStates_SaveOther() {
	gchar *SStateFile;

	SStateFile = g_strdup (getenv("HOME"));
	SStateFile = g_strconcat (SStateFile, "/.pcsx/sstates/", NULL);

    FileSel = gtk_file_chooser_dialog_new(_("Select State File"),
			NULL, GTK_FILE_CHOOSER_ACTION_SAVE,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_SAVE, GTK_RESPONSE_OK,
            NULL);
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (FileSel), SStateFile);

    if (gtk_dialog_run (GTK_DIALOG (FileSel)) == GTK_RESPONSE_OK) {
        gchar *filename;
        char Text[MAXPATHLEN+20];

        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (FileSel));
        gtk_widget_destroy(FileSel);
        destroy=1;
        gtk_widget_destroy(Window);
        destroy=0;
        gtk_main_quit();
        while (gtk_events_pending()) gtk_main_iteration();
        if (OpenPlugins() == -1) { SysRunGui(); return; }
        if (needreset) {
            SysReset();
            needreset = RESET_REQD_NO;
        }
        GPU_updateLace();

        if (SaveState(filename) == 0)
            sprintf(Text, _("Saved state %s."), filename);
        else
            sprintf(Text, _("Error saving state %s!"), filename);
        GPU_displayText(Text);

        psxCpu->Execute();
		
		g_free (filename);
    }
    else
        gtk_widget_destroy (FileSel);

	g_free (SStateFile);
} 

void OnEmu_Run() {
	if (CheckPlugins() == 1) { 
		ConfigurePlugins();
		return;
	}
	// this stops a segfault when a plugin file changes after being initially loaded
	ReleasePlugins();
	LoadPlugins();
	destroy=1;
	gtk_widget_destroy(Window);
	destroy=0;
	gtk_main_quit();
	while (gtk_events_pending())
		gtk_main_iteration();
	if (OpenPlugins() == -1) {
		SysRunGui();
		return;
	}
	if (needreset) {
		SysReset();
		needreset = FALSE;
	}
	psxCpu->Execute();
}

void OnEmu_Reset() {
	needreset = 1;
}

void OnConf_Gpu() {
	gtk_widget_set_sensitive(Window, FALSE);
	while (gtk_events_pending()) gtk_main_iteration();
	if ((strlen(Config.Gpu) != 0) && (strlen(Config.Spu) != 0) && (strlen(Config.Cdr) != 0) && (strlen(Config.Pad1) != 0) && (strlen(Config.Pad2) != 0))  { GPU_configure(); } else { ConfigurePlugins(); }
	gtk_widget_set_sensitive(Window, TRUE);
}

void OnConf_Spu() {
	gtk_widget_set_sensitive(Window, FALSE);
	while (gtk_events_pending()) gtk_main_iteration();
	if ((strlen(Config.Gpu) != 0) && (strlen(Config.Spu) != 0) && (strlen(Config.Cdr) != 0) && (strlen(Config.Pad1) != 0) && (strlen(Config.Pad2) != 0)) { SPU_configure(); } else { ConfigurePlugins(); }
	gtk_widget_set_sensitive(Window, TRUE);
}

void OnConf_Cdr() {
	gtk_widget_set_sensitive(Window, FALSE);
	while (gtk_events_pending()) gtk_main_iteration();
	if ((strlen(Config.Gpu) != 0) && (strlen(Config.Spu) != 0) && (strlen(Config.Cdr) != 0) && (strlen(Config.Pad1) != 0) && (strlen(Config.Pad2) != 0)) { CDR_configure(); } else { ConfigurePlugins(); }
	gtk_widget_set_sensitive(Window, TRUE);
}

void OnConf_Pads() {
	gtk_widget_set_sensitive(Window, FALSE);
	while (gtk_events_pending()) gtk_main_iteration();
	if ((strlen(Config.Gpu) != 0) && (strlen(Config.Spu) != 0) && (strlen(Config.Cdr) != 0) && (strlen(Config.Pad1) != 0) && (strlen(Config.Pad2) != 0)) {
		PAD1_configure();
		if (strcmp(Config.Pad1, Config.Pad2)) PAD2_configure();
	} 
	else { 
		ConfigurePlugins();
	}
	gtk_widget_set_sensitive(Window, TRUE);
}

GtkWidget *NetDlg;

void OnNet_Clicked (GtkDialog *dialog, gint arg1, gpointer user_data)
{
	if (arg1 == GTK_RESPONSE_OK)
	{
		GetComboText(NetConfS.Combo, NetConfS.plist, Config.Net);

		SaveConfig();
	
		/* ADB Commented out - we should release/load the plugin
		   when we select them, not when we open or close the config screen
		   Either way, net plugin is currently disabled, so this is irrelevant
		ReleasePlugins();
		LoadPlugins();
		needreset = 1;*/
	}

	gtk_widget_destroy(GTK_WIDGET (dialog));
	NetDlg = NULL;
	gtk_widget_set_sensitive(Window, TRUE);
}

void OnConf_Net() {
	GladeXML *xml;
	GtkWidget *widget;

	if (NetDlg != NULL) {
		gtk_window_present (GTK_WINDOW (NetDlg));
		return;
	}

	xml = glade_xml_new(PACKAGE_DATA_DIR "pcsx.glade2", "NetDlg", NULL);

	if (!xml)
	{
		g_warning(_("Error: Glade interface could not be loaded!"));
		return;
	}

	NetDlg = glade_xml_get_widget(xml, "NetDlg");

	FindNetPlugin(xml);

	/* Setup a handler for when Close or Cancel is clicked */
	g_signal_connect_data(GTK_OBJECT(NetDlg), "response",
			GTK_SIGNAL_FUNC(OnNet_Clicked), xml, (GClosureNotify)g_object_unref, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_ConfNet");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnNet_Conf), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_ConfAbout");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnNet_About), xml, NULL, G_CONNECT_AFTER);
}

GtkWidget *McdDlg;
GtkWidget *Entry1,*Entry2;
GtkWidget *List1,*List2;
GtkWidget *BtnPaste;
GTimer *Gtimer;
GtkTreeSelection *selection1;
GtkTreeSelection *selection2;
int timer;
McdBlock Blocks[2][15];
int IconC[2][15];
enum {
    CL_ICON,
    CL_TITLE,
    CL_STAT,
    CL_ID,
    CL_NAME,
    NUM_CL
};

static void
add_columns (GtkTreeView *treeview)
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
//  g_print ("painting table");
  /* column for icon */
  renderer = gtk_cell_renderer_pixbuf_new ();

  column = gtk_tree_view_column_new_with_attributes (_("Icon"),
						     renderer,
						     "pixbuf", CL_ICON,
						     NULL);

   gtk_tree_view_append_column (treeview, column);

  /* column for title */
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Title"),
						     renderer,
						     "text",
						     CL_TITLE,
						     NULL);
  gtk_tree_view_append_column (treeview, column);

  /* column for status */
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Status"),
						     renderer,
						     "text",
						     CL_STAT,
						     NULL);
  gtk_tree_view_append_column (treeview, column);

  /* column for id */
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("ID"),
						     renderer,
						     "text",
						     CL_ID,
						     NULL);
  gtk_tree_view_append_column (treeview, column);

  /* column for Name */
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Name"),
						     renderer,
						     "text",
						     CL_NAME,
						     NULL);
  gtk_tree_view_append_column (treeview, column);
							    
}



GdkPixbuf *SetIcon(short *icon, int i) {
	GdkPixmap *pixmap;
	GdkImage  *image;
	GdkVisual *visual;
	GdkGC     *gc;
	int x, y, c;

	visual = gdk_window_get_visual(McdDlg->window);

	if (visual->depth == 8) return NULL;

	image = gdk_image_new(GDK_IMAGE_NORMAL, visual, 16, 16);

	for (y=0; y<16; y++) {
		for (x=0; x<16; x++) {
			c = icon[y*16+x];
			c = ((c&0x001f) << 10) | ((c&0x7c00) >> 10) | (c&0x03e0);
			if (visual->depth == 16)
				c = (c&0x001f) | ((c&0x7c00) << 1) | ((c&0x03e0) << 1);
			else if (visual->depth == 24 || visual->depth == 32)
				c = ((c&0x001f) << 3) | ((c&0x03e0) << 6) | ((c&0x7c00) << 9);
				
			gdk_image_put_pixel(image, x, y, c);
		}
	}

	pixmap = gdk_pixmap_new(McdDlg->window, 16, 16, visual->depth);

	gc = gdk_gc_new(pixmap);
	gdk_draw_image(pixmap, gc, image, 0, 0, 0, 0, 16, 16);
	gdk_gc_destroy(gc);
	gdk_image_destroy(image);
	
	return gdk_pixbuf_get_from_drawable    (NULL,
	                                        GDK_PIXMAP (pixmap),
		                                NULL,
						0,0,0,0,-1,-1);
//	gtk_clist_set_pixmap(GTK_CLIST(List), i-1, 0, pixmap, NULL);
}

void LoadListItems(int mcd, GtkWidget *List) {
	int i;
/*
	gtk_clist_clear(GTK_CLIST(List));
*/
	GtkListStore *store= gtk_list_store_new (NUM_CL,GDK_TYPE_PIXBUF, G_TYPE_STRING,
					     G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING);
	GtkTreeIter iter;
	GdkPixbuf *pixbuf;

	for (i=1; i<16; i++) {
		McdBlock *Info;
		gchar *text[5];

		Info = &Blocks[mcd-1][i-1];
		IconC[mcd-1][i-1] = 0;

		if ((Info->Flags & 0xF0) == 0xA0) {
			if ((Info->Flags & 0xF) >= 1 &&
				(Info->Flags & 0xF) <= 3) {
				text[2] = _("Deleted");
			} else text[2] = _("Free");
		} else if ((Info->Flags & 0xF0) == 0x50)
			text[2] = _("Used");
		else { text[2] = _("Free"); }

		text[0] = "";
		text[1] = Info->Title;
		text[3] = Info->ID;
		text[4] = Info->Name;
/*
		gtk_clist_insert(GTK_CLIST(List), i-1, text);
*/
//		if (Info->IconCount == 0) continue;
		pixbuf = SetIcon(Info->Icon, i);
		
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
				CL_ICON,pixbuf,
				CL_TITLE,Info->Title,
				CL_STAT,text[2],
				CL_NAME,Info->Name,
				CL_ID,Info->ID,
				-1);
	}

	gtk_tree_view_set_model (GTK_TREE_VIEW (List), GTK_TREE_MODEL (store));
	g_object_unref (G_OBJECT (store));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (List), TRUE);
	gtk_widget_show(List);
}


void UpdateListItems(int mcd, GtkWidget *List) {
	LoadListItems(mcd, List);
}


/*
{
	int i,j;

	for (i=1; i<16; i++) {
		McdBlock *Info;
		gchar *text[5];

		Info = &Blocks[mcd-1][i-1];
		IconC[mcd-1][i-1] = 0;

		if ((Info->Flags & 0xF0) == 0xA0) {
			if ((Info->Flags & 0xF) >= 1 &&
				(Info->Flags & 0xF) <= 3) {
				text[2] = "Deleted";
			} else text[2] = "Free";
		} else if ((Info->Flags & 0xF0) == 0x50)
			text[2] = "Used";
		else { text[2] = "Free"; }

		text[0] = "";
		text[1] = Info->Title;
		text[3] = Info->ID;
		text[4] = Info->Name;

		for (j=0; j<5; j++)
			gtk_clist_set_text(GTK_CLIST(List), i-1, j, text[j]);

		if (Info->IconCount == 0) continue;

		SetIcon(Info->Icon, i);
	}
}
*/
void LoadMcdDlg() {
	int i;

	for (i=1; i<16; i++) GetMcdBlockInfo(1, i, &Blocks[0][i-1]);
	for (i=1; i<16; i++) GetMcdBlockInfo(2, i, &Blocks[1][i-1]);
	LoadListItems(1, List1);
	LoadListItems(2, List2);
}

void UpdateMcdDlg() {
	int i;

	for (i=1; i<16; i++) GetMcdBlockInfo(1, i, &Blocks[0][i-1]);
	for (i=1; i<16; i++) GetMcdBlockInfo(2, i, &Blocks[1][i-1]);
	UpdateListItems(1, List1);
	UpdateListItems(2, List2);
}

void StopTimer() {
	g_timer_stop(Gtimer); timer = 0;
}

void OnMcd_Clicked (GtkDialog *dialog, gint arg1, gpointer user_data)
{
	StopTimer();

	if (arg1 == GTK_RESPONSE_CLOSE)
	{
		const char *tmp;
		GtkWidget *Btn, *widget;
	
		tmp = gtk_entry_get_text(GTK_ENTRY(Entry1));
		strcpy(Config.Mcd1, tmp);
		tmp = gtk_entry_get_text(GTK_ENTRY(Entry2));
		strcpy(Config.Mcd2, tmp);
	
		SaveConfig();
	} 

	LoadMcds(Config.Mcd1, Config.Mcd2);

	gtk_widget_destroy(McdDlg);
	McdDlg = NULL;
}

void OnMcd_FS(GtkWidget *widget, gpointer user_data) {
	gint memcard = (int *) user_data;
	
	FileSel = gtk_file_chooser_dialog_new (_("Select Memory Card"),
			NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);

	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (FileSel), getenv("HOME"));
	
	if (gtk_dialog_run (GTK_DIALOG (FileSel)) == GTK_RESPONSE_ACCEPT) {
		const gchar *File;

		File = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileSel));
		LoadMcd(memcard, (char*)File);
		UpdateMcdDlg();
	}
	gtk_widget_destroy(FileSel);
}
/*
void OnMcd_FS1(GtkWidget *widget, gpointer user_data) {
	FileSel = gtk_file_chooser_dialog_new (_("Select Memory Card 1"),
			NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);

	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (FileSel), getenv("HOME"));
	
	if (gtk_dialog_run (GTK_DIALOG (FileSel)) == GTK_RESPONSE_ACCEPT) {
		const gchar *File;

		File = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileSel));
		LoadMcd(1, (char*)File);
		UpdateMcdDlg();
	}
	gtk_widget_destroy(FileSel);
}

void OnMcd_FS2(GtkWidget *widget, gpointer user_data) {
	FileSel = gtk_file_chooser_dialog_new (_("Select Memory Card 2"),
			NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);

	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (FileSel), getenv("HOME"));
	
	if (gtk_dialog_run (GTK_DIALOG (FileSel)) == GTK_RESPONSE_ACCEPT) {
		const gchar *File;

		File = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileSel));
		LoadMcd(2, (char*)File);
		UpdateMcdDlg();
	}
	gtk_widget_destroy(FileSel);
}
*/

void OnMcd_Format(GtkWidget *widget, gpointer user_data) {
	const char *str;
	GtkWidget *message_dialog;
	gint memcard = (int *) user_data;
	gint result;
	
	message_dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL,
		GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE,
		_("Format this Memory Card?"),
		NULL);
	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message_dialog),
		_("If you format the memory card, the contents will be overwritten."),
		NULL);
	gtk_dialog_add_buttons (GTK_DIALOG (message_dialog),
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		_("Format card"), GTK_RESPONSE_YES,
		NULL);
	
	result = gtk_dialog_run (GTK_DIALOG (message_dialog));
	gtk_widget_destroy (message_dialog);
	if (result == GTK_RESPONSE_YES) {
		if (memcard == 1)
			str = gtk_entry_get_text(GTK_ENTRY(Entry1));
		else
			str = gtk_entry_get_text(GTK_ENTRY(Entry2));
		CreateMcd((char*)str);
		LoadMcd(memcard, (char*)str);
		UpdateMcdDlg();
	}
}

void OnMcd_Reload1(GtkWidget *widget, gpointer user_data) {
	const char *str;

	str = gtk_entry_get_text(GTK_ENTRY(Entry1));
	LoadMcd(1, (char*)str);
	UpdateMcdDlg();
}

void OnMcd_Reload2(GtkWidget *widget, gpointer user_data) {
	const char *str;

	str = gtk_entry_get_text(GTK_ENTRY(Entry2));
	LoadMcd(2, (char*)str);
	UpdateMcdDlg();
}

static int copy = 0, copymcd = 0;

void OnMcd_CopyTo1(GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
	gint *i;

	/* If the item selected is not reported as a 'Free' slot */
	if (gtk_tree_selection_get_selected (selection1, &model, &iter)) {
		path = gtk_tree_model_get_path (model, &iter);
		i = gtk_tree_path_get_indices (path);
//		g_print("index1 %d\n",*i);
		copy    = *i;
		copymcd = 1;
		gtk_tree_path_free (path);
		gtk_widget_set_sensitive(BtnPaste, TRUE);
	}
}

void OnMcd_CopyTo2(GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
	gint *i;

	/* If the item selected is not reported as a 'Free' slot */
	if (gtk_tree_selection_get_selected (selection1, &model, &iter)) {
		path = gtk_tree_model_get_path (model, &iter);
		i = gtk_tree_path_get_indices (path);
//		g_print("index2 %d\n",*i);
		copy    = *i;
		copymcd = 2;
		gtk_tree_path_free (path);
		gtk_widget_set_sensitive(BtnPaste, TRUE);
	}
}

void OnMcd_Paste(GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
	gint *i;
	const char *str;

	GtkWidget *message_dialog;
	gint result;
	
	message_dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL,
		GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE,
		_("Paste this selection?"),
		NULL);
	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message_dialog),
		_("Are you sure you want to paste this selection?"),
		NULL);
	gtk_dialog_add_buttons (GTK_DIALOG (message_dialog),
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		_("Paste"), GTK_RESPONSE_YES,
		NULL);
	
	result = gtk_dialog_run (GTK_DIALOG (message_dialog));
	gtk_widget_destroy (message_dialog);
	if (result != GTK_RESPONSE_YES)
		return;
	
	if (copymcd == 1) {
		str = gtk_entry_get_text(GTK_ENTRY(Entry1));
		if (gtk_tree_selection_get_selected (selection1, &model, &iter)) {
			path = gtk_tree_model_get_path (model, &iter);
			i = gtk_tree_path_get_indices (path);

			//i = GTK_CLIST(List1)->focus_row;

			// save dir data + save data
			memcpy(Mcd1Data + (*i+1) * 128, Mcd2Data + (copy+1) * 128, 128);
			SaveMcd((char*)str, Mcd1Data, (*i+1) * 128, 128);
			memcpy(Mcd1Data + (*i+1) * 1024 * 8, Mcd2Data + (copy+1) * 1024 * 8, 1024 * 8);
			SaveMcd((char*)str, Mcd1Data, (*i+1) * 1024 * 8, 1024 * 8);
		} else return;
	} else { // 2
		str = gtk_entry_get_text(GTK_ENTRY(Entry2));
		if (gtk_tree_selection_get_selected (selection2, &model, &iter)) {
			path = gtk_tree_model_get_path (model, &iter);
			i = gtk_tree_path_get_indices (path);

			//i = GTK_CLIST(List2)->focus_row;

			// save dir data + save data
			memcpy(Mcd2Data + (*i+1) * 128, Mcd1Data + (copy+1) * 128, 128);
			SaveMcd((char*)str, Mcd2Data, (*i+1) * 128, 128);
			memcpy(Mcd2Data + (*i+1) * 1024 * 8, Mcd1Data + (copy+1) * 1024 * 8, 1024 * 8);
			SaveMcd((char*)str, Mcd2Data, (*i+1) * 1024 * 8, 1024 * 8);
		} else return;
	}
	UpdateMcdDlg();
}

void OnMcd_Delete1(GtkWidget *widget, gpointer user_data) {
	McdBlock *Info;
	int mcd = 1;
	int i, xor = 0, j;
	unsigned char *data, *ptr;
	const char *str;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;

	if (gtk_tree_selection_get_selected (selection1, &model, &iter)) {
		path = gtk_tree_model_get_path (model, &iter);
		i = *gtk_tree_path_get_indices (path);

		str = gtk_entry_get_text(GTK_ENTRY(Entry1));
		data = Mcd1Data;

		i++;

		ptr = data + i * 128;

		Info = &Blocks[mcd-1][i-1];

		if ((Info->Flags & 0xF0) == 0xA0) {
			if ((Info->Flags & 0xF) >= 1 &&
				(Info->Flags & 0xF) <= 3) { // deleted
				*ptr = 0x50 | (Info->Flags & 0xF);
			} else return;
		} else if ((Info->Flags & 0xF0) == 0x50) { // used
				*ptr = 0xA0 | (Info->Flags & 0xF);
		} else { return; }

		for (j=0; j<127; j++) xor^=*ptr++;
		*ptr = xor;

		SaveMcd((char*)str, data, i * 128, 128);
		UpdateMcdDlg();
	}
}

void OnMcd_Delete2(GtkWidget *widget, gpointer user_data) {
	McdBlock *Info;
	int mcd = 2;
	int i, xor = 0, j;
	unsigned char *data, *ptr;
	const char *str;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;

	if (gtk_tree_selection_get_selected (selection2, &model, &iter)) {
		path = gtk_tree_model_get_path (model, &iter);
		i = *gtk_tree_path_get_indices (path);

		str = gtk_entry_get_text(GTK_ENTRY(Entry2));
//		i = GTK_CLIST(List2)->focus_row;
		data = Mcd2Data;

		i++;

		ptr = data + i * 128;

		Info = &Blocks[mcd-1][i-1];

		if ((Info->Flags & 0xF0) == 0xA0) {
			if ((Info->Flags & 0xF) >= 1 &&
				(Info->Flags & 0xF) <= 3) { // deleted
				*ptr = 0x50 | (Info->Flags & 0xF);
			} else return;
		} else if ((Info->Flags & 0xF0) == 0x50) { // used
				*ptr = 0xA0 | (Info->Flags & 0xF);
		} else { return; }

		for (j=0; j<127; j++) xor^=*ptr++;
		*ptr = xor;

		SaveMcd((char*)str, data, i * 128, 128);
		UpdateMcdDlg();
	}
}

void UpdateMcdIcon(int mcd, GtkWidget *List) {
#if 0
	int i;
	GtkListStore *store= gtk_list_store_new (NUM_CL,GDK_TYPE_PIXBUF, G_TYPE_STRING,
					     G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING);
	GtkTreeIter iter;
	GdkPixbuf *pixbuf;

	gtk_tree_clear_items(GTK_TREE_VIEW (List));

	for (i=1; i<16; i++) {
		McdBlock *Info;
		int *count;

		Info = &Blocks[mcd-1][i-1];
		count = &IconC[mcd-1][i-1];

		if (Info->IconCount <= 1) continue;

		(*count)++;
		if (*count == Info->IconCount) *count = 0;

	{
		gchar *text[5];

		Info = &Blocks[mcd-1][i-1];
		IconC[mcd-1][i-1] = 0;

		if ((Info->Flags & 0xF0) == 0xA0) {
			if ((Info->Flags & 0xF) >= 1 &&
				(Info->Flags & 0xF) <= 3) {
				text[2] = _("Deleted");
			} else text[2] = _("Free");
		} else if ((Info->Flags & 0xF0) == 0x50)
			text[2] = _("Used");
		else { text[2] = _("Free"); }

		text[0] = "";
		text[1] = Info->Title;
		text[3] = Info->ID;
		text[4] = Info->Name;

//		pixbuf=SetIcon(Info->Icon, i);
		pixbuf=SetIcon(&Info->Icon[*count*16*16], i);
		
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
				CL_ICON,pixbuf,
				CL_TITLE,Info->Title,
				CL_STAT,text[2],
				CL_NAME,Info->Name,
				CL_ID,Info->ID,
				-1);
	}

//		SetIcon(&Info->Icon[*count*16*16], i);
	}
/*	gtk_tree_view_set_model (GTK_TREE_VIEW (List), GTK_TREE_MODEL (store));
	g_object_unref (G_OBJECT (store));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (List), TRUE);*/
//	gtk_widget_show(List);

#endif
}

void OnConf_Mcds() {

	GladeXML *xml;
	GtkWidget *widget;
	gchar *str;

	if (McdDlg != NULL) {
		gtk_window_present (GTK_WINDOW (McdDlg));
		return;
	}

	xml = glade_xml_new(PACKAGE_DATA_DIR "pcsx.glade2", "McdsDlg", NULL);

	if (!xml)
	{
		g_warning("We could not load the interface!");
		return;
	}

	McdDlg = glade_xml_get_widget(xml, "McdsDlg");

	gtk_window_set_title(GTK_WINDOW(McdDlg), _("Memory Card Manager"));
	
	Entry1 = glade_xml_get_widget(xml, "GtkEntry_Mcd1");
	if (!strlen(Config.Mcd1)) {
		str = g_strdup (getenv("HOME"));
		str = g_strconcat (str, DEFAULT_MEM_CARD_1, NULL);
		strcpy(Config.Mcd1, str);
    }
	gtk_entry_set_text(GTK_ENTRY(Entry1), Config.Mcd1);

	Entry2 = glade_xml_get_widget(xml, "GtkEntry_Mcd2");
	if (!strlen(Config.Mcd2)) {
		str = g_strdup (getenv("HOME"));
		str = g_strconcat (str, DEFAULT_MEM_CARD_2, NULL);
		strcpy(Config.Mcd2, str);
    }
	gtk_entry_set_text(GTK_ENTRY(Entry2), Config.Mcd2);

	BtnPaste = glade_xml_get_widget(xml, "GtkButton_McdPaste");
	gtk_widget_set_sensitive(BtnPaste, FALSE);

	List1 = glade_xml_get_widget(xml, "GtkCList_McdList1");
	List2 = glade_xml_get_widget(xml, "GtkCList_McdList2");
	add_columns(GTK_TREE_VIEW (List1));
	add_columns(GTK_TREE_VIEW (List2));

	/* memory vs cpu */
	selection1 = gtk_tree_view_get_selection (GTK_TREE_VIEW (List1));
	gtk_tree_selection_set_mode (selection1, GTK_SELECTION_SINGLE);
	
	selection2 = gtk_tree_view_get_selection (GTK_TREE_VIEW (List2));
	gtk_tree_selection_set_mode (selection2, GTK_SELECTION_SINGLE);
	
	LoadMcdDlg();

	/* Setup a handler for when Close or Cancel is clicked */
	g_signal_connect_data(GTK_OBJECT(McdDlg), "response",
			GTK_SIGNAL_FUNC(OnMcd_Clicked), xml, (GClosureNotify)g_object_unref, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_Format1");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_Format), (gpointer) 1, NULL, G_CONNECT_AFTER);
			//GTK_SIGNAL_FUNC(OnMcd_Format1), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_Format2");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_Format), (gpointer) 1, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_SelMcd1");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_FS), (gpointer) 1, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_SelMcd2");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_FS), (gpointer) 2, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_Reload1");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_Reload1), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_Reload2");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_Reload2), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_CopyTo1");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_CopyTo1), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_CopyTo2");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_CopyTo2), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_Delete1");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_Delete1), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_Delete2");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_Delete2), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkButton_McdPaste");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnMcd_Paste), xml, NULL, G_CONNECT_AFTER);

	Gtimer = g_timer_new(); timer = 1;

	while (gtk_events_pending()) gtk_main_iteration();

	while (timer) {
		unsigned long usecs;

		g_timer_elapsed(Gtimer, &usecs);
		if (usecs > 250000) {
			UpdateMcdIcon(1, List1);
			UpdateMcdIcon(2, List2);
			g_timer_reset(Gtimer);
		}

		while (gtk_events_pending()) gtk_main_iteration();
		usleep(10000);
	}
}

GtkWidget *CpuDlg;
GtkWidget *PsxCombo;
GList *psxglist;
char *psxtypes[] = {
	"NTSC",
	"PAL"
};

/* When the auto-detect CPU type is selected, disable the NTSC/PAL selection */
void OnCpu_PsxAutoClicked (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *combo;
	GladeXML *xml = user_data;
	combo = glade_xml_get_widget(xml, "GtkCombo_PsxType");

	gtk_widget_set_sensitive (combo,
			!(gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget))));
}

void OnCpu_Clicked (GtkDialog *dialog, gint arg1, gpointer user_data)
{
	if (arg1 == GTK_RESPONSE_CLOSE)
	{
		GtkWidget *Btn, *widget;
		GladeXML *xml = user_data;
		int tmp;
		long t;

		widget = glade_xml_get_widget(xml, "GtkCombo_PsxType");
	
		tmp = gtk_combo_box_get_active (GTK_COMBO_BOX (widget));
		if (tmp == -1)	
			tmp = PSX_TYPE_NTSC;
		
		if (!strcmp("NTSC",psxtypes[tmp])) Config.PsxType = PSX_TYPE_NTSC;
			else Config.PsxType = PSX_TYPE_PAL;

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_Xa");
		Config.Xa = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_Sio");
		Config.Sio = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_Mdec");
		Config.Mdec = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_Cdda");
		Config.Cdda = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_PsxAuto");
		Config.PsxAuto = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		t = Config.Cpu;
		Btn = glade_xml_get_widget(xml, "GtkCheckButton_Cpu");
		Config.Cpu = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));
		if (t != Config.Cpu) {
			psxCpu->Shutdown();
			#ifdef PSXREC
			if (Config.Cpu)	psxCpu = &psxInt;
			else psxCpu = &psxRec;
			#else
			psxCpu = &psxInt;
			#endif
			if (psxCpu->Init() == -1) {
				SysClose();
				exit(1);
			}
			psxCpu->Reset();
		}

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_PsxOut");
		Config.PsxOut = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_SpuIrq");
		Config.SpuIrq = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_RCntFix");
		Config.RCntFix = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		Btn = glade_xml_get_widget(xml, "GtkCheckButton_VSyncWA");
		Config.VSyncWA = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(Btn));

		SaveConfig();
	} 

	gtk_widget_destroy(CpuDlg);
	CpuDlg = NULL;
	if (Window != NULL)
		gtk_widget_set_sensitive(Window, TRUE);

}

void OnConf_Cpu() {
	GtkWidget *Btn;
	GladeXML *xml;

	if (CpuDlg != NULL) {
		gtk_window_present (GTK_WINDOW (CpuDlg));
		return;
	}

	xml = glade_xml_new(PACKAGE_DATA_DIR "pcsx.glade2", "CpuDlg", NULL);

	if (!xml)
	{
		g_warning("We could not load the interface!");
		return;
	}

	CpuDlg = glade_xml_get_widget(xml, "CpuDlg");
	
	PsxCombo = glade_xml_get_widget(xml, "GtkCombo_PsxType");
	gtk_combo_box_set_active (GTK_COMBO_BOX (PsxCombo), Config.PsxType);
	gtk_widget_set_sensitive (GTK_WIDGET (PsxCombo), !Config.PsxAuto);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_Xa");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.Xa);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_Sio");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.Sio);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_Mdec");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.Mdec);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_Cdda");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.Cdda);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_PsxAuto");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.PsxAuto);

	g_signal_connect_data(GTK_OBJECT(Btn), "toggled",
			GTK_SIGNAL_FUNC(OnCpu_PsxAutoClicked), xml, NULL, G_CONNECT_AFTER);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_Cpu");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.Cpu);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_PsxOut");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.PsxOut);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_SpuIrq");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.SpuIrq);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_RCntFix");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.RCntFix);

	Btn = glade_xml_get_widget(xml, "GtkCheckButton_VSyncWA");
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Btn), Config.VSyncWA);

	/* Setup a handler for when Close or Cancel is clicked */
	g_signal_connect_data(GTK_OBJECT(CpuDlg), "response",
			GTK_SIGNAL_FUNC(OnCpu_Clicked), xml, (GClosureNotify)g_object_unref, G_CONNECT_AFTER);
}

/*void OnConfConf_Ok() {
	GetComboText(GpuConfS.Combo, GpuConfS.plist, Config.Gpu);
	GetComboText(SpuConfS.Combo, SpuConfS.plist, Config.Spu);
	GetComboText(CdrConfS.Combo, CdrConfS.plist, Config.Cdr);
	GetComboText(Pad1ConfS.Combo, Pad1ConfS.plist, Config.Pad1);
	GetComboText(Pad2ConfS.Combo, Pad2ConfS.plist, Config.Pad2);
	GetComboText(BiosConfS.Combo, BiosConfS.plist, Config.Bios);

	SaveConfig();
	* ADB Commented out - we should release/load the plugin
	   when we select them, not when we open or close the config screen
*	ReleasePlugins();
	LoadPlugins();

	needreset = 1;
	gtk_widget_destroy(ConfDlg);
	gtk_main_quit();
}

void OnConfConf_Cancel() {
	gtk_widget_destroy(ConfDlg);
	gtk_main_quit();
}*/

#define ConfPlugin(src, confs, plugin, name, parent) \
	void *drv; \
	src conf; \
	char file[MAXPATHLEN]; \
	gchar *dotdir; \
 \
	dotdir = g_strdup (getenv("HOME")); \
	dotdir = g_strconcat (dotdir, PLUGINS_DIR, NULL); \
 \
	GetComboText(confs.Combo, confs.plist, plugin); \
	strcpy(file, dotdir); \
	strncat(file, plugin, 80); \
	drv = SysLoadLibrary(file); \
	if (drv == NULL) return; \
 \
	gtk_widget_set_sensitive(parent, FALSE); \
	while (gtk_events_pending()) gtk_main_iteration(); \
	conf = (src) SysLoadSym(drv, name); \
	if (conf) { \
		conf(); \
	} else SysMessage(_("This plugin doesn't need to be configured.")); \
	SysCloseLibrary(drv); \
	gtk_widget_set_sensitive(parent, TRUE); \
	g_free (dotdir);

#define TestPlugin(src, confs, plugin, name, parent) \
	void *drv; \
	src conf; \
	int ret = 0; \
	char file[MAXPATHLEN]; \
	gchar *dotdir; \
 \
	dotdir = g_strdup (getenv("HOME")); \
	dotdir = g_strconcat (dotdir, PLUGINS_DIR, NULL); \
 \
	GetComboText(confs.Combo, confs.plist, plugin); \
	strcpy(file, dotdir); \
	strncat(file, plugin, 80); \
	drv = SysLoadLibrary(file); \
	if (drv == NULL) return; \
 \
	gtk_widget_set_sensitive(parent, FALSE); \
	while (gtk_events_pending()) gtk_main_iteration(); \
	conf = (src) SysLoadSym(drv, name); \
	if (SysLibError() == NULL) ret = conf(); \
	SysCloseLibrary(drv); \
	if (ret == 0) \
		 SysMessage(_("This plugin reports that it should work correctly.")); \
	else SysMessage(_("This plugin reports that it won't work correctly.")); \
	gtk_widget_set_sensitive(parent, TRUE); \
	g_free (dotdir);

void OnConfConf_GpuConf(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(GPUconfigure, GpuConfS, Config.Gpu, "GPUconfigure", ConfDlg);
}

void OnConfConf_GpuTest(GtkWidget *widget, gpointer user_data) {
	TestPlugin(GPUtest, GpuConfS, Config.Gpu, "GPUtest", ConfDlg);
}

void OnConfConf_GpuAbout(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(GPUabout, GpuConfS, Config.Gpu, "GPUabout", ConfDlg);
}

void OnConfConf_SpuConf(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(SPUconfigure, SpuConfS, Config.Spu, "SPUconfigure", ConfDlg);
}

void OnConfConf_SpuTest(GtkWidget *widget, gpointer user_data) {
	TestPlugin(SPUtest, SpuConfS, Config.Spu, "SPUtest", ConfDlg);
}

void OnConfConf_SpuAbout(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(SPUabout, SpuConfS, Config.Spu, "SPUabout", ConfDlg);
}

void OnConfConf_CdrConf(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(CDRconfigure, CdrConfS, Config.Cdr, "CDRconfigure", ConfDlg);
}

void OnConfConf_CdrTest(GtkWidget *widget, gpointer user_data) {
	TestPlugin(CDRtest, CdrConfS, Config.Cdr, "CDRtest", ConfDlg);
}

void OnConfConf_CdrAbout(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(CDRabout, CdrConfS, Config.Cdr, "CDRabout", ConfDlg);
}

void OnConfConf_Pad1Conf(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(PADconfigure, Pad1ConfS, Config.Pad1, "PADconfigure", ConfDlg);
}

void OnConfConf_Pad1Test(GtkWidget *widget, gpointer user_data) {
	TestPlugin(PADtest, Pad1ConfS, Config.Pad1, "PADtest", ConfDlg);
}

void OnConfConf_Pad1About(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(PADabout, Pad1ConfS, Config.Pad1, "PADabout", ConfDlg);
}

void OnConfConf_Pad2Conf(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(PADconfigure, Pad2ConfS, Config.Pad2, "PADconfigure", ConfDlg);
}

void OnConfConf_Pad2Test(GtkWidget *widget, gpointer user_data) {
	TestPlugin(PADtest, Pad2ConfS, Config.Pad2, "PADtest", ConfDlg);
}

void OnConfConf_Pad2About(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(PADabout, Pad2ConfS, Config.Pad2, "PADabout", ConfDlg);
}

void OnNet_Conf(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(NETconfigure, NetConfS, Config.Net, "NETconfigure", NetDlg);
}

void OnNet_Test(GtkWidget *widget, gpointer user_data) {
	TestPlugin(NETtest, NetConfS, Config.Net, "NETtest", NetDlg);
}

void OnNet_About(GtkWidget *widget, gpointer user_data) {
	ConfPlugin(NETabout, NetConfS, Config.Net, "NETabout", NetDlg);
}

void OnPluginPath_Changed(GtkWidget *wdg, gpointer data) {
	gchar *path;

	path = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (wdg));
	strcpy(Config.PluginsDir, path);

	/* TODO The next statement is redundant, since gtk_file_chooser_get_
	   current_folder always chops the '/' */
	if (Config.PluginsDir[strlen(Config.PluginsDir)-1] != '/')
		strcat(Config.PluginsDir, "/");
	
	ScanPlugins(Config.PluginsDir);
	UpdatePluginsBIOS();

	// Pick some defaults, if they're available and no Config.x is set
	if ((GpuConfS.plist[0]) && (strlen(Config.Gpu) == 0)) { strcpy(Config.Gpu, GpuConfS.plist[0]); printf("picking default GPU plugin: %s\n", GpuConfS.plist[0]); }
	if ((SpuConfS.plist[0]) && (strlen(Config.Spu) == 0)) { strcpy(Config.Spu, SpuConfS.plist[0]); printf("picking default SPU plugin: %s\n", SpuConfS.plist[0]); }
	if ((CdrConfS.plist[0]) && (strlen(Config.Cdr) == 0)) { strcpy(Config.Cdr, CdrConfS.plist[0]); printf("picking default CD-ROM plugin: %s\n", CdrConfS.plist[0]); }
	if ((Pad1ConfS.plist[0]) && (strlen(Config.Pad1) == 0)) { strcpy(Config.Pad1, Pad1ConfS.plist[0]); printf("picking default Controller 1 plugin: %s\n", Pad1ConfS.plist[0]); }
	if ((Pad2ConfS.plist[0]) && (strlen(Config.Pad2) == 0)) { strcpy(Config.Pad2, Pad2ConfS.plist[0]); printf("picking default Controller 2 plugin: %s\n", Pad2ConfS.plist[0]); }
	if ((BiosConfS.plist[0]) && (strlen(Config.Bios) == 0)) { strcpy(Config.Bios, BiosConfS.plist[0]); printf("picking default BIOS: %s\n", BiosConfS.plist[0]); }
	
	UpdatePluginsBIOS_UpdateGUI(data);
	g_free (path);
}

void OnBiosPath_Changed(GtkWidget *wdg, gpointer data) {
	const gchar *File;

	File = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (wdg));
	strcpy(Config.BiosDir, File);
	if (Config.BiosDir[strlen(Config.BiosDir)-1] != '/')
		strcat(Config.BiosDir, "/");

	UpdatePluginsBIOS();
	UpdatePluginsBIOS_UpdateGUI(data);
}

void OnConf_Clicked (GtkDialog *dialog, gint arg1, gpointer user_data)
{
	if (arg1 == GTK_RESPONSE_OK)
	{
		GetComboText(GpuConfS.Combo, GpuConfS.plist, Config.Gpu);
		GetComboText(SpuConfS.Combo, SpuConfS.plist, Config.Spu);
		GetComboText(CdrConfS.Combo, CdrConfS.plist, Config.Cdr);
		GetComboText(Pad1ConfS.Combo, Pad1ConfS.plist, Config.Pad1);
		GetComboText(Pad2ConfS.Combo, Pad2ConfS.plist, Config.Pad2);
		GetComboText(BiosConfS.Combo, BiosConfS.plist, Config.Bios);

		/* ADB TODO Validation */

		SaveConfig();

		/* ADB Commented out - we should release/load the plugin
		   when we select them, not when we open or close the config screen
		*/
		ReleasePlugins();
		// Ryan: inelegant hack to stop the 'Could not load' error.
		// if all plugins aren't available, don't try to load them.
		if ((strlen(Config.Gpu) == 0) && (strlen(Config.Spu) == 0) && (strlen(Config.Cdr) == 0) && (strlen(Config.Pad1) == 0) && (strlen(Config.Pad2) == 0) && (strlen(Config.Bios) == 0)) {
			LoadPlugins();
		}

		needreset = 1;
	} 

	gtk_widget_destroy(ConfDlg);
	if (Window != NULL)
		gtk_widget_set_sensitive(Window, TRUE);
	ConfDlg = NULL;

}

void OnConf_Conf() {
	GladeXML *xml;
	GtkWidget *widget;

	const gchar *path;

	// update some essentials first
	ScanPlugins(Config.PluginsDir);
	UpdatePluginsBIOS();


	if (ConfDlg != NULL) {
		gtk_window_present (GTK_WINDOW (ConfDlg));
		return;
	}

	xml = glade_xml_new(PACKAGE_DATA_DIR "pcsx.glade2", "ConfDlg", NULL);

	if (!xml)
	{
		g_warning(_("Error: Glade interface could not be loaded!"));
		return;
	}

	UpdatePluginsBIOS_UpdateGUI(xml);

	ConfDlg = glade_xml_get_widget(xml, "ConfDlg");

	gtk_window_set_title(GTK_WINDOW(ConfDlg), _("Configure PCSX"));

	/* Set the paths in the file choosers to be based on the saved configurations */
	widget = glade_xml_get_widget(xml, "GtkFileChooser_Bios");
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (widget), 
			Config.BiosDir);
	
	widget = glade_xml_get_widget(xml, "GtkFileChooser_Plugin");
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (widget), 
			Config.PluginsDir);
	
	/*printf("pluginsdir = %s", Config.PluginsDir);
	if (strlen(Config.PluginsDir) == 0) {
		path = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (widget));
		strcpy(Config.PluginsDir, path);
		g_free(path);
	}*/

	/* Assign callbacks; when fully migrated to GladeXML, don't need to do this */
	g_signal_connect_data(GTK_OBJECT(ConfDlg), "response",
			GTK_SIGNAL_FUNC(OnConf_Clicked), xml, (GClosureNotify)g_object_unref, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_ConfGpu");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_GpuConf), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_ConfSpu");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_SpuConf), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_ConfPad1");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_Pad1Conf), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_ConfPad2");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_Pad2Conf), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_ConfCdr");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_CdrConf), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_AboutGpu");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_GpuAbout), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_AboutSpu");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_SpuAbout), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_AboutPad1");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_Pad1About), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_AboutPad2");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_Pad2About), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "btn_AboutCdr");
	g_signal_connect_data(GTK_OBJECT(widget), "clicked",
			GTK_SIGNAL_FUNC(OnConfConf_CdrAbout), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkFileChooser_Bios");
	g_signal_connect_data(GTK_OBJECT(widget), "current_folder_changed",
			GTK_SIGNAL_FUNC(OnBiosPath_Changed), xml, NULL, G_CONNECT_AFTER);

	widget = glade_xml_get_widget(xml, "GtkFileChooser_Plugin");
	g_signal_connect_data(GTK_OBJECT(widget), "current_folder_changed",
			GTK_SIGNAL_FUNC(OnPluginPath_Changed), xml, NULL, G_CONNECT_AFTER);

	/* Wait for user to close window before proceeding; this is for startup, when
	   configuration window opens if plugins are mis-configured, to prevent other
	   windows from opening in the background. */
	gtk_dialog_run (GTK_DIALOG (ConfDlg));

	if ((strlen(Config.Gpu) != 0) && (strlen(Config.Spu) != 0) && (strlen(Config.Cdr) != 0) && (strlen(Config.Pad1) != 0) && (strlen(Config.Pad2) != 0))  { ReleasePlugins(); LoadPlugins(); }

}

void OnHelp_About() {
	GladeXML *xml;

	if (AboutDlg != NULL) {
		gtk_window_present (GTK_WINDOW (AboutDlg));
		return;
	}
	
	xml = glade_xml_new(PACKAGE_DATA_DIR "pcsx.glade2", "AboutDlg", NULL);

	if (!xml)
	{
		g_warning("We could not load the interface!");
		return;
	}

	AboutDlg = glade_xml_get_widget(xml, "AboutDlg");
	
	gtk_dialog_run (GTK_DIALOG (AboutDlg));
}

void OnHelp_Close() {
	gtk_widget_destroy(AboutDlg);
	AboutDlg = NULL;
}

#define ComboAddPlugin(type) { \
	type##ConfS.plugins+=2; \
	strcpy(type##ConfS.plist[type##ConfS.plugins-1], name); \
	strcpy(type##ConfS.plist[type##ConfS.plugins-2], ent->d_name); \
	type##ConfS.glist = g_list_append(type##ConfS.glist, type##ConfS.plist[type##ConfS.plugins-1]); \
}

void populate_combo_box (GtkWidget *widget, GList *list) {
	GtkListStore *store;
	GtkCellRenderer *renderer;
	store = gtk_list_store_new (1, G_TYPE_STRING);

	/* Clear existing data from combo box */
	gtk_cell_layout_clear (GTK_CELL_LAYOUT (widget));

	renderer = gtk_cell_renderer_text_new();
        gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (widget), renderer, FALSE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (widget), renderer, "text", 0);
		
	while (list != NULL) {
		GtkTreeIter iter;
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter, 0, (char *)list->data, -1);
		list = list->next;
	}
	gtk_combo_box_set_model (GTK_COMBO_BOX (widget), GTK_TREE_MODEL (store));
}

#define ConfCreatePConf(name, type) \
	/* Populate the relevant combo widget with the list of plugins. \
	   If no plugins available, disable the combo and its controls */ \
	type##ConfS.Combo = glade_xml_get_widget(xml, "GtkCombo_" name); \
	if (type##ConfS.glist != NULL) { \
		populate_combo_box (type##ConfS.Combo, type##ConfS.glist); \
		FindComboText(type##ConfS.Combo, type##ConfS.plist, Config.type); \
		gtk_widget_set_sensitive (type##ConfS.Combo, TRUE); \
		controlwidget = glade_xml_get_widget(xml, "btn_Conf" name); \
		gtk_widget_set_sensitive (controlwidget, TRUE); \
		controlwidget = glade_xml_get_widget(xml, "btn_About" name); \
		gtk_widget_set_sensitive (controlwidget, TRUE); \
	} else { \
		gtk_widget_set_sensitive (type##ConfS.Combo, FALSE); \
		controlwidget = glade_xml_get_widget(xml, "btn_Conf" name); \
		gtk_widget_set_sensitive (controlwidget, FALSE); \
		controlwidget = glade_xml_get_widget(xml, "btn_About" name); \
		gtk_widget_set_sensitive (controlwidget, FALSE); }

void ScanPlugins(gchar* scandir) {
	// scan for plugins and configuration tools
	DIR *dir;
	struct dirent *ent;
	char file[MAXPATHLEN];
	gchar *linkname;

	linkname = g_strdup (getenv("HOME"));
	linkname = g_strconcat (linkname, PLUGINS_DIR, NULL);

	dir = opendir(scandir);
	if (dir == NULL) {
		printf(_("Could not open directory: '%s'\n"), scandir);
		/* ADB TODO Should throw an error message and a file chooser for
		   a new plugin directory */
		return;
	}

	while ((ent = readdir(dir)) != NULL) {
		sprintf(file, "%s%s", scandir, ent->d_name);

		if ((strstr(file, ".so") == NULL) && (strstr(file, "cfg") == NULL)) continue;

		/* Create a symlink from this file to the directory ~/.pcsx/plugin */
		linkname = g_strconcat (linkname, ent->d_name, NULL);
		printf("symlink: %s -> %s\n", file, linkname);
		symlink(file, linkname);

		// if it's a config tool, make one in the cfg dir as well
		// this allows plugins with retarded cfg finding to work :- )
		if (strstr(file, "cfg") != NULL) {

			linkname = g_strdup (getenv("HOME"));
			linkname = g_strconcat (linkname, PLUGINS_CFG_DIR, ent->d_name, NULL);

			printf("symlink: %s -> %s\n", file, linkname);
			symlink(file, linkname);
		}
		
		/* reset linkname value */
		linkname = g_strdup (getenv("HOME"));
		linkname = g_strconcat (linkname, PLUGINS_DIR, NULL);
	}

	closedir(dir);
	
	g_free (linkname);
}

void check_symlinks_in_path (char* dotdir)
{
	DIR *dir;
	struct dirent *ent;
	struct stat stbuf;
	gchar *linkname/*,name[256]*/;

	dir = opendir(dotdir);
	if (dir == NULL) {
		SysMessage(_("Could not open directory: '%s'\n"), dotdir);
		return;
	}

	/* Check for any bad links in the directory. If the remote
	   file no longer exists, remove the link */
	while ((ent = readdir(dir)) != NULL) {
		linkname = g_strconcat (dotdir, ent->d_name, NULL);

		if (stat(linkname, &stbuf) == -1) {
			/* File link is bad, remove it */
			printf("unlink: %s\n", linkname);
			unlink(linkname);
		}
	}
	closedir(dir);
	g_free (linkname);
}

int CheckPlugins() {
	struct stat stbuf;
	char pluginfile[MAXPATHLEN];
	
	// make sure there are choices for all of the plugins
	if ((strlen(Config.Gpu) == 0) || (strlen(Config.Spu) == 0) || (strlen(Config.Cdr) == 0) || (strlen(Config.Pad1) == 0) || (strlen(Config.Pad2) == 0)) {
		return 1;
	}
	// and make sure they can all be accessed
	// if they can't be, wipe the variable and return 1
	strncpy(pluginfile, getenv("HOME"), MAXPATHLEN-(strlen("/.pcsx/plugins/")+strlen(Config.Gpu)));
	strcat(pluginfile, "/.pcsx/plugins/");
	strcat(pluginfile, Config.Gpu);
	if (stat(pluginfile, &stbuf) == -1) { Config.Gpu[0] = '\0'; return 1; }
	strncpy(pluginfile, getenv("HOME"), MAXPATHLEN-(strlen("/.pcsx/plugins/")+strlen(Config.Spu)));
	strcat(pluginfile, "/.pcsx/plugins/");
	strcat(pluginfile, Config.Spu);
	if (stat(pluginfile, &stbuf) == -1) { Config.Spu[0] = '\0'; return 1; }
	strncpy(pluginfile, getenv("HOME"), MAXPATHLEN-(strlen("/.pcsx/plugins/")+strlen(Config.Cdr)));
	strcat(pluginfile, "/.pcsx/plugins/");
	strcat(pluginfile, Config.Cdr);
	if (stat(pluginfile, &stbuf) == -1) { Config.Cdr[0] = '\0'; return 1; }
	strncpy(pluginfile, getenv("HOME"), MAXPATHLEN-(strlen("/.pcsx/plugins/")+strlen(Config.Pad1)));
	strcat(pluginfile, "/.pcsx/plugins/");
	strcat(pluginfile, Config.Pad1);
	if (stat(pluginfile, &stbuf) == -1) { Config.Pad1[0] = '\0'; return 1; }
	strncpy(pluginfile, getenv("HOME"), MAXPATHLEN-(strlen("/.pcsx/plugins/")+strlen(Config.Pad2)));
	strcat(pluginfile, "/.pcsx/plugins/");
	strcat(pluginfile, Config.Pad2);
	if (stat(pluginfile, &stbuf) == -1) { Config.Pad2[0] = '\0'; return 1; }
	// if everything is happy, return 0
	return 0;
}
	
void UpdatePluginsBIOS() {
	DIR *dir;
	struct dirent *ent;
	struct stat stbuf;
	void *Handle;
	char name[256];
	gchar *linkname;
	gchar *dotpluginsdir;
	gchar *dotpluginscfgdir;
	gchar *dotdir;

	dotpluginsdir = g_strdup (getenv("HOME"));
	dotpluginsdir = g_strconcat (dotpluginsdir, PLUGINS_DIR, NULL);

	GpuConfS.plugins  = 0; SpuConfS.plugins  = 0; CdrConfS.plugins  = 0;
	Pad1ConfS.plugins = 0; Pad2ConfS.plugins = 0; BiosConfS.plugins = 0;
	GpuConfS.glist  = NULL; SpuConfS.glist  = NULL; CdrConfS.glist  = NULL;
	Pad1ConfS.glist = NULL; Pad2ConfS.glist = NULL; BiosConfS.glist = NULL;

	/* Check for bad links in ~/.pcsx/plugins/ */
	check_symlinks_in_path (dotpluginsdir);
	
	/* Check for bad links in ~/.pcsx/plugins/cfg */
	dotpluginscfgdir = g_strdup (dotpluginsdir);
	dotpluginscfgdir = g_strconcat (dotpluginscfgdir, "cfg/", NULL);
	check_symlinks_in_path (dotpluginscfgdir);
	
	/* Load and get plugin info */
	dir = opendir(dotpluginsdir);
	if (dir == NULL) {
		printf(_("Could not open directory: '%s'\n"), dotpluginsdir);
		return;
	}
	while ((ent = readdir(dir)) != NULL) {
		long type,v;
		linkname = g_strconcat (dotpluginsdir, ent->d_name, NULL);
		
		// only libraries past this point, not config tools
		if (strstr(linkname, ".so") == NULL) continue;

		Handle = dlopen(linkname, RTLD_NOW);
		if (Handle == NULL) { printf("%s\n", dlerror()); continue; }

		PSE_getLibType = (PSEgetLibType) dlsym(Handle, "PSEgetLibType");
		if (dlerror() != NULL) {
			if (strstr(linkname, "gpu") != NULL) type = PSE_LT_GPU;
			else if (strstr(linkname, "cdr") != NULL) type = PSE_LT_CDR;
			else if (strstr(linkname, "spu") != NULL) type = PSE_LT_SPU;
			else if (strstr(linkname, "pad") != NULL) type = PSE_LT_PAD;
			else continue;
		}
		else type = PSE_getLibType();

		PSE_getLibName = (PSEgetLibName) dlsym(Handle, "PSEgetLibName");
		if (dlerror() == NULL) {
			sprintf(name, "%s", PSE_getLibName());
			PSE_getLibVersion = (PSEgetLibVersion) dlsym(Handle, "PSEgetLibVersion");
			if (dlerror() == NULL) {
				char ver[32];

				v = PSE_getLibVersion();
				sprintf(ver, " %ld.%ld.%ld",v>>16,(v>>8)&0xff,v&0xff);
				strcat(name, ver);
			}
		}
		else strcpy(name, ent->d_name);

		if (type & PSE_LT_CDR) {
			ComboAddPlugin(Cdr);
		}
		if (type & PSE_LT_GPU) {
			ComboAddPlugin(Gpu);
		}
		if (type & PSE_LT_SPU) {
			ComboAddPlugin(Spu);
		}
		if (type & PSE_LT_PAD) {
			PADquery query = (PADquery)dlsym(Handle, "PADquery");
			if (query() & 0x1) {
				ComboAddPlugin(Pad1);
			}
			if (query() & 0x2) {
				ComboAddPlugin(Pad2);
			}
		}
	}
	closedir(dir);

	/* The BIOS list always contains the PCSX BIOS */
	BiosConfS.plugins+=2;
	strcpy(BiosConfS.plist[BiosConfS.plugins-1], _("Internal HLE Bios"));
	strcpy(BiosConfS.plist[BiosConfS.plugins-2], "HLE");
	BiosConfS.glist = g_list_append(BiosConfS.glist, BiosConfS.plist[BiosConfS.plugins-1]);

	dir = opendir(Config.BiosDir);
	if (dir == NULL) {
		SysMessage(_("Could not open directory: '%s'\n"), Config.BiosDir);
		return;
	}

	while ((ent = readdir(dir)) != NULL) {
		struct stat buf;

		sprintf (linkname, "%s%s", Config.BiosDir, ent->d_name);
		if (stat(linkname, &buf) == -1) continue;
		if (buf.st_size != (1024*512)) continue;

		BiosConfS.plugins+=2;
		strcpy(BiosConfS.plist[BiosConfS.plugins-1], ent->d_name);
		strcpy(BiosConfS.plist[BiosConfS.plugins-2], ent->d_name);
		BiosConfS.glist = g_list_append(BiosConfS.glist, BiosConfS.plist[BiosConfS.plugins-1]);
	}
	closedir(dir);

	/* also scan the .pcsx/bios/ directory */
	dotdir = g_strdup (getenv("HOME"));
	dotdir = g_strconcat (dotdir, "/.pcsx/bios/", NULL);
	
	dir = opendir(dotdir);

	if (dir == NULL) {
		printf(_("Could not open directory: '%s'\n"), dotdir);
		return;
	}

	while ((ent = readdir(dir)) != NULL) {
		struct stat buf;

		sprintf (linkname, "%s%s", dotdir, ent->d_name);
		if (stat(linkname, &buf) == -1) continue;
		if (buf.st_size != (1024*512)) continue;

		BiosConfS.plugins+=2;
		strcpy(BiosConfS.plist[BiosConfS.plugins-1], ent->d_name);
		strcpy(BiosConfS.plist[BiosConfS.plugins-2], ent->d_name);
		BiosConfS.glist = g_list_append(BiosConfS.glist, BiosConfS.plist[BiosConfS.plugins-1]);
	}
	closedir(dir);
	
	g_free (dotpluginsdir);
	g_free (dotpluginscfgdir);
	g_free (dotdir);
	g_free (linkname);
}

void UpdatePluginsBIOS_UpdateGUI(GladeXML *xml) {
	/* Populate the plugin combo boxes */
	ConfCreatePConf("Gpu", Gpu);
	ConfCreatePConf("Spu", Spu);
	ConfCreatePConf("Pad1", Pad1);
	ConfCreatePConf("Pad2", Pad2);
	ConfCreatePConf("Cdr", Cdr);
	ConfCreatePConf("Bios", Bios);
}

void FindNetPlugin(GladeXML *xml) {
	DIR *dir;
	struct dirent *ent;
	void *Handle;
	char plugin[MAXPATHLEN],name[MAXPATHLEN];
	gchar *dotdir;
	
	dotdir = g_strdup (getenv("HOME"));
	dotdir = g_strconcat (dotdir, BIOS_DIR, NULL);

	NetConfS.plugins  = 0;
	NetConfS.glist = NULL; 

	NetConfS.plugins+=2;
	strcpy(NetConfS.plist[NetConfS.plugins-1], "Disabled");
	strcpy(NetConfS.plist[NetConfS.plugins-2], "Disabled");
	NetConfS.glist = g_list_append(NetConfS.glist, NetConfS.plist[NetConfS.plugins-1]);

	dir = opendir(dotdir);
	if (dir == NULL) {
		SysMessage(_("Could not open directory: '%s'\n"), dotdir);
		return;
	}
	while ((ent = readdir(dir)) != NULL) {
		long type,v;

		sprintf(plugin, "%s%s", dotdir, ent->d_name);

		if (strstr(plugin, ".so") == NULL) continue;
		Handle = dlopen(plugin, RTLD_NOW);
		if (Handle == NULL) continue;

		PSE_getLibType = (PSEgetLibType) dlsym(Handle, "PSEgetLibType");
		if (dlerror() != NULL) {
			if (strstr(plugin, "net") != NULL) type = PSE_LT_NET;
			else continue;
		}
		else type = PSE_getLibType();

		PSE_getLibName = (PSEgetLibName) dlsym(Handle, "PSEgetLibName");
		if (dlerror() == NULL) {
			sprintf(name, "%s", PSE_getLibName());
			PSE_getLibVersion = (PSEgetLibVersion) dlsym(Handle, "PSEgetLibVersion");
			if (dlerror() == NULL) {
				char ver[32];

				v = PSE_getLibVersion();
				sprintf(ver, " %ld.%ld.%ld",v>>16,(v>>8)&0xff,v&0xff);
				strcat(name, ver);
			}
		}
		else strcpy(name, ent->d_name);

		if (type & PSE_LT_NET) {
			ComboAddPlugin(Net);
		}
	}
	closedir(dir);

	ConfCreatePConf("Net", Net);
	
	g_free (dotdir);
}

void SysMessage(char *fmt, ...) {
	    GtkWidget *Txt, *MsgDlg;
        va_list list;
        char msg[512];

        va_start(list, fmt);
        vsprintf(msg, fmt, list);
        va_end(list);

        if (msg[strlen(msg)-1] == '\n') msg[strlen(msg)-1] = 0;

        if (!UseGui) { printf ("%s\n",msg); return; }
        MsgDlg =  gtk_dialog_new_with_buttons (_("Notice"),
                                         NULL/*GTK_WINDOW(Window)*/,
                                         GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_STOCK_OK,
                                         GTK_RESPONSE_NONE,
                                         NULL);

        gtk_window_set_position (GTK_WINDOW(MsgDlg), GTK_WIN_POS_CENTER);

        Txt = gtk_label_new (msg);
        gtk_label_set_line_wrap(GTK_LABEL(Txt), TRUE);
        gtk_container_add (GTK_CONTAINER (GTK_DIALOG(MsgDlg)->vbox),
                      Txt);

        gtk_widget_show (Txt);
        gtk_widget_show_all (MsgDlg);
        gtk_dialog_run(GTK_DIALOG(MsgDlg));
        gtk_widget_destroy(MsgDlg);

	
}
/* ADB TODO
void SysMessage(gint message_type, gchar *primary, gchar *secondary) {
	GtkWidget *MsgDlg;	
	if (!UseGui) { printf ("%s\n",primary); return; }
	MsgDlg =  gtk_message_dialog_new (NULL,
		GTK_DIALOG_MODAL,
		message_type,
		GTK_BUTTONS_CLOSE,
		primary,
		NULL);
	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (MsgDlg),
		secondary);
	
	gtk_dialog_run(GTK_DIALOG(MsgDlg));
	gtk_widget_destroy(MsgDlg);
}*/
