/*****************************************************************************/
/*                                                                           */
/* nim_drv.c - nim application interface module                                  */
/*                                                                           */ 
/* Differences between HDM8513 and HDM8511:                                  */
/*                                                                           */
/*   1. Steady/Acquisition state carrier loop filter ctrl reg. in 8513       */
/*      ( only 1 register in 8511 regardless of state )                      */
/*   2. Automatic detection of spectrum inversion in 8513                    */
/*   3. Viterbi signal quality reset bit in RESET reg. in 8513               */
/*   4. LNB hold & LNB tone frequency setting reg. in 8513                   */
/*   5. I2C bypass/block function in 8513                                    */
/*   6. Sigma delta 2 reg. only in 8513, both sigma delta 1/2 in 8511        */
/*   7. In test-set-up, IQ input is possible in 8513                         */
/*   8. Byte Sync Control reg. added in 8513                                 */
/*   9. Fixed/auto Viterbi rate, DVB/DSS, BPSK/QPSK mode in 8513             */
/*  10. Viterbi 6/7 added in 8513                                            */
/*                                                                           */
/* Function added in HDM8513A                                                */
/*                                                                           */
/*   1. Internal PLL synthesis logic integrated for system clock generation  */
/*   2. Viterbi decoder output BER register added                            */
/*   3. Transport stream output clock regulated                              */
/*                                                                           */
/* Release note:                                                             */
/*                                                                           */
/* Initial release 1998. 7. 16. H.D. Sung                                    */
/*    - updated for interface test /hss200/test8513, 1998. 9. 21. H.D. Sung  */
/* release 1.0: 1998. 11.23.                                                 */
/*    - nim_start with zig-zag scan added                                    */
/* release 1.1:                                                              */
/* release 1.2: some bug fixed, for dll interface to HCP, 1999. 1. 15. SHD   */
/*              chipsets: sp5659, max2102, hdm8513                           */
/* release 1.2a: prescalar enabled to cover entire lband, 1999. 6. 1. SHD    */
/*                                                                           */
/* release 2.0: for changed chipsets: sp5769, sl1925, hdm8513 1999. 6. 1. SHD*/
/* release 2.1: 1. for changed chipsets: hdm8513a, 1999. 9.1. SHD            */
/*              2. get_qpsk_lock_status() routine slightly changed to        */
/*                 to accomodate the viterbi node sync. viterbi byte sync.   */
/*                 lock field. 1999. 11. 10. SHD                             */
/*              3. p1 switching frequency to 1500Mhz. 2000. 11. 5. SHD       */
/* release 2.2: 1. for changed chipsets CHONGJU HDM8513A, 2000. 11. 5. SHD   */
/*                 - negative triggering of DATA_CLK                         */
/*                 -                                        		     */	
/* Seen bugs:                                                                */
/*            1. auto spectrum detection does not work, chip bug.            */
/*                                                                           */
/* Notes:                                                                    */
/*            1. Carrier loop filter gain; steady state, acquisition         */
/*               state gain parameter index swapped..                        */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/*****************************************************************************/

//#define QPSK_LOCK_DEBUG
//#define SCAN_ZIG_ZAG_DEBUG
//#define SCAN_SUBRANGE_DEBUG

//#include <windows.h>
#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <string.h>
#include "../../share/xptype.h"
#include "../../share/ifdef.h"

#include "pentapi.h"

//#define FWAIT(a) Sleep(a)
//#define FWAIT(a)  mSleep(a)
#define FWAIT(a)  usleep(a * 1000 )
#define ELAPSED(a,b) elapsed(a,b)
#define STAMP(a) stamp(a)


#include "nimdef.h"
#include "misc.h"

#ifdef LIC_API
#undef LIC_API
#endif

#include "nim_drv.h"

achar gg_dev_name[256];
byte gg_qpsk_addr = 0xe8; 	/* default HDM8513A */
double gg_clock = 76000000.0;   /* default HDM8513A */


/* tuner specific definitions ************************************************/

/* tuner address might be different..
#define IIC_TUNER_ADDR 0xc0
*/
#define IIC_TUNER_ADDR 0xc2            /* tuner iic write address                 */

#ifdef ZIF_
#define TUNER_IF 0.0                   /* tuner output frequency                  */
#else 
#define TUNER_IF 479.5E6               /* tuner output frequency                  */
#endif

#define TUNER_REF_RATIO 0x03           /* tuner reference ratio, 250khz respected */
#define TUNER_STEP 250.0E3             /* tuner step                              */

#define TUNER_ACTUAL_STEP 250.0E3      /* tuner real step, TUNER_STEP x 1         */

#define TUNER_PLL_LOCK 0x40            /* tuner pll-lock bit mask                 */
#define MAX_PLL_MSEC_TIME  100         /* tuner max-pll lock time allowded        */
#define TUNER_PLL_ISETTLE_MSEC_TIME 50 /* tuner pll init settle time              */
#define TUNER_PLL_SETTLE_MSEC_TIME 10  /* tuner pll update settle time            */

/* hdm8513(a) specific definitions ******************************/

#if 0 /* both-support */
#ifdef HDM8513A_
#define CLOCK_USED 76000000.0     /* system clock used : 76MHz */
#else
#define CLOCK_USED 60000000.0     /* system clock used  : 60MHz */
#endif
#endif /* End of both-support */

#if 0 /* both-support */
#ifdef HDM_8513A_
#define IIC_HDM_ADDR 0xe8         /* iic hdm8513a write address */
#else
#define IIC_HDM_ADDR 0xe0         /* iic hdm8513 write address */
#endif
#endif /* End of both-support */

#define SCALE_2_20 1048576.0      /* pow(2, 20)                              */
#define SCALE_2_16 65536.0        /* pow(2, 16)                               */
#define ROLL_OFF_FACTOR 0.35      /* raised cosine roll off factor for qpsk  */

/* wide band agc limit definition */
#define MIN_WAGC -128 /* indicates highest signal strength */
#define MAX_WAGC  127 /* indicates lowest signal strength  */

#define WR_REGSZE 32
#define RD_REGSZE 32

/* mask pattern for reg. 0x0f(Control parameters) */
#define MASK_BINARY            0x01
#define MASK_SI                0x02
#define MASK_BIAS_CANCEL       0x04
#define MASK_STRACK            0x08
#define MASK_CTRACK            0x10
#define MASK_SWEEP_HOLD        0x20
#define MASK_NAGC_MODE1        0x40
#define MASK_AUTOMATIC_SI      0x80

/* bit position for reg. 0x0f */
#define POS_BINARY                0
#define POS_SI                    1
#define POS_BIAS_CANCEL           2
#define POS_STRACK                3
#define POS_CTRACK                4
#define POS_SWEEP_HOLD            5
#define POS_NAGC_MODE1            6
#define POS_AUTOMATIC_SI          7

/* mask pattern for reg. 0x10(Reset functions) */
#define MASK_RESET_STFA        0x01
#define MASK_RESET_CPTFA       0x02
#define MASK_RESET_WAGCA       0x04
#define MASK_RESET_NAGCA       0x08
#define MASK_RESET_CSWEEP      0x20
#define MASK_RESET_ERRCNT      0x80

/* bit position for reg. 0x10 */
#define POS_RESET_STFA            0
#define POS_RESET_CPTFA           1
#define POS_RESET_WAGCA           2
#define POS_RESET_NAGCA           3
#define POS_RESET_CSWEEP          5
#define POS_RESET_ERRCNT          7

/* mask pattern for reg. 0x11(Wide band AGC control) */
#define MASK_WAGC_MODE1        0x01
#define MASK_WAGC_INVERT       0x02
#define MASK_WAGC_HOLD         0x04
#define MASK_TONE_HOLD         0x08
#define MASK_IIC_BYPASS        0x10
#define MASK_WAGC_TC           0xe0

/* bit position for reg. 0x11 */
#define POS_WAGC_MODE1            0
#define POS_WAGC_INVERT           1
#define POS_WAGC_HOLD             2
#define POS_TONE_HOLD             3
#define POS_IIC_BYPASS            4
#define POS_WAGC_TC               5

/* value for configuration reg. 0x14(Test Set-Up) */

/* test mode configuration with 3 lsb */
#define OUTPUT_TRISTATE        0x00
#define OUTPUT_IQ_BASEBAND     0x01
#define OUTPUT_CPHASE          0x02
#define OUTPUT_SPHASE          0x03
#define OUTPUT_BEFORE_AFTER_RS 0x04
#define OUTPUT_NAGC            0x05
#define OUTPUT_IQ_ADC          0x06
#define INPUT_IQ_ADC           0x07

#if 0 /* both-support */
#ifdef  HDM_8513A_
#define MASK_AUTO_SET_EBIT     0x08
#define MASK_LOOKUP_FRAME_ERR  0x10
#define MASK_CLK_REG           0x20
#define MASK_CLK_SERIALMODE    0x40

#define POS_AUTO_SET_EBIT         3
#define POS_LOOKUP_FRAME_ERR      4
#define POS_CLK_REG               5
#define POS_CLK_SERIALMODE        6
#endif
#endif /* End of both-support */

/* mask pattern for reg. 0x18(Control parameters for Viterbi & RS decoder) */
#define MASK_PARALLEL_SERIAL   0x01 
#define MASK_SYNC_CHECK        0x02

#define MASK_VITERBI_RATE      0x1c
#define MASK_VITERBI_AUTO      0x20
#define MASK_DVB_DSS           0x40
#define MASK_BPSK_QPSK         0x80

/* bit position for reg. 0x18(Control parameters for Viterbi & RS decoder) */
#define POS_PARALLEL_SERIAL       0 
#define POS_SYNC_CHECK            1

#define POS_VITERBI_RATE          2
#define POS_VITERBI_AUTO          5
#define POS_DVB_DSS               6
#define POS_BPSK_QPSK             7

#if 0 /* both-support */
#ifdef HDM_8513A_
/* mask pattern for reg. 0x1f(Clock generation PLL control) */
#define MASK_PLL_N             0x1f
#define MASK_PLL_M             0x20
#define MASK_PLL_ENABLE        0x40

/* bit position for reg. 0x1f */
#define POS_PLL_N                 0
#define POS_PLL_M                 5
#define POS_PLL_ENABLE            6

#else /* HDM_8513_ */
/* mask pattern for reg. 0x1f(Clock generation PLL control) */
#define MASK_AUTO_SET_EBIT      0x01
#define MASK_LOOKUP_FRAME_ERR   0x02

/* bit position for reg. 0x1f */
#define POS_AUTO_SET_EBIT          0
#define POS_LOOKUP_FRAME_ERR       1
#endif
#endif /* End of both-support */

/* write register index mnemonic */
#define WR_STF2                   0
#define WR_STF1                   1
#define WR_STF0                   2
#define WR_STLGC                  3
#define WR_CF2                    4
#define WR_CF1                    5
#define WR_CF0                    6
#define WR_CLFC_ACQUISITION       7
#define WR_CLFC_STEADY            8
#define WR_STEP1                  9
#define WR_STEP0                 10
#define WR_SPD1                  11
#define WR_SPD0                  12
#define WR_NSTEP                 13
#define WR_NAGC_IVALUE           14
#define WR_CTRL_PARAM            15
#define WR_RESET_PARAM           16
#define WR_CTRL_WAGC             17
#define WR_TONE_FREQ             18
#define WR_SIGMA_DELTA           19
#define WR_TEST_SETUP            20
#define WR_VITERBI_SYNC1         21
#define WR_VITERBI_SYNC2         22
#define WR_VITERBI_SYNC3         23
#define WR_VITERBI_RS_SETUP      24
#define WR_VITERBI_1_2           25
#define WR_VITERBI_2_3           26
#define WR_VITERBI_3_4           27
#define WR_VITERBI_5_6           28
#define WR_VITERBI_6_7           29
#define WR_VITERBI_7_8           30


#if 0 /* both-support */
#ifdef HDM_8513A_
#define WR_PLL_CONTROL           31
#else
#define WR_MISC_CONTROL          31
#endif
#else
#define WR_PLL_CONTROL           31 /* HDM8513A */
#define WR_MISC_CONTROL          31 /* HDM8513  */ 
#endif /* End of both-support */


/* mask pattern for read reg. 0x1b(FEC LOCK) */
#define MASK_VNODE_SYNC        0x01 
#define MASK_F_SYNC            0x02
#define MASK_VBYTE_SYNC        0x04
#define MASK_VPI_AMBIGUITY     0x08
#define MASK_VPI2_AMBIGUITY    0x10

/* bit position for read reg. 0x1b */
#define POS_VNODE_SYNC            0          
#define POS_F_SYNC                1         
#define POS_VBYTE_SYNC            2     
#define POS_VPI_AMBIGUITY         3  
#define POS_VPI2_AMBIGUITY        4 

/* read register index mnemonic */
#define RD_NAGC                   0
#define RD_STF2                   1
#define RD_STF1                   2
#define RD_STF0                   3
#define RD_PTF2                   4
#define RD_PTF1                   5
#define RD_PTF0                   6
#define RD_QPSK_LOCK              7
#define RD_WAGC                   8
#define RD_SF1                    9
#define RD_SF0                   10
#define RD_I_BASEBAND            11
#define RD_Q_BASEBAND            12
#define RD_NP 			 13
/* 14 ~ 23: ununsed */
#define RD_VITERBI_SQ            24
#define RD_VITERBI_RATE          25
#define RD_RSERR_TP              26
#define RD_FEC_LOCK              27
#define RD_CORRECTED_ERRCNT1     28
#define RD_CORRECTED_ERRCNT0     29
#define RD_UNCORRECTED_ERRCNT    30
#define RD_DEVICE_ID             31 /* added for chongju hdm8513a,  */
                                    /* current value should be 0x93 */

/* scan specific definitions *************************************************/

#define QPSK_SWEEP_MSEC_TIME( sym_rate, sym_dwell, num_step ) num_step * sym_dwell * 1000L / sym_rate + 1L

#define DEINTERLEAVER_SYMBOL_TIME 50000     /* experimental; time from qpsk lock to deinterleaver lock         */
#define RS_SYMBOL_TIME 3000                 /* experimental; time from deinterleaver lock to Reed Solomon lock */
#define INVERSION_SETTLE_SYMBOL_TIME 1000000 /* experimental; inversion settling time                           */

#define INVERSION_SETTLE_MSEC_TIME( sym_rate ) INVERSION_SETTLE_SYMBOL_TIME * 1000L / sym_rate + 1L

#define DEINTERLEAVER_MSEC_TIME( sym_rate, viterbi_auto ) ( DEINTERLEAVER_SYMBOL_TIME * 1000L / sym_rate ) * ( 1L + viterbi_auto * 5L ) + 1L

#define RS_MSEC_TIME( sym_rate ) RS_SYMBOL_TIME * 1000L / sym_rate + 1L

#define FEC_MSEC_TIME( sym_rate, viterbi_auto ) DEINTERLEAVER_MSEC_TIME( sym_rate, viterbi_auto ) + RS_MSEC_TIME( sym_rate )

#define INVERSION_MSEC_TIME( sym_rate, viterbi_auto ) ( DEINTERLEAVER_SYMBOL_TIME * ( 1L + viterbi_auto*5L ) + RS_SYMBOL_TIME + INVERSION_SETTLE_SYMBOL_TIME ) * 1000L / sym_rate + 1L

/* default write register map ************************************************/

#if 0 /* both-support */
static const unsigned char wreg_default[WR_REGSZE] =
#else
static unsigned char wreg_default[WR_REGSZE] =
#endif /* End of both-support */
{ 
  0x00, /* 0x00 Symbol timing frequency MSB                                  */
  0x00, /* 0x01 Symbol timing frequency                                      */                     
  0x00, /* 0x02 Symbol timing frequency LSB                                  */
  0x00, /* 0x03 Symbol timing loop gain control                              */
  0x00, /* 0x04 Carrier frequency MSB                                        */
  0x00, /* 0x05 Carrier frequency                                            */
  0x00, /* 0x06 Carrier frequency LSB                                        */
  0x00, /* 0x07 Carrier loop filter control for steady state                 */
  0x00, /* 0x08 Carrier loop filter control for acquisition state            */
  0x00, /* 0x09 Carrier sweep step size MSB                                  */
  0x00, /* 0x0a Carrier sweep step size LSB                                  */
  0x00, /* 0x0b Symbols per dwell MSB                                        */
  0x00, /* 0x0c Symbols per dwell LSB                                        */
  0x00, /* 0x0d Number of search frequencies                                 */
  0x96, /* 0x0e Narrow band AGC initial value                                */
  0x27, /* 0x0f Control parameters, tracking/sweep/nagc disabled..           */
  0x00, /* 0x10 Reset functions                                              */
  0xeb, /* 0x11 WAGC Control(wagc hold, invert mode, lnb hold, i2c bypass off*/
  0x30, /* 0x12 22khz tone default                                           */
  0x00, /* 0x13 Sigma Delta, don't care..                                    */

#ifdef EXTERNAL_ADC_
#if 0 /* both-support */
#ifdef HDM_8513A_
  0x2f, /* 0x14 Test Setup->external ADC, auto_set_ebit=1, lookup_frame_err=0*/
        /*      regulated clock mode enable                                  */
#else /* HDM_8513_ */
  0x07, /* 0x14 Test Setup, external ADC input enable                        */
#endif
#else
  0x2f, /* 0x14 Test Setup->external ADC, auto_set_ebit=1,lookup_frame_err=0 */
        /*      regulated clock mode enable                                  */
#endif /* End of both-support */

#else /* internal ADC */

#if 0 /* both-support */
#ifdef HDM_8513A_
  0xa8, /* 0x14 Test Setup->tristate, auto_set_ebit=1, lookup_frame_err=0    */
        /*      regulated clock mode enabled, negative edge triggering       */
        /*      of DATA_CLK                                                  */
#else /* HDM_8513_ */
  0x00, /* 0x14, Test Setup, tristate                                        */
#endif
  0xa8, /* 0x14 Test Setup->tristate, auto_set_ebit=1, lookup_frame_err=0    */
        /*      regulated clock mode enabled, negative edge triggering       */
        /*      of DATA_CLK                                                  */
#else
#endif /* End of both-support */

#endif 

  0xc2, /* 0x15 Control parameter for synchronization in Viterbi decoder 1   */
  0x01, /* 0x16 Control parameter for synchronization in Viterbi decoder 2   */
  0x01, /* 0x17 Control parameter for synchronization in Viterbi decoder 3   */
  0x00, /* 0x18 Control parameter for Viterbi & R/S decoder                  */
  0x1E, /* 0x19 Rate 1/2 select                                              */
  0x1E, /* 0x1a Rate 2/3 select                                              */
  0x28, /* 0x1b Rate 3/4 select                                              */
  0x3c, /* 0x1c Rate 5/6 select                                              */
  0x3c, /* 0x1d Rate 6/7 select                                              */
  0x3c, /* 0x1e clk_limit to zero, Rate 7/8 select                           */

#if 0 /* both-support */
#ifdef HDM_8513A_ /* Use 76MHz */
  0x53  /* 0x1f clock generation pll control, M=1, N=19 thus 76Mhz           */
#else /* HDM_8513_ : Use 60MHz */
  0x01  /* 0x1f auto_set_ebit = 1, look_up_frame_err = 0                     */
#endif
#else
  0x53  /* 0x1f clock generation pll control, M=1, N=19 thus 76Mhz           */
#endif /* End of both-support */
};

/* static variables **********************************************************/

static byte  hdm8513_wmap[WR_REGSZE]; /* hdm8513 write register mirror */
static byte  tuner_wmap[4];           /* tuner write register mirror   */

/* static function progotypes ************************************************/

static long compliment_2_value( long lData, int iBitCnt );
static long value_2_compliment( long lValue, int iBitCnt );

static boolean write_reg_8513 ( byte offset, byte *dptr, word datasze );
static boolean read_reg_8513  ( byte offset, byte *dptr, word datasze );
static boolean write_reg_tuner( byte *dptr, word datasze );
static boolean read_reg_tuner ( byte *dptr, word datasze );

static boolean tuner_init( tuner_mode_t mode, tuner_t *tuner_param );
static boolean set_viterbi_rate( EN_VITERBI_468 enViterbi );
static boolean set_symbol_per_dwell( word wSymPerDwell );
static boolean set_sweep_start_frequency( long lFrequency );
static boolean set_step_size( long *lStepSize );
static boolean set_carrier_loop( lword lwSymRate );
static boolean set_symbol_loop( lword lwSymRate );
static boolean set_symbol_rate( lword lwSymRate );
static boolean set_number_of_step( byte bNumber );
static boolean set_spectrum_invert( void );
static boolean set_spectrum_normal( void );
static boolean set_spectrum_automatic( void );
static boolean set_spectrum_toggle( void );

static boolean set_normal_operation( void );
static boolean reset_normal_operation( void );
static boolean enable_wagc( void );
static boolean disable_wagc( void );
static boolean enable_nagc_sweep_tracking( void );
static boolean disable_nagc_sweep_tracking( void );

static boolean get_tuner_lock_status( tuner_lock_t *enLock );
static boolean get_qpsk_lock_status( qpsk_lock_t *enLock );
static boolean get_wagc( int *iWagc );
static boolean get_noise_power( byte *bNoisePower );
static boolean get_sweep_frequency( long *lSweep );
static boolean get_tracking_frequency( long *lTracking );
static boolean get_symbol_frequency( long *lSymbol );

static lword qpsk_step_size( lword lwSymbol );
static word symbol_per_dwell( lword lwSymbol );
static lword sub_range_size( lword lwSymbolRate );

static boolean scan_subrange( qpsk_t *stQpsk );
static boolean scan_zig_zag( tuning_t *stTuning );
static boolean correct_drift_frequency( tuning_t *stTuning );
static boolean stop_search_8513( void );

/* ------------------------------------------------------------------------- */
/* function name: write_reg_8513                                             */
/*                                                                           */
/* description: writes value to hdm8513 register                             */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean write_reg_8513( byte offset, byte *dptr, word datasze )
{

#if 0
	aint i;
	boolean ferror = false;

	for(i = 0 ; i < datasze ; i++ ){
		printf("write_reg_8513[%s]chip:0x%x,offset=0x%x,data[%d]=0x%x\n",gg_dev_name,gg_qpsk_addr,offset,i, dptr[i]);
		//getchar( );
		ferror |= sm_send_with_offset( gg_dev_name, gg_qpsk_addr, offset, &dptr[i], 1 );
	}
	return  ferror;
#else
	 return (sm_send_with_offset( gg_dev_name, gg_qpsk_addr, offset, dptr, (int)datasze ) );
#endif
	

}

/* ------------------------------------------------------------------------- */
/* function name: read_reg_8513                                              */
/*                                                                           */
/* description: reads value from hdm8513 register                            */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean read_reg_8513( byte offset, byte *dptr, word datasze )
{
//	printf("read_reg_8513[%s]chip:0x%x,offset=0x%x,count=%d\n",gg_dev_name,gg_qpsk_addr,offset,datasze);
	return ( sm_read_with_offset( gg_dev_name, gg_qpsk_addr, offset, dptr, (int)datasze ) );

}

/* ------------------------------------------------------------------------- */
/* function name: write_reg_tuner                                            */
/*                                                                           */
/* description: writes value to tuner register                               */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean write_reg_tuner( byte *dptr, word datasze )
{
	boolean ferror = false;

	/* i2c bypass ON to access to tuner */
	hdm8513_wmap[WR_CTRL_WAGC] |= 1 << POS_IIC_BYPASS & MASK_IIC_BYPASS;
	ferror = write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );

	//printf("write_reg_tuner[%s]chip:0x%x\n",gg_dev_name,gg_qpsk_addr);
	//getchar( );
	ferror |= sm_send_without_offset( gg_dev_name, IIC_TUNER_ADDR, dptr, (int)datasze );

	/* i2c bypass OFF to block possible noise caused by host interface*/
	hdm8513_wmap[WR_CTRL_WAGC] &= ~(1 << POS_IIC_BYPASS & MASK_IIC_BYPASS);
	ferror |= write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: read_reg_tuner                                             */
/*                                                                           */
/* description: reads value from tuner register                              */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean read_reg_tuner( byte *dptr, word datasze )
{
	boolean ferror = false;

	/* i2c bypass ON to access to tuner */
	hdm8513_wmap[WR_CTRL_WAGC] |= 1 << POS_IIC_BYPASS & MASK_IIC_BYPASS;
	ferror = write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );

	//printf("read_reg_tuner[%s]chip:0x%x\n",gg_dev_name,gg_qpsk_addr);
	//getchar( );
	ferror |= sm_read_without_offset( gg_dev_name, IIC_TUNER_ADDR, dptr, (int)datasze );

	/* i2c bypass OFF to block possible noise caused by host interface*/
	hdm8513_wmap[WR_CTRL_WAGC] &= ~(1 << POS_IIC_BYPASS & MASK_IIC_BYPASS);
	ferror |= write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );

	return ferror;
	}

/* ------------------------------------------------------------------------- */
/* function name: set_symbol_rate                                            */
/*                                                                           */
/* description: write symbol rate to a register related                      */
/*                                                                           */
/* date: 1998. 9. 23.                                                        */
/* Note: 1. unit of lwSymRate is Hz                                          */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_symbol_rate( lword lwSymRate )
{
	boolean ferror = false;
	lword lwRate = lwSymRate;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	char sbuf[200];
#endif

	lwRate = (lword) ( (lwRate / gg_clock) * SCALE_2_20);

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_symbol_rate : lwSymRate: %d\n", lwSymRate );
	printf( sbuf );
	/* elcomski */
	sprintf( sbuf , " set_symbol_rate : lwRate: %d\n", lwRate );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_STF2] = (byte)( lwRate >> 12 & 0xff );
	hdm8513_wmap[WR_STF1] = (byte)( lwRate >> 4 & 0xff );
	hdm8513_wmap[WR_STF0] = (byte)( lwRate << 4 & 0xf0 );
	ferror |= write_reg_8513( WR_STF2, &hdm8513_wmap[WR_STF2], 3 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_symbol_loop                                            */
/*                                                                           */
/* description: write symbol tracking loop gain,                             */
/*              gain parameter is a function of symbol rate.                 */
/*                                                                           */
/* date: 1998. 9. 23.                                                        */
/* Note: 1. unit of lwSymRate is Hz                                          */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_symbol_loop( lword lwSymRate )
{
	boolean ferror = false;
	lword lwRate = lwSymRate;
	double fSymRate;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	char sbuf[200];
#endif

	fSymRate = lwRate /gg_clock; /* for later use */

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_symbol_loop : lwSymRate: %d\n", lwSymRate );
	printf( sbuf );
	/* elcomski */
	sprintf( sbuf , " set_symbol_loop : fSymRate: %f\n", fSymRate );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_STLGC] = ( fSymRate > 0.4   ) ? 0xb6 : 0xa5;
	hdm8513_wmap[WR_STLGC] = ( fSymRate < 0.2   ) ? 0x94 : hdm8513_wmap[WR_STLGC];
	hdm8513_wmap[WR_STLGC] = ( fSymRate < 0.1   ) ? 0x83 : hdm8513_wmap[WR_STLGC];
	hdm8513_wmap[WR_STLGC] = ( fSymRate < 0.05  ) ? 0x72 : hdm8513_wmap[WR_STLGC];
	hdm8513_wmap[WR_STLGC] = ( fSymRate < 0.025 ) ? 0x61 : hdm8513_wmap[WR_STLGC];
	ferror |= write_reg_8513( WR_STLGC, &hdm8513_wmap[WR_STLGC], 1 );

  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_carrier_loop                                           */
/*                                                                           */
/* description: write carrier tracking loop gain,                            */
/*              gain parameters are function of symbol rate.                 */
/*                                                                           */
/* date: 1998. 9. 23.                                                        */
/* Note: 1. unit of lwSymRate is Hz                                          */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_carrier_loop( lword lwSymRate )
{
	boolean ferror = false;
	lword lwRate = lwSymRate;
	double fSymRate;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	char sbuf[200];
#endif

	fSymRate = lwRate /gg_clock; /* for later use */


#ifdef ZIF_ /* fix to 0x55.. temporally....... */

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_carrier_loop : lwSymRate: %d\n", lwSymRate );
	printf( sbuf );
	/* elcomski */
	sprintf( sbuf , " set_carrier_loop : fSymRate: %f\n", fSymRate );
	printf( sbuf );
#endif

	/* steady state carrier loop filter gain setting */
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate > 0.4   ) ? 0x55 : 0x55;
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.2   ) ? 0x55 : hdm8513_wmap[WR_CLFC_STEADY];
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.1   ) ? 0x55 : hdm8513_wmap[WR_CLFC_STEADY];
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.05  ) ? 0x55 : hdm8513_wmap[WR_CLFC_STEADY];
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.025 ) ? 0x55 : hdm8513_wmap[WR_CLFC_STEADY];
#else
	/* steady state carrier loop filter gain setting */
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate > 0.4   ) ? 0x47 : 0x47;
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.2   ) ? 0x47 : hdm8513_wmap[WR_CLFC_STEADY];
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.1   ) ? 0x15 : hdm8513_wmap[WR_CLFC_STEADY];
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.05  ) ? 0x15 : hdm8513_wmap[WR_CLFC_STEADY];
	hdm8513_wmap[WR_CLFC_STEADY] = ( fSymRate < 0.025 ) ? 0x15 : hdm8513_wmap[WR_CLFC_STEADY];
#endif

	/* acquisition state carrier loop filter gain setting */
	hdm8513_wmap[WR_CLFC_ACQUISITION] = ( fSymRate > 0.4   ) ? 0xa8 : 0xa8;
	hdm8513_wmap[WR_CLFC_ACQUISITION] = ( fSymRate < 0.2   ) ? 0xa8 : hdm8513_wmap[WR_CLFC_ACQUISITION];
	hdm8513_wmap[WR_CLFC_ACQUISITION] = ( fSymRate < 0.1   ) ? 0x77 : hdm8513_wmap[WR_CLFC_ACQUISITION];
	hdm8513_wmap[WR_CLFC_ACQUISITION] = ( fSymRate < 0.05  ) ? 0x56 : hdm8513_wmap[WR_CLFC_ACQUISITION];
	hdm8513_wmap[WR_CLFC_ACQUISITION] = ( fSymRate < 0.025 ) ? 0x56 : hdm8513_wmap[WR_CLFC_ACQUISITION];

	ferror |= write_reg_8513( WR_CLFC_STEADY, &hdm8513_wmap[WR_CLFC_STEADY], 1 );
	ferror |= write_reg_8513( WR_CLFC_ACQUISITION, &hdm8513_wmap[WR_CLFC_ACQUISITION], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_step_size                                              */
/*                                                                           */
/* description: convert step size[Hz] to formatted data                      */
/*              and write its result to the write reg. & reg. mirror         */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_step_size( long *lStepSize )
{
	boolean ferror = false;
	long lStep = *lStepSize, lStep1, lStep2;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[200];
#endif

	lStep1 = (long) ((lStep / gg_clock) * SCALE_2_16 );
	*lStepSize = (long)(lStep1 / SCALE_2_16 * gg_clock);
	lStep2 = value_2_compliment( lStep1, 16 );

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_step_szie : lStep2 : %d\n", lStep2 );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_STEP1] = (byte) (lStep2 >> 8 & 0xff);
	hdm8513_wmap[WR_STEP0] = (byte) (lStep2 & 0xff);
	ferror |= write_reg_8513( WR_STEP1, &hdm8513_wmap[WR_STEP1], 2 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_sweep_start_frequency                                  */
/*                                                                           */
/* description: convert sweep start frequency[Hz] to formatted data          */
/*              and write its result to the write reg. & mirror.             */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_sweep_start_frequency( long lFrequency )
{
	boolean ferror = false;
	long lFrq = lFrequency, lFrq1, lFrq2;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[200];
#endif
	
	lFrq1 = (long)((lFrq / gg_clock) * SCALE_2_20 );
	lFrq2 = value_2_compliment( lFrq1, 20 );

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_sweep_start_frequecy : lFrq2 : %d\n", lFrq2 );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_CF2] = (byte)(lFrq2 >> 12 & 0xff );
	hdm8513_wmap[WR_CF1] = (byte)(lFrq2 >>  4 & 0xff );
	hdm8513_wmap[WR_CF0] = (byte)(lFrq2 <<  4 & 0xf0 );
	ferror |= write_reg_8513( WR_CF2, &hdm8513_wmap[WR_CF2], 3 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: get_symbol_frequency                                       */
/*                                                                           */
/* description: get/convert symbol tracking frequency accumulator data to    */
/*              a comprehensive value[SPS].                                  */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean get_symbol_frequency( long *lSymbol )
{
  boolean ferror = false;
  byte bSTFA[3];
  long lSym, lSym1, lSym2;

  ferror |= read_reg_8513( RD_STF2, &bSTFA[0], 3 );
  lSym  = bSTFA[0]; lSym <<= 8;
  lSym |= bSTFA[1]; lSym <<= 8;
  lSym |= bSTFA[2]; lSym >>= 4;
  lSym1 = compliment_2_value( lSym, 20 );
  lSym2 = (long)((lSym1 / SCALE_2_20) * gg_clock);

  *lSymbol = lSym2;
  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: get_tracking_frequency                                     */
/*                                                                           */
/* description: get/convert phase tracking frequency accumulator data to     */
/*              a comprehensive value[hz].                                   */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean get_tracking_frequency( long *lTracking )
{
  boolean ferror = false;
  byte bPTFA[3];
  long lTrac, lTrac1, lTrac2;

  ferror |= read_reg_8513( RD_PTF2, &bPTFA[0], 3 );
  lTrac  = bPTFA[0]; lTrac <<= 8;
  lTrac |= bPTFA[1]; lTrac <<= 8;
  lTrac |= bPTFA[2]; lTrac >>= 4;
  lTrac1 = compliment_2_value( lTrac, 20 );
  lTrac2 = (long)((lTrac1 / SCALE_2_20) * gg_clock);

  *lTracking = lTrac2;
  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: get_sweep_frequency                                        */
/*                                                                           */
/* description: get/convert sweep frequency accumulator data to              */
/*              a comprehensive value[hz].                                   */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean get_sweep_frequency( long *lSweep )
{
  byte bSF[2];
  long lSw = 0, lSw1, lSw2;
  boolean ferror = false;

  ferror |= read_reg_8513( RD_SF1, &bSF[0], 2 );
  lSw  = bSF[0]; lSw <<= 8;
  lSw |= bSF[1];
  lSw1 = compliment_2_value( lSw, 16 );
  lSw2 = (long)((lSw1 / SCALE_2_16) * gg_clock);

  *lSweep = lSw2;
  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: get_wagc                                                   */
/*                                                                           */
/* description: Gets value from wide band agc register                       */
/*              and convert it to a comprehensive value.                     */
/*              The lower signal amplitude, the larger wagc value will be    */
/*              returned.                                                    */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean get_wagc( int *iWagc )
{
  boolean ferror = false;
  byte bVal = 0;

  ferror |= read_reg_8513( RD_WAGC, &bVal, 1 );
  if( !ferror )
    *iWagc = compliment_2_value( bVal, 8 );

  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: value_2_compliment                                         */
/*                                                                           */
/* description: Utility function that convert a comprehensive value          */
/*              to a compliment data.                                        */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1. lData is MSB/msb first and the bit value is valid with size      */
/*          of "iBitCnt" from the start of LSB/lsb.                          */
/*       2. if positive data, no conversion is performed.                    */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static long value_2_compliment( long lValue, int iBitCnt )
{
  int iCnt = iBitCnt;
  long lVal = lValue, lMask = 0;

  if( lVal < 0 )
  {
    while( iCnt-- > 0 )
    {
      lMask |= 1; if( iCnt != 0 ) lMask <<= 1;
    }
    lVal *= -1;
    lVal ^= 0xffffffff;
    lVal += 1;
    lVal &= lMask;
  }
  return lVal;
}

/* ------------------------------------------------------------------------- */
/* function name: compliment_2_value                                         */
/*                                                                           */
/* description: Utility function that convert 2's complement data            */
/*              to a comprehensive value.                                    */
/*                                                                           */
/* date: 1998. 7. 25.                                                        */
/* Note: 1. lData is MSB/msb first and the bit value is valid with size      */
/*          of "iBitCnt" from the start of LSB/lsb.                          */
/*       2. if positive data, no conversion is performed.                    */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static long compliment_2_value( long lData, int iBitCnt )
{
  int iCnt = iBitCnt;
  long lValue = lData, lMask=0;

  if( iCnt > 0 && iCnt <= 32 )
  {
    while( iCnt-- > 0 )
    {
      lMask |= 1; if( iCnt != 0 ) lMask <<= 1;
    }

    if( lValue & (1 << (iBitCnt-1)) ) /* minus value */
    {
      lValue ^= 0xffffffff; lValue += 1;
      lValue &= lMask; lValue *= -1;
    }
  }
  return lValue;
}

/* ------------------------------------------------------------------------- */
/* function name: set_symbol_per_dwell                                       */
/*                                                                           */
/* description: write the number of symbol per dwell                         */
/*              dwell means the basic unit for determining lock statitics    */
/*                                                                           */
/* date: 1998. 9. 23.                                                        */
/* Note: 1. unit of wSymPerDwell is just the symbol number                   */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_symbol_per_dwell( word wSymPerDwell )
{
	boolean ferror = false;
	word wSpd = wSymPerDwell;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	char sbuf[200];

	/* elcomski */
	sprintf( sbuf , " set_symbol_per_dwell : wSpd : %d\n", wSpd );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_SPD1] = wSpd >> 8 & 0xff;
	hdm8513_wmap[WR_SPD0] = wSpd & 0xff;
	ferror |= write_reg_8513( WR_SPD1, &hdm8513_wmap[WR_SPD1], 2 );
	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_number_of_step                                         */
/*                                                                           */
/* description: write the number of search frequency to the reg. related.    */
/*                                                                           */
/* date: 1998. 9. 23.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_number_of_step( byte bNumber )
{
	boolean ferror = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[200];

	/* elcomski */
	sprintf( sbuf , " set_number_of_step : nNumber : %d\n", bNumber );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_NSTEP] = bNumber;
	ferror |= write_reg_8513( WR_NSTEP, &hdm8513_wmap[WR_NSTEP], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_viterbi_rate                                           */
/*                                                                           */
/* description: write the the Viterbi rate                                   */
/*                                                                           */
/* date: 1998. 9. 23.                                                        */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_viterbi_rate( EN_VITERBI_468 enViterbi )
{
	boolean ferror = false;
	EN_VITERBI_468 enV468 =  enViterbi;
	EN_VITERBI_8513 enV8513 = VFIXED_1_2;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	char sbuf[200];

	/* elcomski */
	sprintf( sbuf , " set_viterbi_rate : enV468 : %d\n", enV468 );
	printf( sbuf );
#endif
	
	if( enV468 == FEC_AUTOMATIC ){
		hdm8513_wmap[WR_VITERBI_RS_SETUP] |= 1 << POS_VITERBI_AUTO & MASK_VITERBI_AUTO;
	}else{
		hdm8513_wmap[WR_VITERBI_RS_SETUP] &= ~(1 << POS_VITERBI_AUTO & MASK_VITERBI_AUTO);
	}

	switch( enV468 )
	{
		case FEC_INNER_1_2: enV8513 = VFIXED_1_2; break;
		case FEC_INNER_2_3: enV8513 = VFIXED_2_3; break;
		case FEC_INNER_3_4: enV8513 = VFIXED_3_4; break;
		case FEC_INNER_5_6: enV8513 = VFIXED_5_6; break;
		case FEC_INNER_7_8: enV8513 = VFIXED_7_8; break;
		default: break;
	}

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_viterbi_rate : enV8513 : %d\n", enV8513 );
	printf( sbuf );
#endif

	/* clear previous viterbi rate & set newer one */
	hdm8513_wmap[WR_VITERBI_RS_SETUP] &= ~MASK_VITERBI_RATE;
	hdm8513_wmap[WR_VITERBI_RS_SETUP] |= (byte)enV8513 << POS_VITERBI_RATE & MASK_VITERBI_RATE;
	ferror |= write_reg_8513( WR_VITERBI_RS_SETUP, &hdm8513_wmap[WR_VITERBI_RS_SETUP], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: tuner_init                                                 */
/*                                                                           */
/* description: write frequency to tuner so as to catch up the signal with   */
/*              tuner pll                                                    */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note: 1. note of difference between init mode and update mode             */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean tuner_init( tuner_mode_t mode, tuner_t *tuner_param )
{
	boolean ferror = false;
	byte bTunerStatus=0;

	struct timeval ulStamp;

	word wProgrammableDivider;
	unsigned long temp;

	tuner_param->stStatus.enLock = TUNER_PLL_UNLOCKED;

	if( mode == INIT )
	{
		/* tuning with frequency */
		temp =tuner_param->lwFrequency;
		wProgrammableDivider = tuner_param->stStatus.wProgrammableDivider =
		(word) ((tuner_param->lwFrequency+TUNER_IF)/TUNER_ACTUAL_STEP);
	}
	else /* update mode */
	{
		/* calculating programmable divider with prescaler enable */
		wProgrammableDivider =
		(int)tuner_param->stStatus.wProgrammableDivider + tuner_param->stStatus.iStepIncrement;
	}

	tuner_wmap[0] = (byte)( wProgrammableDivider>>8 & 0x7f ) ;
	tuner_wmap[1] = (byte)( wProgrammableDivider & 0xff );
	tuner_wmap[2] = 0x80 ;
	tuner_wmap[2] |= TUNER_REF_RATIO;

#if 1/* no saw filter present */
	/* select saw filter bandwidth */
	switch( tuner_param->enSawBW )
	{
		case TSAW_SINGLE: tuner_wmap[3] = 0x00; break;
		case TSAW_DUAL_1: tuner_wmap[3] = 0x02; break;
		case TSAW_DUAL_2: tuner_wmap[3] = 0x04; break;
		default: tuner_wmap[3] = 0x04; break;
	}
#endif

	tuner_wmap[3] = 0x80; /* charge pump current, c1=1, c0=0 */
	if( tuner_param->lwFrequency <= 1500000000L )
		tuner_wmap[3] |= 0x02; /* p1 select */
	

	ferror |= write_reg_tuner( &tuner_wmap[0], 4 );
	if( !ferror )
	{
		STAMP( &ulStamp );
		do
		{
			FWAIT(1);
			ferror |= read_reg_tuner( &bTunerStatus, 1 );
			if( bTunerStatus & TUNER_PLL_LOCK ){
				tuner_param->stStatus.enLock = TUNER_PLL_LOCKED;
				break;
			}
		} while ( !ferror && !ELAPSED( ulStamp, MAX_PLL_MSEC_TIME ) );
	}
	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: stop_search_8513                                           */
/*                                                                           */
/* description: write parameters to hdm8513 so as to stop tracking and 		 */
/*              output bitstream as well.                                    */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean stop_search_8513( void )
{
  boolean ferror = false;

  /* stops nagc/frequency tracking/symbol tracking/frequency sweeping */
  ferror |= disable_nagc_sweep_tracking();

  /* holds wagc */
  ferror |= disable_wagc();

  /* disable normal operation with reset register */
  ferror |= reset_normal_operation();

  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_normal_operation                                       */
/*                                                                           */
/* description: resume normal operation with reset register.                 */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_normal_operation( void )
{
	boolean ferror = false;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	char sbuf[200];
#endif


	hdm8513_wmap[WR_RESET_PARAM] |=  ( 1 << POS_RESET_STFA & MASK_RESET_STFA );
	hdm8513_wmap[WR_RESET_PARAM] |=  ( 1 << POS_RESET_CPTFA & MASK_RESET_CPTFA );
	hdm8513_wmap[WR_RESET_PARAM] |=  ( 1 << POS_RESET_WAGCA & MASK_RESET_WAGCA );
	hdm8513_wmap[WR_RESET_PARAM] |=  ( 1 << POS_RESET_NAGCA & MASK_RESET_NAGCA );
	hdm8513_wmap[WR_RESET_PARAM] |=  ( 1 << POS_RESET_CSWEEP & MASK_RESET_CSWEEP );
	hdm8513_wmap[WR_RESET_PARAM] |=  ( 1 << POS_RESET_ERRCNT & MASK_RESET_ERRCNT );
	ferror |= write_reg_8513( WR_RESET_PARAM, &hdm8513_wmap[WR_RESET_PARAM], 1 );

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_normal_operation \n" );
	printf( sbuf );
#endif

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: reset_normal_operation                                     */
/*                                                                           */
/* description: disable normal operation with reset register.                */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean reset_normal_operation( void )
{
  boolean ferror = false;

  hdm8513_wmap[WR_RESET_PARAM] &= ~( 1 << POS_RESET_STFA & MASK_RESET_STFA );
  hdm8513_wmap[WR_RESET_PARAM] &= ~( 1 << POS_RESET_CPTFA & MASK_RESET_CPTFA );
  hdm8513_wmap[WR_RESET_PARAM] &= ~( 1 << POS_RESET_WAGCA & MASK_RESET_WAGCA );
  hdm8513_wmap[WR_RESET_PARAM] &= ~( 1 << POS_RESET_NAGCA & MASK_RESET_NAGCA );
  hdm8513_wmap[WR_RESET_PARAM] &= ~( 1 << POS_RESET_CSWEEP & MASK_RESET_CSWEEP );
  hdm8513_wmap[WR_RESET_PARAM] &= ~( 1 << POS_RESET_ERRCNT & MASK_RESET_ERRCNT );
  ferror |= write_reg_8513( WR_RESET_PARAM, &hdm8513_wmap[WR_RESET_PARAM], 1 );

  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: enable_wagc                                                */
/*                                                                           */
/* description: make wide band agc circuit active.                           */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean enable_wagc( void )
{

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[200];
#endif

	boolean ferror = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " enable_wagc \n" );
	printf( sbuf );
#endif
	
	hdm8513_wmap[WR_CTRL_WAGC] |= ( 1 << POS_WAGC_HOLD & MASK_WAGC_HOLD );
	ferror |= write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: enable_nagc_sweep_tracking                                 */
/*                                                                           */
/* description: resume symbol tracking/carrier tracking/sweeping/nagc        */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean enable_nagc_sweep_tracking( void )
{

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[200];
#endif

	boolean ferror = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " enable_nagc_sweep_tracking \n" );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_CTRL_PARAM] |=  ( 1 << POS_STRACK & MASK_STRACK ); 
	hdm8513_wmap[WR_CTRL_PARAM] |=  ( 1 << POS_CTRACK & MASK_CTRACK );
	hdm8513_wmap[WR_CTRL_PARAM] |=  ( 1 << POS_NAGC_MODE1 & MASK_NAGC_MODE1 );
	hdm8513_wmap[WR_CTRL_PARAM] &= ~( 1 << POS_SWEEP_HOLD & MASK_SWEEP_HOLD );
	ferror |= write_reg_8513( WR_CTRL_PARAM, &hdm8513_wmap[WR_CTRL_PARAM], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: disable_wagc                                               */
/*                                                                           */
/* description: make wide band agc circuit inactive.                         */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean disable_wagc( void )
{

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski  */
	char sbuf[100];
#endif

	boolean ferror = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " disable_wagc \n" );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_CTRL_WAGC] &= ~( 1 << POS_WAGC_HOLD & MASK_WAGC_HOLD );
	ferror |= write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: disable_nagc_sweep_tracking                                */
/*                                                                           */
/* description: stops symbol tracking/carrier tracking/sweeping/nagc         */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean disable_nagc_sweep_tracking( void )
{

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski  */
	char sbuf[100];
#endif

	boolean ferror = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " disable_nagc_sweep_tracking \n" );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_CTRL_PARAM] &= ~( 1 << POS_STRACK & MASK_STRACK ); 
	hdm8513_wmap[WR_CTRL_PARAM] &= ~( 1 << POS_CTRACK & MASK_CTRACK );
	hdm8513_wmap[WR_CTRL_PARAM] &= ~( 1 << POS_NAGC_MODE1 & MASK_NAGC_MODE1 );
	hdm8513_wmap[WR_CTRL_PARAM] |=  ( 1 << POS_SWEEP_HOLD & MASK_SWEEP_HOLD );
	ferror |= write_reg_8513( WR_CTRL_PARAM, &hdm8513_wmap[WR_CTRL_PARAM], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_spectrum_invert                                        */
/*                                                                           */
/* description: set qpsk to spectrum invert mode                             */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note: 1. complement the in-phase channel input                            */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_spectrum_invert( void )
{
#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[100];
#endif
	boolean ferror = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " set_spectrum_invert\n" );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_CTRL_PARAM] &= ~( 1 << POS_SI & MASK_SI );
	ferror = write_reg_8513( WR_CTRL_PARAM, &hdm8513_wmap[WR_CTRL_PARAM], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_spectrum_normal                                        */
/*                                                                           */
/* description: set qpsk to spectrum normal mode                             */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_spectrum_normal( void )
{
	boolean ferror = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[100];
	
	/* elcomski */
	sprintf( sbuf , " set_spectrum_normal \n" );
	printf( sbuf );
#endif

	hdm8513_wmap[WR_CTRL_PARAM] |= ( 1 << POS_SI & MASK_SI );
	ferror = write_reg_8513( WR_CTRL_PARAM, &hdm8513_wmap[WR_CTRL_PARAM], 1 );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_spectrum_toggle                                        */
/*                                                                           */
/* description: set qpsk to spectrum toggle( opposite the current mode )     */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_spectrum_toggle( void )
{
  boolean ferror = false;

  hdm8513_wmap[WR_CTRL_PARAM] ^= ( 1 << POS_SI & MASK_SI );
  ferror = write_reg_8513( WR_CTRL_PARAM, &hdm8513_wmap[WR_CTRL_PARAM], 1 );

  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: set_spectrum_automatic                                     */
/*                                                                           */
/* description: set qpsk to spectrum automatic mode                          */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note: 1. using this function is prohibited.. for later use..              */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean set_spectrum_automatic( void )
{
  boolean ferror = false;

  hdm8513_wmap[WR_CTRL_PARAM] |= ( 1 << POS_AUTOMATIC_SI & MASK_AUTOMATIC_SI );
  ferror = write_reg_8513( WR_CTRL_PARAM, &hdm8513_wmap[WR_CTRL_PARAM], 1 );

  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: get_tuner_lock_status                                      */
/*                                                                           */
/* description: reads various lock flags and result in *enLock               */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean get_tuner_lock_status( tuner_lock_t *enLock )
{
	boolean ferror = false;
	byte bTlock;

	*enLock = TUNER_PLL_UNLOCKED;
	/* read tuner lock status */
	ferror |= read_reg_tuner( &bTlock, 1 );
	if( bTlock & TUNER_PLL_LOCK ) *enLock = TUNER_PLL_LOCKED;


	return ferror; 
}

/* ------------------------------------------------------------------------- */
/* function name: get_qpsk_lock_status                                       */
/*                                                                           */
/* description: reads various lock flags and result in *enLock               */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean get_qpsk_lock_status( qpsk_lock_t *enLock )
{

#ifdef QPSK_LOCK_DEBUG 
	/* elcomski */
	char sbuf[100];
#endif 

	boolean ferror = false;
	byte bQlock=0, bFlock=0, bTmp;
	int i;
	
  
	/* read qpsk lock status */
	for( i = 0 ; i < 3 ; i++ )
	{
		ferror = read_reg_8513( RD_QPSK_LOCK, &bTmp, 1 );
		bQlock |= bTmp;

#ifdef QPSK_LOCK_DEBUG 
		/* elcomski */
		sprintf(sbuf, " get_qpsk_lock_status : qpsk bTmp = 0x%x \n", bTmp);
		printf(sbuf);
#endif

		/* read FEC lock status */
		ferror |= read_reg_8513( RD_FEC_LOCK, &bTmp, 1 );
		bFlock |= bTmp;
		
#ifdef QPSK_LOCK_DEBUG 
		/* elcomski */
		sprintf(sbuf, " get_qpsk_lock_status : FEC bTmp = 0x%x \n", bTmp );
		printf(sbuf);

		/* elcomski */
		sprintf(sbuf, " get_qpsk_lock_status : ferror = 0x%x \n", ferror );
		printf(sbuf);
#endif
		if( ferror ) break;
		FWAIT(1);
	}

	
#ifdef QPSK_LOCK_DEBUG 
	/* elcomski */
	sprintf(sbuf, " get_qpsk_lock_status : Qpsk lock status: bQlock = 0x%x \n", bQlock );
	printf(sbuf);
	/* elcomski */
	sprintf(sbuf, " get_qpsk_lock_status : FEC lock status: bQlock = 0x%x \n", bFlock );
	printf(sbuf);
#endif

	if( bQlock & 0x01 )
	{
		if( (bFlock & MASK_F_SYNC) && (bFlock & MASK_VNODE_SYNC) && (bFlock & MASK_VBYTE_SYNC) )
		{
			*enLock = ALL_LOCKED;
		}
		else
		{
			*enLock = QPSK_LOCKED_FEC_UNLOCKED;
		}
	}
	else
	{
		*enLock = QPSK_UNLOCKED;
	}

#ifdef QPSK_LOCK_DEBUG 
	/* elcomski */
	sprintf(sbuf," get_qpsk_lock_status: enLock = %d \n", *enLock );
	printf(sbuf);
#endif

  return ferror; 
}

/* ------------------------------------------------------------------------- */
/* function name: get_noise_power                                            */
/*                                                                           */
/* description: get the noise power from the register related.               */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean get_noise_power( byte *bNoisePower )
{
  boolean ferror = false;

  ferror |= read_reg_8513( RD_NP, bNoisePower, 1 );
  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: qpsk_step_size                                             */
/*                                                                           */
/* description: calculate qpsk step size to accomodate proper carrier loop   */
/*              filter acquisition range.                                    */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* ------------------------------------------------------------------------- */
static lword qpsk_step_size( lword lwSymbol )
{
  long lStep = lwSymbol / 1000;
  /* if symbol rate is 10Msps, the step size is 10khz */

  return ( lStep );
}

/* ------------------------------------------------------------------------- */
/* function name: symbol_per_dwell                                           */
/*                                                                           */
/* description: calculate symbol per dwell to accomodate proper carrier loop */
/*              filter acquisition range.                                    */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note: 1.                                                                  */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static word symbol_per_dwell( lword lwSymbol )
{
  return 1024;
}

/* ------------------------------------------------------------------------- */
/* function name: sub_range_size                                             */
/*                                                                           */
/* description: determine subrange size which is a multiple of tuner         */
/*              step size.                                                   */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note: 1. Subrange size should be less than 4/symbol rate to avoid         */
/*          false lock, we choose multiples of tuner step size which         */
/*          is nearest to 8/symbol rate.                                     */
/*       2. In low symbol rate application, large tuner step size            */
/*          might not overcome false lock condition                          */
/*          for example, tuner step size larger than 250khz in 1Msps         */
/*          is prohibited.                                                   */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static lword sub_range_size( lword lwSymbolRate )
{
  lword lwTmp;
  double fTmp;

  fTmp = lwSymbolRate / 8.;
  lwTmp = (lword)(fTmp / TUNER_ACTUAL_STEP);
  if( lwTmp == 0 ) lwTmp = 1;
  /* return step count, instead of real frequency
  return ( (lword) ( lwTmp * TUNER_ACTUAL_STEP ) );
  */
  return lwTmp;
}

/* ------------------------------------------------------------------------- */
/* function name: scan_subrange                                              */
/*                                                                           */
/* description: scan 1 sub range interval for a calculated interval          */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean scan_subrange( qpsk_t *stQpsk )
{
	boolean ferror = false, fViterbi = false;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	char sbuf[200];
#endif

	struct timeval ulStamp;
	long int ulTmp;

	long lNumStep, lStep, lF;
	long lF1 = stQpsk->stStatus.lSweepStartFrequency, lF2 = stQpsk->stStatus.lSweepEndFrequency;

	lword lwSymbol = stQpsk->lwSymbol;
	EN_VITERBI_468 enViterbi = stQpsk->enViterbi;
	qpsk_lock_t enQlock;

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , "\n================ (START) scan_subrange ================\n");
	printf( sbuf );
#endif

	/* to determine wait time */
	if( enViterbi == FEC_AUTOMATIC ) fViterbi = true;

	/* determine qpsk step size */
	lStep = qpsk_step_size( lwSymbol );

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " lwSymbol : %d\n", lwSymbol );
	printf( sbuf );
	/* elcomski */
	sprintf( sbuf , " lStep  : %d\n", lStep );
	printf( sbuf );
#endif

	/* calculate number of step in subrange scan */
	if( (lF = lF2 - lF1) < 0 ){
		lStep *= -1L;
	}

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " lF  : %d\n", lF );
	printf( sbuf );
#endif

	/* set step size */
	if( !ferror ) ferror = set_step_size( &lStep );

	lNumStep = lF / lStep /* + 1 */; /* 0 corresponds to one sweep point */

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , " lNumStep  : %d\n", lNumStep );
	printf( sbuf );
#endif

	/* set sweep start frequency */
	if( !ferror ) ferror = set_sweep_start_frequency( lF1 );

	/* set number of search frequency */
	if( !ferror ) ferror = set_number_of_step( (byte)lNumStep );

	if( !ferror ) ferror = set_spectrum_normal();
	stQpsk->stStatus.enSpectrum = NORMAL_SPECTRUM;

	/* resume wagc/nagc/sweep/tracking */
	if( !ferror ) ferror = enable_wagc();

	if( !ferror ) ferror = enable_nagc_sweep_tracking();


#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , "check I: error  : %d\n", ferror );
	printf( sbuf );
#endif

	/* TTest Start */
#if 0
	getchar( );
	do{
		read_reg_8513( RD_QPSK_LOCK, &Temp, 1 );
		sprintf(sbuf, "%d", Temp); 
		printf(sbuf);
	}while( Temp != 0x01 );
	/* Test End */
#endif
	if( !ferror )
	{
		STAMP( &ulStamp );
		do
		{
			FWAIT( 1 );
			if( !( ferror = get_qpsk_lock_status( &enQlock ) ) )
			{
#ifdef SCAN_SUBRANGE_DEBUG
				/* elcomski */
				sprintf( sbuf , "check II: error  : %d\n", ferror );
				printf( sbuf );
#endif

				if( enQlock == QPSK_LOCKED_FEC_UNLOCKED )
				{
					ulTmp = FEC_MSEC_TIME( lwSymbol, fViterbi );
#ifdef SCAN_SUBRANGE_DEBUG
					/* elcomski */
					sprintf( sbuf , "FWAIT->ulTmp  : %d\n", ulTmp );
					printf( sbuf );
#endif

					/* wait for valid data & sync detection */
					FWAIT( ulTmp );

					if( !( ferror = get_qpsk_lock_status( &enQlock ) ) )
					{
						if( enQlock == ALL_LOCKED )
						{	
							stQpsk->stStatus.enSpectrum = NORMAL_SPECTRUM;
						}
						else /* qpsk lock ok, but fec not locked */
						{
							if( !( ferror = set_spectrum_invert() ) )
							{

								ulTmp = INVERSION_MSEC_TIME( lwSymbol, fViterbi );

#ifdef SCAN_SUBRANGE_DEBUG
								/* elcomski */
								sprintf( sbuf , "FWAIT->ulTmp  : %d\n", ulTmp );
								printf( sbuf );
#endif

								/* wait for settling spectrum inversioned data */
								FWAIT( ulTmp );

								if( !( ferror = get_qpsk_lock_status( &enQlock ) ) )
								{
									if( enQlock == ALL_LOCKED )
									{
										stQpsk->stStatus.enSpectrum = INVERTED_SPECTRUM;
									}
								}
							}
						}
					}
				}
			}
			ulTmp = 3L * QPSK_SWEEP_MSEC_TIME( lwSymbol, symbol_per_dwell(lwSymbol), lNumStep  );
		} while( !ELAPSED( ulStamp, ulTmp ) && enQlock == QPSK_UNLOCKED && !ferror );
	}

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , "check III: error  : %d\n", ferror );
	printf( sbuf );
#endif
	if( !ferror )
	{
		stQpsk->stStatus.enLock = enQlock;
		if( enQlock != ALL_LOCKED )
		{
			/* stops/hold wagc/nagc/sweep/tracking */
			ferror |= disable_nagc_sweep_tracking();
			ferror |= disable_wagc();
		}
	}

#ifdef SCAN_SUBRANGE_DEBUG
	/* elcomski */
	sprintf( sbuf , "================ scan_subrange (END) ================\n\n");
	printf( sbuf );
#endif
	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: scan_zig_zag                                               */
/*                                                                           */
/* description: zig-zag scan with a given parameter                          */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean scan_zig_zag( tuning_t *stTuning )
{
	boolean ferror = false;
	long lSubrangeSize, lTstep, lNumSubrange;
	long lDriftFrq = stTuning->lLnbDriftFrequency;
	int iCnt = 0;

	tuner_t stTuner = stTuning->tuner_param;
	qpsk_t stQpsk = stTuning->qpsk_param;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	char sbuf[200];

	/* elcomski */
	sprintf( sbuf , "------------------------------------------\n" );
	printf( sbuf );
#endif

	/* calculate subrange size */
	lTstep =  sub_range_size( stQpsk.lwSymbol );
	lSubrangeSize = (long)(TUNER_ACTUAL_STEP  * lTstep);

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " sub_range_size: lTstep : %d\n", lTstep );
	printf( sbuf );
#endif
	
	/* calculate number of subrange */
	lNumSubrange = lDriftFrq * 2 / lSubrangeSize + 1;
	if( lNumSubrange % 2 == 0 ) lNumSubrange++; /* for symmetricity */

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " lNumSubrange: %d\n", lNumSubrange );
	printf( sbuf );
#endif

	/* set qpsk sweep start/end frequency to cover 1 subrange */
	stQpsk.stStatus.lSweepStartFrequency = lSubrangeSize / 2 * -1;
	stQpsk.stStatus.lSweepEndFrequency = lSubrangeSize / 2;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , " lSweepStartFreq: %d\n", stQpsk.stStatus.lSweepStartFrequency );
	printf( sbuf );
	/* elcomski */
	sprintf( sbuf , " lSweepEndFreq: %d\n", stQpsk.stStatus.lSweepEndFrequency );
	printf( sbuf );
#endif

	/* set generic qpsk parameters */
	ferror = set_symbol_rate( stQpsk.lwSymbol );
	if( !ferror ) ferror = set_symbol_loop( stQpsk.lwSymbol );
	if( !ferror ) ferror = set_carrier_loop( stQpsk.lwSymbol );
	if( !ferror ) ferror = set_symbol_per_dwell( symbol_per_dwell(stQpsk.lwSymbol) );
	if( !ferror ) ferror = set_viterbi_rate( stQpsk.enViterbi );
	if( !ferror ) ferror = set_normal_operation( );

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , "----------------------------------------\n" );
	printf( sbuf );
#endif

	if( !ferror )
	{
		do
		{
			/* force tuner to center the current subrange */
			if( iCnt == 0 ) /* first tries */
			{
#ifdef SCAN_ZIG_ZAG_DEBUG
				/* elcomski */
				sprintf( sbuf , "\n Tuner Init( iCnt = %d )\n",iCnt );
				printf( sbuf );
#endif

				stTuner.stStatus.iStepIncrement = 0;
				ferror = tuner_init( INIT, &stTuner ); 
				if( ferror ) break;
				FWAIT( TUNER_PLL_ISETTLE_MSEC_TIME );
			} else {
#ifdef SCAN_ZIG_ZAG_DEBUG
				/* elcomski */
				sprintf( sbuf , "\n\n\n Tuner Init Try( iCnt = %d )\n",iCnt );
				printf( sbuf );
#endif

				if( ( stTuner.stStatus.iStepIncrement *= -1 ) >= 0 )
					stTuner.stStatus.iStepIncrement += lTstep;

				ferror = tuner_init( UPDATE, &stTuner );
				if( ferror ) break;
				FWAIT( TUNER_PLL_SETTLE_MSEC_TIME );
			}

#ifdef SCAN_ZIG_ZAG_DEBUG
			/* elcomski */
			sprintf( sbuf , " ---> Tuner Lock : %d\n", stTuner.stStatus.enLock );
			printf( sbuf );
#endif

			if( stTuner.stStatus.enLock != TUNER_PLL_LOCKED ) break;

			/* now, start qpsk sweeping for a current subrange */
			stQpsk.stStatus.lSweepStartFrequency *= -1;
			stQpsk.stStatus.lSweepEndFrequency *= -1;

			/* now, start qpsk sweeping for a current subrange */
			ferror = scan_subrange( &stQpsk );
			if ( ferror ) break;

#ifdef SCAN_ZIG_ZAG_DEBUG
			/* elcomski */
			sprintf( sbuf , " ---> Qpsk  Lock : %d\n", stQpsk.stStatus.enLock );
			printf( sbuf );
#endif

			if( stQpsk.stStatus.enLock == ALL_LOCKED ) break;
		} while( ++iCnt < lNumSubrange );
	} /* end of if */

	/* fill qpsk/tuner status */
	stTuning->qpsk_param.stStatus = stQpsk.stStatus;
	stTuning->tuner_param.stStatus = stTuner.stStatus;

#ifdef SCAN_ZIG_ZAG_DEBUG
	/* elcomski */
	sprintf( sbuf , "\n\n  Tuner Lock : %d\n", stTuning->tuner_param.stStatus.enLock );
	printf( sbuf );
	/* elcomski */
	sprintf( sbuf , "  Qpsk Lock : %d\n\n", stTuning->qpsk_param.stStatus.enLock );
	printf( sbuf );
#endif

  /* center signal to tuner SAW filter */
#if 1
	if( !ferror )
	if( stQpsk.stStatus.enLock == ALL_LOCKED )
	ferror = correct_drift_frequency( stTuning );
#endif

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: correct_drift_frequency                                    */
/*                                                                           */
/* description: Correct frequency to center signal at tuner SAW filter       */
/*              Frequency correcting may occur if the frequency drift        */
/*              exceeds the tuner step size                                  */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note: 1. this function might be used for periodic nim monitor             */
/*                                                                           */
/* ------------------------------------------------------------------------- */
static boolean correct_drift_frequency( tuning_t *stTuning )
{
  boolean ferror = false;

  long lRecFrq, lFrq, lTstep;
  tuner_t stTuner = stTuning->tuner_param;
  qpsk_t stQpsk = stTuning->qpsk_param;

  /* get currently receiving frequency */
  lRecFrq = stQpsk.stStatus.lSweepStartFrequency;
  ferror = get_sweep_frequency( &lFrq );
  lRecFrq += lFrq;
  if( !ferror )
    ferror = get_tracking_frequency( &lFrq );
  lRecFrq += lFrq;

#if 1 /* get current receiving frequency */

  if( !ferror )
  {
    /* get current tuner frequency */
    stTuning->nim_status.lCurrentFrequency = (long)( ( (int)stTuner.stStatus.wProgrammableDivider + stTuner.stStatus.iStepIncrement ) * TUNER_ACTUAL_STEP - TUNER_IF );
    /* add qpsk receiver frequency */
    if( stQpsk.stStatus.enSpectrum == INVERTED_SPECTRUM )
      stTuning->nim_status.lCurrentFrequency += lRecFrq*(-1);
	else
      stTuning->nim_status.lCurrentFrequency += lRecFrq;
  }

#endif

  /* calculate the steps needed for centering signal */
  lTstep = ( lRecFrq > 0 ? (long)(lRecFrq / TUNER_ACTUAL_STEP + 0.5) : (long)(lRecFrq / TUNER_ACTUAL_STEP - 0.5) );

  if( stQpsk.stStatus.enSpectrum == NORMAL_SPECTRUM )
    lTstep *= -1;
  if( !ferror )
  {
    if( lTstep )
    {
      do /* compensate frequency step by step */
      {
        if( lTstep > 0 ) { stTuner.stStatus.iStepIncrement++; lTstep--; }
        else { stTuner.stStatus.iStepIncrement--; lTstep++; }
        ferror = tuner_init( UPDATE, &stTuner );
        FWAIT( TUNER_PLL_SETTLE_MSEC_TIME );
      } while( !ferror && lTstep );
    }
  }

  FWAIT( 200 );
  if( !ferror )
    ferror = get_tuner_lock_status( &stTuner.stStatus.enLock );
  if( !ferror )
    ferror = get_qpsk_lock_status( &stQpsk.stStatus.enLock );

  /* fill qpsk/tuner status */
  stTuning->qpsk_param.stStatus = stQpsk.stStatus;
  stTuning->tuner_param.stStatus = stTuner.stStatus;

  return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: nim_initialize                                             */
/*                                                                           */
/* description: initialize environment variables and registers as well.      */
/*                                                                           */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
//EXPORT boolean WINAPI nim_initialize( void )
EXPORT boolean /*WINAPI*/ nim_initialize( achar * dev_name )
{
	int i;
	boolean ferror = false;

	strcpy( gg_dev_name, dev_name );

	/* h/w reset or something here... */

	/* write 8513 register with its default value */
	for( i = 0 ; i < WR_REGSZE ; i++ ) hdm8513_wmap[i] = wreg_default[i];
	ferror |= write_reg_8513( 0, &hdm8513_wmap[0], WR_REGSZE );

	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: nim_start                                                  */
/*                                                                           */
/* description: starts nim operation to find qpsk signal                     */
/*  - parameters needed                                                      */
/*      lLnbDriftFrequency: maximum allowable lnb drift frequency[hz]        */
/*      lwFrequency: signal frequency                                        */
/*      enSawBW: tuner saw filter bandwidth selection                        */
/*      lwSymbol: symbol rate[sps]                                           */
/*      enViterbi: viterbi code rate                                         */
/* - status returned                                                         */
/*      stStatus of type tuning_qpsk_status_t/tuning_tuner_status_t          */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note: 1. recommended drift frequency is 1 ~ 5 Mhz                         */
/*                                                                           */
/* ------------------------------------------------------------------------- */
//EXPORT boolean WINAPI nim_start( tuning_t *param )
EXPORT boolean /*WINAPI*/ nim_start( achar *dev_name, tuning_t *param )
{
	boolean ferror = false;
	long lF1, lF2, lSym;
	int iSaw, iVit;

	strcpy( gg_dev_name, dev_name );

	if( param->lLnbDriftFrequency > 0 &&
		param->tuner_param.lwFrequency >= 950000000 &&
		param->tuner_param.lwFrequency <= 2150000000 &&
		param->tuner_param.enSawBW <= TSAW_DUAL_2 &&
		param->qpsk_param.lwSymbol > 0 &&
		param->qpsk_param.lwSymbol <= 55000000 &&
		param->qpsk_param.enViterbi <= FEC_INNER_7_8 )
		{
#if 1 /* for debugging purpose only */
		lF1 = param->lLnbDriftFrequency;
		lF2 = param->tuner_param.lwFrequency;
		lSym = param->qpsk_param.lwSymbol;
		iSaw = (int)param->tuner_param.enSawBW;
		iVit = (int)param->qpsk_param.enViterbi;

#if 0
		printf(" lF1 = %lu \n", lF1 );
		printf(" lF2 = %lu \n", lF2 );
		printf(" lSym = %lu \n", lSym );
		printf(" iSaw = %d \n", iSaw );
		printf(" iVit = %d \n", iVit );
#endif 

#endif
		ferror = scan_zig_zag( param );
	}
	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: nim_stop                                                   */
/*                                                                           */
/* description: stops sweep process and agc                                  */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
//EXPORT boolean WINAPI nim_stop( void )
EXPORT boolean /*WINAPI*/ nim_stop( achar *dev_name )
{
    strcpy( gg_dev_name, dev_name );

    return ( stop_search_8513() );
}

/* ------------------------------------------------------------------------- */
/* function name: nim_control                                                */
/*                                                                           */
/* description:                                                              */
/*                                                                           */
/*   NIM_LNBCNTL    : controls lnb                                           */
/*     - AC_CTRL        - lnb 22khz ac control                               */
/*     - DC_CTRL        - lnb dc control                                     */
/*   NIM_BERT       : forces NIM to BER test mode                            */
/*     - TEST_VITERBIOUT                                                     */
/*     - TEST_TRISTATE                                                       */
/*   NIM_SPECTRUM   : set spectrum mode                                      */
/*     - INVERT         - set to inversioned spectrum mode                   */
/*     - NORMAL         - set to normal spectrum mode                        */
/*     - TOGGLE         - set to toggled spectrum mode                       */
/*     - AUTO           - set to auto spectrum detection mode                */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note: 1. auto spectrum detection mode not available!                      */
/*                                                                           */
/* ------------------------------------------------------------------------- */
EXPORT boolean /*WINAPI*/ nim_control( byte command, byte mode, byte *param )
{
	boolean ferror = false;

	switch( command )
	{
		case NIM_LNBCNTL:
		{
			switch( mode )
			{
				case AC_CTRL:
				{
					if( (LNB_CTRL) param[0] == LNB_BAND_HIGH ) {
						/* force 22k tone off */
						hdm8513_wmap[WR_CTRL_WAGC] |= 1 << POS_TONE_HOLD & MASK_TONE_HOLD;
						ferror |= write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );
					} else if( (LNB_CTRL) param[0] == LNB_BAND_LOW ) {
						/* force 22k tone on */
						hdm8513_wmap[WR_CTRL_WAGC] &= ~(1 << POS_TONE_HOLD & MASK_TONE_HOLD);
						ferror |= write_reg_8513( WR_CTRL_WAGC, &hdm8513_wmap[WR_CTRL_WAGC], 1 );
					}
				}
				break;

				case DC_CTRL:
				{
					if( (LNB_CTRL) param[0] == LNB_POL_VERTICAL ) {
						; /* force left-hand polarization */
						/* .. nim board not implemented */
					}else if( (LNB_CTRL) param[0] == LNB_POL_HORIZONTAL ) {
						; /* force right-hand polarization */
						/* .. nim board not implemented */
					}
				}
				break;
			
				default: break;	
			}
		}
		break;

		case NIM_BERT:
		{
			switch( mode )
			{
				case TEST_VITERBIOUT:
					hdm8513_wmap[WR_TEST_SETUP] = OUTPUT_BEFORE_AFTER_RS;
					ferror |= write_reg_8513( WR_TEST_SETUP, &hdm8513_wmap[WR_TEST_SETUP], 1 );
				break;

				case TEST_TRISTATE:
					hdm8513_wmap[WR_TEST_SETUP] = OUTPUT_TRISTATE;
					ferror |= write_reg_8513( WR_TEST_SETUP, &hdm8513_wmap[WR_TEST_SETUP], 1 );
				break;

				default: break;
			}
		}
		break;

		case NIM_SPECTRUM:
		{
			switch( mode )
			{
				case SPECTRUM_INVERT: ferror |= set_spectrum_invert();
				break;  
				case SPECTRUM_NORMAL: ferror |= set_spectrum_normal();
				break;
				case SPECTRUM_TOGGLE: ferror |= set_spectrum_toggle();
				break;
				case SPECTRUM_AUTOMATIC: ferror |= set_spectrum_automatic();
				break;

				default: break;
			}
		}
		break; 

		default: 
		break;
	}
	return ferror;
}

/* ------------------------------------------------------------------------- */
/* function name: nim_get_lock_status                                        */
/*                                                                           */
/* description: reads various lock flags and result in *enTlock              */
/*              and *enQlock.                                                */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
EXPORT boolean /*WINAPI*/ nim_get_lock_status( achar *dev_name, byte qpsk_addr, tuner_lock_t *enTlock, qpsk_lock_t *enQlock )
{
	boolean ferror = false;

	gg_qpsk_addr = qpsk_addr;
	strcpy( gg_dev_name, dev_name );

	ferror |= get_tuner_lock_status( enTlock );
	ferror |= get_qpsk_lock_status( enQlock );
	
	return ferror; 
}

/* ------------------------------------------------------------------------- */
/* function name: nim_get_register                                           */
/*                                                                           */
/* description: reads register value with size of wSize to *pbValReg         */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note:                                                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */
EXPORT boolean /*WINAPI*/ nim_get_register( achar *dev_name, word wOffset, byte *pbValReg, word wSize )
{

	boolean ferror = false;

	strcpy( gg_dev_name, dev_name );

	ferror |= read_reg_8513( (byte)wOffset, pbValReg, wSize );

	return ferror;
}


/* ------------------------------------------------------------------------- */
/* function name: nim_get_quality                                            */
/*                                                                           */
/* description: read the related register and convert it to a comprehensive  */
/*              quantitity. FINE mode can be used after QPSK lock is         */
/*              achieved and its result is a direct SNR of unit dB.          */
/*              COARSE mode can be used when signal is deeply paded in noise */
/*              and its result is a relative signal power.                   */
/*              When WAGC inverted mode is selected, the larger value of WAGC*/
/*              means the higher input signal power.(range: -128 ~ 127)      */
/*                                                                           */
/*                                                                           */
/* date: 1998. 7. 16.                                                        */
/* Note: 1. noise power to SNR conversion using polynormial approximation.   */
/*       2. note of result a.b[dB] to application interface using DLL.       */
/*                                                                           */
/* ------------------------------------------------------------------------- */
#define CX3 -0.00002384
#define CX2  0.006964
#define CX1 -0.747
#define CX0 34.5
EXPORT double /*WINAPI*/ nim_get_quality( achar *dev_name, byte qpsk_addr, byte mode )
{
	byte bVal;
	int iWagc;
	double fQuality = 0;

	gg_qpsk_addr = qpsk_addr;
	strcpy( gg_dev_name, dev_name );

	switch( mode )
	{
		case FINE: /* when in tracking mode */
		{
			/* read noise power register */
			get_noise_power( &bVal );
			/* noise power to SNR[dB] conversion */
			fQuality  = (double)bVal*bVal*bVal*CX3;
			fQuality += (double)bVal*bVal*CX2;
			fQuality += (double)bVal*CX1;
			fQuality += CX0;
		}
		break;

		case COARSE: /* when in acquisition mode */
		/* Read wide band agc register,
		 Caution! this register value has only meaning when
		 wide band agc is used in Mode 1 */
		{
			get_wagc( &iWagc );
			fQuality = (double)iWagc;
		}
		break;
		
		default:
		break;
	}
	return fQuality;
}

/* ------------------------------------------------------------------------- */
/* function name: nim_monitor                                                */
/*                                                                           */
/* description: Periodic nim monitor for lnb frequency drift correction.     */
/*                                                                           */
/* date: 1998. 11. 23.                                                       */
/* Note: 1. recommended period is 1 ~ 10 sec.                                */
/*                                                                           */
/* ------------------------------------------------------------------------- */
EXPORT boolean /*WINAPI*/ nim_monitor( tuning_t *stTuning )
{
  boolean ferror = false;
  qpsk_lock_t enQlock;

  ferror = get_qpsk_lock_status( &enQlock );
  if( !ferror )
  {
    if( enQlock == ALL_LOCKED )
    {
      ferror = correct_drift_frequency( stTuning );
    }
  }

  /* fill status */
  if( !ferror )
    stTuning->qpsk_param.stStatus.enLock = enQlock;
  return ferror;
}

/*
 * Func : nim_set_values
 * Context : 
 */
EXPORT void /*WINAPI*/ nim_set_values( byte qpsk_addr )
{
 	gg_qpsk_addr = qpsk_addr ;

	if( qpsk_addr == 0xe8 ) {
	/* ------------------------------------ */
	/* HDM8513A */	
	/* ------------------------------------ */
		/* Set system clock to 76MHz */
		gg_clock = 76000000.0;
		
		/* Set HDM8513A dependent value write register */ 
#ifdef EXTERNAL_ADC_
		wreg_default[0x14] = 0x2f;
#else
		wreg_default[0x14] = 0xa8;
#endif
		/* clock generation pll control, M=1, N=19 thus 76Mhz */
		wreg_default[0x1f] = 0x53;
	}else {
	/* ------------------------------------ */
	/* HDM 8513 */	
	/* ------------------------------------ */
		/* Set system clock to 60MHz */
		gg_clock = 60000000.0;

		/* Set HDM8513 dependent value write register */ 
#ifdef EXTERNAL_ADC_
		wreg_default[0x14] = 0x07;
#else
		wreg_default[0x14] = 0x00;
#endif
		/* clock generation pll control, M=1, N=15 thus 60Mhz */
		wreg_default[0x1f] = 0x01;
		
	}
}

/*
 * Func : nim_set_values
 * Context : 
 */
EXPORT boolean nim_get_qpsk_lock_status( achar *dev_name, byte qpsk_addr, qpsk_lock_t *enQlock )
{
	boolean ferror = false;

	strcpy( gg_dev_name, dev_name );

	gg_qpsk_addr = qpsk_addr;

	ferror |= get_qpsk_lock_status( enQlock );
	return ferror; 
}

/*
BOOL WINAPI	DllMain(HINSTANCE hInstDll, DWORD fdwReason, LPVOID fImpLoad)
{

	switch (fdwReason)
	{
	case DLL_PROCESS_ATTACH:
		// to do:
		break;
	case DLL_THREAD_ATTACH:
		// to do:
		break;
	case DLL_PROCESS_DETACH:
		// to do:
		break;
	}
	return
}
*/
