/*
 * imagesource_montage.cpp
 * Composites multiple images into a single image.
 * Supports Random Access if and only if all source images also support it.
 *
 * Copyright (c) 2004, 2005 by Alastair M. Robinson
 * Distributed under the terms of the GNU General Public License -
 * see the file named "COPYING" for more details.
 *
 * 2005-03-25 - Separated out the n-up image fitting into imagesource_signature.*
 *
 */


#include <iostream>

#include "imagesource_montage.h"

using namespace std;

class ISMontage_Component
{
	public:
	ISMontage_Component(ImageSource_Montage *header,ImageSource *source,int xpos,int ypos);
	~ISMontage_Component();
	ISDataType *GetRow(int row);
	protected:
	ImageSource_Montage *header;
	ImageSource *source;
	int xpos,ypos;
	ISMontage_Component *next,*prev;
	friend class ImageSource_Montage;
};


ISDataType *ISMontage_Component::GetRow(int row)
{
	if((row<ypos) || (row>=(ypos+source->height)))
		return(NULL);
	else
		return(source->GetRow(row-ypos));
}


ISMontage_Component::ISMontage_Component(ImageSource_Montage *header,ImageSource *src,int xpos,int ypos)
	: header(header), source(src), xpos(xpos), ypos(ypos), next(NULL), prev(NULL)
{
	if((next=header->first))
		next->prev=this;
	header->first=this;
	
	if(header->height<(ypos+source->height))
		header->height=ypos+source->height;

	if(header->width<(xpos+source->width))
		header->width=xpos+source->width;
}


ISMontage_Component::~ISMontage_Component()
{
	if(source)
		delete source;
	if(prev)
		prev->next=next;
	else
		header->first=next;
	if(next)
		next->prev=prev;
}


ImageSource_Montage::ImageSource_Montage(IS_TYPE type,int resolution)
	: ImageSource(), first(NULL)
{
	xres=resolution;
	yres=resolution;
	this->type=type;

	switch(type)
	{
		case IS_TYPE_BW:
		case IS_TYPE_GREY:
			samplesperpixel=1;
			break;
		case IS_TYPE_RGB:
			samplesperpixel=3;
			break;
		case IS_TYPE_CMYK:
			samplesperpixel=4;
			break;
		default:
			throw "Unsupported image type";
	}

	randomaccess=true;

	width=height=0;
}


ImageSource_Montage::~ImageSource_Montage()
{
	while(first)
		delete first;
}


void ImageSource_Montage::Add(ImageSource *is,int xpos,int ypos)
{
	if(STRIP_ALPHA(is->type)!=type)
		throw "Can't yet mix different colour spaces on one page";
	new ISMontage_Component(this,is,xpos,ypos);

	randomaccess&=is->randomaccess;
}


ISDataType *ImageSource_Montage::GetRow(int row)
{
	if(!rowbuffer)
		MakeRowBuffer();

	ISDataType *src;
	ISDataType *dst=rowbuffer;

	switch(STRIP_ALPHA(type))
	{
		case IS_TYPE_RGB:
			for(int i=0;i<width*samplesperpixel;++i)
				dst[i]=IS_SAMPLEMAX;
			break;
		default:
			for(int i=0;i<width*samplesperpixel;++i)
				dst[i]=0;
			break;
	}

	ISMontage_Component *mc=first;
	while(mc)
	{
		if((src=mc->GetRow(row)))
		{
			if(HAS_ALPHA(mc->source->type))
			{
				for(int i=0;i<mc->source->width;++i)
				{
					int a=src[(i+1)*mc->source->samplesperpixel-1];
//					cerr << "Alpha: " << a << endl;
					int ia=IS_SAMPLEMAX-a;
					int xp=(mc->xpos+i)*samplesperpixel;
					int sp=i*mc->source->samplesperpixel;
					for(int j=0;j<samplesperpixel;++j)
					{
						int t=dst[xp+j];
						t*=ia;
						int t2=src[sp+j];
						t2*=a;
						dst[xp+j]=(t+t2)/IS_SAMPLEMAX;
					}
				}
			}
			else
			{
				for(int i=0;i<mc->source->width*samplesperpixel;++i)
					dst[mc->xpos*samplesperpixel+i]=src[i];
			}
		}
		mc=mc->next;
	}

	return(rowbuffer);
}
