/*
 * pp_pageextent.cpp - provides a custom widget for controlling a PageExtent.
 * Uses the GPrinter from PhotoPrint_State to display a list of pagesizes
 * and to get the page size.  The margins are applied independently to the
 * supplied PageExtent.
 * Copyright (c) 2004 by Alastair M. Robinson
 * Distributed under the terms of the GNU General Public License -
 * see the file named "COPYING" for more details.
 *
 */

#include <string.h>

#include <gtk/gtkframe.h>
#include <gtk/gtksizegroup.h>
#include <gtk/gtklabel.h>

#include "stpui_widgets/dimension.h"
#include "stpui_widgets/stpui_combo.h"

#include "photoprint_state.h"
#include "pp_pageextent.h"

enum {
	CHANGED_SIGNAL,
	LAST_SIGNAL
};

static guint pp_pageextent_signals[LAST_SIGNAL] = { 0 };

static void pp_pageextent_class_init (pp_PageExtentClass *klass);
static void pp_pageextent_init (pp_PageExtent *stpuicombo);


static void lmargin_changed(GtkWidget *wid,gpointer *ob)
{
	pp_PageExtent *lo=(pp_PageExtent *)ob;
	PageExtent *pe=lo->pe;
	Dimension *d=DIMENSION(lo->lmargin);
	int v=int(dimension_get_pt(d));
	pe->SetMargins(v,pe->rightmargin,pe->topmargin,pe->bottommargin);
	g_signal_emit(G_OBJECT (ob),pp_pageextent_signals[CHANGED_SIGNAL], 0);
}


static void rmargin_changed(GtkWidget *wid,gpointer *ob)
{
	pp_PageExtent *lo=(pp_PageExtent *)ob;
	PageExtent *pe=lo->pe;
	Dimension *d=DIMENSION(lo->rmargin);
	int v=int(dimension_get_pt(d));
	pe->SetMargins(pe->leftmargin,v,pe->topmargin,pe->bottommargin);
	g_signal_emit(G_OBJECT (ob),pp_pageextent_signals[CHANGED_SIGNAL], 0);
}


static void tmargin_changed(GtkWidget *wid,gpointer *ob)
{
	pp_PageExtent *lo=(pp_PageExtent *)ob;
	PageExtent *pe=lo->pe;
	Dimension *d=DIMENSION(lo->tmargin);
	int v=int(dimension_get_pt(d));
	pe->SetMargins(pe->leftmargin,pe->rightmargin,v,pe->bottommargin);
	g_signal_emit(G_OBJECT (ob),pp_pageextent_signals[CHANGED_SIGNAL], 0);
}


static void bmargin_changed(GtkWidget *wid,gpointer *ob)
{
	pp_PageExtent *lo=(pp_PageExtent *)ob;
	PageExtent *pe=lo->pe;
	Dimension *d=DIMENSION(lo->bmargin);
	int v=int(dimension_get_pt(d));
	pe->SetMargins(pe->leftmargin,pe->rightmargin,pe->topmargin,v);
	g_signal_emit(G_OBJECT (ob),pp_pageextent_signals[CHANGED_SIGNAL], 0);
}


static void pagesize_changed(GtkWidget *wid,gpointer *ob)
{
	pp_PageExtent *lo=(pp_PageExtent *)ob;

	lo->state->layout->UpdatePageSize();

	pp_pageextent_refresh(lo);
	g_signal_emit(G_OBJECT (lo),pp_pageextent_signals[CHANGED_SIGNAL], 0);
}


void pp_pageextent_refresh(pp_PageExtent *ob)
{
	dimension_set_pt(DIMENSION(ob->lmargin),ob->pe->leftmargin);
	dimension_set_pt(DIMENSION(ob->rmargin),ob->pe->rightmargin);
	dimension_set_pt(DIMENSION(ob->tmargin),ob->pe->topmargin);
	dimension_set_pt(DIMENSION(ob->bmargin),ob->pe->bottommargin);

	stpui_combo_refresh(STPUI_COMBO(ob->pagesize));
}


void pp_pageextent_set_unit(pp_PageExtent *ob,enum Units unit)
{
	dimension_set_unit(DIMENSION(ob->lmargin),unit);
	dimension_set_unit(DIMENSION(ob->rmargin),unit);
	dimension_set_unit(DIMENSION(ob->tmargin),unit);
	dimension_set_unit(DIMENSION(ob->bmargin),unit);
}


GtkWidget*
pp_pageextent_new (PageExtent *pe,PhotoPrint_State *state,GtkSizeGroup *sizegroup)
{
	pp_PageExtent *ob=PP_PAGEEXTENT(g_object_new (pp_pageextent_get_type (), NULL));
	gtk_box_set_spacing(GTK_BOX(ob),5);

	ob->pe=pe;
	ob->state=state;
	
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *vbox;
	GtkWidget *frame;


	// PageSize Selector

	frame=gtk_frame_new("Page Size");
	gtk_container_set_border_width(GTK_CONTAINER(frame),5);
	gtk_box_pack_start(GTK_BOX(ob),frame,FALSE,FALSE,0);
	gtk_widget_show(frame);

	ob->pagesize=stpui_combo_new(ob->state->printer.stpvars,"PageSize",NULL);
	gtk_container_set_border_width(GTK_CONTAINER(ob->pagesize),5);
	g_signal_connect(G_OBJECT(ob->pagesize),"changed",G_CALLBACK(pagesize_changed),ob);
	gtk_container_add(GTK_CONTAINER(frame),ob->pagesize);
	gtk_widget_show(ob->pagesize);


	//  Margins frame
	
	frame=gtk_frame_new("Margins");
	gtk_box_pack_start(GTK_BOX(ob),frame,FALSE,FALSE,0);
	gtk_container_set_border_width(GTK_CONTAINER(frame),5);
	gtk_widget_show(frame);

	vbox=gtk_vbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);


	//    LeftMargin spin button

	hbox=gtk_hbox_new(FALSE,0);
	
	label=gtk_label_new("Left:");
	gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,5);
	gtk_widget_show(label);

	enum Units unit=state->GetUnits();

	ob->lmargin=dimension_new(0.0,300.0,unit);
	g_signal_connect(G_OBJECT(ob->lmargin),"value-changed",G_CALLBACK(lmargin_changed),ob);
	gtk_widget_show(ob->lmargin);
	
	gtk_size_group_add_widget(GTK_SIZE_GROUP(sizegroup),ob->lmargin);
	gtk_box_pack_start(GTK_BOX(hbox),ob->lmargin,FALSE,FALSE,5);

	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,5);
	gtk_widget_show(hbox);

	//    RightMargin spin button

	hbox=gtk_hbox_new(FALSE,0);
	
	label=gtk_label_new("Right:");
	gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,5);
	gtk_widget_show(label);

	ob->rmargin=dimension_new(0.0,300.0,unit);
	g_signal_connect(G_OBJECT(ob->rmargin),"value-changed",G_CALLBACK(rmargin_changed),ob);
	gtk_widget_show(ob->rmargin);

	gtk_size_group_add_widget(GTK_SIZE_GROUP(sizegroup),ob->rmargin);
	gtk_box_pack_start(GTK_BOX(hbox),ob->rmargin,FALSE,FALSE,5);

	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,5);
	gtk_widget_show(hbox);

	//    TopMargin spin button

	hbox=gtk_hbox_new(FALSE,0);
	label=gtk_label_new("Top:");
	gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,5);
	gtk_widget_show(label);

	ob->tmargin=dimension_new(0.0,300.0,unit);
	g_signal_connect(G_OBJECT(ob->tmargin),"value-changed",G_CALLBACK(tmargin_changed),ob);
	gtk_widget_show(ob->tmargin);

	gtk_size_group_add_widget(GTK_SIZE_GROUP(sizegroup),ob->tmargin);
	gtk_box_pack_start(GTK_BOX(hbox),ob->tmargin,FALSE,FALSE,5);

	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,5);
	gtk_widget_show(hbox);

	//    BottomMargin spin button

	hbox=gtk_hbox_new(FALSE,0);
	
	label=gtk_label_new("Bottom:");
	gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,5);
	gtk_widget_show(label);

	ob->bmargin=dimension_new(0.0,300.0,unit);
	g_signal_connect(G_OBJECT(ob->bmargin),"value-changed",G_CALLBACK(bmargin_changed),ob);
	gtk_widget_show(ob->bmargin);

	gtk_size_group_add_widget(GTK_SIZE_GROUP(sizegroup),ob->bmargin);
	gtk_box_pack_start(GTK_BOX(hbox),ob->bmargin,FALSE,FALSE,5);

	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,5);
	gtk_widget_show(hbox);

	pp_pageextent_refresh(ob);

	return(GTK_WIDGET(ob));
}


GType
pp_pageextent_get_type (void)
{
	static GType stpuic_type = 0;

	if (!stpuic_type)
	{
		static const GTypeInfo pp_pageextent_info =
		{
			sizeof (pp_PageExtentClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) pp_pageextent_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof (pp_PageExtent),
			0,
			(GInstanceInitFunc) pp_pageextent_init,
		};
		stpuic_type = g_type_register_static (GTK_TYPE_VBOX, "pp_PageExtent", &pp_pageextent_info, (GTypeFlags)0);
	}
	return stpuic_type;
}


static void
pp_pageextent_class_init (pp_PageExtentClass *klass)
{
	pp_pageextent_signals[CHANGED_SIGNAL] =
	g_signal_new ("changed",
		G_TYPE_FROM_CLASS (klass),
		GSignalFlags(G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION),
		G_STRUCT_OFFSET (pp_PageExtentClass, changed),
		NULL, NULL,
		g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}


static void
pp_pageextent_init (pp_PageExtent *ob)
{
	ob->pe=NULL;
}
