#include <iostream>

#include "profilemanager.h"
#include "../support/searchpathdbhandler.h"

using namespace std;


// ProfileManager


ConfigTemplate ProfileManager::Template[]=
{
 	ConfigTemplate("DefaultRGBProfile","sRGB Color Space Profile.icm"),
	ConfigTemplate("DefaultRGBProfileActive",int(1)),
	ConfigTemplate("DefaultCMYKProfile","USWebCoatedSWOP.icc"),
	ConfigTemplate("DefaultCMYKProfileActive",int(1)),
	ConfigTemplate("PrinterProfile","devicelink.icm"),
	ConfigTemplate("PrinterProfileActive",int(1)),
	ConfigTemplate("ExportProfile",""),
	ConfigTemplate("ExportProfileActive",int(0)),
	ConfigTemplate("MonitorProfile",""),
	ConfigTemplate("MonitorProfileActive",int(0)),
	ConfigTemplate("RenderingIntent",int(INTENT_PERCEPTUAL)),
	ConfigTemplate("ProfilePath","/usr/share/color/icc:/usr/local/share/color/icc/:$HOME/.color/icc"),
	ConfigTemplate()
};


ProfileManager::ProfileManager(ConfigFile *inifile,const char *section) : ConfigDB(Template), SearchPath(), first(NULL)
{
	new SearchPathDBHandler(inifile,section,this,this,"ProfilePath");
	AddPath(FindString("ProfilePath"));
}


ProfileManager::~ProfileManager()
{
}


CMSProfile *ProfileManager::GetProfile(const char *name)
{
	CMSProfile *result=NULL;
	char *fn=Search(name);
	if(fn && strlen(fn))
	{
		try
		{
			result=new CMSProfile(fn);
		}
		catch(const char *err)
		{
			free(fn);
			throw err;
		}
	}
	if(fn)
		free(fn);
	return(result);
}


CMSProfile *ProfileManager::GetProfile(enum CMColourDevice target)
{
	const char *profilename=NULL;
	switch(target)
	{
		case CM_COLOURDEVICE_DISPLAY:
			if(FindInt("MonitorProfileActive"))
				profilename=FindString("MonitorProfile");
			break;
		case CM_COLOURDEVICE_EXPORT:
			if(FindInt("ExportProfileActive"))
				profilename=FindString("ExportProfile");
			break;
		case CM_COLOURDEVICE_PRINTER:
			if(FindInt("PrinterProfileActive"))
				profilename=FindString("PrinterProfile");
			break;
		case CM_COLOURDEVICE_DEFAULTRGB:
			if(FindInt("DefaultRGBProfileActive"))
				profilename=FindString("DefaultRGBProfile");
			break;
		case CM_COLOURDEVICE_DEFAULTCMYK:
			if(FindInt("DefaultCMYKProfileActive"))
				profilename=FindString("DefaultCMYKProfile");
			break;
		default:
			throw "Unknown target device!";
	}
	if(profilename)
		return(GetProfile(profilename));
	else
		return(NULL);
}


CMSProfile *ProfileManager::GetDefaultProfile(IS_TYPE colourspace)
{
	const char *profilename=NULL;
	switch(STRIP_ALPHA(colourspace))
	{
		case IS_TYPE_RGB:
			if(FindInt("DefaultRGBProfileActive"))
				profilename=FindString("DefaultRGBProfile");
			break;
		case IS_TYPE_CMYK:
			if(FindInt("DefaultCMYKProfileActive"))
				profilename=FindString("DefaultCMYKProfile");
			break;
	}
	if(profilename)
		return(GetProfile(profilename));
	else
		return(NULL);
}


CMTransformFactory *ProfileManager::GetTransformFactory()
{
	return(new CMTransformFactory(*this));
}


// CMTransformFactory

// CMTransformFactoryNode

CMTransformFactoryNode::CMTransformFactoryNode(CMTransformFactory *header,CMSTransform *transform,MD5Digest &d1,MD5Digest &d2,int intent)
	: header(header), prev(NULL), next(NULL), transform(transform), digest1(d1), digest2(d2), intent(intent)
{
	prev=header->first;
	if((prev=header->first))
	{
		while(prev->next)
			prev=prev->next;
		prev->next=this;
	}
	else
		header->first=this;
}


CMTransformFactoryNode::~CMTransformFactoryNode()
{
	cerr << "Removing FactoryNode" << endl;

	if(next)
		next->prev=prev;
	if(prev)
		prev->next=next;
	else
		header->first=next;

	if(transform)
		delete transform;
}


// CMTransformFactory proper


CMTransformFactory::CMTransformFactory(ProfileManager &pm)
	: manager(pm), first(NULL)
{
}


CMTransformFactory::~CMTransformFactory()
{
	while(first)
		delete first;
}


CMSTransform *CMTransformFactory::Search(MD5Digest *srcdigest,MD5Digest *dstdigest,int intent)
{
	CMTransformFactoryNode *tfn=first;
	while(tfn)
	{
		if((*srcdigest==tfn->digest1)&&(*dstdigest==tfn->digest2)&&(intent==tfn->intent))
			return(tfn->transform);
		tfn=tfn->next;
	}
	return(NULL);
}


CMSTransform *CMTransformFactory::GetTransform(enum CMColourDevice target,IS_TYPE type,int intent)
{
	CMSProfile *srcprofile=manager.GetDefaultProfile(type);
	CMSTransform *t=NULL;
	try
	{
		t=GetTransform(target,srcprofile,intent);
	}
	catch(const char *err)
	{
	}
	delete srcprofile;
	return(t);
}


CMSTransform *CMTransformFactory::GetTransform(enum CMColourDevice target,CMSProfile *srcprofile,int intent)
{
	// If a NULL profile is supplied, we currently bail out.
	// Theoretically we could continue if the target's profile is a DeviceLink,
	// or we could assume a colourspace to match the target's profile,
	// and fall back gracefully.

	if(!srcprofile)
		return(NULL);

	CMSProfile *destprofile=manager.GetProfile(target);

	return(GetTransform(destprofile,srcprofile,intent));
}

CMSTransform *CMTransformFactory::GetTransform(CMSProfile *destprofile,CMSProfile *srcprofile,int intent)
{
	// No point whatever in continuing without an output device profile...
	if(!destprofile)
		return(NULL);

	CMSTransform *transform=NULL;
	if(intent==INTENT_DEFAULT)
		intent=manager.FindInt("RenderingIntent");

	cerr << "Using intent: " << intent << endl;

	// We use MD5 digests to compare profiles for equality.
	MD5Digest *d1,*d2;
	d2=destprofile->GetMD5();

	if(destprofile->IsDeviceLink())
	{
		cerr << "Device link profile detected" << endl;
		// Device link profiles make life awkward if we have to use a source profile
		// (which we must do in the case of an image having an embedded profile).
		// What we do here is convert from the source profile to the appropriate
		// colour space's default profile, and then apply the devicelink.

		CMSProfile *defprofile=NULL;
		if(srcprofile)
			defprofile=manager.GetDefaultProfile(srcprofile->GetColourSpace());

		// If we have both source and default profiles, and they're not equal,
		// create a multi-profile transform: src -> default -> devicelink.
		if((srcprofile)&&(defprofile)&&(*srcprofile->GetMD5()!=*defprofile->GetMD5()))
		{
			cerr << "Source and default profiles don't match - building transform chain..." << endl;
			CMSProfile *profiles[3];
			profiles[0]=srcprofile;
			profiles[1]=defprofile;
			profiles[2]=destprofile;
			d1=srcprofile->GetMD5();
			
			// Search for an existing transform by source / devicelink MD5s...
			transform=Search(d1,d2,intent);
			if(!transform)
			{
				cerr << "No suitable cached transform found - creating a new one..." << endl;
				transform=new CMSTransform(profiles,3,intent);
				new CMTransformFactoryNode(this,transform,*d1,*d2,intent);
			}
		}
		else
		{
			cerr << "Source and default profiles match - using devicelink in isolation..." << endl;
			// If there's no default profile, or the source and default profiles match
			// then we can just use the devicelink profile in isolation.
			d1=d2;
			transform=Search(d1,d2,intent);
			if(!transform)
			{
				cerr << "No suitable cached transform found - creating a new one..." << endl;
				transform=new CMSTransform(destprofile,intent);
				new CMTransformFactoryNode(this,transform,*d1,*d2,intent);
			}
		}
		if(defprofile)
			delete defprofile;
	}
	else
	{
		// The non-device link case is much easier to deal with...
		d1=srcprofile->GetMD5();

		// Don't bother transforming if src/dest are the same profile...
		if(*d1==*d2)
			return(NULL);

		transform=Search(d1,d2,intent);
		if(!transform)
		{
			cerr << "No suitable cached transform found - creating a new one..." << endl;
			transform=new CMSTransform(srcprofile,destprofile,intent);
			new CMTransformFactoryNode(this,transform,*d1,*d2,intent);
		}
	}
	delete destprofile;

	return(transform);
}


void CMTransformFactory::Flush()
{
	while(first)
		delete first;
}


// Path handling

static const char *findextension(const char *filename)
{
	int t=strlen(filename)-1;
	int c;
	for(c=t;c>0;--c)
	{
		if(filename[c]=='.')
			return(filename+c);
	}
	return(filename);
}


const char *ProfileManager::GetNextFilename(const char *prev)
{
	const char *result=prev;
	while((result=SearchPath::GetNextFilename(result)))
	{
		const char *ext=findextension(result);
		if(strncasecmp(ext,".ICM",4)==0)
			return(result);
		if(strncasecmp(ext,".icm",4)==0)
			return(result);
		if(strncasecmp(ext,".ICC",4)==0)
			return(result);
		if(strncasecmp(ext,".icc",4)==0)
			return(result);
	}
	return(result);
}


void ProfileManager::AddPath(const char *path)
{
	FlushProfileInfoList();
	SearchPath::AddPath(path);
}


void ProfileManager::RemovePath(const char *path)
{
	FlushProfileInfoList();
	SearchPath::RemovePath(path);
}


void ProfileManager::ClearPaths()
{
	FlushProfileInfoList();
	SearchPath::ClearPaths();
}


ProfileInfo *ProfileManager::GetFirstProfileInfo()
{
	if(!first)
		BuildProfileInfoList();
	return(first);
}


void ProfileManager::BuildProfileInfoList()
{
	cerr << "Building ProfileInfo List:" << endl;
	const char *f=NULL;
	FlushProfileInfoList();
	while((f=GetNextFilename(f)))
	{
		if(!(FindProfileInfo(f)))
			new ProfileInfo(*this,f);
	}
}


void ProfileManager::FlushProfileInfoList()
{
	while(first)
		delete first;
}


ProfileInfo *ProfileManager::FindProfileInfo(const char *filename)
{
	ProfileInfo *pi=first;
	while(pi)
	{
		const char *fn=pi->filename;
		if(strcmp(fn,filename)==0)
		{
			cerr << "Found " << filename << endl;
			return(pi);
		}
		pi=pi->Next();
	}
	return(NULL);
}


// ProfileInfo


ProfileInfo::ProfileInfo(ProfileManager &pm,const char *filename)
	: profilemanager(pm), next(NULL), prev(NULL), filename(NULL), iscached(false), description(NULL), isdevicelink(false)
{
	if((next=profilemanager.first))
		next->prev=this;
	profilemanager.first=this;
	this->filename=strdup(filename);
}


ProfileInfo::~ProfileInfo()
{
	if(filename)
		free(filename);
	if(description)
		free(description);
	if(prev)
		prev->next=next;
	else
		profilemanager.first=next;
	if(next)
		next->prev=prev;
}


ProfileInfo *ProfileInfo::Next()
{
	return(next);
}


void ProfileInfo::GetInfo()
{
	if(iscached)
		return;
	cerr << "Getting profile info for " << filename << endl;
	char *fn=profilemanager.Search(filename);
 	if(fn && strlen(fn))
 	{
 		CMSProfile profile(fn);
 		free(fn);
 		colourspace=profile.GetColourSpace();
 		isdevicelink=profile.IsDeviceLink();
 		description=strdup(profile.GetDescription());
 		iscached=true;
 	}
}


const char *ProfileInfo::GetFilename()
{
	GetInfo();
	return(filename);
}


const char *ProfileInfo::GetDescription()
{
	GetInfo();
	return(description);
}


bool ProfileInfo::IsDeviceLink()
{
	GetInfo();
	return(isdevicelink);
}


IS_TYPE ProfileInfo::GetColourSpace()
{
	GetInfo();
	return(colourspace);
}


int ProfileManager::GetIntentCount()
{
	return(CMS_GetIntentCount());
}

const char *ProfileManager::GetIntentName(int intent)
{
	if(intent==INTENT_DEFAULT)
		return("Default");
	else
		return(CMS_GetIntentName(intent));
}

const char *ProfileManager::GetIntentDescription(int intent)
{
	if(intent==INTENT_DEFAULT)
		return("");
	else
		return(CMS_GetIntentDescription(intent));
}
