# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Robert Kaye
#
#
# ***** END LICENSE BLOCK *****

import wx, sys, re, os, copy
from wx.gizmos import TreeListCtrl
from picard.ui import images, dirctrl
from picard import album, artist, track, albummanager, events, util, cluster, debug, wpath
from tunepimp import tunepimp, metadata
from tunepimp import track as tptrack
import traceback

treeControlId = wx.NewId()
albumsNodeDragText = "albums-node-drag-text"
numColorGradients = 10

class AlbumDropTarget(wx.DropTarget):

    def __init__(self, albumPanel, config):
        wx.DropTarget.__init__(self)
        self.textObject = wx.TextDataObject()
        self.fileObject = wx.FileDataObject()
        self.dataObject = wx.DataObjectComposite()
        self.dataObject.Add(self.textObject)
        self.dataObject.Add(self.fileObject)
        self.SetDataObject(self.dataObject)
        self.albumPanel = albumPanel
        self.wpath = wpath.wpath(config)

    def OnData(self, x, y, default):
        if self.GetData():
            text = self.textObject.GetText()
            if text:
                self.albumPanel.OnDropText(x, y, text)
                self.textObject.SetText("")
            else:
                for filename in self.fileObject.GetFilenames():
                    if self.wpath.isdir(filename):
                        self.albumPanel.OnDropText(x, y, u"dir:" + filename)
                    else:
                        isSupported = False
                        temp = filename.lower()
                        for ext in self.albumPanel.extList:
                            if temp.endswith(ext):
                                isSupported = True
                                break
                        if isSupported:
                            self.albumPanel.OnDropText(x, y, u"file:" + filename)
        return default
        
    def OnDrop(self, x, y):
        return True

    def OnDragOver(self, x, y, default):
        return wx.DragCopy

class TaggerDropSource(wx.DropSource):

    def __init__(self, parent, albumPanel):
        wx.DropSource.__init__(self, parent)
        self.albumPanel = albumPanel

    def GiveFeedback(self, effect):
        self.albumPanel.checkForScrolling()
        return False

class AlbumPanel(wx.Panel):


    """
    Each tree node can store extra data (SetPyData, GetPyData).  Here's what we store:
    NODE                 - PYDATA               - DRAGTEXT
    * pendingFiles       - Album                - n/a
     * a file            - TPFile               - fileid:<fileid>
    * unmatchedFiles     - Album                - n/a
     * a file            - TPFile               - fileid:<fileid>
    * errorFiles         - Album                - n/a
     * a file            - TPFile               - fileid:<fileid>
    * clusters                                  - n/a
     * a cluster         - ClusterAlbum         - cluster:<clusterid>
      * a file           - TPFile               - fileid:<fileid>
    * "albums"           - None                 - albumsNodeDragText
    * an album           - Album                - album:<mbid>
     * an unlinked track - Track                - n/a
     * a linked track    - Track                - fileid:<fileid>
     * unmatched files   - AlbumUnmatched       - albumunmatched:<mbid>
      * a file           - TPFile               - fileid:<fileid>
    """

    menuClearErrorId = wx.NewId()
    menuForceSaveId = wx.NewId()
    menuReloadAlbumId = wx.NewId()
    menuGenerateCuesheetId = wx.NewId()
    menuGeneratePlaylistId = wx.NewId()
    menuAnalyzeId = wx.NewId()

    def __init__(self, parent, config, id, extList):
        # Borders look dumb on windows, and dumb without on linux.
        if sys.platform == "win32":
           border = wx.NO_BORDER
        else:
           border = wx.SIMPLE_BORDER
        wx.Panel.__init__(self, parent, id, style=border)

        self.defaultColProps = [ .5, .1, .4]
        self.albumManager = None

        self.nodesToRemove = 0

        # Indexes mbids -> treeNodes
        self.albumIndex = {}

        # indexes mbids -> treenodes
        self.trackIndex = {}

        # indexes fileids -> treenodes
        self.fileIndex = {}

        # indexes unmatched album ids -> treenodes
        self.unmatchedIndex = {}

        self.savedColor = None
        self.savedFontcolor = None
        self.colors = [ [ 0x83, 0x83, 0xda], [ 0x90, 0x90, 0xda], [ 0xa8, 0xa8, 0xda], [ 0xb0, 0xb0, 0xda], \
                   [ 0xb8, 0xb8, 0xda], [ 0xc0, 0xc0, 0xda], [ 0xc8, 0xc8, 0xda], [ 0xd0, 0xd0, 0xda], \
                   [ 0xd8, 0xd8, 0xd8], [ 0xf0, 0xf0, 0xf0], [ 0xff, 0xff, 0xff] ]
        self.config = config
        self.tunePimp = config.getTunePimp()
        self.frame = config.getTagger()
        self.extList = extList
        self.tree = wx.gizmos.TreeListCtrl(self, treeControlId, style = wx.TR_DEFAULT_STYLE |
                                   wx.TR_HIDE_ROOT | wx.TR_NO_LINES | wx.TR_HAS_BUTTONS |
                                   wx.SIMPLE_BORDER | wx.TR_MULTIPLE)
        self.tree.SetLineSpacing(1)
        wx.EVT_SIZE(self, self.OnSize)
        wx.EVT_LIST_COL_END_DRAG(self.tree, treeControlId, self.OnColEndDrag)
        wx.EVT_TREE_BEGIN_DRAG(self.tree, treeControlId, self.OnDragInit)
        wx.EVT_TREE_SEL_CHANGED(self.tree, treeControlId, self.OnSelChanged)
        wx.EVT_MENU(self, self.menuClearErrorId, self.OnClearError)
        wx.EVT_MENU(self, self.menuForceSaveId, self.OnForceSave)
        wx.EVT_MENU(self, self.menuReloadAlbumId, self.OnReloadAlbum)
        wx.EVT_MENU(self, self.menuGenerateCuesheetId, self.OnGenerateCuesheet)
        wx.EVT_MENU(self, self.menuGeneratePlaylistId, self.OnGeneratePlaylist)
        wx.EVT_MENU(self, self.menuAnalyzeId, self.OnAnalyze)
        wx.EVT_TREE_ITEM_RIGHT_CLICK(self, treeControlId, self.OnRightDown)
        wx.EVT_TREE_DELETE_ITEM(self, treeControlId, self.OnTreeDeleteItem)
        
        self.tree.Bind(wx.EVT_TREE_KEY_DOWN, self.OnKeyDown)

        self.SetDropTarget(AlbumDropTarget(self, self.config))

        boxSizer = wx.BoxSizer(wx.VERTICAL)
        boxSizer.Add(self.tree, 1, wx.EXPAND)
        self.SetSizer(boxSizer)

        isz = (13,13)
        il = wx.ImageList(isz[0], isz[1])
        self.folderIconIndex = il.Add(wx.ArtProvider_GetBitmap(wx.ART_FOLDER,      wx.ART_OTHER, isz))
        self.openFolderIconIndex = il.Add(wx.ArtProvider_GetBitmap(wx.ART_FILE_OPEN,   wx.ART_OTHER, isz))
        self.fileIconIndex = il.Add(wx.ArtProvider_GetBitmap(wx.ART_REPORT_VIEW, wx.ART_OTHER, isz))
        self.cdIconIndex = il.Add(images.getCDIconBitmap())
        self.noFileIconIndex = il.Add(images.getNoFileIconBitmap())
        self.unknownTrackIconIndex = il.Add(images.getUnknownTrackIconBitmap())
        self.savedTrackIconIndex = il.Add(images.getSavedTrackIconBitmap())
        self.pendingTrackIconIndex = il.Add(images.getPendingTrackIconBitmap())
        self.errorTrackIconIndex = il.Add(images.getHasErrorTrackIconBitmap())
        self.similarityIcons = []
        self.similarityIcons.append(il.Add(images.get50Bitmap()))
        self.similarityIcons.append(il.Add(images.get60Bitmap()))
        self.similarityIcons.append(il.Add(images.get70Bitmap()))
        self.similarityIcons.append(il.Add(images.get80Bitmap()))
        self.similarityIcons.append(il.Add(images.get90Bitmap()))
        self.similarityIcons.append(il.Add(images.get100Bitmap()))

        self.tree.SetImageList(il)
        self.il = il

        # These default values will only be used in case the config gets borked
        try:
            self.colProps = self.config.persistColumnProps.split("/")
            if len(self.colProps) != 3:
                raise ValueError

            self.colProps = [float(val) for val in self.colProps]
            for prop in self.colProps:
                if prop <= 0:
                    raise ValueError
        except ValueError:
            self.colProps = self.defaultColProps

        # create some columns
        self.tree.AddColumn(_("Album"))
        self.tree.AddColumn(_("Time"))
        self.tree.AddColumn(_("Artist"))
        self.tree.SetMainColumn(0) # the one with the tree in it...
        self.OnSize(None) # Set the column widths

        self.root = self.tree.AddRoot("")
        self.tree.Expand(self.root)

        # The "albums" node has no data object associated with it
        child = self.tree.AppendItem(self.root, _("Albums"))
        self.tree.SetPyData(child, None)
        self.tree.SetItemHasChildren(child, True)
        self.tree.SetItemImage(child, self.folderIconIndex, which = wx.TreeItemIcon_Normal)
        self.tree.SetItemImage(child, self.openFolderIconIndex, which = wx.TreeItemIcon_Expanded)
        self.tree.Expand(child)
        self.albumsNode = child

        self.Fit()

        self.settingsChanged()

    def dumpState(self, n):
        n("albumpanel state")
        n = n.nest()
        n("albumIndex = %s" % repr(self.albumIndex))
        n("trackIndex = %s" % repr(self.trackIndex))
        n("fileIndex = %s" % repr(self.fileIndex))
        n("unmatchedIndex = %s" % repr(self.unmatchedIndex))

        def walktree(node, n):
            n("%s = %s" % (repr(node), repr(self.tree.GetItemText(node))))
            n = n.nest()
            (node, cookie) = self.tree.GetFirstChild(node)
            while node:
                walktree(node, n)
                node = self.tree.GetNextSibling(node)

        walktree(self.tree.GetRootItem(), n)

    def setAlbumManager(self, albumManager):
        self.albumManager = albumManager

    def setTPManager(self, tpmanager):
        self.tpmanager = tpmanager

    def settingsChanged(self):

        self.colors[0] = [ self.config.settingBadMatchBGColor.Blue(), 
                           self.config.settingBadMatchBGColor.Green(), 
                           self.config.settingBadMatchBGColor.Red() ]
        self.colors[numColorGradients] = [ self.config.settingGoodMatchBGColor.Blue(), 
                                           self.config.settingGoodMatchBGColor.Green(), 
                                           self.config.settingGoodMatchBGColor.Red() ]
        for i in xrange(1, numColorGradients):
            for j in xrange(3):
                step = (self.colors[numColorGradients][j] - self.colors[0][j]) / numColorGradients 
                self.colors[i][j] = self.colors[0][j] + (step * i)

        for item in self.fileIndex.values():
            self.setTrackIcon(item)

    def OnKeyDown(self, event):
        keyEvent = event.GetKeyEvent()
        if keyEvent.GetKeyCode() == wx.WXK_DELETE:
            if (self.config.toolbar.isRemoveEnabled()):
                self.remove()
        else:
            event.Skip()
        
    def GetPyData(self, node):
        debug.debug("GetPyData for %s" % repr(node), "wx.getpydata")
        debug.debug("GetPyData text=%s" % self.tree.GetItemText(node), "wx.getpydata")
        answer = self.tree.GetPyData(node)
        debug.debug("GetPyData for %s = %s" % (repr(node), repr(answer)), "wx.getpydata")
        return answer

    def DeleteNode(self, node):
        t = self.tree.GetItemText(node)
        debug.debug("DeleteNode %s %s" % (repr(node), repr(t)), "wx.deletenode")

        def checkDict(name, dict):
            debug.debug("Checking self.%s ..." % name)
            if node in dict.values():
                debug.debug("Error!  Node to be deleted (%s, %s) is still in self.%s"
                    % ( repr(node), repr(t), name ))

        if debug.isDebugClassEnabled("wx.deletenode.check.before"):
            checkDict("albumIndex", self.albumIndex)
            checkDict("fileIndex", self.fileIndex)
            checkDict("trackIndex", self.trackIndex)
            checkDict("unmatchedIndex", self.unmatchedIndex)

        if self.tree.IsSelected(node):
            debug.debug("DeleteNode deleting selected node")
            self.setCurrentObject(None)
            self.tree.Unselect()

        #dellist = []
        #def walktree(node):
        #    (node, cookie) = self.tree.GetFirstChild(node)
        #    while node:
        #       dellist.append(node)
        #       node = self.tree.GetNextSibling(node)
        #       walktree(node)
        #walktree(node)
        #dellist.reverse()
        #for n in dellist: self.tree.Delete(n)

        # TODO self.tree.DeleteChildren(node) - recursively?
        self.tree.Delete(node)
        debug.debug("DeleteNode done", "wx.deletenode")
        node = None

        if debug.isDebugClassEnabled("wx.deletenode.check.after"):
            checkDict("albumIndex", self.albumIndex)
            checkDict("fileIndex", self.fileIndex)
            checkDict("trackIndex", self.trackIndex)
            checkDict("unmatchedIndex", self.unmatchedIndex)

    def OnSelChanged(self, evt):
        """A new tree node has been selected

        NOTE: this doesn't tell us about all changes to what's selected;
        for example, if the currently selected node is deleted, this is not
        called."""

        sel = self.getCurrentSelection()
        self.showCoverArt(sel)
        self.setCurrentObject(sel)

    def setCurrentObject(self, sel):
        """Set the currently active object (e.g. a TPFile, an Album, etc)"""

        save = remove = analyze = False

        if not isinstance(sel, list):
            sel = [sel]
        
        for obj in sel:
            debug.debug("setCurrentObject: %s" % repr(obj))
            n = obj.__class__.__name__
            if n == "Album":
                if not obj.isSpecial():
                    if obj.getNumUnsavedTracks() > 0:
                        save = True
                    remove = True
            elif n == "Track":
                if obj.isLinked():
                    remove = True
                if obj.hasChanged():
                    save = True
            elif n == "AlbumUnmatched":
                remove = True
            elif n == "Error":
                remove = True
            elif n == "TPFile":
                remove = True
                if obj.getPUID() == None:
                    analyze = True
            elif n == "ClusterAlbum":
                remove = True

        self.frame.enableSave(save)
        self.frame.enableRemove(remove)
        self.frame.enableAnalyze(analyze)

        if len(sel) == 1:
            data = sel[0]
        else:
            data = None

        wx.PostEvent(self.frame, events.ShowItemDetailsEvent(data))
        wx.WakeUpIdle()

    def OnTreeDeleteItem(self, evt):

        debug.debug("OnTreeDeleteItem: evt=%s" % repr(evt), "wx.delete")
        debug.debug("OnTreeDeleteItem: evt item=%s" % repr(evt.GetItem()), "wx.delete")
        data = self.GetPyData(evt.GetItem())
        debug.debug("OnTreeDeleteItem: evt item data=%s" % repr(data), "wx.delete")

    def OnSize(self, evt):
        self.tree.SetSize(self.GetSize())
        for i in xrange(3):
            width = int(self.colProps[i] * self.GetSize().GetWidth())
            if width > 0:
                self.tree.SetColumnWidth(i, width)

    def checkForScrolling(self):
        x, y = wx.GetMousePosition()
        x, y = self.tree.ScreenToClientXY(x, y)
        self.insideLowerScrollRect(wx.Point(x, y))
        self.insideUpperScrollRect(wx.Point(x, y))

    def insideUpperScrollRect(self, point):
        if self.tree.GetMainWindow().HasScrollbar(wx.VERTICAL):
            headerRect = self.tree.GetHeaderWindow().GetRect()
            rect = self.tree.GetRect()
            x1 = rect.left
            y1 = rect.top - 20 + (headerRect.bottom - headerRect.top)
            x2 = rect.right
            y2 = rect.top + (headerRect.bottom - headerRect.top)

            if point.x >= x1 and point.x <= x2 and point.y >= y1 and point.y <= y2:
                sw = self.tree.GetMainWindow()
                x,y = sw.GetViewStart()
                xu,yu = sw.GetScrollPixelsPerUnit()
                sw.Scroll(x, y-yu)

    def insideLowerScrollRect(self, point):
        if self.tree.GetMainWindow().HasScrollbar(wx.VERTICAL):
            rect = self.tree.GetRect()
            rect.top = rect.bottom
            rect.bottom = rect.top + 20
            if rect.Inside(point):
                sw = self.tree.GetMainWindow()
                x,y = sw.GetViewStart()
                xu,yu = sw.GetScrollPixelsPerUnit()
                sw.Scroll(x, y+yu)

    def OnRightDown(self, event):
        node = event.GetItem()
        if node:
            self.tree.SelectItem(node)
            obj = self.GetPyData(node)
            popupMenu = None
            if obj.__class__.__name__ == "Track":
                popupMenu = wx.Menu()
                if (obj.hasError()):
                    popupMenu.Append(self.menuClearErrorId, _("Clear Error"))
                popupMenu.Append(self.menuForceSaveId, _("Force Save"))

            elif obj.__class__.__name__ == "TPFile" and obj.getPUID() == None:
                popupMenu = wx.Menu()
                item = popupMenu.Append(self.menuAnalyzeId, _("Analyze"))
                if self.config.disableFingerprinting:
                    item.Enable(False)

            elif obj.__class__.__name__ == "Album" and not obj.isSpecial():
                popupMenu = wx.Menu()
                popupMenu.Append(self.menuReloadAlbumId, _("Reload album from main server"))
                popupMenu.Append(self.menuForceSaveId, _("Force Save"))
                if obj.getDuration() and obj.getTrack(0) and obj.getTrack(0).getFileId() >= 0:
                    popupMenu.Append(self.menuGenerateCuesheetId, _("Generate cuesheet"))
                popupMenu.Append(self.menuGeneratePlaylistId, _("Generate playlist"))

            if popupMenu:
                pt = event.GetPoint()
                headerRect = self.tree.GetHeaderWindow().GetRect()
                self.PopupMenuXY(popupMenu, pt[0], pt[1] + headerRect.bottom - headerRect.top)

    def OnClearError(self, event):
        sel = self.getCurrentSelection()
        for obj in sel:
            if obj.__class__.__name__ == "Track":
                if obj.hasError() and obj.isLinked():
                    wx.PostEvent(self.frame, events.ClearErrorEvent(obj.getFileId()))
                    wx.WakeUpIdle()

    def OnForceSave(self, event):
        sel = self.getCurrentSelection()
        for obj in sel:
            if obj.__class__.__name__ == "Track":
                if obj.isLinked():
                    wx.PostEvent(self.frame, events.ForceSaveEvent(obj.getFileId()))
                    wx.WakeUpIdle()
            if obj.__class__.__name__ == "Album":
                for track in obj.tracks:
                    if track.isLinked():
                        wx.PostEvent(self.frame, events.ForceSaveEvent(track.getFileId()))
                        wx.WakeUpIdle()

    def OnReloadAlbum(self, event):
        sel = self.getCurrentSelection()
        for obj in sel:
            if obj.__class__.__name__ == "Album":
                wx.PostEvent(self.frame, events.ReloadAlbumEvent(obj.getId()))
                wx.WakeUpIdle()

    def OnGenerateCuesheet(self, event):
        sel = self.getCurrentSelection()
        for obj in sel:
            if obj.__class__.__name__ == "Album":
                wx.PostEvent(self.frame, events.GenerateCuesheetEvent(obj.getId()))
                wx.WakeUpIdle()
                
    def OnGeneratePlaylist(self, event):
        sel = self.getCurrentSelection()
        for obj in sel:
            if obj.__class__.__name__ == "Album":
                wx.PostEvent(self.frame, events.GeneratePlaylistEvent(obj.getId()))
                wx.WakeUpIdle()
                
    def analyze(self):                
        sel = self.getCurrentSelection()
        fileIds = []
        for obj in sel:
            if obj.__class__.__name__ == "TPFile":
                if obj.getPUID() == None:
                    fileIds.append(obj.getFileId())
        if len(fileIds) > 0:                
            wx.PostEvent(self.frame, events.AnalyzeFile2Event(fileIds))
            wx.WakeUpIdle()
                
    def OnAnalyze(self, event):
        self.analyze()
                
    def OnColEndDrag(self, event):
        self.colProps[event.GetColumn()] = float(self.tree.GetColumnWidth(event.GetColumn())) / float(self.GetSize().GetWidth())
        self.config.persistColumnProps = '/'.join([str(v) for v in self.colProps])

    def OnDragInit(self, event):

        if (event.GetItem() == self.albumIndex[albummanager.pendingFiles]):
            event.Veto()
            event.Skip()
            return

        dragText = ""
        obj = self.GetPyData(event.GetItem())

        if obj.__class__.__name__ == "Track":
            if obj.isLinked():
                dragText = "fileid:" + str(obj.getFileId())
        if obj.__class__.__name__ == "TPFile":
            dragText = "fileid:" + str(obj.getFileId())
        elif obj.__class__.__name__ == "Album":
            dragText = "album:" + obj.getId()
        elif obj.__class__.__name__ == "AlbumUnmatched":
            dragText = "albumunmatched:" + obj.getId()
        elif obj.__class__.__name__ == "ClusterAlbum":
            dragText = "cluster:" + obj.getId()
        elif event.GetItem() == self.albumsNode:
            dragText = albumsNodeDragText

        if dragText:
            tdo = wx.PyTextDataObject(dragText)
            tds = TaggerDropSource(self.tree, self)
            tds.SetData(tdo)
            tds.DoDragDrop()
        else:
            event.Veto()

        event.Skip()

    def OnDropText(self, x, y, data):
        """
        Called when a drag+drop operation ends - i.e. the "drop"

        We should try to make sense of as many as possible of these
        combinations.  That's a lot of combinations.

        Draggable items:
            file:<filename>
            files:<filenames separated by \n>
            dir:<dirname>
            dirs:<dirnames separated by \n>
            album:<mbid> (unmatched files)
                fileid:<int> (unmatched file)
            album:<mbid> (error files)
                fileid:<int> (error file)
            album:<mbid> (clusters)
                cluster:<clusterid>
                    fileid:<int> (clustered file)
            albumsNodeDragText
            album:<mbid> (real album)
                fileid:<int> (album matched file)
                albumunmatched:<mbid>
                    fileid:<int> (album unmatched file)

        Drop targets:
            empty space
            pendingFiles album
                pending file
            unmatched files album
                unmatched file
            clusters album
                cluster
                    clustered track
            albumsNode
            album
                album track (linked)
                album track (unlinked)
                unmatched files
                    unmatched file on album
        """

        assert(data.__class__.__name__ == 'unicode')

        # If we can't find a tree object, bail
        try:
            (xSize, ySize) = self.tree.GetHeaderWindow().GetSizeTuple()
            dropTreeItem = self.tree.HitTest(wx.Point(x, y - ySize))[0]
            dropTreeObj = self.GetPyData(dropTreeItem)
        except:
            return

        isPendingFiles = (dropTreeObj == self.albumManager.get(albummanager.pendingFiles))
        isUnmatchedFiles = (dropTreeObj == self.albumManager.get(albummanager.unmatchedFiles))
        isErrorFiles = (dropTreeObj == self.albumManager.get(albummanager.errorFiles))
        isClusteredFiles = (dropTreeObj == self.albumManager.get(albummanager.clusteredFiles))

        # Right now I can't think of any use for dragging anything onto the
        # "Album clusters" node
        if isClusteredFiles or isErrorFiles: return
        # Nor "albums"
        if dropTreeItem == self.albumsNode: return

        debug.debug("OnDropText: dropping %s onto %s" % (repr(data), repr(dropTreeObj)),
            "albumpanel.drop")

        if isPendingFiles or dropTreeObj is None:
            debug.debug("Dropping onto pendingfiles", "albumpanel.drop")

            if data.startswith(u"file:"):
                fileName = data[5:]
                wx.PostEvent(self.frame, events.FileAddEvent([fileName]))
                wx.WakeUpIdle()
                return

            if data.startswith(u"files:"):
                fileNames = data[6:].split(u"\n")
                wx.PostEvent(self.frame, events.FileAddEvent(fileNames))
                wx.WakeUpIdle()
                return

            if data.startswith(u"dir:"):
                dirName = data[4:]
                wx.PostEvent(self.frame, events.DirAddEvent([dirName]))
                wx.WakeUpIdle()
                return

            if data.startswith(u"dirs:"):
                dirNames = data[5:].split(u"\n")
                wx.PostEvent(self.frame, events.DirAddEvent(dirNames))
                wx.WakeUpIdle()
                return

            if data.startswith(u"fileid:"):
                fileId = int(data[7:])
                wx.PostEvent(self.frame, events.AnalyzeFileEvent([fileId]))
                wx.WakeUpIdle()
                return

            if data.startswith(u"album:") and dropTreeObj.__class__.__name__ == "Album":
                src_al = self.albumManager.get( data[data.find(":")+1:] )
                if src_al == self.albumManager.get(albummanager.errorFiles):
                    wx.PostEvent(self.frame, events.AnalyzeFileEvent(src_al.getUnmatchedFiles()))
                    wx.WakeUpIdle()
                    return

            # Nothing else should be dropped to pending files
            return

        # No files or directories should get dropped onto unmatched or clustered files
        if ( \
                data.startswith(u"file:") \
                or data.startswith(u"files:") \
                or data.startswith(u"dir:") \
                or data.startswith(u"dirs:") \
            ) and \
            (isUnmatchedFiles or isClusteredFiles or isErrorFiles):
            return

        if data.startswith(u"albumunmatched:") and isUnmatchedFiles:
            unmatchedId = data[data.find(":")+1:]
            wx.PostEvent(self.frame, events.MoveAlbumUnmatchedToUnmatchedEvent(unmatchedId))
            wx.WakeUpIdle()
            return

        if data.startswith(u"fileid:") and isUnmatchedFiles:
            fileId = int(data[data.find(":")+1:])
            wx.PostEvent(self.frame, events.MoveFilesToUnmatchedEvent([fileId]))
            wx.WakeUpIdle()
            return

        if data.startswith(u"file:") and dropTreeObj.__class__.__name__ == "Track":
            fileName = data[data.find(":")+1:]
            album = dropTreeObj.getAlbum()
            if album:
                seq = album.findSeqOfTrack(dropTreeObj.getId())
                if seq != None: 
                    wx.PostEvent(self.frame, events.FileAddEvent([fileName], album.getId(), seq))
                else:
                    wx.PostEvent(self.frame, events.FileAddEvent([fileName]))
            else:
                wx.PostEvent(self.frame, events.FileAddEvent([fileName]))
            wx.WakeUpIdle()
            return

        if data.startswith(u"fileid:") and dropTreeObj.__class__.__name__ == "Track":
            fileId = int(data[data.find(":")+1:])
            parentItem = self.tree.GetItemParent(dropTreeItem)
            if parentItem:
                parent = self.GetPyData(parentItem)
                if parent and parent.__class__.__name__ == "Album" and \
                    self.tree.GetItemText(parentItem).startswith(albummanager.unmatchedFilesText):
                        pass
                else:
                    if parent and parent.__class__.__name__ == "Album":
                        wx.PostEvent(self.frame, events.LinkFileEvent(dropTreeObj.getId(),
                                    fileId))
                        wx.WakeUpIdle()
                    else:
                        if parent and parent.__class__.__name__ == "ClusterAlbum":
                            wx.PostEvent(self.frame, events.AddFileToClusterEvent(fileId,
                                        parent.getId()))
                            wx.WakeUpIdle()

            return

        if data.startswith(u"fileid:") and dropTreeObj.__class__.__name__ == "ClusterAlbum":
            fileId = int(data[data.find(":")+1:])
            wx.PostEvent(self.frame, events.AddFileToClusterEvent(fileId, dropTreeObj.getId()))
            wx.WakeUpIdle()
            return

        if data.startswith(u"cluster:") and isUnmatchedFiles:

            clusterId = data[8:]
            node = None
            try:
                node = self.albumIndex[clusterId]
            except KeyError:
                return

            clusterAlbum = self.GetPyData(node)
            if clusterAlbum:
                fileList = clusterAlbum.getFileIds()
                wx.PostEvent(self.frame, events.MoveFilesToUnmatchedEvent(fileList))
                wx.WakeUpIdle()
            return

        if data.startswith(u"cluster:") and dropTreeObj.__class__.__name__ == "Album" \
            and not isClusteredFiles:
            clusterId = data[8:]
            node = None
            try:
                node = self.albumIndex[clusterId]
            except KeyError:
                return

            clusterAlbum = self.GetPyData(node)
            if clusterAlbum:
                fileList = clusterAlbum.getFileIds()
                wx.PostEvent(self.frame, events.MatchFileIdsToAlbumEvent(fileList, dropTreeObj.getId()))
                wx.WakeUpIdle()
            return

        if data.startswith(u"files:") and dropTreeObj.__class__.__name__ == "Album":
            fileList = data[6:].split("\n")
            albumId = dropTreeObj.getId()
            if albumId:
                wx.PostEvent(self.frame, events.FileAddEvent(fileList, albumId, -1))
            else:
                wx.PostEvent(self.frame, events.FileAddEvent(fileList))
            wx.WakeUpIdle()
            return

        if data.startswith(u"file:") and dropTreeObj.__class__.__name__ == "Album":
            fileName = data[data.find(":")+1:]
            albumId = dropTreeObj.getId()
            if albumId:
                wx.PostEvent(self.frame, events.FileAddEvent([fileName], albumId, -1))
            else:
                wx.PostEvent(self.frame, events.FileAddEvent([fileName]))
            wx.WakeUpIdle()
            return

        if data.startswith(u"dir:") and dropTreeObj.__class__.__name__ == "Album":
            dirName = data[data.find(":")+1:]
            wx.PostEvent(self.frame, events.MatchDirToAlbumEvent(dirName, dropTreeObj.getId()))
            wx.WakeUpIdle()
            return

        if data.startswith(u"dirs:") and dropTreeObj.__class__.__name__ == "Album":
            dirList = data[5:].split("\n")
            albumId = dropTreeObj.getId()
            for dirName in dirList:
                wx.PostEvent(self.frame, events.MatchDirToAlbumEvent(dirName, albumId))
            wx.WakeUpIdle()
            return

        if data.startswith(u"fileid:") and dropTreeObj.__class__.__name__ == "Album":
            fileId = int(data[data.find(":")+1:])
            tpTrack = self.tunePimp.getTrack(fileId)
            tpTrack.lock()
            fileName = tpTrack.getFileName()
            tpTrack.unlock()
            self.tunePimp.releaseTrack(tpTrack)
            wx.PostEvent(self.frame, events.MatchFilesToAlbumEvent([fileName], dropTreeObj.getId()))
            wx.WakeUpIdle()
            return

        #if data.startswith(u"album:") and isUnmatchedFiles:
        #    # XXX aren't we dragging an album?  this says "move clusters"
        #    wx.PostEvent(self.frame, events.MoveClustersToUnmatchedEvent())
        #    wx.WakeUpIdle()
        #    return

        if data.startswith(u"album:") and dropTreeObj.__class__.__name__ == "Album":
            sourceAlbum = self.albumManager.get(data[data.find(":")+1:])
            if sourceAlbum == dropTreeObj:
                return

            fileIds = []

            for i in xrange(0, sourceAlbum.getNumTracks()):
                t = sourceAlbum.getTrack(i)
                if t and t.isLinked():
                    fileIds.append(t.getFileId())
            for fileId in sourceAlbum.getUnmatchedFiles():
                fileIds.append(fileId)

            if isUnmatchedFiles:
                for fileId in fileIds:
                    tpfile = self.tpmanager.findFile(fileId)
                    tpfile.moveToAlbumAsUnlinked(dropTreeObj)
                return

            for fileId in fileIds:
                tpTrack = self.tunePimp.getTrack(fileId)
                tpTrack.lock()
                fileName = tpTrack.getFileName()
                tpTrack.unlock()
                self.tunePimp.releaseTrack(tpTrack)
                wx.PostEvent(self.frame, events.MatchFilesToAlbumEvent([fileName], dropTreeObj.getId()))

            wx.WakeUpIdle()
            return

        # TODO: drop cluster onto cluster (merge clusters)
        # TODO: drop cluster onto "unmatched albums" (un-cluster)

        debug.debug("OnDropText: dropping %s onto %s not handled" % (repr(data), repr(dropTreeObj)),
            "albumpanel.drop.unhandled")

    def getCurrentSelection(self):
        """Returns the PyData associated with the currently selected node, or None.
        
        Called internally, and by the parent frame, and by the metadata panel."""

        node = self.tree.GetSelections()
        debug.debug("getCurrentSelection node = %s" % repr(node))
        return [self.GetPyData(n) for n in node]

    def save(self):
        """Called when the 'save' icon has been clicked in the toolbar"""

        list = self.getCurrentSelection()
        if not len(list):
           debug.debug("albumpanel save called, but nothing is selected")
           return

        tracksToSave = []

        for sel in list:
            if sel.__class__.__name__ == "Album":
                if sel.isSpecial():
                    continue

                num = sel.getNumTracks()
                for i in xrange(num):
                    tr = sel.getTrack(i)
                    if tr and tr.isLinked() and tr.hasChanged():
                        tracksToSave.append(tr.getFileId())

            if sel.__class__.__name__ == "Track":
                if sel.isLinked() and sel.hasChanged():
                    tracksToSave.append(sel.getFileId())

        debug.debug("Posting SaveFileEvent(%s)" % repr(tracksToSave), "event.save")
        wx.PostEvent(self.frame, events.SaveFileEvent(tracksToSave))
        wx.WakeUpIdle()

    def remove(self):
        """Called when the 'X' icon has been clicked in the toolbar"""

        list = self.getCurrentSelection()
        if not len(list):
            debug.debug("albumpanel remove called, but nothing is selected")
            return

        tracks = []
        albums = []
        for sel in list:
            if sel.__class__.__name__ == "TPFile" or sel.__class__.__name__ == "Track":
                tracks.append(sel)
            elif sel:
                albums.append(sel)

        nextNode = None
        for sel in tracks:
            if sel.__class__.__name__ == "TPFile":
                node = self.tree.GetSelection()
                nextNode = self.tree.GetNextSibling(node)
                if not nextNode:
                    nextNode = self.tree.GetPrevSibling(node)
                wx.PostEvent(self.frame, events.RemoveTPFileCommand(sel.getFileId()))
            elif sel.__class__.__name__ == "Track":
                fileId = sel.getFileId()
                if sel.isLinked():
                    wx.PostEvent(self.frame, events.RemoveTPFileCommand(fileId))

        for sel in albums:
            id = sel.getId()
            if sel.__class__.__name__ == "Album" and not sel.isSpecial():
                self.nodesToRemove += 1
                wx.PostEvent(self.frame, events.AlbumRemoveEvent(id))
            elif sel.__class__.__name__ == "AlbumUnmatched":
                wx.PostEvent(self.frame, events.AlbumRemoveUnmatchedFilesEvent(id))
            elif sel.__class__.__name__ == "ClusterAlbum":
                wx.PostEvent(self.frame, events.RemoveClusterAlbumEvent(id))
            else:
                debug.debug("albumpanel remove unhandled: %s" % repr(sel))
            
        wx.WakeUpIdle()
        if nextNode and self.nodesToRemove == 0:
            self.tree.SelectItem(nextNode)

        

    def setTrackIcon(self, node):

        if not node:
            return

        data = self.GetPyData(node)
        if not data:
            self.tree.SetItemImage(node, self.noFileIconIndex)
            self.tree.SetItemBackgroundColour(node, self.savedColor)
            self.tree.SetItemTextColour(node, self.savedFontColor)
            #self.tree.SetItemText(node, "", 1)
            return

        if data.__class__.__name__ == "AlbumUnmatched":
            self.tree.SetItemImage(node, self.unknownTrackIconIndex)
            self.tree.SetItemBackgroundColour(node, self.savedColor)
            self.tree.SetItemTextColour(node, self.savedFontColor)
            return

        if data.__class__.__name__ == "TPFile":
            index = self.unknownTrackIconIndex
            parent = self.tree.GetItemParent(node)
            if parent:
                pdata = self.GetPyData(parent)
                if pdata.__class__.__name__ == "Album" and pdata.getId() == albummanager.errorFiles:
                    index = self.errorTrackIconIndex
            
            self.tree.SetItemImage(node, index)
            self.tree.SetItemBackgroundColour(node, self.savedColor)
            self.tree.SetItemTextColour(node, self.savedFontColor)
            return

        tr = data
        if tr.hasError():
            self.tree.SetItemImage(node, self.errorTrackIconIndex)
            self.tree.SetItemBackgroundColour(node, self.savedColor)
            self.tree.SetItemTextColour(node, self.savedFontColor)
            return

        if not tr.isLinked():
            self.tree.SetItemImage(node, self.noFileIconIndex)
            self.tree.SetItemBackgroundColour(node, self.savedColor)
            self.tree.SetItemTextColour(node, self.savedFontColor)
            return

        if tr.isUnmatched():
            self.tree.SetItemImage(node, self.unknownTrackIconIndex)
            self.tree.SetItemBackgroundColour(node, self.savedColor)
            self.tree.SetItemTextColour(node, self.savedFontColor)
            return

        if tr.hasChanged():
            sim = tr.getSimilarity()
            index = 0
            if sim > 50:
                index = int((sim - 50) / numColorGradients)

            self.tree.SetItemImage(node, self.similarityIcons[index])
            temp = self.colors[int(sim / numColorGradients)]
            self.tree.SetItemBackgroundColour(node, wx.Color(temp[2], temp[1], temp[0]))
            self.tree.SetItemTextColour(node, self.config.settingFontColor)
        else:
            self.tree.SetItemImage(node, self.savedTrackIconIndex)
            self.tree.SetItemBackgroundColour(node, self.savedColor)
            self.tree.SetItemTextColour(node, self.savedFontColor)

        self.tree.SetItemText(node, tr.getDurationString(), 1)

    def showCoverArt(self, sel):
        asin = None
        if sel:
            if sel.__class__.__name__ == "unicode":
                asin = sel
            else:
                for data in sel:
                    prev_asin = asin
                    try:
                        if data.__class__.__name__ == "Track":
                            asin = data.getAlbum().getAmazonAsin()
                        elif data.__class__.__name__ == "Album":
                            asin = data.getAmazonAsin()
                        elif data.__class__.__name__ == "AlbumUnmatched":
                            asin = self.albumManager.get(data.getId()).getAmazonAsin()
                        else:
                            asin = u"<mblogo>"
                    except:
                        pass
                    if prev_asin != None and prev_asin != asin:
                        asin = u"<mblogo>"
                        break
        if asin == None:
            asin = u""
        wx.PostEvent(self.frame, events.CoverArtShowEvent(asin))
        wx.WakeUpIdle()

    def updateAlbumCount(self, node):

        data = self.GetPyData(node)
        if data and data.__class__.__name__ == "Track":
            node = self.tree.GetItemParent(node)
            data = self.GetPyData(node)

        if data and data.__class__.__name__ == "Album":
            pass
        else:
            if data and data.__class__.__name__ == "ClusterAlbum":
                text = data.getName() + " (%d)" % (data.getNumFiles())
                self.tree.SetItemText(node, text, 0)

    def updateTrackCounts(self):

        try:
            node = self.albumIndex[albummanager.pendingFiles]
        except KeyError:
            return

        count = self.tree.GetChildrenCount(node, False)
        self.tree.SetItemText(node, _(albummanager.pendingFilesText) + (u" (%d)" % count))

        try:
            node = self.albumIndex[albummanager.unmatchedFiles]
        except KeyError:
            return

        count = self.tree.GetChildrenCount(node, False)
        self.tree.SetItemText(node, _(albummanager.unmatchedFilesText) + (u" (%d)" % count))

        try:
            node = self.albumIndex[albummanager.errorFiles]
        except KeyError:
            return

        count = self.tree.GetChildrenCount(node, False)
        self.tree.SetItemText(node, _(albummanager.errorFilesText) + (u" (%d)" % count))

        try:
            node = self.albumIndex[albummanager.clusteredFiles]
        except KeyError:
            return

        count = self.tree.GetChildrenCount(node, False)
        self.tree.SetItemText(node, _(albummanager.clusteredFilesText) + (u" (%d)" % count))

    def addAlbumUnmatched(self, mbid, fileList):

        for fileId in fileList:
            self.moveFilesToUnmatched([ fileId ], mbid)

    def findOrCreateUnmatchedForAlbum(self, albumId):
        try:
            return self.unmatchedIndex[albumId]
        except KeyError:
            alNode = self.albumIndex[albumId]

            newNode = self.tree.AppendItem(alNode, "[ " + _("Unmatched files for this album") + " ]")
            self.tree.SetPyData(newNode, album.AlbumUnmatched(self.config, albumId))
            self.tree.SetItemText(newNode, "", 1)
            self.tree.SetItemText(newNode, "", 2)
            self.setTrackIcon(newNode)
            self.unmatchedIndex[albumId] = newNode

            return newNode

    def moveFilesToUnmatched(self, fileIdList, albumId=None):
        """
        Move one or more files either to "unmatched files" (albumId is None),
        or to "unmatched files for this album" (album is not None)
        """

        # Figure out where the unmatched file needs to go
        newNode = None
        if albumId:
            try:
                al = self.albumManager.get(albumId)
            except KeyError:
                return
        else:
            try:
                newNode = self.albumIndex[albummanager.unmatchedFiles]
                al = self.albumManager.get(albummanager.unmatchedFiles)
            except KeyError:
                return

        for fileId in fileIdList:
            tpfile = self.tpmanager.findFile(fileId)
            tpfile.moveToAlbumAsUnlinked(al)

    def moveFilesToError(self, fileIdList):
        """
        Move one or more files either to "error files"
        """

        # Figure out where the error file needs to go
        newNode = None
        try:
            newNode = self.albumIndex[albummanager.errorFiles]
            al = self.albumManager.get(albummanager.errorFiles)
        except KeyError:
            return

        for fileId in fileIdList:
            tpfile = self.tpmanager.findFile(fileId)
            tpfile.moveToAlbumAsUnlinked(al)

    def moveFilesToPending(self, fileIdList):
        """
        Move one or more files to "new files"
        """

        # Figure out where the error file needs to go
        newNode = None
        try:
            newNode = self.albumIndex[albummanager.pendingFiles]
            al = self.albumManager.get(albummanager.pendingFiles)
        except KeyError:
            return

        for fileId in fileIdList:
            tpfile = self.tpmanager.findFile(fileId)
            tpfile.moveToAlbumAsUnlinked(al)

    def linkFile(self, fileId, trackId):

        try:
            al = self.albumManager.getFromTrackId(trackId)
        except LookupError:
            return

        try:
            tpfile = self.tpmanager.findFile(fileId)
        except LookupError:
            return

        seq = al.findSeqOfTrack(trackId)
        tpfile.moveToAlbumAsLinked(al, seq)

    def addAlbum(self, mbid, show=False):

        if not self.savedColor:
            self.savedColor = self.tree.GetItemBackgroundColour(self.root)
            self.savedFontColor = self.tree.GetItemTextColour(self.root)

        if self.albumManager == None:
            return

        try:
            al = self.albumManager.get(mbid)
        except LookupError:
            return

        if al.isSpecial():
            # This is something of a hack; the "Albums" node is added to the tree
            # as soon as we start, but the nodes for the other preset items
            # (pendingFiles, unmatchedFiles, clusteredFiles) are added via this
            # "addAlbum" hook.  To get them in the right order, here's some
            # non-obvious code:
            idx = self.tree.GetChildrenCount(self.root, False)-1
            child = self.tree.InsertItemBefore(self.root, idx, al.getName())
        else:
            child = self.tree.AppendItem(self.albumsNode, al.getName())

        self.tree.SetPyData(child, al)
        self.tree.SetItemText(child, al.getDurationString(), 1)
        self.tree.SetItemText(child, al.getArtist().getName(), 2)
        self.tree.SetItemHasChildren(child, True)

        if al.isSpecial():
            self.tree.SetItemImage(child, self.folderIconIndex, which = wx.TreeItemIcon_Normal)
            self.tree.SetItemImage(child, self.openFolderIconIndex, which = wx.TreeItemIcon_Expanded)
        else:
            self.tree.SetItemImage(child, self.cdIconIndex, which = wx.TreeItemIcon_Normal)
            self.tree.SetItemImage(child, self.cdIconIndex, which = wx.TreeItemIcon_Expanded)
            self.tree.SetItemBold(child)

        self.albumIndex[mbid] = child

        for i in xrange(0, al.getNumTracks()):
            tr = al.getTrack(i)
            if tr:
                text = unicode(tr.getNum()) + u'. ' + tr.getName()
                item = self.tree.AppendItem(child, text)
                self.tree.SetPyData(item, tr)
                self.tree.SetItemText(item, tr.getDurationString(), 1)
                self.tree.SetItemText(item, tr.getArtist().getName(), 2)
                self.setTrackIcon(item)
                self.trackIndex[tr.getId()] = item

        if show or al.isSpecial():
            self.tree.Expand(child)
            self.tree.SelectItem(child)
            self.tree.EnsureVisible(child)
        self.updateTrackCounts()

    def albumRemoved(self, mbid):

        try:
            node = self.albumIndex[mbid]
        except LookupError:
            return

        nextNode = None
        if self.nodesToRemove == 1:
            nextNode = self.tree.GetNextSibling(node)
            if not nextNode:
                nextNode = self.tree.GetPrevSibling(node)

        child, cookie = self.tree.GetFirstChild(node)
        while child:
            tr = self.GetPyData(child)
            if tr and tr.getFileId() >= 0:
                del self.fileIndex[tr.getFileId()]
                wx.PostEvent(self.frame, events.RemoveTPFileCommand(tr.getFileId()))
            del self.trackIndex[tr.getId()]

            child = self.tree.GetNextSibling(child)

        wx.WakeUpIdle()

        del self.albumIndex[mbid]
        self.DeleteNode(node)

        self.nodesToRemove -= 1

        if nextNode:
            self.tree.SelectItem(nextNode)

    def updateFile(self, fileId):

        try:
            node = self.fileIndex[fileId]
        except KeyError:
            pass
        else:
            self.setTrackIcon(node)
            # Set the object again so that the toolbar gets updated too
            data = self.GetPyData(node)
            self.setCurrentObject(data)

    def updateAlbum(self, mbid):
        """
        Called when an album load has completed (maybe successfully, maybe not)
        """

        if self.albumManager == None:
            return

        try:
            al = self.albumManager.get(mbid)
        except LookupError:
            return

        node = None
        try:
            node = self.albumIndex[mbid]
        except KeyError:
            self.addAlbum(mbid)
            return

        self.tree.SetItemText(node, al.getName(), 0)
        self.tree.SetItemText(node, al.getDurationString(), 1)
        self.tree.SetItemText(node, al.getArtist().getName(), 2)

        for i in xrange(0, al.getNumTracks()):
             tr = al.getTrack(i)
             if tr:
                 text = unicode(tr.getNum()) + u'. ' + tr.getName()
                 trackNode = None
                 try:
                     trackNode = self.trackIndex[tr.getId()]
                 except KeyError:
                     pass

                 # TODO sort nodes properly
                 if trackNode:
                     self.tree.SetItemText(trackNode, text, 0)
                     self.tree.SetItemText(trackNode, tr.getDurationString(), 1)
                     self.tree.SetItemText(trackNode, tr.getArtist().getName(), 2)
                 else:
                     trackNode = self.tree.AppendItem(node, text)
                     self.tree.SetPyData(trackNode, tr)
                     self.tree.SetItemText(trackNode, tr.getDurationString(), 1)
                     self.tree.SetItemText(trackNode, tr.getArtist().getName(), 2)
                     self.setTrackIcon(trackNode)
                     self.trackIndex[tr.getId()] = trackNode

        self.albumManager.checkUnmatchedAlbumFiles(al)
        data = self.GetPyData(node)
        wx.PostEvent(self.frame, events.ShowItemDetailsEvent(data))
        wx.WakeUpIdle()
        self.updateTrackCounts()
        self.updateAlbumCount(node)
        
        sel = self.getCurrentSelection()
        if al in sel:        
            self.setCurrentObject(sel)

        self.checkLoadedAlbumAgainstClusters(al)

    def albumTrackCountsChanged(self, mbid):

        if self.albumManager == None:
            return

        try:
            al = self.albumManager.get(mbid)
        except LookupError:
            # If the album isn't there, it's presumably already gone, in which case
            # there should be a "remove album" event following shortly
            return

        node = None
        try:
            node = self.albumIndex[mbid]
        except KeyError:
            self.addAlbum(mbid)
            return

        text = ""

        if al.getId() == albummanager.pendingFiles:
            text = "%s (%d)" % (al.getName(), al.getNumUnmatchedFiles())
        elif al.getId() == albummanager.unmatchedFiles:
            text = "%s (%d)" % (al.getName(), al.getNumUnmatchedFiles())
        elif al.getId() == albummanager.errorFiles:
            text = "%s (%d)" % (al.getName(), al.getNumUnmatchedFiles())
        else:
            text = al.getName() + " (%d / %d" % (al.getNumLinkedTracks() , al.getNumTracks())
            unmatched = al.getNumUnmatchedFiles()
            if unmatched:
                    text = text + "; %d?" % (unmatched)
            unsaved = al.getNumUnsavedTracks()
            if unsaved:
                    text = text + "; %d*" % (unsaved)
            text = text + ")"

        self.tree.SetItemText(node, text, 0)

    def removeAlbumUnmatched(self, mbid):

        try:
            al = self.albumManager.get(mbid)
        except LookupError:
            return

        nextNode = None
        try:
            nextNode = self.albumIndex[mbid]
        except KeyError:
            return

        for fileId in al.getUnmatchedFiles():
            wx.PostEvent(self.frame, events.RemoveTPFileCommand(fileId))

        wx.WakeUpIdle()

        if nextNode:
            self.tree.SelectItem(nextNode)

    def moveAlbumUnmatchedToUnmatched(self, mbid):
        """Moves album-specific unmatched tracks to general unmatched tracks"""

        try:
            al = self.albumManager.get(mbid)
            unmatched_album = self.albumManager.get(albummanager.unmatchedFiles)
        except LookupError:
            return

        for fileId in al.getUnmatchedFiles():
            tpfile = self.tpmanager.findFile(fileId)
            tpfile.moveToAlbumAsUnlinked(unmatched_album)

    def clusterAdded(self, clusterAlbum):
        debug.debug("ui clusterAdded %s" % clusterAlbum.getId())

        try:
            node = self.albumIndex[albummanager.clusteredFiles]
        except KeyError:
            return

        child = self.tree.AppendItem(node, clusterAlbum.getName())
        self.tree.SetPyData(child, clusterAlbum)
        self.tree.SetItemText(child, "", 1)
        self.tree.SetItemText(child, clusterAlbum.getArtistName(), 2)

        self.tree.SetItemImage(child, self.cdIconIndex, which = wx.TreeItemIcon_Normal)
        self.tree.SetItemImage(child, self.cdIconIndex, which = wx.TreeItemIcon_Expanded)

        self.albumIndex[clusterAlbum.getId()] = child
        self.updateAlbumCount(child)

        # Update count on "clusters" node
        self.updateTrackCounts()

        self.tree.Expand(node)

    def clusterRemoved(self, clusterAlbum):
        debug.debug("ui clusterRemoved %s" % clusterAlbum.getId())
        try:
            node = self.albumIndex[clusterAlbum.getId()]
        except KeyError:
            return
        del self.albumIndex[clusterAlbum.getId()]
        self.DeleteNode(node)

        # Update count on "clusters" node
        self.updateTrackCounts()

    def clusterChanged(self, clusterAlbum):
        debug.debug("ui clusterChanged %s" % clusterAlbum.getId())
        try:
            node = self.albumIndex[clusterAlbum.getId()]
        except KeyError:
            return
        self.updateAlbumCount(node)

    def clusterFileAdded(self, clusterAlbum, fileId):
        debug.debug("ui clusterFileAdded %s %d" % (clusterAlbum.getId(), fileId))
        assert(fileId not in self.fileIndex)

        try:
            clusterNode = self.albumIndex[clusterAlbum.getId()]
        except KeyError:
            return

        tpfile = self.tpmanager.findFile(fileId)

        tpTrack = self.tunePimp.getTrack(fileId)
        tpTrack.lock()
        ldata = tpTrack.getLocalMetadata()
        fileName = tpTrack.getFileName()
        tpTrack.unlock()
        self.tunePimp.releaseTrack(tpTrack)

        text = unicode(ldata.trackNum) + u'. ' + ldata.track
        dur = u"%d:%02d" % ((ldata.duration / 60000), ((ldata.duration % 60000) / 1000))

        item = self.tree.AppendItem(clusterNode, text)
        self.tree.SetPyData(item, tpfile)
        self.tree.SetItemText(item, dur, 1)
        self.tree.SetItemText(item, ldata.artist, 2)
        self.setTrackIcon(item)

        self.fileIndex[fileId] = item

    def clusterFileRemoved(self, clusterAlbum, fileId):
        debug.debug("ui clusterFileRemoved %s %d" % (clusterAlbum.getId(), fileId))
        try:
            node = self.fileIndex[fileId]
        except KeyError:
            return
        del self.fileIndex[fileId]
        self.DeleteNode(node)

    def removeClusterAlbum(self, clusterAlbumId):
        try:
            node = self.albumIndex[clusterAlbumId]
        except KeyError:
            return

        nextNode = self.tree.GetNextSibling(node)
        if not nextNode:
            nextNode = self.tree.GetPrevSibling(node)

        clusterAlbum = self.GetPyData(node)
        if clusterAlbum:
            for fileId in clusterAlbum.getFileIds():
                wx.PostEvent(self.frame, events.RemoveTPFileCommand(fileId))

        wx.WakeUpIdle()

        if nextNode:
            self.tree.SelectItem(nextNode)

    def moveClustersToUnmatched(self):

        try:
            node = self.albumIndex[albummanager.clusteredFiles]
        except KeyError:
            return

        node, cookie = self.tree.GetFirstChild(node)
        if not node:
            return

        # Make a list first, so I don't traverse the tree while deleting items from it
        while node:
            clusterAlbum = self.GetPyData(node)
            if clusterAlbum:
                fileList = clusterAlbum.getFileIds()
                wx.PostEvent(self.frame, events.MoveFilesToUnmatchedEvent(fileList))

            node = self.tree.GetNextSibling(node)

        wx.WakeUpIdle()

    def checkLoadedAlbumAgainstClusters(self, album):
        """Automatically move clusters to newly added albums.
        """

        try:
            node = self.albumIndex[albummanager.clusteredFiles]
        except KeyError:
            return

        node, cookie = self.tree.GetFirstChild(node)
        if not node:
            return

        # TODO: similarity comparison needs to be UTF-8 safe!
        mdata = metadata.metadata(self.config.getTunePimp())
        while node:
            cluster = self.GetPyData(node)
            sim = 0.6 * mdata.similarity(album.getName().encode('ascii', 'replace'), \
                                          cluster.getName().encode('ascii', 'replace')) + \
                  0.35 * mdata.similarity(album.getArtist().getName().encode('ascii', 'replace'), \
                                          cluster.getArtistName().encode('ascii', 'replace'))
            if album.getNumTracks() == cluster.getNumFiles():
                sim += 0.05
            if sim >= self.config.settingAlbumLoadClusterMatchThreshold:
                fileList = cluster.getFileIds()
                wx.PostEvent(self.frame, events.MatchFileIdsToAlbumEvent(fileList, album.getId()))
                wx.WakeUpIdle()
                break
                
            node = self.tree.GetNextSibling(node)

    def albumFilesAdded(self, mbid, fileList):
        debug.debug("album files added: %s %s" % (mbid, fileList))
        
        al = self.albumManager.get(mbid)
        alnode = self.albumIndex[mbid]

        for fileId in fileList:
            assert(fileId not in self.fileIndex)
            tpfile = self.tpmanager.findFile(fileId)

            if al.getId() == albummanager.pendingFiles:
                tr = tpfile.getTrack()
                fileName = tr.getFileName()
                tpfile.releaseTrack(tr)

                trnode = self.tree.AppendItem(alnode, os.path.basename(fileName))
                self.tree.SetItemText(trnode, "?:??", 1)
                self.tree.SetItemText(trnode, "?", 2)
                self.tree.SetItemImage(trnode, self.pendingTrackIconIndex)
                self.tree.SetItemBackgroundColour(trnode, self.savedColor)
                self.fileIndex[fileId] = trnode

            elif al.getId() == albummanager.unmatchedFiles:
                tr = tpfile.getTrack()
                artistName = tr.getLocalMetadata().artist
                ldata = tr.getLocalMetadata()
                fileName = tr.getFileName()
                tpfile.releaseTrack(tr)

                dur = u"%d:%02d" % ((ldata.duration / 60000), ((ldata.duration % 60000) / 1000))

                trnode = self.tree.AppendItem(alnode, os.path.basename(fileName))
                self.tree.SetItemText(trnode, dur, 1)
                self.tree.SetItemText(trnode, artistName, 2)
                self.tree.SetItemImage(trnode, self.pendingTrackIconIndex)
                self.tree.SetItemBackgroundColour(trnode, self.savedColor)
                self.fileIndex[fileId] = trnode

                self.tree.SetPyData(trnode, tpfile)
                self.setTrackIcon(trnode)

            elif al.getId() == albummanager.errorFiles:
                tr = tpfile.getTrack()
                artistName = tr.getLocalMetadata().artist
                ldata = tr.getLocalMetadata()
                fileName = tr.getFileName()
                tpfile.releaseTrack(tr)

                dur = u"%d:%02d" % ((ldata.duration / 60000), ((ldata.duration % 60000) / 1000))

                trnode = self.tree.AppendItem(alnode, os.path.basename(fileName))
                self.tree.SetItemText(trnode, dur, 1)
                self.tree.SetItemText(trnode, artistName, 2)
                self.tree.SetItemImage(trnode, self.errorTrackIconIndex)
                self.tree.SetItemBackgroundColour(trnode, self.savedColor)
                self.fileIndex[fileId] = trnode

                self.tree.SetPyData(trnode, tpfile)
                self.setTrackIcon(trnode)

            elif not al.isSpecial():
                tr = tpfile.getTrack()
                trackid = tr.getServerMetadata().trackId
                tpfile.releaseTrack(tr)

                seq = al.findSeqOfTrack(trackid)
                if seq is not None:
                    # Joined to a track
                    trnode = self.trackIndex[trackid]
                    track = al.getTrackFromFileId(fileId)

                    self.fileIndex[fileId] = trnode
                    self.tree.SetPyData(trnode, track)
                    self.setTrackIcon(trnode)
                else:
                    # an unmatched file
                    unmatchedNode = self.findOrCreateUnmatchedForAlbum(al.getId())

                    tr = tpfile.getTrack()
                    artistName = tr.getLocalMetadata().artist
                    fileName = tr.getFileName()
                    ldata = tr.getLocalMetadata()
                    tpfile.releaseTrack(tr)

                    dur = u"%d:%02d" % ((ldata.duration / 60000), ((ldata.duration % 60000) / 1000))

                    trnode = self.tree.AppendItem(unmatchedNode, os.path.basename(fileName))
                    self.tree.SetItemText(trnode, dur, 1)
                    self.tree.SetItemText(trnode, artistName, 2)
                    self.tree.SetItemImage(trnode, self.pendingTrackIconIndex)
                    self.tree.SetItemBackgroundColour(trnode, self.savedColor)

                    self.fileIndex[fileId] = trnode
                    self.tree.SetPyData(trnode, tpfile)
                    self.setTrackIcon(trnode)

    def albumFilesRemoved(self, mbid, fileList):
        debug.debug("album files removed: %s %s" % (mbid, fileList))

        alnode = self.albumIndex[mbid]

        unmatchednode = None
        try:
            unmatchednode = self.unmatchedIndex[mbid]
        except KeyError:
            pass

        for fileId in fileList:
            assert(fileId in self.fileIndex)
            trnode = self.fileIndex[fileId]
            del self.fileIndex[fileId]

            if self.GetPyData(trnode).__class__.__name__ == "Track":
                self.setTrackIcon(trnode)
                data = self.GetPyData(trnode)
                wx.PostEvent(self.frame, events.ShowItemDetailsEvent(data))
                wx.WakeUpIdle()
            else:
                # album unmatched, error, pending or unmatched
                self.DeleteNode(trnode)

        if unmatchednode:
            if self.tree.GetChildrenCount(unmatchednode) == 0:
                del self.unmatchedIndex[mbid]
                self.DeleteNode(unmatchednode)

        wx.WakeUpIdle()

