# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED,G AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Robert Kaye
#
#
# ***** END LICENSE BLOCK *****
import wx, sys, os, locale, copy
from picard.ui import dirctrl, albumpanel, options
from picard import util, events, dircache, debug, wpath

class SelectHerePanel(wx.Panel):

    def __init__(self, parent):

        # Borders look dumb on windows, and dumb without on linux.
        border = wx.SIMPLE_BORDER
        if sys.platform == "win32":
           border = wx.NO_BORDER

        wx.Panel.__init__(self, parent, -1, wx.DefaultPosition, wx.DefaultSize, border)

        boxSizer = wx.BoxSizer(wx.HORIZONTAL)
        boxSizer.Add(wx.StaticText(self, -1, _("Select or drag a folder from below") + ":", 
                     wx.DefaultPosition, wx.DefaultSize, wx.ALIGN_CENTRE), 
                     0, wx.TOP | wx.BOTTOM | wx.LEFT, 4)
        self.SetSizer(boxSizer)

class DragFromHerePanel(wx.Panel):

    def __init__(self, parent):
    
        # Borders look dumb on windows, and dumb without on linux.
        border = wx.SIMPLE_BORDER
        if sys.platform == "win32":
           border = wx.NO_BORDER

        wx.Panel.__init__(self, parent, -1, wx.DefaultPosition, wx.DefaultSize, border)

        boxSizer = wx.BoxSizer(wx.HORIZONTAL)
        boxSizer.Add(wx.StaticText(self, -1, _("Drag files from below") + ":", 
                     wx.DefaultPosition, wx.DefaultSize, wx.ALIGN_CENTRE), 
                     0, wx.LEFT | wx.TOP | wx.BOTTOM, 4)
        self.SetSizer(boxSizer)

class FileSelectionPanel(wx.Panel):

    fileSplitterId = wx.NewId()
    fileListId = wx.NewId()
    menuRefreshId = wx.NewId()

    def __init__(self, parent, config, albumPanel, id, dirControlId, extList):
        wx.Panel.__init__(self, parent, id)
        self.wp = wpath.wpath(config)
        self.splitter = parent
        self.albumPanel = albumPanel
        self.frame = config.getTagger()
        self.config = config
        self.curDir = ""
        self.selected = []
        self.lastSelected = []
        self.dirCache = {}
        self.extList = extList
        self.dirCache = dircache.DirCache(config)

        wx.EVT_LIST_BEGIN_DRAG(self, self.fileListId, self.OnBeginDrag)
        wx.EVT_LIST_ITEM_ACTIVATED(self, self.fileListId, self.OnDblClick)
        wx.EVT_LIST_ITEM_SELECTED(self, self.fileListId, self.OnClick)
        wx.EVT_SIZE(self, self.OnSize)
        wx.EVT_LIST_ITEM_RIGHT_CLICK(self, self.fileListId, self.OnRightDown)
        wx.EVT_MENU(self, self.menuRefreshId, self.OnRefresh)
        self.splitter = wx.SplitterWindow(self, self.fileSplitterId, wx.DefaultPosition, wx.DefaultSize, 
                                         wx.SP_LIVE_UPDATE | wx.SP_3D)
        wx.EVT_CHAR(self, self.OnChar)

        boxSizer = wx.BoxSizer(wx.VERTICAL)
        boxSizer.Add(SelectHerePanel(self), 0, wx.EXPAND)

        dirSizer = wx.BoxSizer(wx.VERTICAL)
        self.dirPanel = wx.Panel(self.splitter, -1, style=wx.NO_BORDER)
        self.dirControl = dirctrl.TaggerDirCtrl(self.dirPanel, self.frame, self.albumPanel,
                                                dirControlId, config, self.dirCache)
        dirSizer.Add(self.dirControl, 1, wx.EXPAND)
        self.dirPanel.SetSizer(dirSizer)

        fileSizer = wx.BoxSizer(wx.VERTICAL)
        filePanel = wx.Panel(self.splitter, -1, style=wx.NO_BORDER)
        self.fileControl = wx.ListCtrl(filePanel, self.fileListId, style=wx.LC_REPORT | wx.LC_NO_HEADER)
        self.fileControl.InsertColumn(0, "")
        self.fileControl.SetColumnWidth(0, wx.LIST_AUTOSIZE)
        fileSizer.Add(DragFromHerePanel(filePanel), 0, wx.EXPAND)
        fileSizer.Add(self.fileControl, 1, wx.EXPAND)
        filePanel.SetSizer(fileSizer)

        self.splitter.SetMinimumPaneSize(150)
        self.splitter.SplitHorizontally(self.dirPanel, filePanel, self.config.persistFileSplitterPos);

        boxSizer.Add(self.splitter, 1, wx.EXPAND);
        self.SetSizer(boxSizer)

    def getSashPosition(self):
        return self.splitter.GetSashPosition()

    def refreshPaths(self, paths):

        selectedPaths = self.dirControl.getSelectedPaths()
        savedPath = u''
        if selectedPaths:
            savedPath = selectedPaths[0]

        for path in paths:
            self.dirCache.invalidate(path) 
            if self.curDir == path:
                self.showDirectory(path)
            self.dirControl.refreshPath(path)

        self.dirControl.showPath(savedPath)

    def showDirectory(self, dir):

        self.fileControl.DeleteAllItems()
        if not dir:
            return

        index = 0
        for file in self.dirCache.getFiles(dir):
            if self.wp.isfile(self.wp.join(dir, file)):
                isSupported = False
                temp = file.lower()
                for ext in self.extList:
                    if temp.endswith(ext):
                        isSupported = True
                        break

                self.fileControl.InsertStringItem(index, file)
                if not isSupported: self.fileControl.SetItemTextColour(index, wx.Colour(128, 128, 128))
                self.fileControl.SetItemData(index, isSupported)
                index = index + 1
            elif self.wp.isdir(self.wp.join(dir, file)):
                pass
            else:
                if not self.wp.access(file, os.F_OK):
                    self.fileControl.InsertStringItem(index, file)
                    self.fileControl.SetItemTextColour(index, wx.Colour(255, 128, 128))
                    self.fileControl.SetItemData(index, 0)
                    index = index + 1

        self.curDir = dir

    def OnSize(self, event):
        size = event.GetSize()
        self.fileControl.SetColumnWidth(0, size.GetWidth())
        event.Skip()

    def OnClick(self, event):

        self.lastSelected = self.selected
        self.selected = []
        index = -1
        while 1:
            index = self.fileControl.GetNextItem(index, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
            if index < 0:
                break

            self.selected.append(index)

	debug.debug("fileselection OnClick, new sel=%s, last sel=%s\n" % (repr(self.selected), repr(self.lastSelected)))

    def OnDblClick(self, event):
        index = self.fileControl.GetNextItem(-1, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
        if index >= 0:
            isSupported = self.fileControl.GetItemData(index)
            if isSupported:
                path = self.wp.join(self.curDir, self.fileControl.GetItemText(index))
                if self.wp.isfile(path):
                    wx.PostEvent(self.frame, events.FileAddEvent([path]))
                    wx.WakeUpIdle()

    def OnBeginDrag(self, event): 
    	selTemp = self.selected

        # Filter out any of the unsupported files (that are shown in grey)
        sel = []
        for index in selTemp:
            isSupported = self.fileControl.GetItemData(index)
            if isSupported:
                sel.append(index)

        if len(sel) == 0: return

	debug.debug("fileselection OnBeginDrag sel=%s\n" % repr(sel))

        dropText = ""
        for index in sel:
            if dropText:
               dropText = dropText + "\n"
            dropText = dropText + self.wp.join(self.curDir, self.fileControl.GetItemText(index))

        if len(sel) > 1:
            dropText = "files:" + dropText
        else:
            dropText = "file:" + dropText

        tdo = wx.PyTextDataObject(copy.copy(dropText))

        tds = albumpanel.TaggerDropSource(self, self.albumPanel) 
        tds.SetData(tdo) 
        tds.DoDragDrop() 
        event.Skip()

    def OnRightDown(self, event):
        popupMenu = wx.Menu()
        popupMenu.Append(self.menuRefreshId, _("Refresh"))
        self.fileControl.PopupMenu(popupMenu, event.GetPoint())

    def OnRefresh(self, event):
        self.refreshPaths([self.curDir])

    def selectAll(self):
        for index in xrange(self.fileControl.GetItemCount()):
            self.fileControl.SetItemState(index, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED);

    def OnChar(self, event):
        if event.GetKeyCode() == ord('a') and event.CmdDown():
            self.selectAll()
