/*
 * PIMPPA - Backup util
 *
 * Files which have "file_backup==0" [or the value specified
 * with "-r" on the commandline] will be included in the backup.
 * Included files will be set a new file_backup ID [or the one 
 * specified with "-r"] ... 
 *
 * Files will be included until media size is reached. Media
 * size can be specified on the commandline (in Megabytes).
 * 
 * Then a shadow directory of the included files is created
 * to a specified location ("-p path"). You can then create 
 * an ISO-image out of the created directory structure, 
 * stream it to tape or whatever.
 *
 * Note1: Without "-n", this utility tries to fill the backup media 
 *        to the brim. In any case, it can't know which files should 
 *        go together. For example, disk archives belonging to the 
 *        same product may end up on different backups.
 *
 * Note2: After using this utility, its suggested to do atleast 
 *        "du -L" in the destination directory to see that everything
 *        went ok.
 *
 */

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "pimppa.h"

#define DEFAULT_MEDIA_SIZE		625
#define DEFAULT_LOCATION 		"/tmp/test"

int be_nice=0;					// Do not fill backup media to the brim? 
int be_random=0;				// Randomize filearea order?

struct arealist					// List to keep area ID's in
{
	char area_id;
	struct arealist *next;
};

struct areadata
{
	int area_id;
	char area_name[PATH_MAX];
	char area_path[PATH_MAX];
};

struct arealist *sList=NULL;		// Explicitly skip these areas
struct arealist *tList=NULL;		// Explicitly take these areas

int output_formatted_row(FILE *dp, MYSQL_ROW sql_row, unsigned long size, int area_id);
int randomize(struct areadata *areas, int area_amount);

int escape_outfile(char *input, char *output)
{
	int in=0,out=0;

	if(!input || !output)
		return(0);
	
	while(input[in])
	{
		if(input[in]=='\\')
		{
			output[out++]='\\';
			output[out++]='\\';
			in++;
			continue;
		}
		if(input[in]=='\t')
		{
			output[out++]='\\';
			output[out++]='\t';
			in++;
			continue;
		}
		if(input[in]=='\n')
		{
			output[out++]='\\';
			output[out++]='\n';
			in++;
			continue;
		}
		
		output[out++]=input[in++];
	}

	output[out]=0;

	return(out);
}

	

int usage(char *prg_name)
{
	fprintf(stderr, "Usage: %s <options>\n\n%s",	prg_name,
	"-a <area_id>   Explicitly include this area\n"
	"-n             Be nice, stop at first file not fitting on backup\n"
	"-p <path>      Path to make the shadow directory into\n"
	"-r <ID>        Specify an explicit backup ID number\n"
	"-R             Randomize area order\n"
	"-c <area_id>   Skip this file area\n"
	"-s <size>      Media size in megabytes (1MB==1024*1024 bytes)\n");

	return(1);
}

/*
 * Returns the location of last component
 *
 */
int mymkdir(char *component, char *dest_path, char *new_path)
{
	char temp[PATH_MAX];
	int i;
	
	strcpy(temp, component);

	i=strlen(temp)-1;
	if(temp[i]=='/')
	{
		temp[i]=0;
		i--;
	}
		
	while(i>0 && temp[i]!='/')
		i--;
	i++;

	if(p_checkp(dest_path))
		sprintf(new_path, "%s%s", dest_path, &temp[i]); 
	else
		sprintf(new_path, "%s/%s", dest_path, &temp[i]);

	mkdir(new_path, 0774);
	return(i);
}

/*
 * Makes the backup.
 *
 */
int make_backup(int backup_id, int media_size, char *dest_path)
{
	struct areadata *areas=NULL;
	int i,area_amount=0, go=1, size_reached=0, last_area=0;
	int total_bytes=0,total_files=0, total_dirs=0;
	int old_id=0;
	MYSQL *db;
	MYSQL_RES *sql_result;
	MYSQL_ROW sql_row;
	char filelist[PATH_MAX];
	char new_path[PATH_MAX];
	char new_whole[PATH_MAX];
	char old_whole[PATH_MAX];
	struct arealist *tmp;
	FILE *dp;

	media_size=media_size*1024*1024;

	db=p_connect();
	if(!db)
		return(-1);

/******************** Get last visited area ***********************/

	last_area=atoi(p_getmisc(db, P_KEY_PBACK));
	if(last_area<0)
		last_area=0;

/******************** Select fileareas ****************************/

	// Skip these areas
	tmp=sList;
	old_whole[0]=0;
	i=0;
	while(tmp)
	{
		i+=sprintf(&old_whole[i], "area_id!=%d AND ", tmp->area_id);
		tmp=tmp->next;
	}
	
	// Take these areas
	tmp=tList;
	while(tmp)
	{
		i+=sprintf(&old_whole[i], "area_id=%d %s", tmp->area_id,
					(tmp->next ? "OR " : "AND "));
		tmp=tmp->next;
	}
	

//	fprintf(stderr, "SELECT area_id, area_path, area_name 
	p_query(db, "SELECT area_id, area_path, area_name " 
		    "FROM p_areas "
		    "WHERE NOT (area_flags & %ld) "
		    "  AND %s 1 "
		    "ORDER BY area_name", 
		AREA_INCOMING,
		old_whole);

	if(mysql_error(db)[0])
		fprintf(stderr, "%s\n", mysql_error(db));
	
	sql_result=mysql_store_result(db);
	if(sql_result)
	{
		area_amount=mysql_num_rows(sql_result);
		areas=malloc(area_amount*sizeof(struct areadata));
		if(!areas)
		{
			mysql_free_result(sql_result);
			mysql_close(db);
			return(0);
		}
			
		for(i=0;i<area_amount;i++)
		{
			sql_row=mysql_fetch_row(sql_result);
			areas[i].area_id=atoi(sql_row[0]);
			strcpy(areas[i].area_path, sql_row[1]);
			if(!p_checkp(sql_row[1]))
				strcat(areas[i].area_path, "/");
			strcpy(areas[i].area_name, sql_row[2]);
//			printf("%d %s\n", areas[i].area_id, areas[i].area_path);
		}

		mysql_free_result(sql_result);
	}

	if(be_random)
	{
		printf("Randomizing...\n");
		randomize(areas, area_amount);
	}
//	printf("LastA: %d\n", last_area);
//	for(i=0;i<area_amount;i++)
//		printf("%d - %d : %s\n", i, areas[i].area_id, areas[i].area_path);

/************ Get new backup ID ********************/

	if(backup_id==0)						// Get new ID
	{
		p_query(db, "SELECT max(file_backup) "
			    "FROM p_files");
		sql_result=mysql_store_result(db);
		if(sql_result)
		{
			sql_row=mysql_fetch_row(sql_result);
			if(sql_row)
				backup_id=atoi(sql_row[0]);
			mysql_free_result(sql_result);
		}

		backup_id++;
		fprintf(stderr, "New backup ID is %d\n", backup_id);
	}
	else
		old_id=backup_id;

/***************************** Find files ******************************/

	sprintf(filelist, "%s/allfiles.txt", dest_path);
	dp=fopen(filelist, "w");
	if(!dp)
	{
		fprintf(stderr, "Unable to open %s\n", filelist);
		mysql_close(db);
		return(1);
	}

	if(tList)
		last_area=0;

	while(go && !size_reached)
	{
		for(i=last_area;i<area_amount && go;i++)
		{
			printf("Checking %3d: %s\n", areas[i].area_id, areas[i].area_path);

			p_query(db, "SELECT file_id, file_name, file_size, " 
				    "  file_date, file_desc "
				    "FROM p_files "
				    "WHERE file_area=%d AND file_backup=%d "
				    "  AND NOT (file_flags & %ld) "
				    "  AND file_integ!=%d "
				    "ORDER BY file_name", 
				areas[i].area_id, old_id,
				FILE_OFFLINE, 
				INTEG_NEW);
			if(mysql_error(db)[0])
				fprintf(stderr, "%s\n", mysql_error(db));
		
			sql_result=mysql_store_result(db);
			if(sql_result)
			{
				int temp_size, dir_made=0;
				struct stat st;
			
				while((sql_row=mysql_fetch_row(sql_result)) && go)
				{
					if(!dir_made)
					{
						int lastcomploc;
						char areaname[512];
						char areapath[512];

						escape_outfile(areas[i].area_name, areaname);
						escape_outfile(areas[i].area_path, areapath);
					
						lastcomploc=mymkdir(areas[i].area_path, 
									dest_path, new_path);
						dir_made=1;
						fprintf(dp, "#\n#\t%s\t%d\t%s\t%s\n#\n", 
								areaname,
								areas[i].area_id,
								areapath,
								&(areas[i].area_path[lastcomploc]));
						total_dirs++;
					}
					
					sprintf(new_whole, "%s/%s", new_path, sql_row[1]);
					sprintf(old_whole, "%s%s", areas[i].area_path, sql_row[1]);

//					printf("Got %s %s\n", new_whole, old_whole);
	
					if(stat(old_whole, &st)!=0)		// File not found
						continue;

					temp_size=atoi(sql_row[2]);
					if(total_bytes+temp_size<media_size)
						total_bytes+=temp_size;
					else
					{
						size_reached=1;
						
						if(be_nice)						// Quit here
						{
							printf("Media size reached\n");
							go=0;
							break;
						}
						else		       // Try to find small files that fit	
							continue;
					}
		
					output_formatted_row(dp, sql_row, temp_size, 
										areas[i].area_id);
			
					p_query(db, "UPDATE p_files "
						    "SET file_backup=%d "
						    "WHERE file_id=%s",
						backup_id, sql_row[0]);
				
					symlink(old_whole, new_whole);

					total_files++;
				}
				mysql_free_result(sql_result);
			}
		}

		if(last_area>0)
		{
			go=1;
			last_area=0;
		}

		if(!old_id)
			go=0;
		else
			old_id=0;			// See if we can add new stuff to the backup
	}
	
	if(i==area_amount)
		last_area=0;
	else
		last_area=i;

	fclose(dp);

/******************** Set last visited area ********************/

	if(last_area>0)
		last_area--;

	p_query(db, "REPLACE INTO p_misc (misc_key, misc_data) "
		    "VALUES ('%s', '%d')", 
		P_KEY_PBACK, last_area);

	mysql_close(db);

	if(areas)
		free(areas);

	fprintf(stderr, "Total: %d dirs, %d files and %d bytes added.\n",
	total_dirs, total_files, total_bytes);

	return(1);
}

int main(int argc, char *argv[])
{
	int go=1;
	char dest_path[PATH_MAX];
	int media_size=DEFAULT_MEDIA_SIZE;		// In Megabytes
	int backup_id=0;						// Get new ID

	strcpy(dest_path, DEFAULT_LOCATION);

	while(go)
	{
		switch(getopt(argc, argv, "a:c:hnr:Rs:p:V"))
		{
			case 'a':
			{
				struct arealist *tmp;

				tmp=malloc(sizeof(struct arealist));
				tmp->area_id=atoi(optarg);
				tmp->next=NULL;

				if(!tList)
					tList=tmp;
				else
				{
					tmp->next=tList;
					tList=tmp;
				}
				printf("Explicitly taking area %d\n", tmp->area_id);
			}
			break;
			case 'c':
			{
				struct arealist *tmp;
				
				tmp=malloc(sizeof(struct arealist));
				tmp->area_id=atoi(optarg);
				tmp->next=NULL;

				if(!sList)
					sList=tmp;
				else
				{
					tmp->next=sList;
					sList=tmp;
				}
				printf("Skipping area %d\n", tmp->area_id);
			}
			break;
			case 'n':
				fprintf(stderr, "Ok, were nice today, stopping at the first nonfitting file.\n");
				be_nice=1;
				break;
			case 'r':
				backup_id=atoi(optarg);
				if(backup_id>0)
					fprintf(stderr, "Replacing/updating backup #%d\n", 
							backup_id);
				break;
			case 'R':
				be_random=1;
				break;
			case 's':
				media_size=atoi(optarg);
				if(!media_size)
					media_size=DEFAULT_MEDIA_SIZE;
				break;
			case 'p':
				strcpy(dest_path, optarg);
				break;
			case 'V':
				printf("%s %s %s\n", PACKAGE, argv[0], VERSION);
				return(0);
				break;
			case -1:
				go=0;
				break;
			case 'h':
			default:
				go=0;
				usage(argv[0]);
				return(0);
				break;
		}
	}

	fprintf(stderr, "Creating shadow (max %dMB) to %s...\n", 
			media_size, dest_path);
	make_backup(backup_id, media_size, dest_path);

	return(1);
}

/*
 *			p_query(sprintf(querydb, "SELECT file_id, file_name, file_size
 *									file_uldate, file_description
 *
 */
int output_formatted_row(FILE *dp, MYSQL_ROW sql_row, unsigned long size, int area_id)
{
	char filename[512];
	char date[512];
	char desc[4096];
			
//	 p_query(&db, "SELECT file_id, file_name, file_size, 
//							file_date, file_desc

	escape_outfile(sql_row[1], filename);
	escape_outfile(sql_row[3], date);
	escape_outfile(sql_row[4], desc);

	fprintf(dp, "%s\t%ld\t%s\t%d\t%s\n", filename, size, date, area_id, desc);

	return(1);
}
	
int randomize(struct areadata *areas, int area_amount)
{
	int i,j;
	struct areadata tmp;

	srand(time(NULL));

//	printf("Amount: %d\n", area_amount);

	for(i=0;i<area_amount;i++)
	{
		j=i+((float)(area_amount-i)*rand()/(RAND_MAX+1.0));
		memcpy(&tmp, &areas[i], sizeof(struct areadata));
		memcpy(&areas[i], &areas[j], sizeof(struct areadata));
		memcpy(&areas[j], &tmp, sizeof(struct areadata));
//		printf("%d %s\n", areas[i].area_id, areas[i].area_path);
	}

	return(1);
}
