/*
 * PIMPPA - Cleaner
 *
 * Performs various delete operations on files.
 *
 * Use with caution!
 *
 */
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <limits.h>
#include <getopt.h>

#include "pimppa.h"

int Verbose=0;
int Simulate=0;
	
MYSQL *src_db;

#define ACTION_PURGEDUPES	0
#define ACTION_CLEANDIR		1
#define ACTION_FAILDEL		2
#define ACTION_BACKUPDEL	3
#define ACTION_NUMBERCHECK  4
#define ACTION_PURIFY		5
#define	ACTION_SLAUGHTER	6
#define ACTION_OFFLINE		7
#define ACTION_TRASH		8
#define AMOUNT_ACTIONS		9

int purify(void);
int offline(void);
int trash(void);

void usage(char *name)
{
	fprintf(stderr, "Usage: %s <options>\n\n"
					"-a <area_id>     Purge duplicates from <area_id>\n"
					"-d               Delete files from current dir found in db\n"
					"-f               Delete files which failed integrity check\n"
					"-i <backup_id>   Delete files associated with backup_id\n"
					"-n <area_id>     Delete filenames with no numbers from <area_id>\n"
					"-o               Delete all offline files without backup id\n"
					"-p               'Purify' current dir, see doc.\n"
					"-s               Simulate only, don't delete anything\n"
					"-t               Remove trash from database (files w/ file_area=-1)\n"
					"-z               Zuper slaughter, immense\n"
					"-v               Verbose execution\n", name);
}

/*
 * Deletes files from incoming areas which are offline,
 * creates negative assign patterns for all of them, 
 * and deletes the entries from 'p_files'.
 *
 * Use with EXTREME caution.
 *
 */
int slaughter(void)
{
	char pathbuf[PATH_MAX];
	MYSQL *dst_db;
	MYSQL_RES *sql_res, *sql_res2;
	MYSQL_ROW sql_row;
	int total_files=0, deleted_files=0;
	struct stat st; 
	int deleted=0;
	int minlen=0;
	char *value;

	src_db=p_connect();
	if(!src_db)
		return(-1);

	dst_db=p_connect();
	if(!dst_db)
	{
		mysql_close(src_db);
		return(-1);
	}

	value=p_getmisc(src_db, P_KEY_MINASSNAMELEN);
	if(value)
		minlen=atoi(value);
	else
		minlen=atoi(P_MIN_ASS_NAMELENGTH);

	p_query(src_db, "SELECT area_path, file_name, file_id, file_area, "
			"  area_context "
			"FROM p_files, p_areas "
			"WHERE (area_flags & %ld) "
			"  AND file_area=area_id",
		AREA_INCOMING);
	sql_res=mysql_use_result(src_db);
	if(!sql_res)
	{
		printf("Nothing found...\n");
		mysql_close(src_db);
		mysql_close(dst_db);
		return(0);
	}	

	printf("Checking files from incoming areas...\n");

	
	while((sql_row=mysql_fetch_row(sql_res)))
	{
		deleted=0;
	
		if(Verbose)
			printf("Checking %s [%s]... ", sql_row[1], sql_row[3]);

		sprintf(pathbuf, "%s%s%s", 
				sql_row[0], 
				(p_checkp(sql_row[0]) ? "" : "/"),
				sql_row[1]);
		if(stat(pathbuf, &st)!=0)			// Not present
		{
			p_query(dst_db, "SELECT DISTINCT 1 "
					"FROM p_files "
					"WHERE file_name='%s' "
					"  AND file_area<>%s",
				sql_row[1], sql_row[3]);
			sql_res2=mysql_store_result(dst_db);
			// Doesn't exist?
			if(sql_res2 && mysql_num_rows(sql_res2)==0) {
				if(!Simulate) {
					p_query(dst_db, "DELETE FROM p_files "
							"WHERE file_id=%s",
						sql_row[2]);
					p_assign(dst_db, sql_row[1], minlen, -1,
						atoi(sql_row[4]));
				}

				if(Verbose)
					printf(" deleted\n");
	
				deleted_files++;
				deleted=1;
			}
			if(sql_res2)
			  mysql_free_result(sql_res2);
		}
			
		if(Verbose && !deleted)
			printf("Ok.\n");

		total_files++;
		
		if(Verbose && total_files%500==0)
			printf("%d files checked, %d deleted...\n", 
					total_files, deleted_files);
	}

	mysql_free_result(sql_res);

	printf("Done. %d files checked, %d deleted.\n", total_files, deleted_files);

	mysql_close(src_db);
	mysql_close(dst_db);

	return(total_files);
}

int faildel(void)
{
	char pathbuf[PATH_MAX];
	MYSQL *dst_db;
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	int total_files=0;

	src_db=p_connect();
	if(!src_db)
		return(-1);

	dst_db=p_connect();
	if(!dst_db)
	{
		mysql_close(src_db);
		return(-1);
	}
	
	fprintf(stderr, "Finding nonbackupped failed files...\n");

	p_query(src_db, "SELECT area_path, file_name, file_id " 
			"FROM p_files, p_areas "
			"WHERE file_integ=%d AND file_backup=0 "
			"  AND file_area=area_id",
		INTEG_FAILED);
	sql_res=mysql_store_result(src_db);
	if(!sql_res)
	{
		printf("Nothing found...\n");
		mysql_close(src_db);
		mysql_close(dst_db);
		return(0);
	}	

	fprintf(stderr, "Deleting files...\n");

	while((sql_row=mysql_fetch_row(sql_res)))
	{
		if(!Simulate)
		{
			sprintf(pathbuf, "%s%s%s", 
					sql_row[0], 
					(p_checkp(sql_row[0]) ? "" : "/"),
					sql_row[1]);
			unlink(pathbuf);
		
//			fprintf(stderr, "Hier\n");

			p_query(dst_db, "UPDATE p_files "
					"SET file_flags=(file_flags | %ld) "
					"WHERE file_id=%s",
				FILE_DELETE, sql_row[2]);
			
//			fprintf(stderr, "Pass\n");
		}
	
		total_files++;
		
		if(Verbose)
			printf("%s%s%s deleted...\n", 
					sql_row[0], 
					(p_checkp(sql_row[0]) ? "" : "/"),
					sql_row[1]);
		
		if(total_files%500==0)
			fprintf(stderr, "%d files deleted...\n", total_files);
	}

	mysql_free_result(sql_res);
	
	if(!Simulate) {
		p_query(dst_db, "DELETE FROM p_files "
				"WHERE (file_flags & %ld)",
			FILE_DELETE);
	}

	fprintf(stderr, "Done. %d files deleted.\n", total_files);

	mysql_close(src_db);
	mysql_close(dst_db);

	return(total_files);
}

int backupdel(int backup_id)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	char filename[PATH_MAX];
	int deleted_files=0;

	src_db=p_connect();
	if(!src_db)
		return(-1);
	
/******** fetch files with suitable backup id's *******/

	fprintf(stderr, "Deleting files with backup id %d...\n", backup_id);

	p_query(src_db, "SELECT area_path, file_name "
			"FROM p_files, p_areas "
			"WHERE file_backup=%d AND file_area=area_id",
		backup_id);
	if(mysql_error(src_db)[0])
		return(-1);

	sql_res=mysql_use_result(src_db);
	if(sql_res)
	{
		while((sql_row=mysql_fetch_row(sql_res)))
		{
			if(!Simulate)
			{
				sprintf(filename, "%s%s%s", 
						sql_row[0], 
						(p_checkp(sql_row[0]) ? "" : "/"),
						sql_row[1]);
				unlink(filename);
			}

			deleted_files++;
		
			if(deleted_files%500==0)
				fprintf(stderr, "%d files deleted...\n", deleted_files);
		}
		mysql_free_result(sql_res);
	}
	
	fprintf(stderr, "Total %d files deleted.\n", deleted_files);
	
/*********** Mark them as offline *****************/

	fprintf(stderr, "Marking them as offline...\n");

	p_query(src_db, "UPDATE p_files "
			"SET file_flags=(file_flags | %ld) "
			"WHERE file_backup=%d", 
		FILE_OFFLINE, backup_id);
	if(mysql_error(src_db)[0])
		return(-1);
	
	mysql_close(src_db);
				
	fprintf(stderr, "Done.\n");

	return(deleted_files);
}

int numbercheck(int area_id)
{
	MYSQL *dst_db;
	MYSQL_RES *sql_result;
	MYSQL_ROW sql_row;
	int deld_files=0,total_files=0;
	
	src_db=p_connect();
	if(!src_db)
		return(-1);

	dst_db=p_connect();
	if(!dst_db)
	{
		mysql_close(src_db);
		return(-1);
	}

	p_query(src_db, "SELECT file_name FROM p_files "
			"WHERE file_area=%d", area_id);
	sql_result=mysql_use_result(src_db);
	if(sql_result) {
		while((sql_row=mysql_fetch_row(sql_result))) {
			if(!p_checkfornumber(sql_row[0])) {
				char escaped_fn[2*P_LEN_FILE_NAME+1];

				mysql_escape_string(escaped_fn, sql_row[0], strlen(sql_row[0]));
		
				if(!Simulate) {
					p_query(dst_db, "DELETE FROM p_files "
							"WHERE file_name='%s' "
							"  AND file_area=%d",
						escaped_fn, area_id);
				}
				deld_files++;
			}
			total_files++;
		
			if(total_files%500==0)
				fprintf(stderr, "%d files checked...\n", total_files);
		}
		mysql_free_result(sql_result);
	}
		
	fprintf(stderr, "Done, %d files deleted from db...\n", deld_files);
		
	return(deld_files);
}

/*
 * Deletes files from current dir which are found from db
 *
 */
int cleandir(void)
{
	int deleted_files=0;
	DIR *dirri;
	struct dirent *dirdata;
	struct stat st;

	src_db=p_connect();
	if(!src_db)
		return(0);

	dirri=opendir(".");
	if(!dirri)
	{	
		fprintf(stderr, "Unable to open current dir.\n");
		return(-1);
	}

	while((dirdata=readdir(dirri)))
	{
		int dval;
	
		if(stat(dirdata->d_name, &st)!=0)
		{
			fprintf(stderr, "stat() error on %s\n", 
					dirdata->d_name);
			continue;
		}
			
		if(!(S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
			|| S_ISDIR(st.st_mode))
			continue;

		dval=p_dupecheck(src_db, dirdata->d_name, -1);
		if(dval>=0 && dval!=INTEG_FAILED)
		{
			if(!Simulate)
				unlink(dirdata->d_name);
			
			if(Verbose)
				printf("Deleted %s\n", dirdata->d_name);
		
			deleted_files++;
		}
	}
	closedir(dirri);
		
	mysql_close(src_db);
	return(deleted_files);
}
	
int purgedupes(int area_id)
{
	MYSQL *dst_db;
	MYSQL_RES *sql_res, *sql_res2;
	MYSQL_ROW sql_row;
	int total_files=0, deleted_files=0;

	src_db=p_connect();
	if(!src_db)
		return(-1);

	dst_db=p_connect();
	if(!dst_db)
	{
		mysql_close(src_db);
		return(-1);
	}

	fprintf(stderr, "Fetching filenames from %d...\n", area_id);
	
	p_query(src_db, "SELECT file_name " 
			"FROM p_files "
			"WHERE file_area=%d", 
		area_id);
	sql_res=mysql_store_result(src_db);
	if(!sql_res)
	{
		printf("Nothing found...\n");
		mysql_close(src_db);
		mysql_close(dst_db);
		return(0);
	}	

	fprintf(stderr, "Checking against other areas...\n");

	while((sql_row=mysql_fetch_row(sql_res)))
	{
		char escaped_fn[2*P_LEN_FILE_NAME+1];

		mysql_escape_string(escaped_fn, sql_row[0], strlen(sql_row[0]));
	
		p_query(dst_db, "SELECT DISTINCT 1 FROM p_files "
				"WHERE file_name='%s' AND file_area<>%d",	
			escaped_fn, area_id);
		sql_res2=mysql_store_result(dst_db);
		if(sql_res2) {
			if(mysql_num_rows(sql_res2)>0) {
				if(!Simulate) {
					p_query(dst_db, "UPDATE p_files "
							"SET file_flags=file_flags|%ld "
							"WHERE file_name='%s' "
							"  AND file_area=%d",
						FILE_DELETE,
						escaped_fn, area_id);
				}
		
				deleted_files++;
			}
			mysql_free_result(sql_res2);
		}
		
		total_files++;
		
		if(total_files%500==0)
			fprintf(stderr, "%d files checked, %d deleted...\n", total_files, deleted_files);
	}

	mysql_free_result(sql_res);
	
	if(Verbose)
		printf("Deleting %d files from db...\n", deleted_files);
		
	if(!Simulate) {
		p_query(dst_db, "DELETE FROM p_files "
				"WHERE file_area=%d "
				"  AND (file_flags & %ld)",
			area_id,
			FILE_DELETE);
	}

	fprintf(stderr, "Done. %d/%d files deleted.\n", deleted_files, total_files);

	mysql_close(src_db);
	mysql_close(dst_db);

	return(total_files);
}

int main(int argc, char *argv[])
{
	int id=-1,go=1,i=0;
	int actions[AMOUNT_ACTIONS];

	if(argc==1)	
	{
		usage(argv[0]);
		return(0);
	}

	for(i=0;i<AMOUNT_ACTIONS;i++)
		actions[i]=-1;

	while(go)
	{
		switch(getopt(argc, argv, "a:dfhi:n:opstvVz"))
		{
			case 'a':
				id=strtol(optarg, (char **)NULL, 10);
				if(id<0 || id==LONG_MAX)
				{
					fprintf(stderr, "Invalid area id...\n");
					return(-1);
				}
				actions[ACTION_PURGEDUPES]=id;
				break;
			case 'd':
				actions[ACTION_CLEANDIR]=0;
				break;
			case 'f':
				actions[ACTION_FAILDEL]=0;
				break;
			case 'i':
				id=strtol(optarg, (char **)NULL, 10);
				if(id<0 || id==LONG_MAX)
				{
					fprintf(stderr, "Invalid backup id...\n");
					return(-1);
				}
				actions[ACTION_BACKUPDEL]=id;
				break;	
			case 'n':
				id=strtol(optarg, (char **)NULL, 10);
				if(id<0 || id==LONG_MAX)
				{
					fprintf(stderr, "Invalid area id...\n");
					return(-1);
				}
				actions[ACTION_NUMBERCHECK]=id;
				break;
			case 's':
				Simulate=1;
				break;
			case 'o':
				actions[ACTION_OFFLINE]=0;
				break;
			case 'p':
				actions[ACTION_PURIFY]=0;
				break;
			case 'v':
				Verbose=1;
				break;
			case 'V':
				printf("%s %s %s\n", PACKAGE, argv[0], VERSION);
				return(0);
				break;
			case 't':
				actions[ACTION_TRASH]=0;
				break;
			case 'z':
				actions[ACTION_SLAUGHTER]=0;
				break;
			case -1:
				go=0;
				break;
			case 'h':
			default:
				usage(argv[0]);
				return(0);
				break;
		}
	}

	if(actions[ACTION_BACKUPDEL]!=-1)
		backupdel(actions[ACTION_BACKUPDEL]);
	if(actions[ACTION_CLEANDIR]!=-1)
		cleandir();
	if(actions[ACTION_OFFLINE]!=-1)
		offline();
	if(actions[ACTION_PURIFY]!=-1)
		purify();
	if(actions[ACTION_SLAUGHTER]!=-1)
		slaughter();
	if(actions[ACTION_TRASH]!=-1)
		trash();
	if(actions[ACTION_PURGEDUPES]!=-1)
		purgedupes(actions[ACTION_PURGEDUPES]);
	if(actions[ACTION_NUMBERCHECK]!=-1)
		numbercheck(actions[ACTION_NUMBERCHECK]);
	if(actions[ACTION_FAILDEL]!=-1)
		faildel();
		
	return(0);
}

int purify(void)
{
	int deleted_files=0,deleted_bytes=0;
	int delete_this;
	DIR *dirri;
	struct dirent *dirdata;
	struct stat st;
	char *value;
	int to_lowercase;

	value=p_getmisc(src_db, P_KEY_TOLOWERCASE);
	if(value)
		to_lowercase=atoi(value);
	else
		to_lowercase=atoi(P_TOLOWERCASE);

	dirri=opendir(".");
	if(!dirri)
	{	
		fprintf(stderr, "Unable to open current dir.\n");
		return(-1);
	}

	while((dirdata=readdir(dirri)))
	{
		char *filename=dirdata->d_name;

		delete_this=0;

		if(stat(filename, &st)!=0)
		{
			fprintf(stderr, "stat() error on %s\n", 
					filename);
			continue;
		}
			
		if(!(S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
			|| S_ISDIR(st.st_mode))
			continue;

        if(Verbose)
            printf("Checking %s ", filename);

		if(!p_checkfilename(src_db, filename))
		{
			if(!Simulate)
				unlink(filename);
			
			deleted_files++;
			deleted_bytes+=st.st_size;
			continue;		
		}
	
		if(to_lowercase)
		{
            char newname[PATH_MAX];

			strcpy(newname, filename);
            p_strtolc(newname);

			if(!Simulate)
			{
            	rename(filename, newname);
				strcpy(filename, newname);
			}
		}

		if(p_getdest(filename, 0)<0)
		{
			if(Verbose)
				printf("Deleted, pattern < 0\n");	
			if(!Simulate)
				unlink(filename);
		
			deleted_files++;
			deleted_bytes+=st.st_size;
				
			continue;
		}
		
		if(Verbose)
			printf("Ok.\n");
	}
	closedir(dirri);
	
	fprintf(stderr, "Total %d files deleted, %d bytes\n",
			deleted_files, deleted_bytes);
		
	return(deleted_files);
}

/*
 * Deletes files with file_area=-1
 *
 */
int trash(void)
{
	src_db=p_connect();
	if(!src_db)
		return(-1);

	p_query(src_db, "DELETE FROM p_files WHERE file_area=-1");

	mysql_close(src_db);

	if(Verbose)
		fprintf(stderr, "Done.\n");
		
	return(0);
}

/*
 * Deletes all offline files which do not have backup id
 * associated with them.
 *
 */
int offline(void)
{
	char pathbuf[PATH_MAX];
	MYSQL *dst_db;
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	int total_files=0, deleted_files=0;
	struct stat st; 
	int deleted=0;
	
	src_db=p_connect();
	if(!src_db)
		return(-1);

	dst_db=p_connect();
	if(!dst_db)
	{
		mysql_close(dst_db);
		return(-1);
	}
	
	printf("Browsing through offline files...\n");

	p_query(src_db, "SELECT area_path, file_name, file_id, file_area " 
			"FROM p_files, p_areas "
			"WHERE (file_flags & %ld) "
			"  AND file_backup=0 "
			"  AND file_area=area_id",
		FILE_OFFLINE);
	sql_res=mysql_use_result(src_db);
	if(!sql_res)
	{
		printf("Nothing found...\n");
		mysql_close(src_db);
		mysql_close(dst_db);
		return(0);
	}	

	while((sql_row=mysql_fetch_row(sql_res)))
	{
		deleted=0;
	
		if(Verbose)
			printf("Checking [%s] %s... ", sql_row[3], sql_row[1]);

		sprintf(pathbuf, "%s%s%s", 
				sql_row[0], 
				(p_checkp(sql_row[0]) ? "" : "/"),
				sql_row[1]);
		if(stat(pathbuf, &st)!=0)			// Not present
		{
			if(!Simulate) {
				p_query(dst_db, "UPDATE p_files "
						"SET file_flags=file_flags|%ld "
						"WHERE file_id=%s",
					FILE_DELETE,
					sql_row[2]);
			}

			if(Verbose)
				printf(" deleted\n");
	
			deleted_files++;
			deleted=1;
		}
			
		if(Verbose && !deleted)
			printf("present, FILE_OFFLINE is wrong.\n");

		total_files++;
		
		if(Verbose && total_files%500==0)
			printf("%d files checked, %d deleted...\n", 
					total_files, deleted_files);
	}

	mysql_free_result(sql_res);

	// Ok, lets really kick the shit out of 'em 
	
	if(Verbose)
		printf("Deleting %d files from db...\n", deleted_files);
		
	if(!Simulate)
	{
		p_query(dst_db, "DELETE FROM p_files "
				"WHERE (file_flags & %ld)",
			FILE_DELETE);
	}

	printf("Done. %d/%d files deleted of those marked offline.\n", deleted_files, total_files);

	mysql_close(src_db);
	mysql_close(dst_db);

	return(total_files);
}

