/* Plagiat gallery creation tool for Jakub Steiner's O.R.I.G.I.N.A.L 
 * photo gallery.
 * Copyright (C) 2005 Robert Staudinger <robert.staudinger@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <string.h>
#include <glade/glade.h>
#include "pgt-view.h"
#include "pgt-galleries.h"
#include "pgt-mainwin.h"
#include "pgt-prefs.h"
#include "pgt-utils.h"

struct _PgtView {
	GObject parent;
	gboolean is_disposed;
	GladeXML *xml;
	PgtMainwin *mainwin;
	PgtGalleries *galleries;
	PgtPrefs *prefs;
	GtkTreeModel *gallerystore;
};

struct _PgtViewClass {
	GObjectClass parent;
	/* main window */
	void (* gallery_changed) 	(PgtView *self);
	void (* album_name_changed) 	(PgtView *self);
	void (* description_changed) 	(PgtView *self);
	void (* photo_dir_changed) 	(PgtView *self);
	void (* option_changed) 	(PgtView *self);
	void (* process) 		(PgtView *self);
	/* gallery window */
	void (* add_gallery) 		(PgtView *self);
	void (* del_gallery) 		(PgtView *self);
};

/* signals */
enum {
	/* main window */
	GALLERY_CHANGED,
	ALBUM_NAME_CHANGED,
	DESCRIPTION_CHANGED,
	PHOTO_DIR_CHANGED,
	OPTION_CHANGED,
	PROCESS,
	/* gallery window */
	ADD_GALLERY,
	DEL_GALLERY,

	_LAST_SIGNAL
};

static guint 	     pgt_view_signals[_LAST_SIGNAL] = { 0 };
static GObjectClass *pgt_view_parent_class = NULL;


static void
add_gallery_cb (PgtView      *self, 
		PgtGalleries *mainwin)
{
	g_signal_emit (self, pgt_view_signals[ADD_GALLERY], 0);
}

static void
del_gallery_cb (PgtView      *self, 
		PgtGalleries *mainwin)
{
	g_signal_emit (self, pgt_view_signals[DEL_GALLERY], 0);
}

static void
galleries_delete_cb (PgtView      *self, 
		     PgtGalleries *galleries)
{
	g_object_unref (self->galleries);
	self->galleries = NULL;
}

static void
manage_galleries_cb (PgtView    *self, 
		     PgtMainwin *mainwin)
{
	GtkWidget *parent = NULL;
	GtkWidget *win = NULL;

	self->galleries = pgt_galleries_new (self->gallerystore);
	g_object_ref (self->galleries);

	g_signal_connect_swapped (self->galleries, "add-gallery", 
				  G_CALLBACK (add_gallery_cb), self);
	g_signal_connect_swapped (self->galleries, "del-gallery", 
				  G_CALLBACK (del_gallery_cb), self);
	g_signal_connect_swapped (self->galleries, "delete", 
				  G_CALLBACK (galleries_delete_cb), self);

	parent = pgt_mainwin_get_window (self->mainwin);
	g_assert (GTK_IS_WINDOW (parent));

	win = pgt_galleries_get_window (self->galleries);
	g_assert (GTK_IS_WINDOW (win));
	
	gtk_window_set_transient_for (GTK_WINDOW (win), GTK_WINDOW (parent));

	gtk_widget_show_all (win);
}

static void
gallery_changed_cb (PgtView    *self, 
		    PgtMainwin *mainwin)
{
	g_signal_emit (self, pgt_view_signals[GALLERY_CHANGED], 0);
}

static void
album_name_changed_cb (PgtView    *self, 
		       PgtMainwin *mainwin)
{
	g_signal_emit (self, pgt_view_signals[ALBUM_NAME_CHANGED], 0);
}

static void
photo_dir_changed_cb (PgtView    *self, 
		      PgtMainwin *mainwin)
{
	g_signal_emit (self, pgt_view_signals[PHOTO_DIR_CHANGED], 0);
}

static void
option_changed_cb (PgtView    *self, 
		   PgtMainwin *mainwin)
{
	g_signal_emit (self, pgt_view_signals[OPTION_CHANGED], 0);
}

static void
process_cb (PgtView    *self, 
	    PgtMainwin *mainwin)
{
	g_signal_emit (self, pgt_view_signals[PROCESS], 0);
}

static void
mainwin_delete_cb (PgtView    *self, 
		   PgtMainwin *mainwin)
{
	g_object_unref (self->mainwin);
	self->mainwin = NULL;
	/* maybe this should be done in mainwin's destroy handler */
	gtk_main_quit ();
}


static void
instance_init (PgtView *self)
{
	self->is_disposed = FALSE;
	self->mainwin = NULL;
	self->galleries = NULL;
	self->prefs = NULL;
	self->gallerystore = NULL;

	if (pgt_runs_from_srcdir ()) {
		self->xml = glade_xml_new ("../data/mainwin.glade", NULL, NULL); 
	}
	else {
		self->xml = glade_xml_new (PLAGIAT_DATADIR "/mainwin.glade", NULL, NULL);
	}
}

static void
instance_dispose (GObject *instance)
{
	PgtView *self = PGT_VIEW (instance);

	if (self->is_disposed)
		return;

	self->is_disposed = TRUE;

	if (self->mainwin) {
		g_object_unref (self->mainwin);
		self->mainwin = NULL;
	}
	if (self->galleries) {
		g_object_unref (self->galleries);
		self->galleries = NULL;
	}
	if (self->prefs) {
		g_object_unref (self->prefs);
		self->prefs = NULL;
	}
	if (self->gallerystore) {
		g_object_unref (self->gallerystore);
		self->gallerystore = NULL;
	}

	pgt_view_parent_class->dispose (G_OBJECT (self));
}

static void
class_init (PgtViewClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	/* hook gobject vfuncs */
	gobject_class->dispose = instance_dispose;

	pgt_view_parent_class = (GObjectClass*) g_type_class_peek_parent (klass);


	/* main window */
	pgt_view_signals [GALLERY_CHANGED] =
		g_signal_new (
			"gallery-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, gallery_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_view_signals [ALBUM_NAME_CHANGED] =
		g_signal_new (
			"album-name-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, album_name_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_view_signals [DESCRIPTION_CHANGED] =
		g_signal_new (
			"description-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, description_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_view_signals [PHOTO_DIR_CHANGED] =
		g_signal_new (
			"photo-dir-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, photo_dir_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_view_signals [OPTION_CHANGED] =
		g_signal_new (
			"option-changed",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, option_changed),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_view_signals [PROCESS] =
		g_signal_new (
			"process",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, process),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	/* gallery window */
	pgt_view_signals [ADD_GALLERY] =
		g_signal_new (
			"add-gallery",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, add_gallery),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);

	pgt_view_signals [DEL_GALLERY] =
		g_signal_new (
			"del-gallery",
			G_OBJECT_CLASS_TYPE (klass),
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (PgtViewClass, del_gallery),
			NULL, NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE, 0);
}

GType
pgt_view_get_gtype (void)
{
        static GType type = 0;
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (PgtViewClass),
                        NULL,           /* base_init */
                        NULL,           /* base_finalize */
                        (GClassInitFunc) class_init,
                        NULL,           /* class_finalize */
                        NULL,           /* class_data */
                        sizeof (PgtView),
                        0,              /* n_preallocs */
                        (GInstanceInitFunc) instance_init,
                };
                type = g_type_register_static (G_TYPE_OBJECT, "PgtView", &info, (GTypeFlags)0);
        }
        return type;
}

PgtView*   
pgt_view_new (void)
{
	return PGT_VIEW (g_object_new (PGT_TYPE_VIEW, NULL));
}

/* no prefs yet
PgtPrefs* 
create_prefs (PgtView *self)
{
	GtkWidget *mainwin = NULL;
	GtkWidget *win = NULL;

	self->prefs = pgt_prefs_new ();
	g_object_ref (self->prefs);

	mainwin = pgt_mainwin_get_window (self->mainwin);
	win = pgt_prefs_get_window (self->prefs);
	
	gtk_window_set_transient_for (GTK_WINDOW (win), GTK_WINDOW (mainwin));
	gtk_widget_show_all (win);

	return self->prefs;
}
*/ 

void	 
pgt_view_create_mainwin (PgtView *self)
{
	GtkWidget *win = NULL;

	self->mainwin = pgt_mainwin_new (self->gallerystore);
	g_object_ref (self->mainwin);

	/* connect signals */	
	g_signal_connect_swapped (self->mainwin, "gallery-changed", 
				  G_CALLBACK (gallery_changed_cb), self);
	g_signal_connect_swapped (self->mainwin, "manage-galleries", 
				  G_CALLBACK (manage_galleries_cb), self);
	g_signal_connect_swapped (self->mainwin, "album-name-changed", 
				  G_CALLBACK (album_name_changed_cb), self);
	g_signal_connect_swapped (self->mainwin, "photo-dir-changed", 
				  G_CALLBACK (photo_dir_changed_cb), self);
	g_signal_connect_swapped (self->mainwin, "option-changed", 
				  G_CALLBACK (option_changed_cb), self);
	g_signal_connect_swapped (self->mainwin, "process", 
				  G_CALLBACK (process_cb), self);
	g_signal_connect_swapped (self->mainwin, "delete", 
				  G_CALLBACK (mainwin_delete_cb), self);

	win = pgt_mainwin_get_window (self->mainwin);
	gtk_widget_show_all (win);
}

gchar const * 
pgt_view_get_gallery_name (PgtView *self)
{
	return pgt_mainwin_get_gallery_name (self->mainwin);
}

gchar const * 
pgt_view_get_album_name (PgtView *self)
{
	return pgt_mainwin_get_album_name (self->mainwin);
}

gchar const * 
pgt_view_get_album_date (PgtView *self)
{
	return pgt_mainwin_get_album_date (self->mainwin);
}

gchar const * 
pgt_view_get_album_desc (PgtView *self)
{
	return pgt_mainwin_get_album_desc (self->mainwin);
}

gchar const * 
pgt_view_get_photo_dir (PgtView *self)
{
	return pgt_mainwin_get_photo_dir (self->mainwin);
}

gboolean 
pgt_view_get_option (PgtView *self, PgtConvertOption opt)
{
	return pgt_mainwin_get_option (self->mainwin, opt);
}

void 
pgt_view_set_progress_value (PgtView *self, gfloat val)
{
	pgt_mainwin_set_progress_value (self->mainwin, val);
}

void 
pgt_view_set_progress_text (PgtView *self, gchar const *text)
{
	pgt_mainwin_set_progress_text (self->mainwin, text);
}

gchar const * 
pgt_view_gallery_get_name (PgtView *self)
{
	g_assert (self && PGT_IS_VIEW (self));
	g_assert (PGT_IS_GALLERIES (self->galleries));
	return pgt_galleries_get_name (self->galleries);
}

gchar const * 
pgt_view_gallery_get_url (PgtView *self)
{
	g_assert (self && PGT_IS_VIEW (self));
	g_assert (PGT_IS_GALLERIES (self->galleries));
	return pgt_galleries_get_url (self->galleries);
}

gchar *
pgt_view_gallery_get_selected_name (PgtView *self)
{
	return pgt_galleries_get_selected_name (self->galleries);
}

void 
pgt_view_internal_set_gallery_store (PgtView      *self, 
				     GtkTreeModel *gallerystore)
{
	self->gallerystore = gallerystore;
	g_object_ref (self->gallerystore);
}
