/*
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   Copyright (C) 2007 by Javier Goday <jgoday@gmail.com>
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "kgetengine.h"
#include "kgetdbus.h"

#include <QtDBus/QDBusConnectionInterface>
#include <QTimer>
#include <KDebug>

#include "plasma/datacontainer.h"

KGetEngine::KGetEngine(QObject* parent, const QVariantList& args)
    : Plasma::DataEngine(parent),
    m_refreshTime(2500)
{
    Q_UNUSED(args);
    	
	m_timer = new QTimer(this);
	m_timer->setSingleShot(false);

	interface = QDBusConnection::sessionBus().interface();

	QObject::connect(interface, 
			SIGNAL(serviceOwnerChanged(const QString &, const QString &, const QString &)),
			this, SLOT(serviceRegistered(const QString &, const QString &, const QString &)));

    connect(m_timer, SIGNAL(timeout()), this, SLOT(updatePercents()));
}

KGetEngine::~KGetEngine()
{
}

void KGetEngine::init()
{
}

bool KGetEngine::sourceRequested(const QString &name)
{
	if (name == "KGet") {	
		setKGetSourceData(name);	
	}
	
	return true;
}

bool KGetEngine::isDBusServiceRegistered()
{
	return interface->isServiceRegistered("org.kde.kget");	
}

void KGetEngine::serviceRegistered(const QString &name, const QString &old_owner, 
						const QString &new_owner)
{
	if(name == "org.kde.kget") {
		setKGetSourceData("KGet");
	}	
}

void KGetEngine::setKGetSourceData(const QString &name)
{	
	if (name == "KGet") {
		clearData(name);
	
		if(isDBusServiceRegistered()) {
			OrgKdeKgetInterface kget_interface(DEFAULT_SERVICE, DEFAULT_PATH,
								QDBusConnection::sessionBus());

            if (!m_timer->isActive()) {
                m_timer->start(m_refreshTime);
            }
			setData(I18N_NOOP(name), I18N_NOOP("error"), false);
		 	setData(I18N_NOOP(name), I18N_NOOP("transfers"),
						 kget_interface.transfers().value());
		}
		else {
            if(m_timer->isActive()) {
                m_timer->stop();
            }
			setData(I18N_NOOP(name), I18N_NOOP("error"), true);
			setData(I18N_NOOP(name), I18N_NOOP("errorMessage"), 
					I18N_NOOP("Is KGet up and running?"));
		}
	}
	checkForUpdates();
}

void KGetEngine::slotTransfersChanged(const QVariantMap &transfers)
{
    setData(I18N_NOOP("KGet"), I18N_NOOP("error"), false);
    setData(I18N_NOOP("KGet"), I18N_NOOP("transfers"),
                                transfers);
    
    checkForUpdates();
}

void KGetEngine::updatePercents()
{
	setKGetSourceData("KGet");
}

void KGetEngine::setRefreshTime(const uint time)
{
    m_refreshTime = time;
    if (m_timer->isActive()) {
        m_timer->stop();
        m_timer->start(m_refreshTime);
    }
}

uint KGetEngine::refreshTime()
{
    return m_refreshTime;
}

#include "kgetengine.moc"
