/**
 *
 * @file     preferences.c
 * @brief    Playground preferences
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Fri Dec 02, 2005 23:51
 *
 * Copyright (C) 2005, 2006 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>

#include "preferences.h"
#include "interface.h"

#include <gconf/gconf-client.h>
#include <glade/glade.h>

#include <string.h>

/* Defines */
#define GCONF_PATH "/apps/playground_applet"
#define GCONF_RUN_START_KEY "/apps/playground_applet/prefs/run_start"
#define GCONF_HIDE_START_KEY "/apps/playground_applet/prefs/hide_start"
#define GCONF_PLAY_START_KEY "/apps/playground_applet/prefs/play_start"
#define GCONF_PLUGIN_KEY "/apps/playground_applet/prefs/plugin"

/* Constants */

enum { DESCRIPTION_COLUMN, FILE_COLUMN, N_COLUMNS };

/* Static function declaration */

static gboolean init_glade_widgets (void);
static gboolean init_gconf_preferences (void);
static gboolean init_gconf_widgets (void);
static void create_plugins_list (void);
static void plugins_list_fill (pg_player_t *player);
static void plugin_select (gchar const *plugin);
static void on_gconf_preferences_change (GConfClient *client, guint cnxn_id,
                                         GConfEntry *entry, gpointer data);

/* Static variables definition */

static GladeXML *xml_ = NULL;

/* Ugly, but we need to keep a copy to use in libglade callbacks */
static pg_player_t *player_ = NULL;

/* Widgets */
static GtkWidget *pref_window_ = NULL;
static GtkWidget *pref_notebook_ = NULL;
static GtkWidget *plugins_list_ = NULL;
static GtkWidget *run_start_cbox_ = NULL;
static GtkWidget *hide_start_cbox_ = NULL;
static GtkWidget *play_start_cbox_ = NULL;

/* Plugins list */
static GtkListStore *list_store_ = NULL;

/* GConf client */
static GConfClient *gconf_ = NULL;

gboolean
preferences_init (pg_player_t *pl)
{
  player_ = pl;

  if (player_ == NULL)
    {
      return FALSE;
    }

  glade_init();

  xml_ = glade_xml_new (PG_DATADIR"/playground.glade", NULL, NULL);
  if (!xml_)
    {
      return FALSE;
    }

  if (!init_glade_widgets ())
    {
      return FALSE;
    }

  /* Fill the plugins list before reading preferences */
  plugins_list_fill (player_);

  if (!init_gconf_preferences ())
    {
      gui_show_error (_("Could not initialize GConf."));
      return FALSE;
    }

  return TRUE;
}

void
preferences_show (void)
{
  /* pref_window_ is set to NULL if window is destroyed */
  if (pref_window_ == NULL)
    {
      preferences_init (player_);
    }
  gtk_widget_show (pref_window_);
}

void
preferences_startup (void)
{
  if (preferences_run_startup ())
    {
      player_start (player_);
    }
  if (preferences_hide_startup ())
    {
      player_show_all (player_, FALSE);
    }
  if (preferences_play_startup ())
    {
      if (!player_is_playing (player_))
        {
          player_play_pause (player_);
        }
    }
  else
    {
      player_stop (player_);
    }
}

gboolean
preferences_run_startup (void)
{
  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (run_start_cbox_));
}

gboolean
preferences_hide_startup (void)
{
  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_start_cbox_));
}

gboolean
preferences_play_startup (void)
{
  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (play_start_cbox_));
}

/* Static functions definition */

static gboolean
init_glade_widgets (void)
{
  pref_window_ = glade_xml_get_widget (xml_, "pref_window");
  if (pref_window_ == NULL)
    {
      return FALSE;
    }

  pref_notebook_ = glade_xml_get_widget (xml_, "pref_notebook");
  if (pref_notebook_ == NULL)
    {
      return FALSE;
    }

  plugins_list_ = glade_xml_get_widget (xml_, "plugins_list");
  if (plugins_list_ == NULL)
    {
      return FALSE;
    }

  run_start_cbox_ = glade_xml_get_widget (xml_, "run_start_cbox");
  if (run_start_cbox_ == NULL)
    {
      return FALSE;
    }

  hide_start_cbox_ = glade_xml_get_widget (xml_, "hide_start_cbox");
  if (hide_start_cbox_ == NULL)
    {
      return FALSE;
    }

  play_start_cbox_ = glade_xml_get_widget (xml_, "play_start_cbox");
  if (play_start_cbox_ == NULL)
    {
      return FALSE;
    }

  glade_xml_signal_autoconnect (xml_);

  create_plugins_list ();

  return TRUE;
}

static gboolean
init_gconf_preferences (void)
{
  if (!gconf_init(0, NULL, NULL))
    {
      return FALSE;
    }

  /* Get the GconfClient, tell it we want to monitor playground prefs */
  gconf_ = gconf_client_get_default ();
  if (gconf_ == NULL)
    {
      return FALSE;
    }

  gconf_client_add_dir (gconf_, GCONF_PATH, GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_notify_add (gconf_, GCONF_PATH "/prefs",
                           on_gconf_preferences_change, NULL, NULL, NULL);

  return init_gconf_widgets ();
}

static gboolean
init_gconf_widgets (void)
{
  if (!gconf_client_key_is_writable (gconf_, GCONF_RUN_START_KEY, NULL))
    {
      gtk_widget_set_sensitive(run_start_cbox_, FALSE);
    }
  if (!gconf_client_key_is_writable (gconf_, GCONF_HIDE_START_KEY, NULL))
    {
      gtk_widget_set_sensitive(hide_start_cbox_, FALSE);
    }
  if (!gconf_client_key_is_writable (gconf_, GCONF_PLAY_START_KEY, NULL))
    {
      gtk_widget_set_sensitive(play_start_cbox_, FALSE);
    }
  if (!gconf_client_key_is_writable (gconf_, GCONF_PLUGIN_KEY, NULL))
    {
      gtk_widget_set_sensitive(plugins_list_, FALSE);
    }

  /* Checkboxes */
  gtk_toggle_button_set_active
    (GTK_TOGGLE_BUTTON (run_start_cbox_),
     gconf_client_get_bool (gconf_, GCONF_RUN_START_KEY, NULL));
  gtk_toggle_button_set_active
    (GTK_TOGGLE_BUTTON (hide_start_cbox_),
     gconf_client_get_bool (gconf_, GCONF_HIDE_START_KEY, NULL));
  gtk_toggle_button_set_active
    (GTK_TOGGLE_BUTTON (play_start_cbox_),
     gconf_client_get_bool (gconf_, GCONF_PLAY_START_KEY, NULL));

  /* Plugins list */
  plugin_select (gconf_client_get_string (gconf_, GCONF_PLUGIN_KEY, NULL));

  return TRUE;
}

static void
create_plugins_list (void)
{
  GtkTreeViewColumn *column = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeSelection *selection = NULL;

  list_store_ = gtk_list_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING);

  gtk_tree_view_set_model (GTK_TREE_VIEW (plugins_list_),
                           GTK_TREE_MODEL (list_store_));

  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes (_("Plugin"), renderer,
                                                     "text",
                                                     FILE_COLUMN,
                                                     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (plugins_list_), column);

  column = gtk_tree_view_column_new_with_attributes (_("Description"),
                                                     renderer, "text",
                                                     DESCRIPTION_COLUMN,
                                                     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (plugins_list_), column);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (plugins_list_));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_BROWSE);
}

static void
plugins_list_fill (pg_player_t *player)
{
  gint i = 0;

  gtk_list_store_clear (list_store_);
  for (i = 0; i < player_plugin_count (player); i++)
    {
      GtkTreeIter iter;
      gtk_list_store_append (list_store_, &iter);
      gtk_list_store_set (list_store_, &iter,
                          FILE_COLUMN,
                          player_plugin_file (player_, i),
                          DESCRIPTION_COLUMN,
                          player_plugin_description (player_, i),
                          -1);
    }
}

static void
plugin_select (gchar const *plugin)
{
  GtkTreeIter iter;
  GtkTreeModel *model = NULL;
  gboolean has_iter = FALSE;

  if (plugin == NULL)
    {
      return;
    }

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (plugins_list_));

  has_iter = gtk_tree_model_get_iter_first (model, &iter);

  while (has_iter)
    {
      gchar *description = NULL;
      gchar *column_plugin = NULL;
      gtk_tree_model_get (model, &iter,
                          FILE_COLUMN, &column_plugin,
                          DESCRIPTION_COLUMN, &description, -1);
      if (strcmp (plugin, column_plugin) == 0)
        {
          GtkTreePath *path = gtk_tree_model_get_path (model, &iter);
          gtk_tree_view_set_cursor (GTK_TREE_VIEW (plugins_list_),
                                    path, NULL, FALSE);
          gtk_tree_path_free (path);

          if (!player_plugin_load (player_, plugin))
            {
              enum { MAX_BUFFER = 512 };
              gchar str[MAX_BUFFER] = "";

              sprintf (str, _("Unable to load \"%s\" plugin."), description);
              gui_show_error (str);
            }
        }
      g_free (description);
      g_free (column_plugin);

      has_iter = gtk_tree_model_iter_next (model, &iter);
    }
}

/* Static function definition (used by GConf)*/
static void
on_gconf_preferences_change (GConfClient *client, guint cnxn_id,
                             GConfEntry *entry, gpointer data)
{
  if (strcmp(entry->key, GCONF_RUN_START_KEY) == 0)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (run_start_cbox_),
                                    gconf_value_get_bool (entry->value));
    }
  if (strcmp(entry->key, GCONF_HIDE_START_KEY) == 0)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (hide_start_cbox_),
                                    gconf_value_get_bool (entry->value));
    }
  if (strcmp(entry->key, GCONF_PLAY_START_KEY) == 0)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (play_start_cbox_),
                                    gconf_value_get_bool (entry->value));
    }
  if (strcmp(entry->key, GCONF_PLUGIN_KEY) == 0)
    {
      plugin_select (gconf_value_get_string (entry->value));
    }
}


/* Public function definition (used by libglade)*/

void
on_window_show (GtkWindow *window, gpointer data)
{
  gtk_notebook_set_current_page (GTK_NOTEBOOK (pref_notebook_), 0);
  player_plugin_reload_list (player_);
  plugins_list_fill (player_);
  plugin_select (gconf_client_get_string (gconf_, GCONF_PLUGIN_KEY, NULL));
}

void
on_window_destroy (GtkWindow *window, gpointer data)
{
  pref_window_ = NULL;
}

void
on_close_clicked (GtkButton *button, gpointer data)
{
  GtkTreeIter iter;
  GtkTreeModel *model = NULL;
  GtkTreeSelection *selection = NULL;
  gchar *plugin = NULL;
  gchar *description = NULL;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (plugins_list_));
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_tree_model_get (model, &iter,
                          FILE_COLUMN, &plugin,
                          DESCRIPTION_COLUMN, &description, -1);
      gconf_client_set_string (gconf_, GCONF_PLUGIN_KEY, plugin, NULL);
      g_free (plugin);
      g_free (description);
    }
  gtk_widget_hide (pref_window_);
}

void
on_run_start_toggled  (GtkToggleButton *checkbox, gpointer data)
{
  gboolean state = gtk_toggle_button_get_active (checkbox);
  gconf_client_set_bool (gconf_, GCONF_RUN_START_KEY, state, NULL);
}

void
on_hide_start_toggled  (GtkToggleButton *checkbox, gpointer data)
{
  gboolean state = gtk_toggle_button_get_active (checkbox);
  gconf_client_set_bool (gconf_, GCONF_HIDE_START_KEY, state, NULL);
}

void
on_play_start_toggled  (GtkToggleButton *checkbox, gpointer data)
{
  gboolean state = gtk_toggle_button_get_active (checkbox);
  gconf_client_set_bool (gconf_, GCONF_PLAY_START_KEY, state, NULL);
}
