/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "libpolyxmass-atomspec.h"
#include "libpolyxmass-config.h"

/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmAtomSpec *
libpolyxmass_atomspec_new (void)
{
  PxmAtomSpec *as = NULL;
  
  as = g_malloc0 (sizeof (PxmAtomSpec));
  
  as->poltype = g_strdup ("");
  as->atomdef = g_strdup ("");
  as->file = g_strdup ("");
  as->source = CONFIG_SYSTEM;

  return as;
}


gboolean
libpolyxmass_atomspec_set_poltype (PxmAtomSpec *as, gchar *poltype)
{
  g_assert (as != NULL && poltype != NULL);

  g_assert (as->poltype != NULL);
  
  g_free (as->poltype);
  
  as->poltype = g_strdup (poltype);
  
  return TRUE;
}


gboolean
libpolyxmass_atomspec_set_atomdef (PxmAtomSpec *as, gchar *atomdef)
{
  g_assert (as != NULL && atomdef != NULL);

  g_assert (as->atomdef != NULL);
  
  g_free (as->atomdef);
  
  as->atomdef = g_strdup (atomdef);
  
  return TRUE;
}


gboolean
libpolyxmass_atomspec_set_file (PxmAtomSpec *as, gchar *file)
{
  g_assert (as != NULL && file != NULL);

  g_assert (as->file != NULL);
  
  g_free (as->file);
  
  as->file = g_strdup (file);
  
  return TRUE;
}


gboolean
libpolyxmass_atomspec_set_source (PxmAtomSpec *as, gint source)
{
  g_assert (as != NULL);
  g_assert (source == CONFIG_SYSTEM || source == CONFIG_USER);
  
  as->source = source;
  
  return TRUE;
}



/*  
    LOCATING FUNCTIONS
*/




/*
  UTILITY FUNCTIONS
*/
gboolean
libpolyxmass_atomspec_reset (PxmAtomSpec *as)
{
  g_assert (as != NULL);
  
  g_assert (as->poltype != NULL);
  g_free (as->poltype);
  as->poltype = g_strdup ("");
  
  g_assert (as->atomdef != NULL);
  g_free (as->atomdef);
  as->atomdef = g_strdup ("");
  
  g_assert (as->file != NULL);
  g_free (as->file);
  as->file = g_strdup ("");
  
  as->source = CONFIG_SYSTEM;

  return TRUE;
}


/*
  The /etc/polyxmass/atom-defs/atom-defs.cat and similar files.
*/
gboolean
libpolyxmass_atomspec_init_from_cat_line (gchar *line,
					  PxmAtomSpec *as) 
{
  gint iter = 0;

  GString * gs = NULL;

  gboolean atomdef_done = FALSE;
  
  g_assert (as != NULL);

  gs = g_string_new ("");

  /* 
     We have a line of the kind 

     "defname=/full/path/to/atoms.xml"
     
     (note that the newline character may be found if the line in
     question is not the last of the file!), we just need to
     deconstruct this into two pieces.
  */
  while (line [iter] != '\x0' && line [iter] != '\n')
    {
      if (line [iter] == '=')
	{
	  /* We should now have the atom definition name in the gs
	   * GString.
	   */
	  if (strlen (gs->str) <= 0)
	    {
	      g_string_free (gs, TRUE);

	      return FALSE;
	    }

	  libpolyxmass_atomspec_set_atomdef (as, gs->str);
	  
	  gs = g_string_truncate (gs, 0);

	  /*
	    g_string_free (gs, TRUE);
	    gs= g_string_new ("");
	  */
	  
	  atomdef_done = TRUE;
	}

      else
	{
	  /* We are iterating in some character, so we should simply
	   * put this into the gs GString.
	   */
	  gs = g_string_append_c (gs, line [iter]);
	}

      iter++;		    
    }

  /* 
     And now we are at the end of the line, so the atom definition
     file name should have completed now, but check that the atomdef
     was done already.
   */
  if (atomdef_done == FALSE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to parse atomspec line: '%s'\n"),
	     __FILE__, __LINE__, line);
      
      g_string_free (gs, TRUE);
      
      return FALSE;
    }

  /* 
     We should now have the atom definition file path+name in the gs
     GString.
   */
  if (strlen (gs->str) <= 0)
    {
      g_string_free (gs, TRUE);
      
      return FALSE;
    }
  
  /* 
     Make sure that the file actually exists. Also be certain that
     the gs->str string is an absolute filename.
  */
  g_assert (TRUE == g_file_test (gs->str, G_FILE_TEST_EXISTS));
  g_assert (TRUE == g_path_is_absolute (gs->str));
  
  /* 
     OK, we faithfully checked this, we can use it!
  */
  libpolyxmass_atomspec_set_file (as, gs->str);
  
  /* We are no longer going to use this allocated GString.
   */
  /* Free the gs GString, as the spec_set_xxx functions allocate
     a string using the param as a template.
  */
  g_string_free (gs, TRUE);
  
  return TRUE;
}


gint
libpolyxmass_atomspec_parse_atom_defs_cat_file (gchar* file,
						GPtrArray *GPA,
						gint pre_ap_pend,
						gint sys_user)
{
  gint count = 0;
  
  gchar *read = NULL;
  gchar *line = NULL;
  
  FILE *cfgp = NULL;

  PxmAtomSpec *as = NULL;
  
  /* 
     We are given a file name were each line gives the name of an atom
     definition and the location of the atom definition file. All we
     have to do is parse each line and for each valid one create a
     PxmAtomSpec item that we append/prepend to the GPA file.
  */
  g_assert (file != NULL);
  g_assert (TRUE == g_file_test (file, G_FILE_TEST_EXISTS));
  g_assert (TRUE == g_path_is_absolute (file));
  g_assert (GPA != NULL);
  g_assert (pre_ap_pend == CONFIG_APPEND || pre_ap_pend == CONFIG_PREPEND);
  g_assert (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_USER);

  cfgp = fopen (file, "r");

  if (cfgp == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to  open file: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return -1;
    }
  
  /* 
     The lines that we have to parse here are in this format:
     defname=/an/absolute/path/atomsdefname.xml, for example:

     basic=/usr/share/polyxmass/atom-defs/atoms.xml
   */

  /* Allocate the memory into which each line of the file will be
   * store for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));

  read = line;
  
  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, cfgp);

      /*
      debug_printf (("file: '%s' -- read line: '%s'\n", 
		     file, read));
      */
      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be in 'read' for
	     us to parse.
	  */
	  /* We first have to check if the line we are reading is
	     actually a comment line, beginning with the '#'
	     character. The same applies if the line is empty (starts
	     with a newline...
	   */
	  if (read [0] == '#' || read [0] == '\n')
	    continue;

	  while (read [0] != '\x0')
	    {
	      if (TRUE == g_ascii_isspace (read [0]))
		read++;
	      else
		break;
	    }

	  if (strlen (read) <= 0)
	    continue;
	  
	  /* 
	     Next check if we are on a line that describes the
	     correspondence between an atom definition name and its
	     definition file. This is most probably the case if '=' is
	     present in the line.
	  */
	  if (NULL != strchr (read, '='))
	    {
	      /* 
		 We are dealing with a defname=/path/to/atoms.xml
		 correspondence.
	      */
	      
	      /* Allocate the PxmAtomSpec instance that we'll
		 initialize using the data read from parsing of the
		 line.
	      */
	      as = libpolyxmass_atomspec_new ();
	      libpolyxmass_atomspec_set_source (as, sys_user);
	      
	      if (FALSE == libpolyxmass_atomspec_init_from_cat_line (read,
								     as))
		{
		  g_free (line);
		  
		  libpolyxmass_atomspec_free (as);
		  
		  fclose (cfgp);
		  
		  return -1;
		}
	     
 	      if (pre_ap_pend == CONFIG_APPEND)
		g_ptr_array_add (GPA, as);
	      else /* if (pre_ap_pend == CONFIG_PREPEND) */
		{
		  /*
		  debug_printf (("array length is: '%d'\n", GPA->len));
		  */
		  g_ptr_array_insert_val (GPA, 0, as);
	      
	      /*
		debug_printf (("added new as: '%s--%s--%s'\n",
		as->poltype, as->atomdef, as->file));
	      */
		}
	      
	      
	      count++;

	      continue;
	    }
	}
    }

  /* Now that we do not need this string anymore, since the file is
   * finished parsing, we can free the memory for reuse of the pointer.
   */
  g_free (line);

  fclose (cfgp);

  return count;
}



gint
libpolyxmass_atomspec_parse_atom_defs_cat_files (GPtrArray *GPA,
						 gint pre_ap_pend,
						 gint sys_user)
{
  const gchar *parsed_file = NULL;
  gchar *parsed_file_utf8 = NULL;
  gchar *catalogue_dir = NULL;
  gchar *absolute_filename = NULL;
  
  GDir*  cat_dir = NULL;
  
  GError *g_error = NULL;

  gint flags = 0;
  gint count = 0;
  
  /*
    We are asked to fill the GPA array with allocated PxmAtomSpec
    objects corresponding to data found in the different atom-def
    catalogue files that are parsed according to function parameter
    requirements.
  */

  g_assert (GPA != NULL);
  g_assert (pre_ap_pend == CONFIG_APPEND || pre_ap_pend == CONFIG_PREPEND);
  g_assert (sys_user == CONFIG_USER 
	    || sys_user == CONFIG_SYSTEM 
	    || sys_user == CONFIG_BOTH);


  if (sys_user == CONFIG_USER || sys_user == CONFIG_BOTH)
    {
      catalogue_dir = libpolyxmass_config_get_user_atom_defs_cat_dir ();

      /* 
	 We only parse the files in the directory if the directory
	 exists...
      */
      if (catalogue_dir != NULL 
	  && TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
	{
	  /* Let's check if there are catalogue files in the directory.
	   */
	  
	  cat_dir = g_dir_open (catalogue_dir,
				flags,
				&g_error);

	  /* And now get the files in this directory and check if 
	     they are actually catalogue files.
	  */
	  parsed_file = g_dir_read_name (cat_dir);
	  
	  while (parsed_file != NULL)
	    {
	      /*
		debug_printf (("parsed_file is: '%s'\n", parsed_file));
	      */

	      /* Translate the name to utf_8.
	       */
	      parsed_file_utf8 = g_filename_to_utf8 (parsed_file,
						     -1,
						     NULL,
						     NULL,
						     &g_error);
	      if (NULL == parsed_file_utf8)
		{
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}

	      /* Make sure the filename actually ends with the proper
		 suffix ("atom-defs-cat").
	      */
	      if (FALSE == g_str_has_suffix (parsed_file_utf8,
                                             "atom-defs-cat"))
		{
		  g_free (parsed_file_utf8);
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}
	      
	      /* We will parse the file, but to do that we have to 
		 construct an absolute path to that file.
	      */
	      absolute_filename = g_strdup_printf ("%s/%s",
						   catalogue_dir,
						   parsed_file_utf8);
	      
	      /*
		Do the parsing of the file into the GPA array.
	      */
	      count += libpolyxmass_atomspec_parse_atom_defs_cat_file
		(absolute_filename,
		 GPA,
		 pre_ap_pend,
		 CONFIG_USER);
	      
	      if (count == -1)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
			 _("%s@%d: failed to parse file: '%s'\n"),
			 __FILE__, __LINE__, absolute_filename);
		}
	      
	      g_free (parsed_file_utf8);
	      g_free (absolute_filename);

	      parsed_file = g_dir_read_name (cat_dir);

	      continue;
	    }
	  
	  /* We can free the catalogue_dir string, we do not need it 
	     anymore.
	  */
	  g_free (catalogue_dir);
	  
	  g_dir_close (cat_dir);
	}
      /* End of
	 if (TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
      */
    }
  /* End of 
     if (sys_user == CONFIG_USER || sys_user == CONFIG_BOTH)
  */
  if (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_BOTH)
    {
      catalogue_dir = libpolyxmass_config_get_system_atom_defs_cat_dir ();

      /* 
	 We only parse the files in the directory if the directory
	 exists...
      */
      if (catalogue_dir != NULL 
	  && TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
	{
	  /* Let's check if there are catalogue files in the directory.
	   */
	  
	  cat_dir = g_dir_open (catalogue_dir,
				flags,
				&g_error);

	  /* And now get the files in this directory and check if 
	     there are actually catalogue files.
	  */
	  parsed_file = g_dir_read_name (cat_dir);
	  
	  while (parsed_file != NULL)
	    {
	      /*
		debug_printf (("parsed_file is: '%s'\n", parsed_file));
	      */

	      /* Translate the name to utf_8.
	       */
	      parsed_file_utf8 = g_filename_to_utf8 (parsed_file,
						     -1,
						     NULL,
						     NULL,
						     &g_error);
	      if (NULL == parsed_file_utf8)
		{
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}

	      /* Make sure the filename actually ends with the proper
		 suffix ("atom-defs-cat").
	      */
	      if (FALSE == g_str_has_suffix (parsed_file_utf8,
                                             "atom-defs-cat"))
		{
		  g_free (parsed_file_utf8);
		  parsed_file = g_dir_read_name (cat_dir);
		  continue;
		}
	      
	      /* We will parse the file, but to do that we have to 
		 construct an absolute path to that file.
	      */
	      absolute_filename = g_strdup_printf ("%s/%s",
						   catalogue_dir,
						   parsed_file_utf8);
	      
	      /*
		Do the parsing of the file into the GPA array.
	      */
	      count += libpolyxmass_atomspec_parse_atom_defs_cat_file
		(absolute_filename,
		 GPA,
		 pre_ap_pend,
		 CONFIG_SYSTEM);
	      
	      if (count == -1)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
			 _("%s@%d: failed to parse file: '%s'\n"),
			 __FILE__, __LINE__, absolute_filename);
		}
	      
	      g_free (parsed_file_utf8);
	      g_free (absolute_filename);
	      
	      parsed_file = g_dir_read_name (cat_dir);

	      continue;
	    }
	  
	  /* We can free the catalogue_dir string, we do not need it 
	     anymore.
	  */
	  g_free (catalogue_dir);

	  g_dir_close (cat_dir);
	}
      /* End of
	 if (TRUE == g_file_test (catalogue_dir, G_FILE_TEST_IS_DIR))
      */
    }
  /* End of 
     if (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_BOTH)
  */

  return count;
}



/* 
   The polchem-defs-atom-defs-dic dictionary files.
*/
gboolean
libpolyxmass_atomspec_init_from_dic_line (gchar *line,
					  PxmAtomSpec *as) 
{
  gint iter = 0;

  GString * gs = NULL;

  gboolean poltype_done = FALSE;
  
  g_assert (as != NULL);

  gs = g_string_new ("");

  /* 
     We have a line of the kind "polchemdef_name=atomdef_name"
     (note that the newline character may be found if the line in
     question is not the last of the file!), we just need to
     deconstruct this into two pieces.
  */
  while (line [iter] != '\x0' && line [iter] != '\n')
    {
      if (line [iter] == '=')
	{
	  /* We should now have the polymer definition name in the gs
	   * GString.
	   */
	  if (strlen (gs->str) <= 0)
	    {
	      g_string_free (gs, TRUE);

	      return FALSE;
	    }

	  libpolyxmass_atomspec_set_poltype (as, gs->str);
	  
	  g_string_free (gs, TRUE);
	  gs = g_string_new ("");
	  
	  poltype_done = TRUE;
	}

      else
	{
	  /* We are iterating in some character, so we should simply
	   * put this into the gs GString.
	   */
	  gs = g_string_append_c (gs, line [iter]);
	}

      iter++;		    
    }

  /* 
     And now we are at the end of the line, so the atom definition
     file name should have completed now, but check that polymer type
     was done already.
   */
  if (poltype_done == FALSE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to parse atomspec line: '%s'\n"),
	     __FILE__, __LINE__, line);
      
      g_string_free (gs, TRUE);
      
      return FALSE;
    }

  /* 
     We should now have the atom definition file name in the gs
     GString.
   */
  if (strlen (gs->str) <= 0)
    {
      g_string_free (gs, TRUE);
      
      return FALSE;
    }
  
  libpolyxmass_atomspec_set_atomdef (as, gs->str);
  
  g_string_free (gs, TRUE);
  
  return TRUE;
}


gint
libpolyxmass_atomspec_parse_polchem_defs_atom_defs_dic_file (gchar* file,
							     GPtrArray *GPA,
							     gint pre_ap_pend)
{
  gint count = 0;
  
  gchar *read = NULL;
  gchar *line = NULL;
  
  FILE *cfgp = NULL;

  PxmAtomSpec *as = NULL;
  
  /* 
     We are given a file name were each line gives the name of a
     polymer definition and the name of an atom definition, separated
     by '='. All we have to do is parse each line and for each valid
     one create a PxmAtomSpec item that we append/prepend to the GPA
     file.
  */
  g_assert (file != NULL);
  g_assert (TRUE == g_file_test (file, G_FILE_TEST_EXISTS));
  g_assert (GPA != NULL);
  g_assert (pre_ap_pend == CONFIG_APPEND || pre_ap_pend == CONFIG_PREPEND);

  cfgp = fopen (file, "r");

  if (cfgp == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to  open file: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return -1;
    }
  
  /* 
     The lines that we have to parse here are in this format:
     polchemdef_name=atomdef_name, for example:

     protein=basic
   */

  /* Allocate the memory into which each line of the file will be
   * stored for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));

  read = line;
  
  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, cfgp);

      /*
      debug_printf (("file: '%s' -- read line: '%s'\n", 
		     file, read));
      */
      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be in 'read' for
	     us to parse.
	  */
	  /* We first have to check if the line we are reading is
	     actually a comment line, beginning with the '#'
	     character. The same applies if the line is empty (starts
	     with a newline...
	   */
	  if (read [0] == '#' || read [0] == '\n')
	    continue;

	  while (read [0] != '\x0')
	    {
	      if (TRUE == g_ascii_isspace (read [0]))
		read++;
	      else
		break;
	    }

	  if (strlen (read) <= 0)
	    continue;
	  
	  /* 
	     Next check if we are on a line that describes the
	     correspondence between a polymer definition n ame and an
	     atom definition name. This is most probably the case if
	     '=' is present in the line.
	  */
	  if (NULL != strchr (read, '='))
	    {
	      /* 
		 We are dealing with a polchemdef_name=atomdef_name
		 correspondence.
	      */
	      
	      /* Allocate the PxmAtomSpec instance that we'll
		 initialize using the data read from parsing of the
		 line.
	      */
	      as = libpolyxmass_atomspec_new ();
	      
	      if (FALSE == libpolyxmass_atomspec_init_from_dic_line (read,
								     as))
		{
		  g_free (line);
		  
		  libpolyxmass_atomspec_free (as);
		  
		  fclose (cfgp);
		  
		  return -1;
		}
	     
 
	      if (pre_ap_pend == CONFIG_APPEND)
		g_ptr_array_add (GPA, as);
	      else /* if (pre_ap_pend == CONFIG_PREPEND) */
		{
		  g_ptr_array_insert_val (GPA, 0, as);
		  /*
		  debug_printf (("array length is: '%d'\n", GPA->len));
		  */
		}
	      

	      /*
		debug_printf (("added new as: '%s--%s--%s'\n",
		as->poltype, as->atomdef, as->file));
	      */
	      
	      count++;

	      continue;
	    }
	}
    }

  /* Now that we do not need this string anymore, since the file is
   * finished parsing, we can free the memory for reuse of the pointer.
   */
  g_free (line);

  fclose (cfgp);

  return count;
}


gint
libpolyxmass_atomspec_parse_polchem_defs_atom_defs_dic_files (GPtrArray *GPA,
							      gint pre_ap_pend,
							      gint sys_user)
{
  const gchar *parsed_file = NULL;
  gchar *parsed_file_utf8 = NULL;
  gchar *dictionary_dir = NULL;
  gchar *absolute_filename = NULL;

  GDir*  dic_dir = NULL;

  GError *g_error = NULL;

  gint flags = 0;
  gint count = 0;
  
  /*
    We are asked to fill the GPA array with allocated PxmAtomSpec
    objects corresponding to data found in the different
    polchem-defs-atom-defs-dic dictionary files that are parsed
    according to function parameter requirements.
  */

  g_assert (GPA != NULL);
  g_assert (pre_ap_pend == CONFIG_APPEND || pre_ap_pend == CONFIG_PREPEND);
  g_assert (sys_user == CONFIG_USER 
	    || sys_user == CONFIG_SYSTEM 
	    || sys_user == CONFIG_BOTH);
  

  if (sys_user == CONFIG_USER || sys_user == CONFIG_BOTH)
    {
      dictionary_dir = 
	libpolyxmass_config_get_user_polchem_defs_atom_defs_dic_dir ();

      /* 
	 We only parse the files in the directory if the directory
	 exists...
      */
      if (dictionary_dir != NULL 
	  && TRUE == g_file_test (dictionary_dir, G_FILE_TEST_IS_DIR))
	{
	  /* Let's check if there are dictionary files in the directory.
	   */
	  
	  dic_dir = g_dir_open (dictionary_dir,
				flags,
				&g_error);

	  /* And now get the files in this directory and check if 
	     they are actually dictionary files.
	  */
	  parsed_file = g_dir_read_name (dic_dir);
	  
	  while (parsed_file != NULL)
	    {
	      /* Translate the name to utf_8.
	       */
	      parsed_file_utf8 = g_filename_to_utf8 (parsed_file,
						     -1,
						     NULL,
						     NULL,
						     &g_error);
	      if (NULL == parsed_file_utf8)
		{
		  parsed_file = g_dir_read_name (dic_dir);
		  continue;
		}

	      /* Make sure the filename actually ends with the proper
		 suffix ("polchem-defs-atom-defs-dic").
	      */
	      if (FALSE == g_str_has_suffix (parsed_file_utf8,
                                             "polchem-defs-atom-defs-dic"))
		{
		  g_free (parsed_file_utf8);
		  parsed_file = g_dir_read_name (dic_dir);
		  continue;
		}
	      
	      /* We will parse the file, but to do that we have to 
		 construct an absolute path to that file.
	      */
	      absolute_filename = g_strdup_printf ("%s/%s",
						   dictionary_dir,
						   parsed_file_utf8);
	      
	      /*
		Do the parsing of the file into the GPA array.
	      */
	      count += 
		libpolyxmass_atomspec_parse_polchem_defs_atom_defs_dic_file
		(absolute_filename,
		 GPA,
		 pre_ap_pend);
	      
	      if (count == -1)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
			 _("%s@%d: failed to parse file: '%s'\n"),
			 __FILE__, __LINE__, absolute_filename);
		}
	      
	      g_free (parsed_file_utf8);
	      g_free (absolute_filename);

	      parsed_file = g_dir_read_name (dic_dir);

	      continue;
	    }
	  
	  /* We can free the dictionary_dir string, we do not need it 
	     anymore.
	  */
	  g_free (dictionary_dir);
	  
	  g_dir_close (dic_dir);
	}
      /* End of
	 if (TRUE == g_file_test (dictionary_dir, G_FILE_TEST_IS_DIR))
      */
    }
  /* End of 
     if (sys_user == CONFIG_USER || sys_user == CONFIG_BOTH)
  */
  if (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_BOTH)
    {
      dictionary_dir = 
	libpolyxmass_config_get_system_polchem_defs_atom_defs_dic_dir ();

      /* 
	 We only parse the files in the directory if the directory
	 exists...
      */
      if (dictionary_dir != NULL 
	  && TRUE == g_file_test (dictionary_dir, G_FILE_TEST_IS_DIR))
	{
	  /* Let's check if there are dictionary files in the directory.
	   */
	  
	  dic_dir = g_dir_open (dictionary_dir,
				flags,
				&g_error);

	  /* And now get the files in this directory and check if 
	     they are actually dictionary files.
	  */
	  parsed_file = g_dir_read_name (dic_dir);
	  
	  while (parsed_file != NULL)
	    {
	      /* Translate the name to utf_8.
	       */
	      parsed_file_utf8 = g_filename_to_utf8 (parsed_file,
						     -1,
						     NULL,
						     NULL,
						     &g_error);
	      if (NULL == parsed_file_utf8)
		{
		  parsed_file = g_dir_read_name (dic_dir);
		  continue;
		}

	      /* Make sure the filename actually ends with the proper
		 suffix ("polchem-defs-atom-defs-dic").
	      */
	      if (FALSE == g_str_has_suffix (parsed_file_utf8,
                                             "polchem-defs-atom-defs-dic"))
		{
		  g_free (parsed_file_utf8);
		  parsed_file = g_dir_read_name (dic_dir);
		  continue;
		}
	      
	      /* We will parse the file, but to do that we have to 
		 construct an absolute path to that file.
	      */
	      absolute_filename = g_strdup_printf ("%s/%s",
						   dictionary_dir,
						   parsed_file_utf8);
	      
	      /*
		Do the parsing of the file into the GPA array.
	      */
	      count += 
		libpolyxmass_atomspec_parse_polchem_defs_atom_defs_dic_file
		(absolute_filename,
		 GPA,
		 pre_ap_pend);
	      
	      if (count == -1)
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
			 _("%s@%d: failed to parse file: '%s'\n"),
			 __FILE__, __LINE__, absolute_filename);
		}
	      
	      g_free (parsed_file_utf8);
	      g_free (absolute_filename);

	      parsed_file = g_dir_read_name (dic_dir);

	      continue;
	    }
	  
	  /* We can free the dictionary_dir string, we do not need it 
	     anymore.
	  */
	  g_free (dictionary_dir);
	  
	  g_dir_close (dic_dir);
	}
      /* End of
	 if (TRUE == g_file_test (dictionary_dir, G_FILE_TEST_IS_DIR))
      */
    }
  /* End of 
     if (sys_user == CONFIG_SYSTEM || sys_user == CONFIG_BOTH)
  */

  return count;
}


gint
libpolyxmass_atomspec_arrays_cross_complete_dic_cat_data (GPtrArray *dicGPA,
							  GPtrArray *catGPA)
{
  gint iter = 0;
  gint jter = 0;
  gint count = 0;

  gboolean found = FALSE;

  PxmAtomSpec *atomspec_dic = NULL;
  PxmAtomSpec *atomspec_cat = NULL;

  /* 
     We get the pointer to a GPtrArray containing PxmAtomSpec
     instances that were obtained parsing the polchem-defs-atom-defs.dic
     dictionary files. This dicGPA array contains incomplete
     PxmAtomSpec objects, as these objects have no data set for the
     'file' member (telling where the atom definition file is
     located).
     
     We get the pointer to a GPtrArray containing PxmAtomSpec
     instances that were obtained parsing the atom-defs.cat catalogue
     files. This catGPA array contains incomplete PxmAtomSpec objects,
     as these objects have no data set for the poltype member (telling
     what polymer definition name (or type) should use what atom
     definition name).
     
     We want to make sure that the dicGPA PxmAtomSpec members do have
     fully characterized member data. For this, we iterate in the
     catGPA and search PxmAtomSpec instances having the same 'atomdef'
     member, and we copy from the found catGPA's PxmAtomSpec instance
     the file member to the dicGPA's PxmAtomSpec instance. 

     This way, the PxmAtomSpec instances of the dicGPA array will be
     fully characterized, and usable by other programs.

     Note that, while it is possible that catGPA contains PxmAtomSpec
     instances of which atomdef member might not be found in the
     dicGPA PxmAtomSpec instances, the converse is not possible. In
     other words, catGPA can contain atom definitions that are not in
     use, since there are no correspondences between polymer
     definition types and these atom definitions in the
     polchem-defs-atom-defs.dic files.
     
     The converse is not possible: if an atom definition is found
     correlated to a polymer chemistry definition name, then that atom
     definition HAS to be found in the catGPA, otherwise, there is an
     error.
  */
     
  g_assert (catGPA != NULL);
  g_assert (dicGPA != NULL);


  /*
    For each PxmAtomSpec instance in catGPA, try to find an instance in
    dicGPA that has the same atomdef member.
  */
  for (iter = 0; iter < dicGPA->len; iter++)
    {
      atomspec_dic = g_ptr_array_index (dicGPA, iter);
      g_assert (atomspec_dic != NULL);

      found = FALSE;

      for (jter = 0; jter < catGPA->len; jter++)
	{
	  atomspec_cat = g_ptr_array_index (catGPA, jter);
	  g_assert (atomspec_cat != NULL);

	  if (0 == strcmp (atomspec_dic->atomdef, 
			   atomspec_cat->atomdef))
	    {
	      /*
		We have found a PxmAtomSpec instance in catGPA that
		has a 'atomdef' member datum matching the 'atomdef'
		member datum from the iterated PxmAtomSpec instance in
		dicGPA. Thus, we can fully characterize the
		PxmAtomSpec instance in dicGPA by copying the 'file'
		member from atomspec_cat to atomspec_dic.

		Note that we want to first assert that this file
		actually exists on disk ! 
	      */

	      g_assert (TRUE == g_file_test (atomspec_cat->file,
					     G_FILE_TEST_EXISTS));

	      g_assert (TRUE ==
			libpolyxmass_atomspec_set_file (atomspec_dic,
							atomspec_cat->file));

	      found = TRUE;
	      count++;

	      break;
	    }

	  continue;
	}
      /* End of 
	 for (jter = 0; jter < catGPA; jter++)
      */

      if (found == FALSE)
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	       _("%s@%d: failed to initialize atom dictionary data\n"),
	       __FILE__, __LINE__);
    }
  /* End of 
     for (iter = 0; iter < dicGPA->len; iter++)
  */  

  return count;
}




























/* FREE'ING FUNCTIONS
 */
gint
libpolyxmass_atomspec_free (PxmAtomSpec *as)
{
  g_assert (as != NULL);
  
  if (as->poltype != NULL)
    g_free (as->poltype);
  
  if (as->atomdef != NULL)
    g_free (as->atomdef);
  
  if (as->file != NULL)
    g_free (as->file);
  
  g_free (as);
  
  return 1;
}




/* GPtrArray-RELATED FUNCTIONS
 */
gint
libpolyxmass_atomspec_GPA_empty (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmAtomSpec *as = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      as = g_ptr_array_remove_index (GPA, 0);
      g_assert (as != NULL);
      libpolyxmass_atomspec_free (as);
      count++;
    }
  
  return count;
}


gint
libpolyxmass_atomspec_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  

  g_assert (GPA != NULL);

  count = libpolyxmass_atomspec_GPA_empty (GPA);
    
  g_ptr_array_free (GPA, TRUE);

  return count;
}



