/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pxmchem-fragspec.h"
#include "pxmchem-monomer.h"
#include "libpolyxmass-plugin.h"
#include "pxmchem-formula.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmFragSpec *
pxmchem_fragspec_new (void)
{
  PxmFragSpec *fgs = g_malloc0 (sizeof (PxmFragSpec));

  fgs->fgrGPA = g_ptr_array_new ();

  return fgs;
}


PxmFragRule *
pxmchem_fragrule_new (void)
{
  PxmFragRule *fgr = g_malloc0 (sizeof (PxmFragRule));
  
  return fgr;
}


PxmFragSpec *
pxmchem_fragspec_new_by_name (gchar *name, GPtrArray *GPA)
{
  PxmFragSpec *fgs = NULL;
  gint iter = 0;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      fgs = g_ptr_array_index (GPA, iter);
            
      if (0 == strcmp (fgs->name, name))
	return pxmchem_fragspec_dup (fgs);
    }
  
  return NULL;
}


PxmFragRule *
pxmchem_fragrule_new_by_name (gchar *name, GPtrArray *GPA)
{
  PxmFragRule *fgr = NULL;
  gint iter = 0;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      fgr = g_ptr_array_index (GPA, iter);
            
      if (0 == strcmp (fgr->name, name))
	return pxmchem_fragrule_dup (fgr);
    }
  
  return NULL;
}


PxmFragSpec *
pxmchem_fragspec_dup (const PxmFragSpec *fgs)
{
  gint iter = 0;

  PxmFragSpec *new_fgs = NULL;
  
  PxmFragRule *fgr = NULL;
  PxmFragRule *new_fgr = NULL;
  
  g_assert (fgs != NULL);
  
  new_fgs = pxmchem_fragspec_new ();
  
  /* The template instance is assumed to be well formed,
     which is without NULL members when there must not be NULL!
  */
  g_assert (fgs->name != NULL);
  new_fgs->name = g_strdup (fgs->name);

  /* A valid frag specif must have a end member equal to
     PXM_FRAG_END_NONE, PXM_FRAG_END_LEFT, PXM_FRAG_END_RIGHT. We do
     not accept that it be set like this : (PXM_FRAG_END_LEFT |
     PXM_FRAG_END_RIGHT)  which is PXM_FRAG_END_BOTH.
   */
  g_assert (fgs->end != PXM_FRAG_END_BOTH);

  new_fgs->end = fgs->end;

  /* Beware that actform may be NULL!
   */
  if (fgs->actform != NULL)
    new_fgs->actform = g_strdup (fgs->actform);
  
  /* Beware that comment may be NULL!
   */
  if (fgs->comment != NULL)
    new_fgs->comment = g_strdup (fgs->comment);
  
  /* A valid frag specif must have an array of frag rules even
   * if it is empty.
   */
  g_assert (fgs->fgrGPA != NULL);
  
  for (iter = 0; iter < fgs->fgrGPA->len; iter++)
    {
      fgr = g_ptr_array_index (fgs->fgrGPA, iter);
      
      new_fgr = pxmchem_fragrule_dup (fgr);
      
      g_ptr_array_add (new_fgs->fgrGPA, new_fgr);
    }
  
  return new_fgs;
}


PxmFragRule *
pxmchem_fragrule_dup (const PxmFragRule *fgr)
{
  PxmFragRule *new_fgr = NULL;
  
  g_assert (fgr != NULL);
  
  new_fgr = pxmchem_fragrule_new ();
  
  /* The template instance is assumed to be well formed,
     which is without NULL members when there must not be NULL!
  */
  g_assert (fgr->name != NULL);
  new_fgr->name = g_strdup (fgr->name);

  /* Beware that prev may be NULL!
   */
  if (fgr->prev != NULL)
    new_fgr->prev = g_strdup (fgr->prev);

  /* Beware that this may be NULL!
   */
  if (fgr->this != NULL)
    new_fgr->this = g_strdup (fgr->this);

  /* Beware that next may be NULL!
   */
  if (fgr->next != NULL)
    new_fgr->next = g_strdup (fgr->next);
  
  g_assert (fgr->actform != NULL);
  new_fgr->actform = g_strdup (fgr->actform);
  
  /* Beware that comment may be NULL!
   */
  if (fgr->comment != NULL)
    new_fgr->comment = g_strdup (fgr->comment);
  
  return new_fgr;
}


gboolean
pxmchem_fragspec_set_name (PxmFragSpec *fgs, gchar *name)
{
  g_assert (fgs != NULL && name != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgs->name != NULL)
    g_free (fgs->name);
  
  fgs->name = g_strdup (name);
  
  return TRUE;
}

gboolean
pxmchem_fragspec_set_end (PxmFragSpec *fgs, PxmFragEnd end)
{
  g_assert (fgs != NULL);
  g_assert (end != PXM_FRAG_END_BOTH);
  
  fgs->end = end;
  
  return TRUE;
}

gboolean
pxmchem_fragspec_set_actform (PxmFragSpec *fgs, gchar *actform)
{
  g_assert (fgs != NULL && actform != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgs->actform != NULL)
    g_free (fgs->actform);
  
  fgs->actform = g_strdup (actform);  

  return TRUE;
}

gboolean
pxmchem_fragspec_set_comment (PxmFragSpec *fgs, gchar *comment)
{
  g_assert (fgs != NULL && comment != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgs->comment != NULL)
    g_free (fgs->comment);
  
  fgs->comment = g_strdup (comment);  

  return TRUE;
}


gboolean
pxmchem_fragrule_set_name (PxmFragRule *fgr, gchar *name)
{
  g_assert (fgr != NULL && name != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgr->name != NULL)
    g_free (fgr->name);
  
  fgr->name = g_strdup (name);   

  return TRUE;
}


gboolean
pxmchem_fragrule_set_prev (PxmFragRule *fgr, gchar *prev)
{
  g_assert (fgr != NULL && prev != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgr->prev != NULL)
    g_free (fgr->prev);
  
  fgr->prev = g_strdup (prev);  

  return TRUE;
}


gboolean
pxmchem_fragrule_set_this (PxmFragRule *fgr, gchar *this)
{
  g_assert (fgr != NULL && this != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgr->this != NULL)
    g_free (fgr->this);
  
  fgr->this = g_strdup (this);  

  return TRUE;
}


gboolean
pxmchem_fragrule_set_next (PxmFragRule *fgr, gchar *next)
{
  g_assert (fgr != NULL && next != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgr->next != NULL)
    g_free (fgr->next);
  
  fgr->next = g_strdup (next);  

  return TRUE;
}


gboolean
pxmchem_fragrule_set_actform (PxmFragRule *fgr, gchar *actform)
{
  g_assert (fgr != NULL && actform != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgr->actform != NULL)
    g_free (fgr->actform);
  
  fgr->actform = g_strdup (actform);  

  return TRUE;
}


gboolean
pxmchem_fragrule_set_comment (PxmFragRule *fgr, gchar *comment)
{
  g_assert (fgr != NULL && comment != NULL);
  
  /* The member data may be NULL, as this function can be called right
     after pxmchem_fragspec_new () which leaves the members NULL
     (except the propGPA which is allocated).
  */
  if (fgr->comment != NULL)
    g_free (fgr->comment);
  
  fgr->comment = g_strdup (comment);  

  return TRUE;
}


/* INTEGRITY CHECKING FUNCTIONS
 */
gboolean
pxmchem_fragspec_validate (PxmFragSpec *fragspec, PxmPolchemdef *polchemdef,
			   gchar **valid)
{
  GString *gs = NULL;
  
  gchar *help = NULL;

  gint iter = 0;

  PxmFragRule *fragrule = NULL;
  
  
  g_assert (fragspec != NULL);
  g_assert (polchemdef != NULL);

  

  /* Note that for integrity reasons, *valid MUST BE NULL to ensure 
   * that it is empty.
   */
  g_assert (valid != NULL);
  g_assert (*valid == NULL);


  gs = g_string_new ("");
  

  /* This is the DTD for the fragspec node.
   * <!ELEMENT fgs (name,end,actform?,comment?,fgr*)>
   */

  /* The name: it must be non-NULL and be longer than 0 chars.
   */
  if (fragspec->name == NULL)
    {
      g_string_append_printf (gs, 
			      _("fragspec has a NULL 'name' member\n"));
    }
  else
    {
      /* Make a copy of the string, so that we can strip it of its
       * spaces and next check its "real" length.
       */
      help = g_strdup (fragspec->name);
      help = g_strstrip (help);
      
      if (strlen (help) <= 0)
	{
	  g_string_append_printf (gs, 
				  _("fragspec has an invalid 'name' member:"
				    " '%s'\n"), fragspec->name);
	}

      g_free (help);
    }
  
  /* The end: it must be one of these: PXM_FRAG_END_NONE |
     PXM_FRAG_END_LEFT | PXM_FRAG_END_RIGHT but cannot be both
     (PXM_FRAG_END_LEFT | PXM_FRAG_END_RIGHT) which is
     PXM_FRAG_END_BOTH.
   */
  if (PXM_FRAG_END_BOTH == fragspec->end)
    g_string_append_printf (gs, 
			    _("fragspec has not a valid 'end' member: '%d'\n"),
			    fragspec->end);

  /* The actform: it cannot be NULL. And must be a chemically valid
    actform.
   */
  if (fragspec->actform == NULL)
    {
      g_string_append_printf (gs, 
			      _("fragspec has a NULL 'actform' member\n"));
    }
  else
    {
      /* The actform is set, check that it is valid.
       */
      /* We do not need to strip the string of its spaces, because
       * the actform checking function will do this itself.
       */
      if (FALSE == pxmchem_actform_check (fragspec->actform, 
					  polchemdef->atomGPA))
	{
	  g_string_append_printf (gs, 
				  _("fragspec has an invalid 'actform' member:"
				    " '%s'\n"), fragspec->actform);
	}
    }
  
  /* The comment: it can be NULL.
   */

  /* The array of fragrules must not be NULL, however fragrules are
   * optional.
   */
  if (fragspec->fgrGPA == NULL)
    {
      g_string_append_printf (gs, 
			      _("fragspec has a NULL array of fragrules\n"));
    }
  
  /* If there are fragrules, we check them individually.
   */
  /* The help pointer must be NULL for the function 
   * pxmchem_fragrule_validate () call below.
   */
  help = NULL;
  
  for (iter = 0; iter < fragspec->fgrGPA->len; iter++)
    {
      fragrule = g_ptr_array_index (fragspec->fgrGPA, iter);
      g_assert (fragrule != NULL);
      
      if (FALSE == pxmchem_fragrule_unique_by_name (fragrule, 
						    fragspec->fgrGPA))
	{
	  g_string_append_printf (gs,
				  _("fragrule at index: '%d'"
				    " is not unique: '%s'\n"),
				  iter, fragrule->name);
	}      

      if (FALSE == 
	  pxmchem_fragrule_validate (fragrule, polchemdef, &help))
	{
	  g_assert (help != NULL);
	  
	  g_string_append_printf (gs, 
				  _("fragspec has an invalid fragrule:\n%s"),
				  help);
	  g_free (help);
	  help = NULL;
	}
    }
  
  /* Finally the validation is finished.
   */
  if (strlen (gs->str) > 0)
    {
      /* String is not empty, which is there were errors.
       */
      *valid = gs->str;
      
      g_string_free (gs, FALSE);
      
      return FALSE;
    }

  g_string_free (gs, TRUE);
  
  return TRUE;
  
}


gboolean
pxmchem_fragspec_unique_by_name (PxmFragSpec *fragspec, 
				 GPtrArray *GPA)
{
  g_assert (fragspec != NULL);
  g_assert (fragspec->name != NULL);
  g_assert (GPA != NULL);
  
  return (pxmchem_fragspec_get_index_top_by_name (fragspec->name, GPA)
	  == 
	  pxmchem_fragspec_get_index_bottom_by_name (fragspec->name, GPA));
}


gboolean
pxmchem_fragrule_validate (PxmFragRule *fragrule, PxmPolchemdef *polchemdef,
			   gchar **valid)
{
  GString *gs = NULL;
  
  gchar *help = NULL;

  
  g_assert (fragrule != NULL);
  g_assert (polchemdef != NULL);

  

  /* Note that for integrity reasons, *valid MUST BE NULL to ensure 
   * that it is empty.
   */
  g_assert (valid != NULL);
  g_assert (*valid == NULL);


  gs = g_string_new ("");
  

  /* This is the DTD for the fragrule node.
   *
   *  <!ELEMENT fgr (name,actform,
   *		 prev-mnm-code?,
   *		 this-mnm-code?,
   *		 next-mnm-code?,
   *		 comment?)>
   */

  /* The name: it must be non-NULL and be longer than 0 chars.
   */
  if (fragrule->name == NULL)
    {
      g_string_append_printf (gs, 
			      _("fragrule has a NULL 'name' member"));
    }
  else
    {
      /* Make a copy of the string, so that we can strip it of its
       * spaces and next check its "real" length.
       */
      help = g_strdup (fragrule->name);
      help = g_strstrip (help);
      
      if (strlen (help) <= 0)
	{
	  g_string_append_printf (gs, 
				  _("fragrule has not a valid 'name' member"));
	}
      
      g_free (help);
    }
  
  /* The actform: it must be non-NULL and be chemically valid.
   */
  if (fragrule->actform == NULL)
    {
      g_string_append_printf (gs, 
			      _("fragrule has a NULL 'actform' member\n"));
    }
  else
    {
      /* We do not need to strip the string of its spaces, because
       * the actform checking function will do this itself.
       */
      if (FALSE == pxmchem_actform_check (fragrule->actform,
					  polchemdef->atomGPA))
	{
	  g_string_append_printf (gs, 
				  _("fragrule has an invalid 'actform' member:"
				    " '%s'\n"), fragrule->actform);
	}
    }
  
  /* The 'prev' code: it can be NULL, but -if set- must be a valid
     monomer code (length- and syntax-wise) and must already exists as
     a valid monomer in the polymer definition.
  */
  if (fragrule->prev != NULL && strlen (fragrule->prev) > 0)
    {
      if (-1 == pxmchem_monomer_get_index_by_code (fragrule->prev,
						   polchemdef->monomerGPA))
	g_string_append_printf (gs, 
				_("fragrule has an invalid 'prev' member:"
				  " '%s'\n"), fragrule->prev);
    }
  else
    {
      /* Item is optional, it is just not set here.
       */
    }
  
  /* The 'this' code: it can be NULL, but -if set- must be a valid
   * monomer code (length- and syntax-wise) and must already exists as
   * a valid monomer in the polymer definition.
   */
  if (fragrule->this != NULL && strlen (fragrule->this) > 0)
    {
      if (-1 == pxmchem_monomer_get_index_by_code (fragrule->this, 
						   polchemdef->monomerGPA))
	g_string_append_printf (gs, 
				_("fragrule has an invalid 'this' member:"
				  " '%s'\n"), fragrule->this);
    }
  else
    {
      /* Item is optional, it is just not set here.
       */
    }
  
  /* The 'next' code: it can be NULL, but -if set- must be a valid
   * monomer code (length- and syntax-wise) and must already exists as
   * a valid monomer in the polymer definition.
   */
  if (fragrule->next != NULL && strlen (fragrule->next) > 0)
    {
      if (-1 == pxmchem_monomer_get_index_by_code (fragrule->next,
						   polchemdef->monomerGPA))
	g_string_append_printf (gs, 
				_("fragrule has an invalid 'next' member:"
				  " '%s'\n"), fragrule->next);
    }
  else
    {
      /* Item is optional, it is just not set here.
       */
    }
  
  /* The comment: it can be NULL.
   */
  
  
  /* Finally the validation is finished.
   */
  if (strlen (gs->str) > 0)
    {
      /* String is not empty, which is there were errors.
       */
      *valid = gs->str;
      
      g_string_free (gs, FALSE);
      
      return FALSE;
    }

  g_string_free (gs, TRUE);
  
  return TRUE;
}


gboolean
pxmchem_fragrule_unique_by_name (PxmFragRule *fragrule, 
				 GPtrArray *GPA)
{
  g_assert (fragrule != NULL);
  g_assert (fragrule->name != NULL);
  g_assert (GPA != NULL);
  
  return (pxmchem_fragrule_get_index_top_by_name (fragrule->name, GPA)
	  == 
	  pxmchem_fragrule_get_index_bottom_by_name (fragrule->name, GPA));
}



/*  LOCATING FUNCTIONS
 */
gint
pxmchem_fragspec_get_index_by_name (gchar *name, GPtrArray *GPA)
{
  return pxmchem_fragspec_get_index_top_by_name (name, GPA);
}


gint
pxmchem_fragspec_get_index_top_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = 0;
  PxmFragSpec *fgs = NULL;
  

  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      fgs = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (fgs->name, name))
	return iter;
    }
  
  return -1;
}


gint
pxmchem_fragspec_get_index_bottom_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = 0;
  PxmFragSpec *fgs = NULL;
  

  g_assert (GPA != NULL);
  
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  fgs = g_ptr_array_index (GPA, iter);
	  
	  if (0 == strcmp (fgs->name, name))
	    return iter;
	}
    }
  
  return -1;
}


gint
pxmchem_fragrule_get_index_by_name (gchar *name, GPtrArray *GPA)
{
  return pxmchem_fragrule_get_index_top_by_name (name, GPA);
}


gint
pxmchem_fragrule_get_index_top_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = 0;
  PxmFragRule *fgr = NULL;
  

  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      fgr = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (fgr->name, name))
	return iter;
    }
  
  return -1;
}


gint
pxmchem_fragrule_get_index_bottom_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = 0;
  PxmFragRule *fgr = NULL;
  

  g_assert (GPA != NULL);
  
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  fgr = g_ptr_array_index (GPA, iter);
	  
	  if (0 == strcmp (fgr->name, name))
	    return iter;
	}
    }
  
  return -1;
}


gint
pxmchem_fragspec_get_index_by_ptr (GPtrArray *GPA, 
				   PxmFragSpec *fgs)
{
  gint iter = 0;
  

  g_assert (GPA != NULL && fgs != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    if ((PxmFragSpec *) g_ptr_array_index (GPA, iter) == fgs)
      return iter;
      
  return -1;
}


gint
pxmchem_fragrule_get_index_by_ptr (GPtrArray *GPA, 
				   PxmFragRule *fgr)
{
  gint iter = 0;
  

  g_assert (GPA != NULL && fgr != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    if ((PxmFragRule *) g_ptr_array_index (GPA, iter) == fgr)
      return iter;
      
  return -1;
}


PxmFragSpec *
pxmchem_fragspec_get_ptr_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = -1;
  PxmFragSpec *fgs = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      fgs = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (fgs->name, name))
	return fgs;
    }
  
  return NULL;
}


PxmFragRule *
pxmchem_fragrule_get_ptr_by_name (gchar *name, GPtrArray *GPA)
{
  gint iter = -1;
  PxmFragRule *fgr = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      fgr = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (fgr->name, name))
	return fgr;
    }
  
  return NULL;
}



/* UTILITY FUNCTIONS
 */



/* XML-format TRANSACTIONS
 */
gchar *
pxmchem_fragspec_format_xml_string_fgs (PxmFragSpec *fgs, 
					gchar *indent, gint offset)
{
  /* The pointer to the fragspec will allow to create a string 
   * representing its member data.
   */
  gint iter = 0;
  gint new_offset = 0;
  
  gchar *lead = NULL;
  gchar *help = NULL;
  
  GString *gs = NULL;

  PxmFragRule *fgr = NULL ;
  
  

  g_assert (fgs != NULL && indent != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create a <fgs> node that should look like this:
   *
   * <fgs>
   *   <name>a</name>
   *   <end>LE</end>
   *   <actform>-C1O1</actform>
   *   <comment>opt_comment</comment>
   *   <fgr>
   *     <name>one_rule</name>
   *     <actform>+H2O</actform>
   *     <prev-mnm-code>M</prev-mnm-code>
   *     <this-mnm-code>Y</this-mnm-code>
   *     <next-mnm-code>T</next-mnm-code>
   *     <comment>opt_comment</comment>
   *   </fgr>
   *   other fgr allowed, none possible also
   * </fgs>
   *   
   */

  /* Open the <fgs> element and immediately insert the data.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);

  g_string_append_printf (gs, "%s<fgs>\n", lead);

  g_free (lead);

  new_offset = offset + 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  g_assert (fgs->name != NULL && strlen (fgs->name) > 0);
  g_string_append_printf (gs, "%s<name>%s</name>\n",
			  lead, fgs->name);
  
  /* The end member, now. Prohibited value:
     (PXM_FRAG_END_LEFT|PXM_FRAG_END_RIGHT), which is equal to 
     PXM_FRAG_END_BOTH.
   */
  g_assert (fgs->end != PXM_FRAG_END_BOTH);
  
  if (0 != (fgs->end & PXM_FRAG_END_NONE))
    g_string_append_printf (gs, "%s<end>%s</end>\n", lead, "NE");
  else if (0 != (fgs->end & PXM_FRAG_END_LEFT))
    g_string_append_printf (gs, "%s<end>%s</end>\n", lead, "LE");
  else if (0 != (fgs->end & PXM_FRAG_END_RIGHT))
     g_string_append_printf (gs, "%s<end>%s</end>\n", lead, "RE");
    

  /* The actform, now.
   */
  g_assert (fgs->actform != NULL && strlen (fgs->actform) > 0);
  g_string_append_printf (gs, "%s<actform>%s</actform>\n",
			  lead, fgs->actform);

  
  /* We now should deal with the GPtrArray of fragrule instances,
   * if any.
   */
  g_assert (fgs->fgrGPA != NULL);
  
  for (iter = 0; iter < fgs->fgrGPA->len; iter++)
    {
      fgr = g_ptr_array_index (fgs->fgrGPA, iter);

      help = pxmchem_fragrule_format_xml_string_fgr (fgr,
						     indent,
						     new_offset);
      if (help != NULL)
	{
	  g_string_append_printf (gs, "%s", help);
	  
	  g_free (help);
	}
      else
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	       _("%s@%d: failed formatting xml string for one fragrule"
	       " of fragspec\n"), __FILE__, __LINE__);
    }


  /* Beware that the comment may be NULL or empty.
   */
  if (fgs->comment != NULL && strlen (fgs->comment) > 0)
    g_string_append_printf (gs, "%s<comment>%s</comment>\n",
			    lead, fgs->comment);
  /* else
     just do not put the xml node in the file.
  */

  g_free (lead);
  
  /* Finally close the fgs element.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);
  
  g_string_append_printf (gs, "%s</fgs>\n", lead);

  g_free (lead);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


gchar *
pxmchem_fragrule_format_xml_string_fgr (PxmFragRule *fgr, 
					gchar *indent, gint offset)
{
  /* The pointer to the fragrule will allow to create a string 
   * representing its member data.
   */
  gint new_offset = 0;
  
  gchar *lead = NULL;
  gchar *help = NULL;
  
  GString *gs = NULL;
  
  

  g_assert (fgr != NULL && indent != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create a <fgs> node that should look like this:
   *
   * <fgr>
   *   <name>one_rule</name>
   *   <actform>+H2O</actform>
   *   <prev-mnm-code>M</prev-mnm-code>
   *   <this-mnm-code>Y</this-mnm-code>
   *   <next-mnm-code>T</next-mnm-code>
   *   <comment>opt_comment</comment>
   * </fgr>
   *   
   */

  /* Open the <fgr> element and immediately insert the data.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);

  g_string_append_printf (gs, "%s<fgr>\n", lead);

  g_free (lead);

  new_offset = offset + 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  g_assert (fgr->name != NULL);
  g_string_append_printf (gs, "%s<name>%s</name>\n",
			  lead, fgr->name);
  
  /* The actform is compulsory, otherwise the frag rule would
   * really serve no purpose!
   */
  g_assert (fgr->actform != NULL && strlen (fgr->actform) > 0);
  g_string_append_printf (gs, "%s<actform>%s</actform>\n",
			  lead, fgr->actform);

  /* The prev, this, next member may be NULL. If they are NULL or
     empty, just do not put the corresponding xml element in the file.
     Note that this is different than the case where an element is
     NULL/empty but for which it is a significant info (for example
     the case where the fragpsec->actform is NULL/empty). In the
     latter case, the xml node has to be put, because no actform is
     chemically relevant.
  */

  /* Beware that prev may be NULL/empty.
   */
  if (fgr->prev != NULL && strlen (fgr->prev) > 0)
    g_string_append_printf (gs, "%s<prev-mnm-code>%s</prev-mnm-code>\n",
			    lead, fgr->prev);
  
  /* Beware that this may be NULL/empty.
   */
  if (fgr->this != NULL && strlen (fgr->this) > 0)
    g_string_append_printf (gs, "%s<this-mnm-code>%s</this-mnm-code>\n",
			    lead, fgr->this);
  
  /* Beware that next may be NULL/empty.
   */
  if (fgr->next != NULL && strlen (fgr->next) > 0)
    g_string_append_printf (gs, "%s<next-mnm-code>%s</next-mnm-code>\n",
			    lead, fgr->next);

  /* Beware that comment may be NULL/empty.
   */
  if (fgr->comment != NULL && strlen (fgr->comment) > 0)
    g_string_append_printf (gs, "%s<comment>%s</comment>\n",
			    lead, fgr->comment);
  
  g_free (lead);
  
  /* Finally close the fgs element.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);
  
  g_string_append_printf (gs, "%s</fgr>\n", lead);

  g_free (lead);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}


PxmFragSpec *
pxmchem_fragspec_render_xml_node_fgs (xmlDocPtr xml_doc,
				      xmlNodePtr xml_node,
				      gpointer user_data)
{
  /* The xml node we are in is structured this way:
   *
   * <fgs>
   *   <name>a</name>
   *   <end>LE</end>
   *   <actform>-C1O1</actform>
   *   <comment>opt_comment</comment>
   *   <fgr>
   *     <name>one_rule</name>
   *     <actform>+H2O</actform>
   *     <prev-mnm-code>M</prev-mnm-code>
   *     <this-mnm-code>Y</this-mnm-code>
   *     <next-mnm-code>T</next-mnm-code>
   *     <comment>opt_comment</comment>
   *   </fgr>
   *   other fgr allowed, none possible also
   * </fgs>
   *
   * And the xml_node parameter points to the 
   *
   * <fgs> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "fgs" and that
   * we'll have to go one step down to the first child of the 
   * current node in order to get to the <name> element.
   *
   */
  PxmFragSpec *fgs = NULL;
  PxmFragRule *fgr = NULL;

  xmlNodePtr xml_child_node= NULL;
    
  gboolean comment_set = FALSE;
  
  gchar *help = NULL;
    
  /* Make sure we have parameters pointing bona fide to the right
   * xml element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == strcmp ((gchar *) xml_node->name, "fgs"));
  
  /* Now go to the first child of current node: <name>.
   */
  xml_node = xml_node->children;

  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <name> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "name"));

  fgs = pxmchem_fragspec_new ();
  
  /* Since we have allocated the fragspec instance at the line above,
     we know that its member data are NULL, so we can make direct
     assignements, without recoursing to the _set_xxx ().
  */
  fgs->name = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (fgs->name != NULL);
  
  /* Now go to the next sibling of current node: <end>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <end> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "end"));

  help = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (help != NULL);
  
  if (0 == strcmp ("NE", help))
    fgs->end = PXM_FRAG_END_NONE;
  else if (0 == strcmp ("LE", help))
    fgs->end = PXM_FRAG_END_LEFT;
  else if (0 == strcmp ("RE", help))
    fgs->end = PXM_FRAG_END_RIGHT;
  else
    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_ERROR,
	   _("%s@%d: fragspec has an invalid 'end' member: '%s'\n"),
	   __FILE__, __LINE__, help);

  g_free (help);
  
  /* Now go to the next sibling of current node: <actform>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <actformn> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "actform"));

  fgs->actform = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (fgs->actform != NULL);


  /* Since the following items are not obligatory, we have to 
   * while () until we have no more items...
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  while (xml_node != NULL)
    {
      if (0 == strcmp ((gchar *) xml_node->name, "comment"))
	{
	  if (comment_set == TRUE)
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		   _("%s@%d: two xml comment elements not allowed\n"),
		   __FILE__, __LINE__);
	  else
	    {
	      fgs->comment = 
		(gchar *) xmlNodeListGetString (xml_doc, 
						xml_node->xmlChildrenNode, 1);
	      g_assert (fgs->comment != NULL);

	      comment_set = TRUE;
	    }
	}
      else if (0 == strcmp ((gchar *) xml_node->name, "fgr"))
	{
	  xml_child_node = xml_node;
	  
	  fgr = pxmchem_fragrule_render_xml_node_fgr (xml_doc,
						      xml_child_node,
						      NULL);
	  if (fgr == NULL)
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		   _("%s@%d: failed to render fragrule from xml node\n"),
		   __FILE__, __LINE__);
	  else
	    g_ptr_array_add (fgs->fgrGPA, fgr);
      	}
      
      xml_node = xml_node->next;

      /* From a rigorous XML parsing point of view, the blanks found in
       * the XML document are considered to be nodes, and we have to detect
       * these and take proper action: go next sibling (next blank) as long
       * as blanks are encountered.
       */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  /* At this point we should have rendered at best the xml
   * <fgs> node into a fully qualified fragspec object.
   */

  return fgs;
}


PxmFragRule *
pxmchem_fragrule_render_xml_node_fgr (xmlDocPtr xml_doc,
				      xmlNodePtr xml_node,
				      gpointer user_data)
{
  /* The xml node we are in is structured this way:
   *
   * <fgr>
   *   <name>one_rule</name>
   *   <actform>+H2O</actform>
   *   <prev-mnm-code>M</prev-mnm-code>
   *   <this-mnm-code>Y</this-mnm-code>
   *   <next-mnm-code>T</next-mnm-code>
   *   <comment>opt_comment</comment>
   * </fgr>
   *   
   * And the xml_node parameter points to the 
   *
   * <fgr> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "fgr" and that
   * we'll have to go one step down to the first child of the 
   * current node in order to get to the <name> element.
   *
   */
  PxmFragRule *fgr = NULL;

  gboolean comment_set = FALSE;
  gboolean this_set = FALSE;
  gboolean prev_set = FALSE;
  gboolean next_set = FALSE;


  /* Make sure we have parameters pointing bona fide to the right
   * xml element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == strcmp ((gchar *) xml_node->name, "fgr"));
  
  /* Now go to the first child of current node: <name>.
   */
  xml_node = xml_node->children;

  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <name> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "name"));

  fgr = pxmchem_fragrule_new ();
  
  /* Since we have allocated the fragrule instance at the line above,
     we know that its member data are NULL, so we can make direct
     assignements, without recoursing to the _set_xxx ().
  */
  fgr->name = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (fgr->name != NULL);
  
  /* Now go to the next sibling of current node: <actform>.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <actform> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "actform"));

  fgr->actform = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (fgr->actform != NULL);

  /* Since the following items are not obligatory, we have to 
   * while () until we have no more items...
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  while (xml_node != NULL)
    {
      if (0 == strcmp ((gchar *) xml_node->name, "prev-mnm-code"))
	{
	  if (prev_set == TRUE)
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		   _("%s@%d: two xml prev elements not allowed\n"),
		   __FILE__, __LINE__);
	  else
	    {
	      fgr->prev = 
		(gchar *) xmlNodeListGetString (xml_doc, 
				      xml_node->xmlChildrenNode, 1);

	      g_assert (fgr->prev != NULL);

	      prev_set = TRUE;
	    }
	}
      else if (0 == strcmp ((gchar *) xml_node->name, "this-mnm-code"))
	{
	  if (this_set == TRUE)
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		   _("%s@%d: two xml this elements not allowed\n"),
		   __FILE__, __LINE__);
	  else
	    {
	      fgr->this = 
		(gchar *) xmlNodeListGetString (xml_doc, 
				      xml_node->xmlChildrenNode, 1);

	      g_assert (fgr->this != NULL);
	      
	      this_set = TRUE;
	    }
	}
      else if (0 == strcmp ((gchar *) xml_node->name, "next-mnm-code"))
	{
	  if (next_set == TRUE)
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		   _("%s@%d: two xml 'next' elements not allowed\n"),
		   __FILE__, __LINE__);
	  else
	    {
	      fgr->next = 
		(gchar *) xmlNodeListGetString (xml_doc, 
				      xml_node->xmlChildrenNode, 1);
	      
	      g_assert (fgr->next != NULL);
	      
	      next_set = TRUE;
	    }
	}
      else if (0 == strcmp ((gchar *) xml_node->name, "comment"))
	{
	  if (comment_set == TRUE)
	    g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		   _("%s@%d: two xml comment elements not allowed\n"),
		   __FILE__, __LINE__);
	  else
	    {
	      fgr->comment = 
		(gchar *) xmlNodeListGetString (xml_doc, 
				      xml_node->xmlChildrenNode, 1);
	      
	      g_assert (fgr->comment != NULL);
	      
	      comment_set = TRUE;
	    }
	}
      
      xml_node = xml_node->next;

      /* From a rigorous XML parsing point of view, the blanks found in
       * the XML document are considered to be nodes, and we have to detect
       * these and take proper action: go next sibling (next blank) as long
       * as blanks are encountered.
       */
      while (TRUE == xmlIsBlankNode (xml_node))
	xml_node = xml_node->next;
    }
  
  /* At this point we should have rendered at best the xml
   * <fgr> node into a fully qualified fragrule object.
   */

  return fgr;
}




/* FREE'ING FUNCTIONS
 */
gboolean
pxmchem_fragspec_free (PxmFragSpec *fgs)
{
  g_assert (fgs != NULL);
  
  if (fgs->name != NULL)
    g_free (fgs->name);
  
  if (fgs->actform != NULL)
    g_free (fgs->actform);
  
  if (fgs->comment != NULL)
    g_free (fgs->comment);
  
  if (fgs->fgrGPA != NULL)
    pxmchem_fragrule_GPA_free (fgs->fgrGPA);
  
  g_free (fgs);
  
  return TRUE;
}
    

gboolean
pxmchem_fragrule_free (PxmFragRule *fgr)
{
  g_assert (fgr != NULL);
  
  if (fgr->name != NULL)
    g_free (fgr->name);
  
  if (fgr->prev != NULL)
    g_free (fgr->prev);
  
  if (fgr->this != NULL)
    g_free (fgr->this);
  
  if (fgr->next != NULL)
    g_free (fgr->next);
  
  if (fgr->actform != NULL)
    g_free (fgr->actform);
  
  if  (fgr->comment != NULL)
    g_free (fgr->comment);
  
  g_free (fgr);
  
  return TRUE;
}




/* GPtrArray-RELATED FUNCTIONS
 */
gint
pxmchem_fragspec_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmFragSpec *fgs = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      fgs = g_ptr_array_remove_index (GPA, 0);
      g_assert (fgs != NULL);
      pxmchem_fragspec_free (fgs);
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}


gint
pxmchem_fragrule_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmFragRule *fgr = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      fgr = g_ptr_array_remove_index (GPA, 0);
      g_assert (fgr != NULL);
      pxmchem_fragrule_free (fgr);
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}









