/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxedit-ui-elemcompos.h"
#include "polyxmass-ui-seqed-widget.h"
#include "polyxedit-ui-seqed-wnd.h"


GtkWidget *
polyxedit_elemcompos_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *seqed_widget = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;
  
  GPtrArray *acGPA = NULL;
  
  
  g_assert (editctxt != NULL);
  

  seqed_widget = 
    g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd), "seqed_widget");
  g_assert (seqed_widget != NULL);
  


  gui_file = 
    g_strdup_printf ("%s/polyxedit-compositions.glade", userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "elemental_composition_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "elemental_composition_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the elemental composition window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);
  


  /* The vertical box where the polymer sequence data widgets are packed, 
     so that we make them invisible in one go.
  */
  widget = glade_xml_get_widget (xml, "polseq_prop_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_prop_vbox", widget);
  

  /* Set the name of the polymer sequence to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_name_entry", widget);

  g_assert (editctxt->polymer->plminfo->name != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->name);

  /* Set the code of the polymer sequence to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_code_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_code_entry", widget);

  g_assert (editctxt->polymer->plminfo->code != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->code);


  /* Set the identity number (actually the seqed_widget pointer) to 
   * its GtkEntry (this is useful when cleavages are made to identify
   * unambiguously the polymer from which results are displayed).
   */
  widget = glade_xml_get_widget (xml, "polseq_id_number_entry");
  g_object_set_data (G_OBJECT (window), 
		     "polseq_id_number_entry",
		     widget);
  help = g_strdup_printf ("%p", seqed_widget);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);




  
  /* Start/End selection values.
   */
  widget = glade_xml_get_widget (xml, "sel_start_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "sel_start_pos_entry",
		     widget);

  widget = glade_xml_get_widget (xml, "sel_end_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "sel_end_pos_entry",
		     widget);
  
  /* The checkbutton that says if the whole sequence should be considered
     or the selection only.
  */
  widget = glade_xml_get_widget (xml, "whole_sequence_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "whole_sequence_checkbutton",
		     widget);

  /* We have a function that can set up the selection region values into 
     their respective entries:
  */
  polyxedit_elemcompos_wnd_update_sequence_region_button (NULL, window);

  /* The checkbutton that says if we want the calculation to be
     incremental or not.
  */
  widget = glade_xml_get_widget (xml, "incremental_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "incremental_checkbutton",
		     widget);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), FALSE);

  widget = glade_xml_get_widget (xml, "elemental_composition_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "elemental_composition_entry",
		     widget);

  /* The update sequence region button.
   */
  widget = glade_xml_get_widget (xml, "update_sequence_region_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "update_sequence_region_button",
		     widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxedit_elemcompos_wnd_update_sequence_region_button), 
     window);

  /* The execute button.
   */
  widget = glade_xml_get_widget (xml, "elemental_composition_execute_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "elemental_composition_execute_button",
		     widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_elemcompos_wnd_execute_button), 
		    window);


  /* We now have to allocate the GPtrArray of atomcount instances
     through which the elemental composition will be calculated. We
     want this GPtrArray to be the propriety of the whole window, and
     not to be localized in the execute_button handler because this
     way it is accessible and remains available if the
     "incremental_checkbutton" widget is active.
  */
  acGPA = g_ptr_array_new ();
  
  /* We know that this array will be populated with PxmAtomcount objects,
     so we know how to free this array when the window is closed.
  */
  g_object_set_data_full (G_OBJECT (window), "acGPA", acGPA,
			  (GDestroyNotify) pxmchem_atomcount_GPA_free);
  
  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_elemcompos_wnd_delete_event), 
		    editctxt);
  
  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_elemcompos_wnd_destroy_event), 
		    editctxt);


  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one composition results' window opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("elemcompos_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_elemcompos_wnd_really_close);
  
  g_free (help);
  
  return window;
}


void
polyxedit_elemcompos_wnd_execute_button (GtkWidget *button,
					  gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;
  GtkWidget *messages_entry = NULL;
  
  gint len = 0;
  
  GPtrArray *acGPA = NULL;
  
  PxmEditCtxt *editctxt = NULL;

  PxmCompcalcRes res = PXM_COMPCALC_FAILURE;
  
  PxmCalcOpt *calcopt = NULL;
  PxmIonizerule *ionizerule = NULL;
      
  gchar *help = NULL;
  gchar *formula = NULL;
  

  
  g_assert (window != NULL);
  
  
  /* First of all get the array of PxmAtomcount instances.
   */
  acGPA = g_object_get_data (G_OBJECT (window), "acGPA");
  g_assert (acGPA != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  g_assert (editctxt->polymer != NULL);
  g_assert (editctxt->polymer->monomerGPA != NULL);
  len = editctxt->polymer->monomerGPA->len;
    
  /* Make a copy of the calcopt/ionizerule instances that belongs to
     the editctxt.
  */
  calcopt = pxmchem_calcopt_dup (editctxt->calcopt);
  ionizerule = pxmchem_ionizerule_dup (editctxt->ionizerule);

  /* We almost certainly will display messages.
   */
  messages_entry = g_object_get_data (G_OBJECT (window), 
				      "messages_entry");
  
  /* We must check if the user wants that the calculation be made for
     the whole sequence or for the selection region of the polymer sequence
     only, or for the interval that is defined with the start_pos and 
     end_pos entry widgets.
  */
  widget = 
    g_object_get_data (G_OBJECT (window), "whole_sequence_checkbutton");
  g_assert (widget != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    {
      calcopt->start_idx = 0;
      calcopt->end_idx = len;
    }
  else
    {
      /* We should check what the start_pos and end_pos entries say:
       */
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);
      
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);
      
      if (strlen (help) <= 0)
	{
	  calcopt->start_idx = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == 
	      libpolyxmass_globals_strtoi (help, &calcopt->start_idx, 10))
	    calcopt->start_idx = 0;
	  else
	    {
	      /* Remember that the user "counts" monomers starting at 1,
		 and we need to count starting at 0, because we use indexes
		 and not positions. So decrement by one the value entered
		 by the user.
	      */
	      calcopt->start_idx--;
	    }
	}
      
      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);
      
      if (strlen (help) <= 0)
	{
	  calcopt->end_idx = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == 
	      libpolyxmass_globals_strtoi (help, &calcopt->end_idx, 10))
	    calcopt->end_idx = 0;
	  else
	    {
	      /* Remember that for the end_idx, the calculations need
		 to have it incremented by one with respect to the real
		 last monomer index. So, contrary to start_idx, we do not
		 change its value.
		 
		 calcopt->end_idx;
	      */
	    }
	}

      /* Now make some final checks:
       */

      if (calcopt->start_idx > len)
	calcopt->start_idx = len;

      if (calcopt->start_idx < 0)
	calcopt->start_idx = 0;
      
      if (calcopt->end_idx > len)
	calcopt->end_idx = len;

      if (calcopt->end_idx < 0)
	calcopt->end_idx = 0;
    }


  /* If the user wants not that the calculations be incremental, 
     we first have to empty the array of atomcounts.
  */
  widget = 
    g_object_get_data (G_OBJECT (window), "incremental_checkbutton");
  g_assert (widget != NULL);
  
  if (FALSE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    pxmchem_atomcount_GPA_empty (acGPA);
  
  /* We finally have all needed to perform the calculation:
   */
  res = pxmchem_elemcompcalc_polymer (editctxt->polymer,
				      editctxt->polchemdefctxt->polchemdef,
				      calcopt,
				      editctxt->ionizerule, 
				      acGPA,
				      &formula);
  if (res != PXM_COMPCALC_SUCCESS)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					     _("Calculation failed"),
					     POLYXMASS_NORM_MSG_TIMEOUT);
    }
  else
    {
      if (formula == NULL)
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
						 _("Calculation failed"),
						 POLYXMASS_NORM_MSG_TIMEOUT);
      else
	{
	  widget = g_object_get_data (G_OBJECT (window), 
				      "elemental_composition_entry");
	  
	  gtk_entry_set_text (GTK_ENTRY (widget), formula);
	  
	  g_free (formula);

	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) window,
	     _("Calculation succeeded"),
	     POLYXMASS_NORM_MSG_TIMEOUT);
	}
    }
  
  /* Since we might have to write a report about this calculation, we
     have to be able to know with which calculation/ionization options
     the calculation was performed. Thus, we store this information as
     full data to the window, for later enquiry. See
     polyxedit_elemcompos_wnd_make_report_text_format() for details !

     Note that the g_object_set_data_full calls below will elicit
     freeing of potentially pre-existing instances using their
     respective freeing functions.
  */
  g_object_set_data_full (G_OBJECT (window), "calcopt", calcopt,
			  (GDestroyNotify) pxmchem_calcopt_free);
  
  g_object_set_data_full (G_OBJECT (window), "ionizerule", ionizerule,
			  (GDestroyNotify) pxmchem_ionizerule_free);
    
  return;
}


void
polyxedit_elemcompos_wnd_update_sequence_region_button (GtkWidget *button,
							gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;
  
  PxmSeqedWidget *seqed_widget = NULL;
  
  
  PxmEditCtxt *editctxt = NULL;

  gchar *help = NULL;

  gint start = 0;
  gint end = 0;
  
  g_assert (window != NULL);

  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");

  seqed_widget = g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd),
				    "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  
  /* Prepare the variables from the sequence editor window, so that
     we can put them immediately in the start/end positions's entry
     windows.
  */
  if (FALSE != polyxmass_seqed_widget_get_selection_indices (GTK_WIDGET (seqed_widget),
							     &start, &end))
    {
      /* Remember that due to the handling of the polymer sequence
	 displayed in the sequence editor, the end value that is 
	 returned is indeed corresponding to the real index of the last
	 selected monomer + 1, so care should be taken not to outbound
	 the polymer sequence array.
      */
      
      /* We show the (start + 1) value because we want to show
	 a position and not an index, while the end needs not be
	 incremented because it is by itself already incremented.
      */      
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      help = g_strdup_printf ("%d", start + 1);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      widget =  g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      help = g_strdup_printf ("%d", end);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);

      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), FALSE);
    }
  else
    {
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);
      
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    }
  
  return;
}



/* REPORTING FUNCTIONS.
 */
gchar *
polyxedit_elemcompos_wnd_make_report (PxmReportOpt* reportopt,
					  PxmWinMngmt* winmngmt)
{
  


  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  g_assert (winmngmt->seqed_widget != NULL);
  g_assert (winmngmt->wnd != NULL);

  /* Great, we are able to start doing the report. We need to know what
     kind of report we want : text or LaTeX.
  */
  if (reportopt->export_format == EXPORT_FORMAT_TEXT)
    return polyxedit_elemcompos_wnd_make_report_text_format (reportopt,
								 winmngmt);
  else if (reportopt->export_format == EXPORT_FORMAT_LATEX)
    return polyxedit_elemcompos_wnd_make_report_latex_format (reportopt,
								  winmngmt);
  else
    g_assert_not_reached ();
    
  return NULL;
}



gchar *
polyxedit_elemcompos_wnd_make_report_text_format (PxmReportOpt* reportopt,
						  PxmWinMngmt* winmngmt)
{
  GtkWidget *window = NULL;
  GtkWidget *entry = NULL;
  GtkWidget *seqed_widget = NULL;

  gchar *report = NULL;
  const gchar *entry_text = NULL;
  
  GString *report_gs = NULL;

  PxmPolymer *polymer = NULL;
  PxmPlminfo *plminfo = NULL;
  PxmCalcOpt *calcopt = NULL;
  PxmIonizerule *ionizerule = NULL;



  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  seqed_widget = winmngmt->seqed_widget;
  g_assert (seqed_widget != NULL);
  
  window = winmngmt->wnd ;
  g_assert (window != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  plminfo = polymer->plminfo;
  g_assert (plminfo != NULL);

  
  report_gs = g_string_new ("");
  
  report_gs = pxmchem_polymer_format_txt_string_polseqinfo (plminfo,
							    reportopt,
							    report_gs);
  
  /* Get the pointers to the structures we know have determined how
     the calculation was performed. However, these pointers are
     non-NULL only if at least one calculation was performed, since
     they are set as full data when the Execute button is pushed.

     Thus, if these are NULL, we return with a message...
  */
  calcopt = g_object_get_data (G_OBJECT (window), "calcopt");
  ionizerule = g_object_get_data (G_OBJECT (window), "ionizerule");
  
  if  (calcopt == NULL || ionizerule == NULL)
    {
      g_warning (_("%s@%d: no calculation took place, nothing to report\n"),
		 __FILE__, __LINE__);
      
      report_gs = 
	g_string_append (report_gs, _("No calculation took place, "
			 "nothing to report\n"));
      
      report = report_gs->str;
      g_string_free (report_gs, FALSE);
      
      return report;
    }
  

  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_CALCOPTIONS) ==
      PXM_PLM_REPORTOPT_CALCOPTIONS)
    {
      report_gs = 
	g_string_append (report_gs, _("Calculations Options:\n"
			 "---------------------\n"));

      if (calcopt->capping & PXM_CAPPING_RIGHT && 
	  calcopt->capping & PXM_CAPPING_LEFT)
	report_gs = 
	  g_string_append (report_gs, _("Capping: BOTH\n"));
      
      else if (calcopt->capping & PXM_CAPPING_RIGHT)
	report_gs = 
	  g_string_append (report_gs, _("Capping: RIGHT\n"));
      
      else if (calcopt->capping & PXM_CAPPING_LEFT)
	report_gs = 
	  g_string_append (report_gs, _("Capping: LEFT\n"));
      
      else
	report_gs = g_string_append (report_gs, _("Capping: NONE\n"));
	
      if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Monomer Chemical Entities: MODIFS\n"));
      else
	report_gs = 
	  g_string_append (report_gs, _("Monomer Chemical Entities: NONE\n"));
	
	
      if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF
	  && calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modifs: BOTH\n"));

      else if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modif: LEFT\n"));

      else if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modif: RIGHT\n"));
	
      else
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modif: NONE\n\n"));

      /* Describe the ionization rule.
       */
      g_string_append_printf (report_gs, 
			      _("Ionization: "
			      "%s, with charge %d and level %d\n\n"),
			      ionizerule->actform,
			      ionizerule->charge,
			      ionizerule->level);
    }
  
  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES) ==
      PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES ||
      (reportopt->polymer_opt & PXM_PLM_REPORTOPT_SELECTION_MASSES) ==
      PXM_PLM_REPORTOPT_SELECTION_MASSES)
      report_gs = 
	g_string_append (report_gs, _("Masses:\n"
			 "-------\n"));

  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES) ==
      PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES)
    {
      g_string_append_printf (report_gs,
			      _("Whole sequence: "
			      "mono mass = %.5lf -- avg mass = %.5lf\n"),
			      polymer->masspair_seq->mono,
			      polymer->masspair_seq->avg);
    }
  
  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_SELECTION_MASSES) ==
      PXM_PLM_REPORTOPT_SELECTION_MASSES)
    {
      g_string_append_printf (report_gs, 
			      _("Selected sequence: "
			      "mono mass = %.5lf -- avg mass = %.5lf\n\n"),
			      polymer->masspair_sel->mono,
			      polymer->masspair_sel->avg);
    }
  
  /* At this point, we can ask that the actual report be performed.
   */

  /* Tell for which sequence, the composition was performed.
   */
  g_string_append_printf (report_gs, 
			  _("Elemental Composition:\n"
			  "----------------------\n\n"
			  "Selected sequence: [%d-%d]\n\n"),
			  calcopt->start_idx + 1, 
			  calcopt->end_idx);
  
  entry = g_object_get_data (G_OBJECT (window), 
			     "elemental_composition_entry");
  
  /* Get the text that corresponds to the formula representing the
     elemental composition.
  */
  entry_text = gtk_entry_get_text (GTK_ENTRY (entry));
  g_assert (entry_text != NULL);
  
  g_string_append_printf (report_gs, 
			  _("Elemental Composition: %s\n\n"),
			  entry_text);
  
  report = report_gs->str;
  g_string_free (report_gs, FALSE);
  
  return report;
}

  
gchar *
polyxedit_elemcompos_wnd_make_report_latex_format (PxmReportOpt* reportopt,
						  PxmWinMngmt* winmngmt)
{

  return NULL;
  
}

  



/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_elemcompos_wnd_really_close (GtkWidget *window)
{

  g_assert (window != NULL);
  
  /* This function is called as a GDestroyNotify callback function when
     the sequence editor window gets closed.
  */
  
  /* As a first step we want to un_register this window.
   */
  polyxmass_winmngmt_un_register_window (window, "POLYXEDIT");
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  gtk_widget_destroy (window);
}



gint 
polyxedit_elemcompos_wnd_delete_event (GtkWidget *window, 
					GdkEventAny *event, 
				       gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  gchar *help = NULL;
  
  g_assert (window != NULL);
  g_assert (editctxt != NULL);
  

  /* As a first step we want to un_register this window.
   */
  polyxmass_winmngmt_un_register_window (window, "POLYXEDIT");
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /* This window pointer was set as a full datum to the sequence editor
     window, which means that we have to remove that pointer, without
     triggering the callback function call.
  */
  help = g_strdup_printf ("elemcompos_wnd-%p", window);
  
  window = 
    g_object_steal_data (G_OBJECT (editctxt->sequence_editor_wnd),
			 help);
  
  g_free (help);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}


gint 
polyxedit_elemcompos_wnd_destroy_event (GtkWidget *window, 
					GdkEventAny *event, 
					gpointer data)
{
  printf ("polyxedit_elemcompos_wnd_destroy_event\n");

  return FALSE;
}


