/*
    ProjectTypeDescription.m

    Implementation of the ProjectTypeDescription class for
    the ProjectManager app.

    Copyright (C) 2006  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "ProjectTypeDescription.h"

#import <Foundation/NSString.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSDictionary.h>

#import "ProjectType.h"

#import "ProjectTemplateDescription.h"

/**
 * The description of a project type.
 *
 * This object is used by a ProjectCreator to represent an available
 * project type. It basically serves as a container for information
 * about a certain project type.
 */
@implementation ProjectTypeDescription

- (void) dealloc
{
  TEST_RELEASE(templates);

  [super dealloc];
}

/**
 * Designated initializers.
 *
 * This method initializes a ProjectTypeDescription object.
 *
 * @param aClass The class which implements the specific project type.
 */
- initWithProjectType: (Class) aClass
{
  if ((self = [self init]) != nil)
    {
      projType = aClass;
    }

  return self;
}

/**
 * Returns the name of the receiver's project type.
 */
- (NSString *) name
{
  return [projType humanReadableProjectTypeName];
}

/**
 * Returns a human-readable description of the receiver's project type.
 */
- (NSString *) description
{
  return [projType projectTypeDescription];
}

/**
 * Returns the icon of the receiver's project type.
 */
- (NSImage *) icon
{
  return [projType projectTypeIcon];
}

/**
 * Returns an array of template descriptions of the receiver's project type.
 *
 * @return An array of ProjectTemplateDescription objects describing
 *      the receiver's project type's project templates.
 */
- (NSArray *) templates
{
  if (templates == nil)
    {
      NSDictionary * desc = [projType projectTemplateDescriptions];
      NSMutableArray * array = [NSMutableArray arrayWithCapacity: [desc count]];
      NSEnumerator * e = [desc keyEnumerator];
      NSString * key;

      while ((key = [e nextObject]) != nil)
        {
          NSDictionary * templateDict = [desc objectForKey: key];

          [array addObject: [[[ProjectTemplateDescription alloc]
            initWithTemplateName: key
                     description: [templateDict objectForKey: @"Description"]
                            icon: [templateDict objectForKey: @"Icon"]
                          parent: self]
            autorelease]];
        }

      ASSIGNCOPY(templates, array);
    }

  return templates;
}

/**
 * Returns the class from which this type description has been generated.
 */
- (Class) projectType
{
  return projType;
}

@end
