 /***************************************************************************
                          tagger_taglib.cpp  -  description
                             -------------------
    begin                : Sun Feb 15 2004
    copyright            : (C) 2004 by Cord Horeis
    email                : kuddel@zipfelmuetzenparty.de
 **************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#if HAVE_TAG

#include "tagger_taglib.h"
#include <qstring.h>
#include <qdatetime.h>
#include <iostream>

#include "configuration.h"

#include <tag.h>
#include <mpegproperties.h>
#include <id3v1tag.h>
#include <id3v2tag.h>
#include <id3v1genres.h>
#include <id3v2framefactory.h>
#include <fileref.h>

Tagger_TagLib::Tagger_TagLib() {};
Tagger_TagLib::~Tagger_TagLib() {};

///////////////////////////////////////////////////////////////////////
//
// Method: getInfo()
//
///////////////////////////////////////////////////////////////////////
bool Tagger_TagLib::getInfo(TRACK *track, bool lazyRead, const QString& fileExt) {

  QString fileName = QString(track->path + "/" + track->filename);
    
  TagLib::FileRef file( QFile::encodeName( fileName ), true, TagLib::AudioProperties::Accurate );
  if (! file.audioProperties() )
    return false;
  
  audioProperties* myAudioProp = new audioProperties;
  myAudioProp->mode = 0;
  myAudioProp->bitrate = 0;
  myAudioProp->samplerate = 0;
  myAudioProp->length = 0;
  myAudioProp->size = 0;
    
  myAudioProp->bitrate = file.audioProperties()->bitrate();
  myAudioProp->samplerate = file.audioProperties()->sampleRate();
  myAudioProp->length = file.audioProperties()->length();
  
  if ( fileExt == "MP3" ) {
    myAudioProp->mimetype =  MIMETYPE_MP3;
    TagLib::MPEG::File *mpegfile = dynamic_cast<TagLib::MPEG::File*>( file.file() );
    if ( mpegfile ) {    
      myAudioProp->version = mpegfile->audioProperties()->version();
      myAudioProp->layer = mpegfile->audioProperties()->layer();
      myAudioProp->mode = mpegfile->audioProperties()->channelMode()  ;
    } else {
      if ( verbose ) qWarning( " Not sure file: %s is MPEG. Disregarding version.", fileName.local8Bit().data() );
    }
  } else {
    myAudioProp->version = 0;
    myAudioProp->layer = 0;
    myAudioProp->mode = file.audioProperties()->channels()==2 ? 0 : 3;
  }
  
  if ( fileExt == "OGG" )
    myAudioProp->mimetype =  MIMETYPE_OGG;
  
  if ( fileExt == "FLAC" )
    myAudioProp->mimetype =  MIMETYPE_FLAC;
  
  //Read songinfo and set it on track.
  if (! getTagOnly( track, fileExt ) )
  {
    delete myAudioProp;
    return false;
  }
    
  setAudioProperties( track, myAudioProp );
  delete myAudioProp;
  return true;
}

///////////////////////////////////////////////////////////////////////
//
// Method: getTagOnly()
//
///////////////////////////////////////////////////////////////////////
bool Tagger_TagLib::getTagOnly( TRACK *track, const QString& fileExt )
{

  QString fileName = QString(track->path + "/" + track->filename);
  songProperties* mySongProp = new songProperties;
  mySongProp->tracknumber = 0;
  mySongProp->year = 0;
  
  bool fileReadFinished = false;
  
  if ( fileExt == "MP3" )
  {
    TagLib::MPEG::File::File mp3File( QFile::encodeName( fileName ), true, TagLib::AudioProperties::Accurate);
    
    if (! mp3File.isValid() )
    {
      delete mySongProp;
      return false;
    }
    
    config->lock();
    bool readID3v2 = config->readID3v2();
    config->unlock();

    //Read id3v1 only if we don't want to read id3v2
    if ( readID3v2 )
    {
      TagLib::ID3v2::Tag::Tag *tag = mp3File.ID3v2Tag();
      if( tag && !tag->isEmpty() )
      {
        mySongProp->title = TStringToQString( tag->title() );
        mySongProp->artist = TStringToQString( tag->artist() );
        mySongProp->album = TStringToQString( tag->album() );
        mySongProp->year = tag->year();
        mySongProp->comment = TStringToQString( tag->comment() );
        mySongProp->genre = TStringToQString( tag->genre() );
        mySongProp->tracknumber = tag->track();
        fileReadFinished = true;
      }
    }
    
    if(! fileReadFinished )
    {
      TagLib::ID3v1::Tag::Tag *tag = mp3File.ID3v1Tag();
      if( tag && !( tag->isEmpty() ) )
      {
        mySongProp->title = TStringToQString( tag->title() );
        mySongProp->artist = TStringToQString( tag->artist() );
        mySongProp->album = TStringToQString( tag->album() );
        mySongProp->year = tag->year();
        mySongProp->comment = TStringToQString( tag->comment() );
        mySongProp->genre = TStringToQString( tag->genre() );
        mySongProp->tracknumber = tag->track();
        fileReadFinished = true;
      }
    }
  }
  if (! fileReadFinished ) //ogg, flac or id3 read failed.
  {
    TagLib::FileRef file( QFile::encodeName( fileName ), true, TagLib::AudioProperties::Accurate );
    if ( file.tag() )
    {
      mySongProp->title = TStringToQString( file.tag()->title() );
      mySongProp->artist = TStringToQString( file.tag()->artist() );
      mySongProp->album = TStringToQString( file.tag()->album() );
      mySongProp->year = file.tag()->year();
      mySongProp->comment = TStringToQString( file.tag()->comment() );
      mySongProp->genre = TStringToQString( file.tag()->genre() );
      mySongProp->tracknumber = file.tag()->track();
      fileReadFinished = true;
    }
  }
  
  if (! fileReadFinished )
  {
    delete mySongProp;
    return false;
  }

  if (verbose )  qWarning( " reading mySongProp->title:%s", mySongProp->title.local8Bit().data() );
  
  setTrack( track, mySongProp );
  delete mySongProp;
  return true;
}
///////////////////////////////////////////////////////////////////////
//
// Method: writeInfo()
//
///////////////////////////////////////////////////////////////////////
bool Tagger_TagLib::writeInfo( TRACK *track, QString ext )
{
  bool write_OK = false;
  QString fileName = QString(track->path + "/" + track->filename);
  
  if ( ext == "MP3" ) // is this a mp3 file?
    {
      TagLib::MPEG::File file( QFile::encodeName( fileName ), true, TagLib::AudioProperties::Accurate );
      TagLib::ID3v1::Tag *v1Tag = file.ID3v1Tag( true );
      if (! v1Tag )
	return false;
      v1Tag->setTitle( QStringToTString( track->title ) );
      v1Tag->setArtist( QStringToTString( track->artist ) );
      v1Tag->setAlbum( QStringToTString( track->album ) );
      v1Tag->setYear( track->year.toInt() );
      v1Tag->setComment( QStringToTString( track->comment ) );
      v1Tag->setGenre( TagLib::ID3v1::genre( track->genre ) );
      v1Tag->setTrack( track->tracknumber );
    
      config->lock();
      bool writeID3v2 = config->writeID3v2();
      config->unlock();
      if ( writeID3v2 )
	{
	  TagLib::ID3v2::Tag *v2Tag = file.ID3v2Tag( true );

	  v2Tag->setTitle( QStringToTString( track->title ) );
	  v2Tag->setArtist( QStringToTString( track->artist ) );
	  v2Tag->setAlbum( QStringToTString( track->album ) );
	  v2Tag->setYear( track->year.toInt() );
	  v2Tag->setComment( QStringToTString( track->comment ) );
	  v2Tag->setGenre( TagLib::ID3v1::genre( track->genre ) );
	  v2Tag->setTrack( track->tracknumber );
	  write_OK = file.save( TagLib::MPEG::File::AllTags );
	}
      else
	file.save();
    }
  else // must be ogg or flac
    {
      TagLib::FileRef file( QFile::encodeName( fileName ), true, TagLib::AudioProperties::Accurate );
      TagLib::Tag *tag = file.tag();
      if (! tag )
	return false;
    
      tag->setTitle( QStringToTString( track->title ) );
      tag->setArtist( QStringToTString( track->artist ) );
      tag->setAlbum( QStringToTString( track->album ) );
      tag->setYear( track->year.toInt() );
      tag->setComment( QStringToTString( track->comment ) );
      tag->setGenre( TagLib::ID3v1::genre( track->genre ) );
      tag->setTrack( track->tracknumber );
      write_OK = file.save();
    }
  
  if (verbose)  qWarning( " writing track->title:%s", track->title.local8Bit().data());

  return write_OK;
}
///////////////////////////////////////////////////////////////////////
//
// Method: setTrack()
//
///////////////////////////////////////////////////////////////////////
bool Tagger_TagLib::setTrack(TRACK* track, songProperties* mySongProp )
{
  track->title   = mySongProp->title.stripWhiteSpace();
  track->artist  = mySongProp->artist.stripWhiteSpace();
  track->album   = mySongProp->album.stripWhiteSpace();
  if ( ( mySongProp->year > 99 ) && ( mySongProp->year < 8000 ) )
    track->year.setNum( mySongProp->year );
  else
    track->year = "";
  track->tracknumber = mySongProp->tracknumber;
  track->comment = mySongProp->comment.stripWhiteSpace();
  if (! mySongProp->genre.isNull() ) 
    track->genre = TagLib::ID3v1::genreIndex( QStringToTString( mySongProp->genre ) );
  else
    track->genre = 255;
    
  if (verbose == 6 ) {
    std::cout << "read Tags: (" << track->filename.local8Bit().data() << ")" << std::endl;
    std::cout << "  Title   : '" << track->title.local8Bit().data() << "'" << std::endl;
    std::cout << "  Artist  : '" << track->artist.local8Bit().data() << "'" << std::endl;
    std::cout << "  Album   : '" << track->album.local8Bit().data() << "'" << std::endl;
    std::cout << "  Year    : " << track->year.local8Bit().data() << std::endl;
    std::cout << "  Track   : " << track->tracknumber << std::endl;
    std::cout << "  Comment : '" << track->comment.local8Bit().data() << "'" << std::endl;
    std::cout << "  Genre   : " << track->genre << std::endl;
  }

  return true;
}

///////////////////////////////////////////////////////////////////////
//
// Method: setAudioProperties()
//
///////////////////////////////////////////////////////////////////////
bool Tagger_TagLib::setAudioProperties(TRACK* track, audioProperties* myAudioProperties )
{
    track->version = myAudioProperties->version;
    track->mode = myAudioProperties->mode;
    track->samplerate = myAudioProperties->samplerate;
    track->bitrate = myAudioProperties->bitrate;
    track->length = myAudioProperties->length;
    track->mimetype = myAudioProperties->mimetype;
    
  if (verbose == 6 ) {
    std::cout << "---------Streaminfo----------" << std::endl;
    std::cout << "  Mode        : " << track->mode << std::endl;
    std::cout << "  Version     : " << track->version << std::endl << std::endl;
    std::cout << "  Samplerate  : " << track->samplerate << std::endl;
    std::cout << "  Bitrate     : " << track->bitrate << std::endl;
    std::cout << "  Length      : " << track->length << std::endl << std::endl;
  }

  return true;
}

#endif /* HAVE_TAG */
