/**
 ** Proll (PROM replacement)
 ** iommu.c: Functions for DVMA management.
 ** Copyright 1999 Pete Zaitcev
 ** This code is licensed under GNU General Public License.
 **/
#include <stdarg.h>
// #include <asm/contregs.h>
#include <asi.h>

#include "pgtsrmmu.h"
#include "iommu.h"		/* Typical SBus IOMMU for sun4m */
#include "phys_jj.h"
#include "vconsole.h"

#include <general.h>		/* __P() */
#include <romlib.h>		/* we are a provider for part of this. */
#include <system.h>		/* our own prototypes */

static __inline__ void iommu_invalidate(struct iommu_regs *regs) {
	regs->tlbflush = 0;
}

#define IOPERM        (IOPTE_CACHE | IOPTE_WRITE | IOPTE_VALID)
#define MKIOPTE(phys) (((((phys)>>4) & IOPTE_PAGE) | IOPERM) & ~IOPTE_WAZ)

/*
 * XXX This is a problematic interface. We alloc _memory_ which is uncached.
 * So if we ever reuse allocations somebody is going to get uncached pages.
 * Returned address is always aligned by page.
 * BTW, we were not going to give away anonymous storage, were we not?
 */
void *iommu_alloc(struct iommu *t, int size, unsigned int *pphys)
{
	void *va;
	unsigned int pa, ba;
	unsigned int npages;
	unsigned int mva, mpa;
	int i;
	unsigned int *iopte;

	npages = (size + (PAGE_SIZE-1)) / PAGE_SIZE;
	va = mem_alloc(&cmem, npages*PAGE_SIZE, PAGE_SIZE);
	if (va == 0) return 0;
	ba = (unsigned int)mem_alloc(&t->bmap, npages*PAGE_SIZE, PAGE_SIZE);
	if (ba == 0) return 0;

	pa = ((unsigned int)va) - PROLBASE + pmem.pbas;

	/*
	 * Change page attributes in MMU to uncached.
	 */
	mva = (unsigned int) va;
	mpa = (unsigned int) pa;
	for (i = 0; i < npages; i++) {
		map_page(pmem.pl1, mva, mpa, 1, pmem.pbas);
		mva += PAGE_SIZE;
		mpa += PAGE_SIZE;
	}

	/*
	 * Map into IOMMU page table.
	 */
	mpa = (unsigned int) pa;
	iopte = &t->page_table[(ba - t->plow)/PAGE_SIZE];
	for (i = 0; i < npages; i++) {
		*iopte++ = MKIOPTE(mpa);
		mpa += PAGE_SIZE;
	}

	*pphys = ba;
	return va;
}

/*
 * Initialize IOMMU
 * This looks like initialization of CPU MMU but
 * the routine is higher in food chain.
 */
void iommu_init(struct iommu *t, unsigned int highbase)
{
	unsigned int *ptab;
	int ptsize;
	struct iommu_regs *regs;
	unsigned int impl, vers;
	unsigned int tmp;

	if ((regs = map_io(PHYS_JJ_IOMMU, sizeof(struct iommu_regs))) == 0) {
		printk("Cannot map IOMMU\n");
		for (;;) { }
	}
	t->regs = regs;

	impl = (regs->control & IOMMU_CTRL_IMPL) >> 28;
	vers = (regs->control & IOMMU_CTRL_VERS) >> 24;

	tmp = regs->control;
	tmp &= ~(IOMMU_CTRL_RNGE);

	tmp |= (IOMMU_RNGE_32MB | IOMMU_CTRL_ENAB);
	t->plow = 0xfe000000;		/* End - 32 MB */
	t->vasize = 0x2000000;		/* 32 MB */

	regs->control = tmp;
	iommu_invalidate(regs);

        /* Allocate IOMMU page table */
	/* Theremendous alignment causes great waste... */
	ptsize = (t->vasize/PAGE_SIZE) *  sizeof(int);
	if ((ptab = mem_zalloc(&cmem, ptsize, ptsize)) == 0) {
		printk("Cannot allocate IOMMU table [0x%x]\n", ptsize);
		for (;;) { }
	}
	t->page_table = ptab;

	/* flush_cache_all(); */
#if 0
iommu_map_dvma_pages_for_iommu(iommu);
static inline void iommu_map_dvma_pages_for_iommu(struct iommu_struct *iommu)
{
        unsigned long kern_end = (unsigned long) high_memory;
        unsigned long first = page_offset;
        unsigned long last = kern_end;
        iopte_t *iopte = iommu->page_table;

        iopte += ((first - iommu->start) >> PAGE_SHIFT);
        while(first <= last) {
                *iopte++ = __iopte(MKIOPTE(mmu_v2p(first)));
                first += PAGE_SIZE;
        }

}
#endif

	/** flush_tlb_all(); **/
	regs->base = ((unsigned int)ptab - PROLBASE + highbase) >> 4;
	iommu_invalidate(regs);

	printk("IOMMU: impl %d vers %d page table at 0x%x of size %d bytes\n",
	    impl, vers, t->page_table, ptsize);

	mem_init(&t->bmap, (char*)t->plow, (char *)0xfffff000);
}
